/*
// Program:  Format
// Version:  0.91g
// (0.90b/c - fixed compiler warnings, pointers - Eric Auer 2003)
// 0.91b/c/d/e - fixed more bugs - Eric.
// Written By:  Brian E. Reifsnyder
// Copyright:  2002 under the terms of the GNU GPL, Version 2
// Module Name:  createfs.c
// Module Description:  Functions to create the file systems.
*/

#define CREATEFS

#include <stdlib.h>
#include <string.h>	/* strlen, memcpy -ea */
#include <dos.h>	/* FP_OFF, FP_SEG */
#include <stdio.h>	/* printf */

#include "hdisk.h"	/* Force_Drive_Recheck */

void E_memcpy(unsigned char far *, char *, size_t);
void E_memcpy(unsigned char far * dst, char * src, size_t cnt) {

  /*
  printf("E_memcpy(%4.4x:%4.4x, %4.4x:%4.4x, %d)\n",
    FP_SEG(dst), FP_OFF(dst), FP_SEG(src), FP_OFF(src), cnt);
  */

  while (cnt-- > 0) {
      dst[0] = src[0];
      dst++;
      src++;
  }
}

#include "bootcode.h"
#include "format.h"
#include "floppy.h"
#include "driveio.h"
#include "btstrct.h"
#include "createfs.h"

void Clear_Sector_Buffer(void); /* -ea */

#define CFSp(a) (void far *)(a)

void Clear_Reserved_Sectors(void);
void Create_Boot_Sect_Start(void);
void Create_Boot_Sect_Ext(void);
void Create_FS_Info_Sector(void);
void Get_FS_Info(void);
void Write_Boot_Sectors(void);
void Write_FAT_Tables(void);
void Write_Root_Directory(void);

unsigned long Create_Serial_Number(void);

FSI file_sys_info;

void Create_File_System()
{
  Get_FS_Info();

  /* Flush DOS disk buffers. */
  regs.h.ah = 0x0d;
  intdos(&regs,&regs);

  if(debug_prog==TRUE) printf("\n[DEBUG]  Creating File System\n");

  /* Clear reserved sectors. */
  Clear_Reserved_Sectors();

  /* Create boot sector structures. */
  Create_Boot_Sect_Start();	/* the initial jump and OEM id */
  Create_Boot_Sect_Ext();	/* FAT id string, LABEL, ... */

  Clear_Sector_Buffer();

  /* *** Add Start of boot sector */
  E_memcpy(&sector_buffer[0], (char *)&bs_start, 11);

  /* *** Add Standard BPB */
  E_memcpy(&sector_buffer[11],(void *)&parameter_block.bpb, 35-11+1);

  /* *** Add FAT32 BPB Extension, if applicable. */
  if (param.fat_type==FAT32)
    {
    /* Root dir always starts in first data cluster (number 2) */
    /* dpb->root_dir_start_cluster = 2; */

    E_memcpy(&sector_buffer[36], (void *)&parameter_block.xbpb, 63-36+1);
    }
  /* *** Add Extended boot sector fields. */
  if(param.fat_type!=FAT32)
    E_memcpy(&sector_buffer[36], (void *)&bs_ext, 61-36+1);
  else
    E_memcpy(&sector_buffer[64], (void *)&bs_ext, 89-64+1);

  /* Add Boot Code */
  if(param.fat_type!=FAT32) {
    E_memcpy(&sector_buffer[62], (void *)&boot_code[0], sizeof(boot_code));
    strcat((char *)&sector_buffer[62+sizeof(boot_code)], &boot_message[0]);
  } else {
    E_memcpy(&sector_buffer[90], (void *)&boot_code[0], sizeof(boot_code));
    strcat((char *)&sector_buffer[90+sizeof(boot_code)], &boot_message[0]);
  }

  /* Add Flag */
  sector_buffer[510]=0x55;
  sector_buffer[511]=0xaa;

  Write_Boot_Sectors();

  Force_Drive_Recheck();

  Write_FAT_Tables();		/* if FAT32, also make root dir chain */

  Write_Root_Directory();	/* directory and label */

  Force_Drive_Recheck();
}

void Create_Boot_Sect_Start()
{
  /* Add Jump Instruction */
  if(param.fat_type==FAT12 || param.fat_type==FAT16)
    {
    /* Add instruction for FAT12 & FAT16 */
    bs_start.jmp[0]=0xeb;
    bs_start.jmp[1]=0x3c;
    bs_start.jmp[2]=0x90;
    }
  else
    {
    /* Add instruction for FAT32 */
    bs_start.jmp[0]=0xeb;
    bs_start.jmp[1]=0x58;
    bs_start.jmp[2]=0x90;
    }

  /* Add OEM ID */
  memcpy(&bs_start.oem_id,"FDOS_1.0", 8);
}

void Create_Boot_Sect_Ext()
{
  unsigned long serial_number;

  /* Add Physical Drive Number. */
  if(param.drive_type==FLOPPY) bs_ext.drive_number=0x00;
  if(param.drive_type==HARD)   bs_ext.drive_number=0x80;

  /* Zero reserved field. */
  bs_ext.reserved=0x00;

  /* Add Signature Byte */
  bs_ext.signature_byte=0x29;

  /* Add Serial Number. */
  serial_number=Create_Serial_Number();
  memcpy(&bs_ext.serial_number[0], &serial_number, 4);

  memcpy(&drive_statistics.serial_number_low,&bs_ext.serial_number[0],2);
  memcpy(&drive_statistics.serial_number_high,&bs_ext.serial_number[2],2);

  /* Add Volume Label */
  if(param.v==TRUE)
    {
    memset( &bs_ext.volume_label[0], ' ', 11);
    memcpy( &bs_ext.volume_label[0], param.volume_label, min(11, strlen(param.volume_label)));
    }
  else
    {
    memcpy( &bs_ext.volume_label[0], "NO NAME    ", 11);
    }

  /* Add System ID */
  memcpy(&bs_ext.file_system_type[0],"FAT1",4);

  if(param.fat_type==FAT12) bs_ext.file_system_type[4]='2';
  if(param.fat_type==FAT16) bs_ext.file_system_type[4]='6';
  if(param.fat_type==FAT32) memcpy(&bs_ext.file_system_type[3],"32",2);

  memcpy(&bs_ext.file_system_type[5],"   ",3);
}

void Create_FS_Info_Sector()
{
  static struct
    {
    unsigned long    lead_signature         ;
    char             reserved_1        [480];
    unsigned long    struc_signature        ;
    unsigned long    free_count             ;
    unsigned long    next_free              ;
    char             reserved_2         [12];
    unsigned long    trailing_signature     ;
    } fs_info_sector;

  unsigned i;
  char far * fsIS_p = (char far *)(&fs_info_sector);

  for (i=0; i<sizeof(fs_info_sector); i++) {
    fsIS_p[i] = 0;
  }    
  /* memset(CFSp(fsIS_p), 0, sizeof(fs_info_sector)); */ /* & changed -ea */

  fs_info_sector.lead_signature     = 0x41615252L; /* RRaA */
  fs_info_sector.struc_signature    = 0x61417272L; /* rrAa */
  fs_info_sector.free_count         = file_sys_info.total_clusters - 1;
                                     /* 1 cluster used by root directory */
  fs_info_sector.next_free          = 0x2;
  fs_info_sector.trailing_signature =0xaa550000L;

  E_memcpy(&sector_buffer[0], (char *)(&fs_info_sector), 512);


}

/* divide 16 or 32 bit total size by cluster size and sub extra areas */
/* ... then divide by cluster size to get number of clusters on drive */
static unsigned long cluster_count(void)
{
  unsigned long totalsect;
  
  totalsect = parameter_block.bpb.total_sectors;
  if (totalsect == 0)
    totalsect = ((unsigned long)(parameter_block.bpb.large_sector_count_high) << 16) |
      parameter_block.bpb.large_sector_count_low;

  /* If FAT32, root dir does not have to start right after FAT, */
  /* HOWEVER, WE always put it there, so this is acceptable:    */
  totalsect -= file_sys_info.start_root_dir_sect;

  if (param.fat_type != FAT32) /* in FAT32, root dir uses normal clusters */
    totalsect -= file_sys_info.number_root_dir_sect;

  if (parameter_block.bpb.sectors_per_cluster==0) {
    printf("YOU FOUND A BUG! BPB has 0 sectors / cluster. Please report.\n");
    exit(23);
  } /* new 0.91b sanity check */

  return (totalsect / parameter_block.bpb.sectors_per_cluster);
}

/* calculate: start_fat_sector, number_fat_sectors,          */
/* start_root_dir_sect, number_root_dir_sect, total_clusters */
void Get_FS_Info()
{
  /* first is easy: FAT starts after boot sector(s). 1 for most FAT12  */
  /* and FAT16, and usually 32 or 64 for FAT32, of which only 2 copies */
  /* of 3 sectors each are used: loader1, statistics and loader2...    */
  file_sys_info.start_fat_sector = parameter_block.bpb.reserved_sectors;

  /* we sanitize this later! */
  file_sys_info.number_fat_sectors = parameter_block.bpb.sectors_per_fat;
  if (file_sys_info.number_fat_sectors == 0) /* FAT32 uses 32 bit instead */
    file_sys_info.number_fat_sectors =
      ((unsigned long)(parameter_block.xbpb.fat_size_high) << 16) |
      parameter_block.xbpb.fat_size_low;
  /* (by definition - actually, FAT32 FATs can be only 16 MB each anyway) */

  /* FAT32 technically can have the root dir anywhere but WE need it HERE */
  /* Among others, cluster_count() needs it there, to be more exact.      */
  file_sys_info.start_root_dir_sect =
    (file_sys_info.number_fat_sectors * parameter_block.bpb.number_of_fats) +
    parameter_block.bpb.reserved_sectors;

  if (param.fat_type != FAT32) /* FAT12 / FAT16 root dir size sanitizing */
    {
      if ((parameter_block.bpb.root_directory_entries & 31) != 0)
        {
        printf("Root directory entries must be a multiple of 32. Fixing.");
        parameter_block.bpb.root_directory_entries |= 31;	/* round up to... */
        parameter_block.bpb.root_directory_entries++;	/* ...next multiple */
        }
    }
  else /* FAT32 root dir size sanitizing */
    {
      unsigned entries_per_cluster = parameter_block.bpb.bytes_per_sector / 32;
      entries_per_cluster *= parameter_block.bpb.sectors_per_cluster;

      if (parameter_block.bpb.root_directory_entries < 512)
        parameter_block.bpb.root_directory_entries = 512; /* not too small! */

      if ((parameter_block.bpb.root_directory_entries & (entries_per_cluster-1)) != 0)
        {
        printf("Root directory must be whole clusters big. Fixing.");
        parameter_block.bpb.root_directory_entries |= (entries_per_cluster-1);
        parameter_block.bpb.root_directory_entries++;	/* round up! */
        }
    }

#if 0
  /* Do not forget later to ALLOCATE CLUSTERS FAT32 root dir! */
  if (param.fat_type == FAT32)
    file_sys_info.number_root_dir_sect = 1024u*32/512;
	/* Pretend there are 1024 entries, to guarantee that */
	/* enough space is cleared. [Whatever this means]    */  
  else
#endif
    file_sys_info.number_root_dir_sect =
      (parameter_block.bpb.root_directory_entries*32) /
      parameter_block.bpb.bytes_per_sector;

  /* divide total 16 or 32 bit sector count by cluster size, */
  /* but subtract FAT and BOOT sizes first. FAT32 only works */
  /* in the current version if root dir starts in 1. cluster */
  file_sys_info.total_clusters = cluster_count();

  if ( ( FAT32 && (file_sys_info.number_root_dir_sect <
         parameter_block.bpb.sectors_per_cluster)
       ) || (parameter_block.bpb.bytes_per_sector != 512)
         || (parameter_block.bpb.number_of_fats<1)
         || (parameter_block.bpb.number_of_fats>2)
     )
   {
   printf("Can only format 512 bytes per sector, 1 or 2 FATs.\n");
   if (param.fat_type == FAT32)
     printf("In FAT32, root dir must be at least 1 cluster big.\n");
   }

}

void Clear_Reserved_Sectors()
{
  int index = 0;

  Clear_Sector_Buffer();

  do
    {
    Drive_IO(WRITE,index,1);

    index++;
    } while(index < file_sys_info.start_fat_sector);
}

unsigned long Create_Serial_Number()
{
  unsigned long serial_number;
  unsigned long serial_number_high;
  unsigned long serial_number_low;
  unsigned long scratch;

  unsigned char month;
  unsigned char day;
  unsigned int year;

  unsigned char hour;
  unsigned char minute;
  unsigned char second;
  unsigned char hundredth;

#ifdef __TURBOC__
  /* Turbo C 2.01 cannot handle multi instruction asm{ ... } blocks -ea */
  /* ... and not to forget, Turbo C has getdate and gettime anyway! -ea */
  {
    struct time ti;
    struct date da;
    
    getdate( &da );
    gettime( &ti );

    year = da.da_year;
    day = da.da_day;
    month = da.da_mon;
    
    hour = ti.ti_hour;
    minute = ti.ti_min;
    second = ti.ti_sec;
    hundredth = ti.ti_hund;
    
  }
#else
  asm{
    mov ah,0x2a
    int 0x21

    mov WORD PTR year,cx
    mov BYTE PTR month,dh
    mov BYTE PTR day,dl

    mov ah,0x2c
    int 0x21

    mov BYTE PTR hour,ch
    mov BYTE PTR minute,cl
    mov BYTE PTR second,dh
    mov BYTE PTR hundredth,dl
    }
#endif

  serial_number_low = hour;
  serial_number_low = serial_number_low << 8;
  serial_number_low += minute;
  scratch = year;
  serial_number_low += scratch;

  serial_number_high = second;
  serial_number_high = serial_number_high << 8;
  serial_number_high += hundredth;
  scratch = month;
  scratch = scratch << 8;
  scratch += day;
  serial_number_high += scratch;

  serial_number = (serial_number_high << 16) + serial_number_low;

  return serial_number;
}

void Write_Boot_Sectors()
{
  int use_backups = 0;

  /* Write boot sector to the first sector of the disk */
  if (debug_prog==TRUE) printf("\n[DEBUG]  Writing Boot Sector ->    0\n");
  Drive_IO(WRITE,0,1);

  if(param.fat_type==FAT32)
    {
    if ( (parameter_block.xbpb.backup_boot_sector<3) ||
         (parameter_block.xbpb.backup_boot_sector >
          (parameter_block.bpb.reserved_sectors-3)) ||
         (parameter_block.xbpb.backup_boot_sector==0xffff) )
      {
      if (parameter_block.xbpb.backup_boot_sector==0xffff)
        {
        printf("FAT32 Backup Boot Sector feature not used.\n");
        }
      else
        {
        printf("NO FAT32 Backup Boot Sector written - position out of range!\n");
        }
      }
    else
      {
      use_backups = parameter_block.xbpb.backup_boot_sector;
      if (debug_prog==TRUE)
        printf("\n[DEBUG]  Writing FAT32 Backup Boot Sector ->    %d\n",
          use_backups);
      Drive_IO(WRITE, use_backups, 1);
      }

    /* This sector only contains a bit of statistics if enabled. */
    if ( ((parameter_block.xbpb.info_sector_number + use_backups) >=
          parameter_block.bpb.reserved_sectors) ||
          (parameter_block.xbpb.info_sector_number == 0xffff)
       )
      {
      if (parameter_block.xbpb.info_sector_number == 0xffff)
        {
        printf("Info sector feature not active.\n");
        }
      else
        {
        printf("Info sector on invalid location? Not writing it.\n");
        }
      }
    else /* info sector active */
      {
      Create_FS_Info_Sector();
      if (debug_prog) printf("\n[DEBUG]  Writing FS Info Sector ->    1\n");
      Drive_IO(WRITE, parameter_block.xbpb.info_sector_number, 1);
      if (use_backups)
        {
        if (debug_prog) printf("\n[DEBUG]  Writing Backup FS Info Sector ->    %d\n",
          use_backups + parameter_block.xbpb.info_sector_number);
        Drive_IO(WRITE, use_backups + parameter_block.xbpb.info_sector_number, 1);
        }
      } /* Info Sector feature */

    /* ** Create 3rd boot sector (2nd boot loader stage dummy). */
    Clear_Sector_Buffer(); /* ??? */
    /* Add Flag (00 at bytes 508 and 509 belongs to flag, too) */
    sector_buffer[510]=0x55;
    sector_buffer[511]=0xaa;

    /* Write sectors for "2nd stage of boot loader". */

    if (debug_prog) printf("\n[DEBUG]  Writing Dummy Stage 2 ->    2\n");
    Drive_IO(WRITE, 2, 1);
    if (use_backups)
      {
      if (debug_prog) printf("\n[DEBUG]  Writing Backup Dummy Stage 2 ->    %d\n",
        use_backups+2);
      Drive_IO(WRITE, use_backups + 2, 1);
      }

    } /* FAT32 */
}

void Write_FAT_Tables()
{
  unsigned long sector;

  /* Configure FAT Tables */
  Clear_Sector_Buffer();
  sector = file_sys_info.start_fat_sector;

  if (debug_prog==TRUE) printf("[DEBUG]  Clearing FAT Sectors ->");
  do
    {
    if (debug_prog==TRUE) printf(" %3d,",sector); /* keep debug output shorter */

    Drive_IO(WRITE,sector,1);

    sector++;
    } while (sector < ( file_sys_info.start_fat_sector
	     + ( 2 * file_sys_info.number_fat_sectors ) ) );

  if (debug_prog==TRUE) printf(" done.\n");

  sector_buffer[0] = drive_specs[param.media_type].media_descriptor;
  sector_buffer[1] = 0xff;
  sector_buffer[2] = 0xff;

  if(param.fat_type==FAT16)
    sector_buffer[3] = 0xff;

  if (param.fat_type==FAT32)
    {
    int sbp, cnum;
    unsigned long nclust;

    sector_buffer[3] = 0x0f;

    sector_buffer[4] = 0xff;
    sector_buffer[5] = 0xff;
    sector_buffer[6] = 0xff;
    sector_buffer[7] = 0x0f;

    sector_buffer[8] = 0xff;
    sector_buffer[9] = 0xff;
    sector_buffer[10]= 0xff;
    sector_buffer[11]= 0x0f;
    
    /* WE always start ROOT DIR CLUSTER CHAIN at 1st cluster! */
    /* ADDED CLUSTER CHAIN CREATION - 0.91e */
    sbp = 12;
    nclust = parameter_block.bpb.root_directory_entries;
    nclust *= 32;
    nclust /= parameter_block.bpb.bytes_per_sector;
    nclust /= parameter_block.bpb.sectors_per_cluster;
    printf("FAT32 root directory: using %ul first clusters.\n");
    cnum = 2; /* first data cluster has number 2, not 0... */
    for ( ; nclust > 0 ; nclust--)
      {
      if (nclust == 1) /* last cluster of root dir? Write End Of Chain! */
        {
        sector_buffer[sbp]   = 0xf8;	/* end of chain */
        sector_buffer[sbp+1] = 0xff;
        sector_buffer[sbp+2] = 0xff;
        sector_buffer[sbp+3] = 0x0f;
        }
      else
        {
        sector_buffer[sbp] = cnum+1;	/* 1 byte is enough here */
        sector_buffer[sbp+1] = 0;	/* +1 because we POINT to the */
        sector_buffer[sbp+2] = 0;	/* NEXT cluster in the chain! */
        sector_buffer[sbp+3] = 0;
        }
      sbp += 4;
      cnum++;
      if (sbp >= 512)
        {
        printf("Root directory bigger than 126 clusters, must abort!\n");
        exit(1);
        }
      } /* for loop for chain creation */
    } /* FAT32 */

  if (debug_prog==TRUE)
    printf("[DEBUG]  Writing FAT1 Headers To Sector ->    %d\n",
      file_sys_info.start_fat_sector);

  Drive_IO(WRITE,file_sys_info.start_fat_sector,1);

  if (debug_prog==TRUE)
    printf("[DEBUG]  Writing FAT2 Headers To Sector ->    %d\n",
      file_sys_info.start_fat_sector + file_sys_info.number_fat_sectors);

  Drive_IO(WRITE,(file_sys_info.start_fat_sector
                  + file_sys_info.number_fat_sectors),1 );
}

void Write_Root_Directory()
{
  int index;
  int root_directory_first_sector;
  int root_directory_num_sectors;
  int space_fill;
  union REGS regs;

  unsigned long sector;

  /* Clear Root Directory Area */
  Clear_Sector_Buffer();

  sector_buffer[0] = 0;  /* -ea */
  sector_buffer[32] = 0; /* -ea */

  sector=file_sys_info.start_root_dir_sect;
  root_directory_first_sector = (unsigned)sector;
  root_directory_num_sectors = (int)(file_sys_info.number_root_dir_sect);

  if (debug_prog==TRUE) printf("[DEBUG]  Clearing Root Directory Sectors -> ");

  do
    {
    if (debug_prog==TRUE) printf(" %3d,",sector); /* shorter debug output */

    Drive_IO(WRITE,sector,1);
    sector++;
    } while (sector<(root_directory_first_sector+root_directory_num_sectors));

  if (debug_prog==TRUE) printf(" done.\n");

  param.first_data_sector
   = root_directory_first_sector+root_directory_num_sectors+1;

  /* Add Volume Label to Root Directory */
  if(param.v==TRUE)
    {
    Clear_Sector_Buffer();
    sector=file_sys_info.start_root_dir_sect;
    index=0;
    space_fill=FALSE;
    do
      {
      if (param.volume_label[(index)]==0x00) space_fill=TRUE;

      if (space_fill==FALSE) sector_buffer[index]=param.volume_label[(index)];
      else sector_buffer[index]=' ';

      index++;
      } while (index<=10);

    sector_buffer[0x0b] = 0x08; /* file attribute: "volume label */

    /* setting of label / file system creation date and time: new 0.91b */

    regs.h.ah=0x2c; /* get time */
    intdos(&regs,&regs); /* now ch=hour cl=minute dh=second dl=subseconds */
    /* word at 0x16 is creation time: secs/2:5 mins:6 hours:5 */
    regs.x.ax = regs.h.dh >> 1;
    regs.x.bx = regs.h.cl;
    regs.x.ax |= regs.x.bx << 5;  /* or in the minute */
    regs.x.bx = regs.h.ch;
    regs.x.ax |= regs.x.bx << 11; /* or in the hour */
    sector_buffer[0x16] = regs.h.al;
    sector_buffer[0x17] = regs.h.ah;

    regs.h.ah=0x2a; /* get date */
    intdos(&regs,&regs); /* now cx=year dh=month dl=day al=wday (0=sun) */
    /* word at 0x18 is creation date: day:5 month:4 year-1980:7 */
    regs.x.ax = regs.h.dl;
    regs.x.bx = regs.h.dh;
    regs.x.cx -= 1980; /* subtract DOS epoch year 1980 */
    regs.x.ax |= regs.x.bx << 5; /* or in the month */
    regs.x.ax |= regs.x.cx << 9; /* or in DOS year! */
    sector_buffer[0x18] = regs.h.al;
    sector_buffer[0x19] = regs.h.ah;
    

    if(debug_prog==TRUE)
      {
      printf("[DEBUG]  Writing volume label to root directory sector ->  %d\n",sector);
      printf("[DEBUG]  Using file system creation date / time as label date / time\n");
      }

    Drive_IO(WRITE,sector,1);
    
    } /* if setting label */
}
