/*
// Program:  Format
// Written By:  Brian E. Reifsnyder
// Version:  0.6 Pre 1
// Copyright:  1999 under the terms of the GNU GPL
*/

/*
/////////////////////////////////////////////////////////////////////////////
//  SPECIAL
/////////////////////////////////////////////////////////////////////////////
*/

#define NAME "Format"
#define VERSION "0.6 Pre 1"

int debug_prog=0;            /* set to 1 for debug mode */

#define MAIN

/*
/////////////////////////////////////////////////////////////////////////////
//  INCLUDES
/////////////////////////////////////////////////////////////////////////////
*/

#include <bios.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>

#include "bootcode.h"
#include "bpb.h"
#include "clstsize.h"
#include "format.h"

#include "pdiskio.h"

/*
/////////////////////////////////////////////////////////////////////////////
//  PROTOTYPES
/////////////////////////////////////////////////////////////////////////////
*/

void ASCII_CD_Number(unsigned long number);
void Ask_User_To_Insert_Disk();
void Clear_Sector_Buffer();
void Clear_Huge_Sector_Buffer();
void Confirm_Hard_Drive_Formatting();
void Display_Drive_Statistics();
void Display_Percentage_Formatted(int cylinder);
void Format_Floppy_Cylinder(int cylinder,int head);
void Initialization();
void Restore_DDPT();
void Set_Floppy_Media_Type();
void Set_Hard_Drive_Media_Parameters();
void Setup_DDPT();

void Unconditional_Format();
void Unconditional_Floppy_Format();
void Unconditional_Hard_Disk_Format();
void Write_Sectors(unsigned long sector_number,int number_of_sectors);
void Write_System_Files();

/*
/////////////////////////////////////////////////////////////////////////////
//  FUNCTIONS
/////////////////////////////////////////////////////////////////////////////
*/

void ASCII_CD_Number(unsigned long number)
{
  int index=0;
  do
    {
    ascii_cd_number[index]=0;

    index++;
    }while(index<14);

  ultoa(number,ascii_cd_number,10);

  /* Add Commas */
  index=13;
  int right_index=13;
  int start_shift=FALSE;

  do
    {
    if(ascii_cd_number[index]>0) start_shift=TRUE;

    if(start_shift==TRUE)
      {
      ascii_cd_number[right_index]=ascii_cd_number[index];
      ascii_cd_number[index]=0;
      right_index--;
      }

    index--;
    }while(index>=0);

  int comma_counter=0;
  index=13;
  int shift_counter;
  int shift_register;
  int shift_temp_register;
  do
    {
    comma_counter++;

    if(ascii_cd_number[index]==0)
      {
      comma_counter=5;
      ascii_cd_number[index]=' ';
      }

    if(comma_counter==4)
      {
      shift_counter=index-1;
      shift_register=ascii_cd_number[index];
      ascii_cd_number[index]=',';
      do
	{
	shift_temp_register=ascii_cd_number[shift_counter];
	ascii_cd_number[shift_counter]=shift_register;
	shift_register=shift_temp_register;

	shift_counter--;
	}while(shift_counter>=0);

      comma_counter=0;
      }

    index--;
    }while(index>=0);
}

void Ask_User_To_Insert_Disk()
{
  printf("\nInsert new diskette for drive %c:\n",param.drive_letter[0]);
  printf("and press ENTER when ready...");

  /* Wait for a key */
  asm{
    mov ah,0x08
    int 0x21
    }

  printf("\n\n");
}

/* Clear Huge Sector Buffer */
void Clear_Huge_Sector_Buffer()
{
  long loop=0;

  do
    {
    huge_sector_buffer[loop]=0;

    loop++;
    }while(loop<(512*32));
}

void Confirm_Hard_Drive_Formatting()
{
  int character=0;

  printf("\nWARNING: ALL DATA ON NON-REMOVABLE DISK\n");
  printf("DRIVE %c: WILL BE LOST!\n",param.drive_letter[0]);
  printf("Proceed with Format (Y/N)?");

  /* Get keypress */
  asm{
    mov ah,0x07
    int 0x21
    mov BYTE PTR character,al
    }

  printf("\n");
  if( (character!='Y') && (character!='y') ) exit(10);
  printf("\n");
}

/* Convert Huge number into 4 LMB integer values */
void Convert_Huge_To_Integers(unsigned long number)
{
  unsigned long temp=number;
  unsigned long mask=0x000000ff;

  integer1=temp & mask;

  temp=number;
  mask=0x0000ff00;
  integer2=(temp & mask) >> 8;

  temp=number;
  mask=0x00ff0000;
  integer3=(temp & mask) >> 16;

  temp=number;
  mask=0xff000000;
  integer4=(temp & mask) >> 24;
}

/* Create File System */
void Create_File_System()
{
  int index;

  unsigned char oem_id[8];
  oem_id[0]='F';
  oem_id[1]='r';
  oem_id[2]='e';
  oem_id[3]='e';
  oem_id[4]=' ';
  oem_id[5]='D';
  oem_id[6]='O';
  oem_id[7]='S';

  if(debug_prog==TRUE) printf("\n[DEBUG]  Creating File System\n");

  Clear_Sector_Buffer();

  /* Add Jump Instruction */
  sector_buffer[0]=0xeb;
  sector_buffer[1]=0x3c;
  sector_buffer[2]=0x90;

  /* Add OEM ID */
  index=3;
  do
    {
    sector_buffer[index]=oem_id[(index-3)];
    index++;
    }while(index<11);

  /* *** Add BPB */
  void *bpb_p=&drive_specs[param.media_type];

  unsigned bpb_seg=FP_SEG(bpb_p);
  unsigned bpb_off=FP_OFF(bpb_p);

  char far *bpb=(char far *)MK_FP(bpb_seg,bpb_off);

  index=11;
  do
    {
    sector_buffer[index]=bpb[(index-10)];

    index++;
    }while(index<=35);

  /* *** Add Extended BPB */
  if(param.media_type==HD) sector_buffer[36]=0x80;

  /* Add Signature Byte */
  sector_buffer[38]=0x29;

  /* Add Serial Number */
  randomize();
  sector_buffer[39]=rand();
  sector_buffer[40]=rand();
  sector_buffer[41]=rand();
  sector_buffer[42]=rand();

  drive_statistics.serial_number_low
   =(sector_buffer[39]) + (sector_buffer[40]<<8);

  drive_statistics.serial_number_high
   =(sector_buffer[41]) + (sector_buffer[42]<<8);

  /* Add Volume Label */
  if(param.v==TRUE)
    {
    index=43;
    int space_fill=FALSE;
    do
      {
      if(param.volume_label[(index-43)]==0x00) space_fill=TRUE;

      if(space_fill==FALSE) sector_buffer[index]=param.volume_label[(index-43)];
      else sector_buffer[index]=' ';

      index++;
      }while(index<=53);
    }
  else
    {
    sector_buffer[43]='N';
    sector_buffer[44]='O';
    sector_buffer[45]=' ';
    sector_buffer[46]='N';
    sector_buffer[47]='A';
    sector_buffer[48]='M';
    sector_buffer[49]='E';
    sector_buffer[50]=' ';
    sector_buffer[51]=' ';
    sector_buffer[52]=' ';
    sector_buffer[53]=' ';
    }

  /* Add System ID */
  sector_buffer[54]='F';
  sector_buffer[55]='A';
  sector_buffer[56]='T';
  sector_buffer[57]='1';

  if(param.fat_type==FAT12) sector_buffer[58]='2';
  else sector_buffer[58]='6';

  sector_buffer[59]=' ';
  sector_buffer[60]=' ';
  sector_buffer[61]=' ';

  /* Add Boot Code */
  index=62;
  do
    {
    sector_buffer[index]=boot_code[(index-62)];

    index++;
    }while(index<=(130+62));

  /* Add Flag */
  sector_buffer[510]=0x55;
  sector_buffer[511]=0xaa;

  /* Write boot sector to the first sector of the disk */
  if(debug_prog==TRUE) printf("\n[DEBUG]  Writing Boot Sector->    0\n");
  Write_Sectors(0,1);

  /* Configure FAT Tables */
  Clear_Sector_Buffer();
  unsigned long sector=1;
  do
    {
    if(debug_prog==TRUE) printf("[DEBUG]  Clearing FAT Sector->  %3d\n",sector);
    Write_Sectors(sector,1);

    sector++;
    }while(sector<(1+(2*drive_specs[param.media_type].sectors_per_fat)));

  sector_buffer[0]=drive_specs[param.media_type].media_descriptor;
  sector_buffer[1]=0xff;
  sector_buffer[2]=0xff;
  if(param.fat_type==FAT16) sector_buffer[3]=0xff;

  if(debug_prog==TRUE) printf("[DEBUG]  Write FAT Headers To Sector->    1\n");
  Write_Sectors(1,1);

  if(debug_prog==TRUE) printf("[DEBUG]  Write FAT Headers To Sector->    %3d\n",(1+drive_specs[param.media_type].sectors_per_fat));
  Write_Sectors( (1+drive_specs[param.media_type].sectors_per_fat),1 );

  /* Clear Root Directory Area */
  Clear_Sector_Buffer();
  sector=(drive_specs[param.media_type].sectors_per_fat*2)+1;
  int root_directory_first_sector=sector;
  int root_directory_num_sectors=drive_specs[param.media_type].root_directory_entries/16;
  do
    {
    if(debug_prog==TRUE) printf("[DEBUG]  Clearing Root Directory Sector->  %3d\n",sector);
    Write_Sectors(sector,1);
    sector++;
    }while(sector<(root_directory_first_sector+root_directory_num_sectors));

  /* Add Volume Label to Root Directory */
  if(param.v==TRUE)
    {
    Clear_Sector_Buffer();
    sector=(drive_specs[param.media_type].sectors_per_fat*2)+1;
    index=0;
    int space_fill=FALSE;
    do
      {
      if(param.volume_label[(index)]==0x00) space_fill=TRUE;

      if(space_fill==FALSE) sector_buffer[index]=param.volume_label[(index)];
      else sector_buffer[index]=' ';

      index++;
      }while(index<=10);

    sector_buffer[11]=0x08;

    if(debug_prog==TRUE) printf("[DEBUG]  Writing Volume Label To Root Directory Sector->  %3d\n",sector);
    Write_Sectors(sector,1);
    }

  /* Flush DOS disk buffers so there is not any phantom information in */
  /* the directory.                                                    */
  asm{
    mov ah,0x0d
    int 0x21
    }
}

void Display_Drive_Statistics()
{
  drive_statistics.allocation_units_available_on_disk
   =drive_statistics.bytes_available_on_disk
   /drive_statistics.bytes_in_each_allocation_unit;

  ASCII_CD_Number(drive_statistics.bytes_total_disk_space);
  printf("\n%13s bytes total disk space\n",ascii_cd_number);

  ASCII_CD_Number(drive_statistics.bytes_available_on_disk);
  printf("%13s bytes available on disk\n",ascii_cd_number);

  printf("\n");

  ASCII_CD_Number(drive_statistics.bytes_in_each_allocation_unit);
  printf("%13s bytes in each allocation unit.\n",ascii_cd_number);

  ASCII_CD_Number(drive_statistics.allocation_units_available_on_disk);
  printf("%13s allocation units available on disk.\n",ascii_cd_number);

  printf("\n");
  printf("Volume Serial Number is %04X-%04X\n"
   ,drive_statistics.serial_number_high,drive_statistics.serial_number_low);
}

void Display_Invalid_Combination()
{
  printf("\nInvalid combination of options...please consult documentation.\n");
  printf("Operation Terminated.\n");
  exit(4);
}

/* Help Routine */
void Display_Help_Screen()
{
  printf("\n%6s Version %s\n",NAME,VERSION);
  printf("Written By:  Brian E. Reifsnyder\n");
  printf("Copyright 1999 under the terms of the GNU GPL.\n\n");
  printf("Syntax:\n\n");
  printf("FORMAT drive: [/V[:label]] [/Q] [/U] [/F:size] [/B | /S]\n");
  printf("FORMAT drive: [/V[:label]] [/Q] [/U] [/T:tracks /N:sectors] [/B | /S]\n");
  printf("FORMAT drive: [/V[:label]] [/Q] [/U] [/1] [/4] [/B | /S]\n");
  printf("FORMAT drive: [/Q] [/U] [/1] [/4] [/8] [/B | /S]\n\n");
  printf(" /V:label   Specifies a volume label for the disk.\n");
  printf(" /Q         Quick formats the disk.  The disk can be UNFORMATed.\n");
  printf(" /U         Unconditionally formats the disk.  The disk cannot be UNFORMATted\n");
  printf(" /F:size    Specifies the size of the floppy disk to format.  Valid sizes are:\n");
  printf("              160, 180, 320, 360, 720, 1200, 1440, or 2880.\n");
  printf(" /B         Kept for compatibility, formally reserved space for the boot files.\n");
  printf(" /S         Copies the operating system files to make the disk bootable.\n");
  printf(" /T:tracks  Specifies the number of tracks on a floppy disk.\n");
  printf(" /N:sectors Specifies the number of sectors on a floppy disk.\n");
  printf(" /1         Formats a single side of a floppy disk.\n");
  printf(" /4         Formats a 360K floppy disk in a 1.2 MB floppy drive.\n");
  printf(" /8         Formats a 5.25\" disk with 8 sectors per track.\n");
}

void Display_Percentage_Formatted(int cylinder)
{
  unsigned long percentage=((100*cylinder)/(drive_specs[param.media_type].cylinders));

  printf("%3d percent completed.\n",percentage);

  if(debug_prog!=TRUE)
    {
    /* Re-position cursor back to the beginning of the line */
    asm{
      /* Get current video display mode */
      mov ah,0x0f
      int 0x10

      /* Get cursor position */
      mov ah,0x03
      int 0x10

      /* Set cursor position to beginning of line */
      mov ah,0x02
      sub dh,1
      int 0x10
      }
    }
}

void Format_Floppy_Cylinder(int cylinder,int head)
{
  /* Set Up Track Address Fields */
  TAF track_address_fields[36];
  void *track_address_fields_p=track_address_fields;

  unsigned track_address_fields_seg=FP_SEG(track_address_fields_p);
  unsigned track_address_fields_off=FP_OFF(track_address_fields_p);

  int index=0;
  do
    {
    track_address_fields[index].cylinder=cylinder;
    track_address_fields[index].head=head;
    track_address_fields[index].sector=(index+1);
    track_address_fields[index].size_code=0x02;

    index++;
    }while(index<=drive_specs[param.media_type].sectors_per_cylinder);

  int drive_number=param.drive_number;


  if(debug_prog==TRUE)
    {
    printf("[DEBUG]  Formatting:  Cylinder->  %2d    Head->  %2d\n",cylinder,head);
    }

  /* Format the Track */
  asm{
    mov ah,0x05
    mov ch,BYTE PTR cylinder
    mov dh,BYTE PTR head
    mov dl,BYTE PTR drive_number
    mov es,WORD PTR track_address_fields_seg
    mov bx,WORD PTR track_address_fields_off
    int 0x13
    }
}

void Initialization()
{
  flags.version=SIX;
  debug.write=TRUE;

  param.drive_letter[0]=NULL;
  param.volume_label[0]=NULL;

  param.drive_type=NULL;
  param.drive_number=NULL;
  param.fat_type=NULL;
  param.media_type=UNKNOWN;

  param.v=FALSE;
  param.q=FALSE;
  param.u=FALSE;
  param.f=FALSE;
  param.b=FALSE;
  param.s=FALSE;
  param.t=FALSE;
  param.n=FALSE;
  param.one=FALSE;
  param.four=FALSE;
  param.eight=FALSE;

  param.size=UNKNOWN;
  param.cylinders=0;
  param.sectors=0;

  Setup_DDPT();
}

/* Save the old file system for possible recovery with unformat */
void Save_File_System()
{
//  printf("Saving UNFORMAT information\n");
}

void Set_Floppy_Media_Type()
{
  param.fat_type=FAT12;

  if(param.f==TRUE)
    {
    if(param.size==160) param.media_type=FD160;
    if(param.size==180) param.media_type=FD180;
    if(param.size==320) param.media_type=FD320;
    if(param.size==360) param.media_type=FD360;
    if(param.size==720) param.media_type=FD720;
    if(param.size==1200) param.media_type=FD1200;
    if(param.size==1440) param.media_type=FD1440;
    if(param.size==2880) param.media_type=FD2880;
    }

  if(param.t==TRUE)
    {
    int index=0;
    do
      {
      if( (param.cylinders==drive_specs[index].cylinders)
       && (param.sectors==drive_specs[index].sectors_per_cylinder) )
	{
	param.media_type=index;
	index=20;
	}

      index++;
      }while(index<8);
    }

  if(param.media_type==UNKNOWN)
    {
    /* Attempt to automatically detect the media type */
    int drive_number=param.drive_number;
    int drive_type=0;

    asm{
      mov ah,0x08
      mov dl,BYTE PTR drive_number
      int 0x13

      mov BYTE PTR drive_type,bl
      }

    if(drive_type==0x01)
      {
      param.size=360;
      param.media_type=FD360;
      }
    if(drive_type==0x02)
      {
      param.size=1200;
      param.media_type=FD1200;
      }
    if(drive_type==0x03)
      {
      param.size=720;
      param.media_type=FD720;
      }
    if(drive_type==0x04)
      {
      param.size=1440;
      param.media_type=FD1440;
      }
    }

  ddpt->sectors_per_cylinder=drive_specs[param.media_type].sectors_per_cylinder;

  drive_statistics.bytes_total_disk_space
   =((unsigned long)drive_specs[param.media_type].bytes_per_sector
   *(unsigned long)drive_specs[param.media_type].total_sectors)
   -((1+(2*(unsigned long)drive_specs[param.media_type].sectors_per_fat)
   +((unsigned long)drive_specs[param.media_type].root_directory_entries/16))
   *(unsigned long)drive_specs[param.media_type].bytes_per_sector);

  drive_statistics.bytes_available_on_disk
   =drive_statistics.bytes_total_disk_space;

  drive_statistics.bytes_in_each_allocation_unit
   =(unsigned long)drive_specs[param.media_type].sectors_per_cluster
   *(unsigned long)drive_specs[param.media_type].bytes_per_sector;
}

void Set_Hard_Drive_Media_Parameters()
{
  int index=0;
  int sub_index=0;

  int drive_number=UNUSED;
  int partition_number=UNUSED;

  unsigned long number_of_sectors;

  param.media_type=HD;
  param.physical_drive_number=UNUSED;

  Read_Partition_Tables();

  /* Search drive_lettering_buffer[8] [27] for the drive letter */
  do
    {
    sub_index=0;
    do
      {
      if(param.drive_letter[0]==drive_lettering_buffer[index][sub_index])
	{
	drive_number=index;
	param.physical_drive_number=index+0x80;
	partition_number=sub_index;
	}

      sub_index++;
      }while(sub_index<27);

    index++;
    }while(index<8);

  if(drive_number==UNUSED)
    {
    printf("\nInvalid drive specification\n");
    exit(1);
    }

  if(partition_number<4)
    {
    /* Drive is in the Primary Partition Table */

    number_of_sectors
     =part_table[drive_number].pri_part_num_sect[partition_number];

    index=0;
    do
      {
      if( (number_of_sectors>=cluster_size_table[index].min_num_sectors)
       && (number_of_sectors<cluster_size_table[index].max_num_sectors) )
	drive_specs[HD].sectors_per_cluster=cluster_size_table[index].sectors_per_cluster;

      index++;
      }while(index<7);

    number_of_sectors=number_of_sectors-(drive_specs[HD].sectors_per_cluster/2);

    if(part_table[drive_number].pri_part_num_type[partition_number]==1)
      {
      param.fat_type=FAT12;

      drive_specs[HD].sectors_per_fat
       =((number_of_sectors/drive_specs[HD].sectors_per_cluster)*3)
       /(drive_specs[HD].bytes_per_sector*2);

      if( (number_of_sectors/drive_specs[HD].sectors_per_cluster*3
       %(drive_specs[HD].bytes_per_sector*2) )!=0)
       drive_specs[HD].sectors_per_fat++;
      }
    if(part_table[drive_number].pri_part_num_type[partition_number]==4)
      {
      param.fat_type=FAT16;

      drive_specs[HD].sectors_per_fat
       =((number_of_sectors/drive_specs[HD].sectors_per_cluster)*2)
       /drive_specs[HD].bytes_per_sector;

      if( (drive_specs[HD].sectors_per_cluster*2
       %drive_specs[HD].bytes_per_sector)!=0) drive_specs[HD].sectors_per_fat++;
      }
    if(part_table[drive_number].pri_part_num_type[partition_number]==6)
      {
      param.fat_type=FAT16;

      drive_specs[HD].sectors_per_fat
       =((number_of_sectors/drive_specs[HD].sectors_per_cluster)*2)
       /drive_specs[HD].bytes_per_sector;

      if( (drive_specs[HD].sectors_per_cluster*2
       %drive_specs[HD].bytes_per_sector)!=0) drive_specs[HD].sectors_per_fat++;
      }

    drive_specs[HD].cylinders=part_table[drive_number].total_cyl;
    drive_specs[HD].sectors_per_cylinder=part_table[drive_number].total_sect;
    drive_specs[HD].number_of_heads=part_table[drive_number].total_head;

    drive_specs[HD].hidden_sectors_low=part_table[drive_number].total_sect;
    drive_specs[HD].hidden_sectors_high=0;

    if(number_of_sectors<=0xffff)
      {
      drive_specs[HD].total_sectors=number_of_sectors;
      }
    else
      {
      drive_specs[HD].large_sector_count_low=number_of_sectors&0x0000ffff;
      drive_specs[HD].large_sector_count_high=(number_of_sectors&0xffff0000)>>16;
      }

    param.size=part_table[drive_number].pri_part_size_in_MB[partition_number];

    param.partition_number=partition_number;
    }
  else
    {
    /* Drive is in the Extended Partition Table */
    partition_number=partition_number-4;

    number_of_sectors
     =part_table[drive_number].log_drive_num_sect[partition_number];

    index=0;
    do
      {
      if( (number_of_sectors>=cluster_size_table[index].min_num_sectors)
       && (number_of_sectors<cluster_size_table[index].max_num_sectors) )
	drive_specs[HD].sectors_per_cluster=cluster_size_table[index].sectors_per_cluster;

      index++;
      }while(index<9);

    number_of_sectors=number_of_sectors-(drive_specs[HD].sectors_per_cluster/2);

    if(part_table[drive_number].log_drive_num_type[partition_number]==1)
      {
      param.fat_type=FAT12;

      drive_specs[HD].sectors_per_fat
       =((number_of_sectors/drive_specs[HD].sectors_per_cluster)*3)
       /(drive_specs[HD].bytes_per_sector*2);

      if( (number_of_sectors/drive_specs[HD].sectors_per_cluster*3
       %(drive_specs[HD].bytes_per_sector*2) )!=0)
       drive_specs[HD].sectors_per_fat++;
      }
    if(part_table[drive_number].log_drive_num_type[partition_number]==4)
      {
      param.fat_type=FAT16;

      drive_specs[HD].sectors_per_fat
       =((number_of_sectors/drive_specs[HD].sectors_per_cluster)*2)
       /drive_specs[HD].bytes_per_sector;

      if( (drive_specs[HD].sectors_per_cluster*2
       %drive_specs[HD].bytes_per_sector)!=0) drive_specs[HD].sectors_per_fat++;
      }
    if(part_table[drive_number].log_drive_num_type[partition_number]==6)
      {
      param.fat_type=FAT16;

      drive_specs[HD].sectors_per_fat
       =((number_of_sectors/drive_specs[HD].sectors_per_cluster)*2)
       /drive_specs[HD].bytes_per_sector;

      if( (drive_specs[HD].sectors_per_cluster*2
       %drive_specs[HD].bytes_per_sector)!=0) drive_specs[HD].sectors_per_fat++;
      }

    drive_specs[HD].cylinders=part_table[drive_number].total_cyl;
    drive_specs[HD].sectors_per_cylinder=part_table[drive_number].total_sect;
    drive_specs[HD].number_of_heads=part_table[drive_number].total_head;

    drive_specs[HD].hidden_sectors_low=part_table[drive_number].total_sect;
    drive_specs[HD].hidden_sectors_high=0;

    if(number_of_sectors<=0xffff)
      {
      drive_specs[HD].total_sectors=number_of_sectors;
      }
    else
      {
      drive_specs[HD].large_sector_count_low=number_of_sectors&0x0000ffff;
      drive_specs[HD].large_sector_count_high=(number_of_sectors&0xffff0000)>>16;
      }

    param.size=part_table[drive_number].log_drive_size_in_MB[partition_number];

    param.partition_number=partition_number+4;
    }

  drive_statistics.bytes_total_disk_space
   =((unsigned long)drive_specs[param.media_type].bytes_per_sector*number_of_sectors)
   -((1+(2*(unsigned long)drive_specs[param.media_type].sectors_per_fat)
   +((unsigned long)drive_specs[param.media_type].root_directory_entries/16))
   *(unsigned long)drive_specs[param.media_type].bytes_per_sector);
  drive_statistics.bytes_available_on_disk
   =drive_statistics.bytes_total_disk_space;

  drive_statistics.bytes_in_each_allocation_unit
   =(unsigned long)drive_specs[param.media_type].sectors_per_cluster
   *(unsigned long)drive_specs[param.media_type].bytes_per_sector;
}

void Setup_DDPT()
{
  unsigned ddpt_seg;
  unsigned ddpt_off;

  /* Get the location of the DDPT */
    asm{
    mov ah,0x35
    mov al,0x1e
    int 0x21

    mov WORD PTR ddpt_seg,es
    mov WORD PTR ddpt_off,bx
    }

  (void far *)ddpt = MK_FP(ddpt_seg,ddpt_off);
}

/* Unconditionally Format the Drive */
void Unconditional_Format()
{
  if(param.drive_type==FLOPPY) Unconditional_Floppy_Format();
  else Unconditional_Hard_Disk_Format();
}

void Unconditional_Floppy_Format()
{
  int index=0;
  do
    {
    if(drive_specs[param.media_type].number_of_heads==2)
     Format_Floppy_Cylinder(index,0);
    Format_Floppy_Cylinder(index,1);

    Display_Percentage_Formatted(index);

    index++;
    }while(index<=drive_specs[param.media_type].cylinders);
}

void Unconditional_Hard_Disk_Format()
{
}

/* Write System Files */
void Write_System_Files()
{
  char sys[9] = {'s','y','s',' ','x',':',13,0,0};

  sys[4]=param.drive_letter[0];

  system(sys);
}

void Write_Sectors(unsigned long sector_number,int number_of_sectors)
{
  unsigned long sector_buffer_off;
  unsigned long sector_buffer_seg;

  if(number_of_sectors==1)
    {
    sector_buffer_off=FP_OFF(sector_buffer);
    sector_buffer_seg=FP_SEG(sector_buffer);
    }
  if(number_of_sectors>1)
    {
    sector_buffer_off=FP_OFF(huge_sector_buffer);
    sector_buffer_seg=FP_SEG(huge_sector_buffer);
    }

  int drive_number=param.drive_number;

  if(param.media_type<HD)
    {
    /* Drive is a floppy disk drive */
    asm{
      mov al,BYTE PTR drive_number
      mov bx,WORD PTR sector_buffer_off
      mov cx,WORD PTR number_of_sectors
      mov dx,WORD PTR sector_number
      mov ds,WORD PTR sector_buffer_seg
      int 0x26
      }
    }
  else
    {
    int error_code=Write_Logical_Sectors(param.drive_letter,sector_number,
     number_of_sectors);

    if(error_code==255)
      {
      printf("\nInvalid drive specification\n");
      exit(1);
      }
    }
}

/*
/////////////////////////////////////////////////////////////////////////////
//  MAIN ROUTINE
/////////////////////////////////////////////////////////////////////////////
*/
void main(int argc, char *argv[])
{
  int loop=0;

  Initialization();

  /* if FORMAT is typed without any options or with /? */
  if( (argc==1) || (argv[1][1]=='?') )
    {
    Display_Help_Screen();
    exit(1);
    }

  /*if FORMAT is typed with a drive letter */
  if(argc>=2)
    {
    int index=3;

    char compare_character[2];

    param.drive_letter[0]=argv[1] [0];

    if(param.drive_letter[0]>='a')
     param.drive_number = param.drive_letter[0] - 'a';
    if( (param.drive_letter[0]>='A') && (param.drive_letter[0]<='Z') )
     param.drive_number = param.drive_letter[0] - 'A';

    if( (param.drive_number<0) || (param.drive_number> 25) )
      {
      printf("\nInvalid drive specification\n");
      exit(4);
      }

    if(param.drive_letter[0]>='a') param.drive_letter[0]=param.drive_letter[0]-32;
    param.drive_letter[1] = ':';

    if(debug_prog==TRUE) printf("\n[DEBUG]  Drive To Format->  %s \n\n",param.drive_letter);

    /* Set the type of disk */
    if(param.drive_number>1) param.drive_type=HARD;
    else param.drive_type=FLOPPY;

    if(argc>=3)
      {
      index=3;
      do
	{
	compare_character[0]=argv[index-1][1];

	/* Determine which switches were entered */
	if( ('v'==compare_character[0]) || ('V'==compare_character[0]) )
	  {
	  param.v=TRUE;

	  loop=0;
	  do
	    {
	    param.volume_label[loop]=argv[index-1][loop+3];
	    if(param.volume_label[loop]>96) param.volume_label[loop]=param.volume_label[loop]-32;

	    loop++;
	    }while(loop<11);
	  }

	if( ('q'==compare_character[0]) || ('Q'==compare_character[0]) )
	  {
	  param.q=TRUE;
	  }

	if( ('u'==compare_character[0]) || ('U'==compare_character[0]) )
	  {
	  param.u=TRUE;
	  }

	if( ('f'==compare_character[0]) || ('F'==compare_character[0]) )
	  {
	  param.f=TRUE;
	  if( ('1'==argv[index-1][3]) && ('6'==argv[index-1][4]) ) param.size=160;
	  if( ('1'==argv[index-1][3]) && ('8'==argv[index-1][4]) ) param.size=180;
	  if( ('3'==argv[index-1][3]) && ('2'==argv[index-1][4]) ) param.size=320;
	  if( ('3'==argv[index-1][3]) && ('6'==argv[index-1][4]) ) param.size=360;
	  if( ('7'==argv[index-1][3]) ) param.size=720;
	  if( ('1'==argv[index-1][3]) && ('2'==argv[index-1][4]) ) param.size=1200;
	  if( ('1'==argv[index-1][3]) && ('2'==argv[index-1][5]) ) param.size=1200;
	  if( ('1'==argv[index-1][3]) && ('4'==argv[index-1][4]) ) param.size=1440;
	  if( ('1'==argv[index-1][3]) && ('4'==argv[index-1][5]) ) param.size=1440;
	  if( ('2'==argv[index-1][3]) ) param.size=2880;
	  }

	if( ('b'==compare_character[0]) || ('B'==compare_character[0]) )
	  {
	  param.b=TRUE;
	  }

	if( ('s'==compare_character[0]) || ('S'==compare_character[0]) )
	  {
	  param.s=TRUE;
	  }

	if( ('t'==compare_character[0]) || ('T'==compare_character[0]) )
	  {
	  param.t=TRUE;
	  if('4'==argv[index-1][3]) param.cylinders=40;
	  if('8'==argv[index-1][3]) param.cylinders=80;
	  }

	if( ('n'==compare_character[0]) || ('N'==compare_character[0]) )
	  {
	  param.n=TRUE;
	  if('8'==argv[index-1][3]) param.sectors=8;
	  if('9'==argv[index-1][3]) param.sectors=9;
	  if( ('1'==argv[index-1][3]) && ('5'==argv[index-1][4]) ) param.sectors=15;
	  if( ('1'==argv[index-1][3]) && ('8'==argv[index-1][4]) ) param.sectors=18;
	  if('3'==argv[index-1][3]) param.sectors=36;
	  }

	if('1'==compare_character[0])
	  {
	  param.one=TRUE;
	  }

	if('4'==compare_character[0])
	  {
	  param.four=TRUE;
	  }

	if('8'==compare_character[0])
	  {
	  param.eight=TRUE;
	  }

	index++;
	}while(index<=argc);
      }

    /* Ensure that valid switch combinations were entered */
    if( (param.b==TRUE) && (param.s==TRUE) ) Display_Invalid_Combination();
    if( (param.v==TRUE) && (param.eight==TRUE) ) Display_Invalid_Combination();
    if( ( (param.one==TRUE) || (param.four==TRUE) ) && ( (param.f==TRUE) || (param.t==TRUE) || (param.n==TRUE) ) ) Display_Invalid_Combination();
    if( ( (param.t==TRUE) && (param.n!=TRUE) ) || ( (param.n==TRUE) && (param.t!=TRUE) ) ) Display_Invalid_Combination();
    if( (param.f==TRUE) && ( (param.t==TRUE) || (param.n==TRUE) ) )Display_Invalid_Combination();
    if( ( (param.one==TRUE) || (param.four==TRUE) ) && (param.eight==TRUE) )Display_Invalid_Combination();
    if( ( (param.four==TRUE) || (param.eight==TRUE) ) && (param.one==TRUE) )Display_Invalid_Combination();
    if( ( (param.eight==TRUE) || (param.one==TRUE) ) && (param.four==TRUE) )Display_Invalid_Combination();

    /* Make sure that a floppy disk is unconditionally formatted if /f /t */
    /* or /n are selected.                                                */
    if( (param.f==TRUE) || (param.t==TRUE) || (param.n==TRUE) )
      {
      param.u=TRUE;
      param.q=FALSE;
      }

    /* Set the media parameters */
    if(param.drive_type==FLOPPY)
      {
      Ask_User_To_Insert_Disk();
      Set_Floppy_Media_Type();
      }
    else
      {
      Confirm_Hard_Drive_Formatting();
      Set_Hard_Drive_Media_Parameters();
      }

    /* Format Drive */
    if( (param.u==TRUE) && (param.q==FALSE) )
      {
      /* Unconditional Format */
      Unconditional_Format();
      printf("Format complete.          \n");
      Create_File_System();
      }

    if( (param.u==FALSE) && (param.q==TRUE) )
      {
      /* Quick Format */
      Save_File_System();
      Create_File_System();
      }

    if( (param.u==TRUE) && (param.q==TRUE) )
      {
      /* Quick Unconditional format */
      printf("QuickFormatting \n");
      Create_File_System();
      printf("Format complete.          \n");
      }

    if( (param.u==FALSE) && (param.q==FALSE) )
      {
      /* Safe Format */
      Save_File_System();
      Create_File_System();
      }

    if(param.s==TRUE) Write_System_Files();

    Display_Drive_Statistics();

    exit(0);
    }
}
