/*
	jemm16c.c
	
	Copyright (c) by tom ehlert 2001-2005 - all rights reserved

	Licensed under the Artistic License version
	
	please see LICENSE.TXT for details

	this code is based on ct0890s (copyright 1990 by harry albrecht, c't)
	this was taken, hacked, patched, tweaked, to make a real EMM386.
*/


#include "jemm16c.h"

extern char szHelp[];

#if INTEGRATED
#define NAMEEXE "JEMMEX"
#define NAME "JemmEx"
#else
#define NAMEEXE "JEMM386"
#define NAME "Jemm386"
#endif

/* public globals */

JEMMINIT jemmini = {0,0,0,-1,MINMEM16K_DEFAULT,0,0,0,0,
        DMABUFFDEFAULT,0xE000,0x4000,0,0,0,0,0,0,1,0,0,-1,0,0,0,0,0};

uchar startup_verbose = 0;
uchar bLoad = 0;

#if INTEGRATED
ulong xms_max = 0;
ushort hma_min = 0;
ushort xms_num_handles = 32;
uchar x_option = 0;
uchar no_above_16 = 0;
#endif

struct reg16 emmreg16;

#if INTEGRATED==0
struct reg16 xmsreg16;
struct reg32 xmsreg32;
#endif

/* local globals */

static ulong dwAllocatedBytes = 0;		/* memory reserved for EMM          */
static ulong memlargest = 0;    /* largest free block in kB */
static ulong memfree    = 0;    /* free extended memory in kB */

static ushort wFRAMEwanted = 0;		/* should be set from commandline */

#if INTEGRATED==0
static ushort xmsspec3 = 0;
#endif

static uchar SystemMemory[256];	/* 256*4K pagesize = 1MB */
static uchar ExcludeTest = 0;   /* X=TEST parameter */
static uchar IncludeTest = 0;   /* I=TEST parameter */
static uchar MinRequest = 0;	/* MIN= has been set */

static char szError[] = {"ERROR\a"};
static char szWarning[] = {"WARNING"};

/*****************************************************************************/

char *skipWhite(char *s)
{
	while (isWhite(*s))
		s++;
	return s;		
}            

/* set memory type , but honour "EXCLUDE=" and "INCLUDE=" types */

static void pascal SetMemoryType(ushort addr, uchar type)
{
	uchar *mem = &SystemMemory[addr >>8];
	
	if (*mem == 'I' && type != 'X')
		;
	else
		*mem = type;
}	

/*
	'R' = RAM
    'E' = EPROM
    'S' = Shadow-RAM activated by UMBPCI
    'G' = GRAPHICS
	'V' = VMWARE allocated, but possibly re-usable via I= (0e800-0ebffh)
	
	'U' = possible UMB space, because nothing else found
	'P' = PAGEFRAME
	'I' = INCLUDE = forced from commandline
	'X' = EXCLUDE = forbidden from commandline
	
	
*/	

/* 
	search memory for ROMS, adapters, graphics,...
	
    builds SystemMemory map

    the "checks" which are done are:

    - memory range C000-EFFF is scanned for ROMS,
      if one is found, and "I=TEST", it is checked if there are pages
      filled with 0x00 or 0xFF.
    - with option "X=TEST", memory range C000-EFFF is also tested if
      content is 0x00 or 0xFF, anything else excludes page
    - memory range C000-EFFF is also checked for RAM. if found pages are
      regarded as "reserved" (must be explicitely included with I= or S=)

*/	

static void ScanSystemMemory(void)
{
	uint mem,i,j;
    uchar far * pmem;
    uchar uc;
	uchar reuse;

    /* scan for ROMS */

	for (mem = 0xc000; mem < 0xf000;) {
		uint romsize;
		pmem = (uchar far *)MK_FP(mem,0);
		
		if ( pmem[0] != 0x55u || pmem[1] != 0xaau) {
								/* no adapter EPROM */
			mem += 2048/16;		/* advance by 2K    */   
			continue;
		}

		/* ensure valid EPROM signature allocates minimum 2K for 0 size */
		/* keep romsize aligned to 2K boundary, rounding up */
        romsize = (((uint)pmem[2] * 2 + 3) / 4);

		if (startup_verbose)
			printf("EPROM at %X, size %u kB\n", mem, romsize);
								/* romsize given in 512 byte*/

		for ( i = 0; i < romsize; i+=2)	{
            if (SystemMemory[mem>>8] == 'X')
                ;
            else
			if (!IncludeTest || (mem & 0xff)) {
				SetMemoryType(mem,'E');
			} else {
				pmem = (uchar far *)MK_FP(mem, 0);
                reuse = 1;
#if CHECK00FFONLY
		/* is the whole 4K block filled with 00 or FF? */
				for (j = 0; j < 4096; j++) {
                    if (*(pmem+j) != 0 && *(pmem+j) != 0xff) {
#else
        /* is the whole 4K block filled with one value? */
        /* and do NOT check the last byte in the page! */
				for (j = 0, uc = *pmem; j < 4095; j++) {
                    if (*(pmem+j) != uc) {
#endif
						SetMemoryType(mem,'E');
						reuse = 0;
						break;
					}
				}
				if (reuse) {
					SetMemoryType(mem,'I');
					mem += 2048/16;	/* advanced by 2K to make 4K total */
				}
			}
			mem += 2048/16;		/* advance by 2K	*/
		}
	}

#if 1
    /* test if RAM is in the non-excluded regions >= 0xC000. */
    i = 0xC0;
    do {
        if (i = TestForSystemRAM(&SystemMemory, i, &mem)) {
            printf("System memory found at %X-%X, region might be in use\n", i, i+mem-1);
            i = (i + mem) >> 8;
        }
    } while (i);
#endif

	/* if ExcludeTest is set, we need to scan all 'U' memory and ensure
     it is all 0 or FF values */

	if (ExcludeTest) {
		for (mem = 0xa0; mem < 0xf0; mem++) {
			if (SystemMemory[mem] == 'U') {
				/* this would be an upper memory block by default */
				/* don't check final byte as this can be garbage */
				pmem = (uchar far *)MK_FP(mem << 8, 0);
                for (i = 0; i < 4095; i++, pmem++) {
                    uc = *pmem;
					if ((uc != 0) && (uc != 0xff)) {
						/* found a nonzero, non-FF value in memory block */
						/* mark the whole block as excluded */
						SystemMemory[mem] = 'X';
						break;
					}
				}
			}
		}
	}

	/* for (i = 0xa0; i < 0xf8; i++)
			printf("%x00 : %c\n",i,SystemMemory[i]); */
}

/* 
	find a contiguous area of 64 KB 
	should handle commandline option like "FRAME=D000"
*/	
static ushort LocatePageFrame(void)
{   
    int page,i;
	ushort frame = 0;
    uchar bSearching = 0;
    uchar bWarning = 0;
    uchar bHardWanted = 0;

    if (wFRAMEwanted)
        bHardWanted = 1;
    else
        wFRAMEwanted = 0xE000;

	page = wFRAMEwanted >> 8;

    /* if a FRAME is explicitely set, ignore RAM/Video pages above A000 */

	for (i = 0; i < 16; i++) {
		if (bHardWanted && (page >= 0x80) && ((SystemMemory[page+i] == 'R') || (SystemMemory[page+i] == 'G')))
            bWarning = 1;
        else
			if (SystemMemory[page+i] != 'U')
				break;
	}
	if (i == 16) {
		frame = page;
		goto frameset;
    }
    if (bHardWanted)
        printf("Selected page frame %04x not accepted, scanning for a valid one...\n", wFRAMEwanted);

	bSearching = 1;

	for (page = 0xa0; page <= 0xE8; page++)	{
		for (i = 0; i < 16; i++) {
			if (SystemMemory[page+i] != 'U')
				break;
		}
		if (i == 16) {
			frame = page;
		}
	}

	if (frame == 0)	{
        printf("%s: no suitable page frame found, EMS functions limited.\n", szWarning);
		jemmini.NoFrame = 1;
		return 0;
	}


frameset:
	if (startup_verbose || bSearching)
		printf("Using page frame %02x00\n", frame);
	if (bWarning && (!bSearching))
        printf("%s: page frame %02x00 might not work reliably\n", szWarning, frame);

	memset(SystemMemory+frame,'P',16);
		
	return frame << 8;
}

/* old:check if there is 16 KB contiguos memory here to be used as UMB */ 
/* new:check if there is 4K KB contiguous memory here to be used as UMB */ 

static int pascal IsUMBMemory(ushort page)
{
    if (jemmini.NoRAM)
        return FALSE;
	if ((SystemMemory[page] == 'U') || (SystemMemory[page] == 'I') ||
	( (jemmini.NoEMS || jemmini.NoFrame) && SystemMemory[page] == 'P') )
        return TRUE;
	else
		return FALSE;
}

/*
	return number of 4k pages for UMB mapping
*/
static int UMBpageswanted(void)
{
    int wanted = 0;
    int i;
	for (i = 0xa0; i < 0xf8; i++)
		if (IsUMBMemory(i)) {
			wanted++;
		}

	return wanted;
}	

#if INTEGRATED==0
/* get free+largest XMS memory */

static int XMSGetMemoryStatus(ushort usev3)
{
	if (usev3) {
        xmscall32(0x88);   /* query free extended memory */
        if (!(xmsreg32.ebx & 0xff)) {
            memlargest = xmsreg32.eax;
            memfree    = xmsreg32.edx;
            return 1;
        }
	}
	xmsreg16.bx = 0;
    xmscall(8);
    if (!(xmsreg16.bx & 0xff)) {  /* query free extended memory */
        memlargest = xmsreg16.ax;
        memfree    = xmsreg16.dx;
        return 1;
    }
    return 0;
}

/* alloc XMS memory block for EMM
*/

static ushort XMSAllocMemory(int usev3, ulong kbneeded)
{
    ushort xmshandle = 0;
    ulong kbtotal;

    for ( ; !xmshandle; jemmini.MinMem16k /= 2) {
        kbtotal = kbneeded + jemmini.MinMem16k * 16;
        if (kbtotal <= 0xFFFFUL) {
            xmsreg16.dx = kbtotal;
            if (xmscall(9)) {
                xmshandle = xmsreg16.dx;
                break;
            }
            if (usev3) goto usev3ver;
            /* try largest free block based allocation */
            if ((memlargest > kbneeded + jemmini.MinMem16k * 8) &&
                (memlargest < kbtotal)) {
                xmsreg16.dx = memlargest;
                if (xmscall(9)) {
                    jemmini.MinMem16k = memlargest - kbneeded;
                    xmshandle = xmsreg16.dx;
                    break;
                }
            }
        } else {
        usev3ver:
            xmsreg32.edx = kbtotal;
            if (xmscall32(0x89)) {
                xmshandle = xmsreg32.edx & 0xFFFF;
                break;
            }
            /* try largest free block based allocation */
            if ((memlargest <= kbneeded + jemmini.MinMem16k * 8) ||
                (memlargest >= kbtotal)) {
                /* outside range, try next loop */
                break;
            }
            xmsreg32.edx = memlargest;
            if (xmscall32(0x89)) {
                jemmini.MinMem16k = (memlargest - kbneeded) / 16;
                xmshandle = xmsreg32.edx & 0xFFFF;
                break;
            }
        }

    }
    return xmshandle;
}
#else


#endif

/* if XMS manager doesnt export handle table (or pool is off),
   calc a reasonable amount of fixed memory to allocate.
*/

static unsigned int GetReasonableFixedMemoryAmount(void)
{
    unsigned int wWanted;

#if INTEGRATED
    if (!(memlargest = I15GetMemoryStatus(1)))
#else
    if (!XMSGetMemoryStatus(xmsspec3))
#endif
        return 0;

    if (memlargest >= (64 * 1024UL))
        wWanted = 32 * 1024;
    else
        wWanted = (unsigned int)(memlargest / 2);
    return wWanted;
}
/*
	allocate memory from XMS/I15
	find highest memory address
    determine monitor load address (must currently be <= 15MB
    + memory required for UMBs
*/

static int AllocAndInitMem(unsigned long kbneeded)
{   
    unsigned u;
    ulong ulcalc;
    ulong PotentialEmsVcpiMemory;
    ulong dwMinOriginal = jemmini.MinMem16k;

    /*
     the DMA buffer must be 64kb aligned. Since EMS pages *must* be
     allocated from memory physically "behind" the DMA buffer, there may
     be some space wasted, max 60-16=44 kB
    */
//    if (jemmini.MinMem16k && (jemmini.DMABufferSize > 4)) {
//    if ((jemmini.MinMem16k || jemmini.NoPool) && (jemmini.DMABufferSize > 4)) {
    if (jemmini.DMABufferSize > 4) {
        u = jemmini.DMABufferSize >= 64 ? 32 : jemmini.DMABufferSize/2;
        kbneeded = kbneeded + u;
        if (startup_verbose)
            printf("%u kB to account for DMA buffer 64 kB alignment\n", u);
    }

#if INTEGRATED==0
    if (!XMSGetMemoryStatus(xmsspec3)) {
        printf("%s: can't get XMS memory status\n", szError);
		return 0;
	}
	if (startup_verbose)
		printf("XMS largest free block %lu kB, XMS free mem %lu kB\n",
               memlargest, memfree);
#else
    if (!(memfree = I15GetMemoryStatus(1))) {
        printf("%s: can't get I15 memory status\n", szError);
		return 0;
    }
    memfree = memfree - 64;	/* dont count HMA */
    memlargest = memfree;
	if (startup_verbose)
		printf("I15 total memory %lu kB\n", memfree);
#endif

/* reality check to throttle requests far beyond available XMS, later actual
	adjustments	are small and need not be compensated for here */
    if (jemmini.MinMem16k) {
        if ((((jemmini.MinMem16k << 4) + kbneeded) > memfree) && (kbneeded < memfree))
            jemmini.MinMem16k = (memfree - kbneeded) >> 4;

/* leave a little extended memory, if possible, for programs that want some XMS */
        if ((memlargest > kbneeded + 384UL) && (memlargest < kbneeded + (jemmini.MinMem16k << 4) + 384UL))
            jemmini.MinMem16k = (memlargest - kbneeded - 384UL) >> 4;
	}

    /* default is: all memory */
    if (memfree > kbneeded)
        PotentialEmsVcpiMemory = memfree - kbneeded;
    else
        PotentialEmsVcpiMemory = 0;

    if (jemmini.NoPool)	{ /* Pool sharing off? */
        if ((jemmini.MinMem16k << 4) < memfree)
            PotentialEmsVcpiMemory = jemmini.MinMem16k << 4; /*  mult with 16 */
	}

    /* MIN= has higher priority than MAX= */
    if (jemmini.MinMem16k > jemmini.MaxMem16k)
        jemmini.MaxMem16k = jemmini.MinMem16k;

    /* MaxMem16k may have been set by MAX=, and above the limit */
    if ((jemmini.MaxMem16k << 4) > PotentialEmsVcpiMemory)
        jemmini.MaxMem16k = (PotentialEmsVcpiMemory >> 4);

    /* MaxMem16k may have been set by MAX=, and below 32 MB! */
    /* this is valid, but then adjust max EMS pages as well */
    if (jemmini.MaxMem16k < jemmini.MaxEMSPages)
        jemmini.MaxEMSPages = jemmini.MaxMem16k;

    /* if MIN= has been set adjust max. EMS pages */
    if (jemmini.MinMem16k > jemmini.MaxEMSPages) {
        if (jemmini.MinMem16k > MAX_EMS_PAGES_POSSIBLE)
            jemmini.MaxEMSPages = MAX_EMS_PAGES_POSSIBLE;
        else
            jemmini.MaxEMSPages = jemmini.MinMem16k;
    }

    if (startup_verbose)
        printf("potential/max. VCPI memory: %lu/%lu kB\n", PotentialEmsVcpiMemory, jemmini.MaxMem16k << 4);
#if 0
    printf("Min=%lu, Max=%lu, EMS=%u\n", jemmini.MinMem16k, jemmini.MaxMem16k, jemmini.MaxEMSPages);
#endif

/*  the memory pooling need ((XMS total / 1.5M) + 1) * 64 bytes
    for pool allocation	table entries
    1.5M is pool allocation maximum memory control,
    64 is pool block size,
    if dynamic XMS allocation is on, 128 more items are needed,
    which represent the maximum number of XMS handles
*/

	ulcalc = jemmini.MaxMem16k * 16 / 1536 + 2; /* no of pool items */
	if (!jemmini.NoPool)
		ulcalc = ulcalc + 128;
	ulcalc = ulcalc << 6;  /*  * 64 = pool mem to alloc */

	/* 4+1 bytes for each EMS page needed */
	/* 255*4 bytes for EMS handle table */
    /* 255*8 bytes for EMS name table */
	/* 64*16 bytes for EMS save status table (EMS_MAXSTATE) */
	ulcalc = ulcalc + 5 * jemmini.MaxEMSPages + 255*4 + 255*8 + 64*4;

    ulcalc = (ulcalc + 1023) / 1024UL;	/* convert bytes back to K */
    ulcalc = (ulcalc + 3) & 0xfffffffc; /* 4k page align */
	if (startup_verbose)
		printf("%lu kB needed for VCPI and EMS handling\n", ulcalc);
    kbneeded += ulcalc;

#if INTEGRATED==0
	/* allocate memory from XMS */
    jemmini.XMSControlHandle = XMSAllocMemory(xmsspec3, kbneeded);
	if (!jemmini.XMSControlHandle) {
        printf("%s: can't allocate enough XMS memory(%lu kB)\n", szError, kbneeded);
		return 0;
    }

	/* lock handle to make a linear adress */
	
	xmsreg16.dx = jemmini.XMSControlHandle;
	if (!xmscall(0x0c))	{
        printf("%s: can't lock XMS memory\n", szError);
        xmscall(0x0A);/* free the block */
		return 0;
    }
#else
    if (!(jemmini.XMSControlHandle = I15AllocMemory(0, kbneeded + jemmini.MinMem16k * 16))) {
        printf("%s: can't allocate enough I15 memory(%lu kB)\n", szError, kbneeded + jemmini.MinMem16k * 16);
		return 0;
    }
#endif

    if (jemmini.MinMem16k < dwMinOriginal)
        printf("%s: MIN has been reduced to %lu kB\n", szWarning, jemmini.MinMem16k * 16);

	dwAllocatedBytes    = ((ulong)(jemmini.MinMem16k * 16 + kbneeded) * 1024);
#if INTEGRATED==0
    jemmini.MonitorStart = ((ulong)xmsreg16.dx << 16) | xmsreg16.bx;
#else
    jemmini.MonitorStart = 0x110000;
#endif
	jemmini.TotalMemory = (ulong)memfree * 1024L + jemmini.MonitorStart;

	return 1;	
	
}   


static void PrintStartup()
{
    printf( NAME " v5.63 [" __DATE__ "]\n");
    return;
}

static void PrintCopyright()
{
    printf(NAME ". Parts (c) tom ehlert 2001-2006 c't/H. Albrecht 1990\n");
    return;
}

/* conditional printf only if startup_verbose */

static void cprintf(char * pszText)
{
    if (startup_verbose)
        printf(pszText);
}

/* called on startup.
	parse commandline "I=B800-BFFF X=E000-EFFF" ...
	search for EPROMS+adapters (network cards)
	determine frame address
	...
	mode = 0 if called as driver
	mode = 1 if called as EXE
	
	return: 0 - Jemm loaded, everything fine
	        1 - Jemm not loaded
*/ 

int TheRealMain(int mode, char *commandline)
{
    char *found;
    int i,j;
    ulong ulFirstPage;
    int bHelp = 0;
    int bOptionSet = 0;

#if INTEGRATED
    jemmini.A20Method = 0xFF;
    jemmini.X2Max = 0xFFFF;
#endif
    if (mode != EXECMODE_EXE) {
        PrintStartup();
        bLoad = 1;
#if FASTBOOT
        if (FindCommand(commandline, "FASTBOOT") ) {
            jemmini.V86Flags = jemmini.V86Flags | V86F_FASTBOOT;
        }
#endif
    } else {
#if UNLOADSUPP
        if (FindCommand(commandline, "UNLOAD") ) {
            XMSinit();
            TryUnload();
            return 1;
        }
#endif
#if LOADSUPP
        if (FindCommand(commandline, "LOAD") )
            bLoad = 1;
        else {
            jemmini.NoVCPI = 0xFF;
#if A20SUPP
            jemmini.NoA20  = 0xFF;
#endif
#if VMESUPP
            jemmini.NoVME  = 0xFF;
#endif
#if PGESUPP
            jemmini.NoPGE  = 0xFF;
#endif
        }
#endif
        if (FindCommand(commandline, "/?") ||
            FindCommand(commandline, "-?") ||
            FindCommand(commandline, "/H") ||
            FindCommand(commandline, "-H") ) {
            bHelp = 1;
        }
    }

/*  printf("'real' commandline is '%s'\n",commandline); */

	if (FindCommand(commandline, "NOVCPI") ) {
		jemmini.NoVCPI = TRUE;
        bOptionSet = 1;
	}
	if (FindCommand(commandline, "VCPI") ) {
		jemmini.NoVCPI = FALSE;
        bOptionSet = 1;
    }
#if INTEGRATED
/* must come before A20 option */
    if (found = FindCommand(commandline, "A20METHOD:") )
        jemmini.A20Method = GetA20Method(found);
#endif
#if A20SUPP
    if (FindCommand(commandline, "NOA20") ) {
        jemmini.NoA20 = 1;
        bOptionSet = 1;
	}
    if (FindCommand(commandline, "A20") ) {
        jemmini.NoA20 = 0;
        bOptionSet = 1;
    }
#endif
#if VMESUPP
    if (FindCommand(commandline, "NOVME") ) {
        jemmini.NoVME = 1;
        bOptionSet = 1;
	}
    if (FindCommand(commandline, "VME") ) {
        jemmini.NoVME = 0;
        bOptionSet = 1;
    }
#endif
#if PGESUPP
    if (FindCommand(commandline, "NOPGE") ) {
        jemmini.NoPGE = 1;
        bOptionSet = 1;
	}
    if (FindCommand(commandline, "PGE") ) {
        jemmini.NoPGE = 0;
        bOptionSet = 1;
	}
#endif
#if INTEGRATED
    if (found = FindCommand(commandline, "HMAMIN=") ) {
        jemmini.HmaMin = GetValue(found, 10, 0);
        if (jemmini.HmaMin > 63)
            jemmini.HmaMin = 63;
    }

    if (found = FindCommand(commandline, "MAXEXT=") )
        xms_max = GetValue(found, 10, 1);

    if (FindCommand(commandline, "NOE801") )
        no_above_16 = 1;

    if (FindCommand(commandline, "NOE820") )
        x_option = 1;

    if (found = FindCommand(commandline, "XMSHANDLES=") ) {
        xms_num_handles = GetValue(found, 10, 0);
        if (xms_num_handles < 8)
            xms_num_handles = 8;
        else
            if (xms_num_handles > 128)
                xms_num_handles = 128;
    }

    if (found = FindCommand(commandline, "X2MAX=") )
        jemmini.X2Max = GetValue(found, 10, 1);
#endif
    if (bLoad) {
        memset(SystemMemory+0x00,'R',0xA0);
        memset(SystemMemory+0xA0,'G',0x20);
        memset(SystemMemory+0xC0,'U',0x30);
        memset(SystemMemory+0xF0,'E',0x10);

        if (FindCommand(commandline, "VERBOSE") ||
            FindCommand(commandline, "/V") ) {
            startup_verbose = 1;
        }

        if (VMwareDetect()) {
		/* exclude E800-EFFF for VMware, if present */
		/* E800-EBFF range may be recoverable via I=, so
    	   use 'V' instead of absolute 'X' */
            memset(SystemMemory+0xE8,'V',4);
            memset(SystemMemory+0xEC,'X',4);
            if (startup_verbose)
                printf("VMware detected\n");
        }

        if (FindCommand(commandline, "NODYN")) {
            jemmini.NoPool = 1; /* pool sharing off */
        }

        if (FindCommand(commandline, "NOINVLPG")) {
            jemmini.NoInvlPg = 1; /* dont use INVLPG opcode */
        }

        if (found = FindCommand(commandline, "MIN=")) {
            jemmini.MinMem16k = GetValue(found,10,TRUE) >> 4;
            if (startup_verbose)
                printf("Wanted preallocated EMS/VCPI memory: %lu kB\n", jemmini.MinMem16k << 4);
            MinRequest = 1;
        }

        if (found = FindCommand(commandline, "MAX=") ) {
            jemmini.MaxMem16k = GetValue(found,10,TRUE) >> 4;
        }


        if (FindCommand(commandline, "NOEMS") )	{
            cprintf("NOEMS: EMS disabled (mostly :-)\n");
            jemmini.NoEMS = TRUE;
        }

        if (FindCommand(commandline, "NOVDS") )	{
            jemmini.NoVDS = TRUE;
        }

        if (FindCommand(commandline, "VDS") ) {
            jemmini.NoVDS = FALSE;
        }

        if (FindCommand(commandline, "FRAME=NONE")) {
            jemmini.NoFrame = TRUE;
        } else {
            if ((found = FindCommand(commandline, "FRAME=")) ||
                (found = FindCommand(commandline, "/P")) ) {
                wFRAMEwanted = GetValue(found,16,FALSE);
                if (startup_verbose)
                    printf("Wanted page frame=%X\n", wFRAMEwanted);
            }
        }

        if (FindCommand(commandline, "X=TEST") ) {
            ExcludeTest = 1;
        }

        if (FindCommand(commandline, "I=TEST") ) {
            IncludeTest = 1;
        }

#if SBSUPP
        if (FindCommand(commandline, "SB") ) {
            jemmini.V86Flags = jemmini.V86Flags | V86F_SB;
        }
#endif
#if EMXSUPP
        if (FindCommand(commandline, "EMX") ) {
            jemmini.V86Flags = jemmini.V86Flags | V86F_EMX;
        }
#endif
#if 0
        if (FindCommand(commandline, "MEMCHECK") ) {
            MEMCHECK = 1;
        }
#endif

        if (FindCommand(commandline, "NOCHECK") ) {
            jemmini.V86Flags = jemmini.V86Flags | V86F_NOCHECK;
        }

        if (FindCommand(commandline, "ALTBOOT") ) {
            jemmini.AltBoot = 1;
        }

        if (FindCommand(commandline, "NOHI") ) {
            jemmini.NoHigh = 1;
        }

        if (FindCommand(commandline, "NOMOVEXBDA") ) {
		/* NOMOVEXBDA is a no-op, but helps MS EMM386 switch compatibility */
        }

        if (FindCommand(commandline, "NORAM") ) {
            jemmini.NoRAM = 1;
        }
        if (FindCommand(commandline, "RAM") ) {
            jemmini.NoRAM = 0;
        }


        if (found = FindCommand(commandline, "D=")) {
            ushort dmasize =  GetValue(found,10,FALSE);
            if (dmasize <= 128)
                jemmini.DMABufferSize = (dmasize+3) & -4;
            else {
                jemmini.DMABufferSize = 128;
                printf("%s: wanted DMA buffer size too large, set to 128 kB\n", szWarning);
            }
        }
        if (found = FindCommand(commandline, "B=")) {
            jemmini.Border =  GetValue(found,16,FALSE);
            if (jemmini.Border < 0x1000)
                jemmini.Border = 0x1000;
        }
					/* "I=a000-afff"  "X=d000-dfff" */
        for (;;) {
            ushort rangestart,rangestop;
            char memtype;
		
            memtype = 'I';
		
            if (!(found = FindCommand(commandline, "I="))) {
                memtype = 'S';
                if (!(found = FindCommand(commandline, "S="))) {
                    memtype = 'X';
                    if (!(found = FindCommand(commandline, "X="))) {
                        break;
                    }
                }
            }

            rangestart =  GetValue(found,16,FALSE);

            if (*found == '-') {
                memcpy( found, found+1,	strlen(found+1) +1);

                rangestop  =  GetValue(found,16,FALSE);
                if ((rangestart < 0xA000) || (rangestart >= rangestop))
                    printf("Rejected %c=%x..%x\n",memtype, rangestart, rangestop);
                else {
                    if (startup_verbose)
                        printf("Accepted %c=%x..%x\n",memtype, rangestart,rangestop);
                    for ( ; rangestart < rangestop; rangestart++)
                        SetMemoryType(rangestart,memtype);
                }
            }
        }
    } /* endif bLoad */

	/******* commandline handling done, are there remainders **********/

	commandline = skipWhite(commandline);
    if (*commandline >= '0' && *commandline <= '9') {
        jemmini.MaxMem16k = GetValue(commandline,10,TRUE) >> 4;
        commandline = skipWhite(commandline);
    }

    if (*commandline) {
        printf("* ignored commandline: '%s'\n", commandline);
        if (mode == EXECMODE_EXE)
            return 1;
    }

    /* shall the monitor be loaded? If no, do either:
		display usage info    or
		status report
	*/
	if (!bLoad) {
        PrintStartup();
        if (bHelp) {
            PrintCopyright();
            printf(szHelp);
        } else {
            if (bOptionSet) {
                if (EmmUpdate())  /* update Jemm status */
                    printf("option(s) passed to installed instance of " NAME "\n");
            } else {
                EmmStatus();  /* display Jemm status */
            }
        }
		return 1;
	}

    /******* options set, now process **********/

	if (IsProtectedMode()) {
		IsEmmInstalled();
		return 1;
	}
#if INTEGRATED==0
	if (!XMSinit())	{
		printf("%s: no XMM found, required\n", szError);
		return 1;
    }
#else
	if (XMSinit())	{
		printf("%s: XMM already installed\n", szError);
		return 1;
    }
#endif
#if 1
    /* its not the responsibility of the EMM to ensure an installed DPMI
     host won't get confused by the EMM to uninstall. But the other way
     is to avoid: install an EMM while a DPMI host is installed.
    */
	if (IsDPMI())	{
        printf("%s: DPMI host detected\n", szError);
		return 1;
    }
#endif
#if INTEGRATED==0
    if (xmscall(0) && (xmsreg16.ax >= 0x300))
        xmsspec3 = 1;

    if (!xmscall(5)) {  	/* enable A20 local */
		printf("%s: enable A20 failed\n", szError);
		return 1;
    }
#else
    EnableA20();
#endif

/* system state ok for loading driver */

    if (jemmini.MaxMem16k == -1)
        jemmini.MaxMem16k = MAXMEM16K_DEFAULT;

    if (jemmini.NoVCPI)
        cprintf("VCPI disabled\n");
    if (jemmini.NoVDS)
		cprintf("VDS disabled\n");
#if 0
    /* auto disable pooling is no longer useful */
    /* if wanted it can be disabled with NODYN */
    if (jemmini.NoVCPI && jemmini.NoEMS)
        jemmini.NoPool = 1;
#endif
    /* if no int 2fh, function 4309h support, disable pool sharing */
#if INTEGRATED==0
    if ((!jemmini.XMSHandleTable) && (!jemmini.NoPool)) {
        jemmini.NoPool = 1;
        printf("%s: XMS host doesn't provide handle array, dynamic memory allocation off!\n", szWarning);
    }
#endif
    if ((jemmini.NoPool) && (!MinRequest)) {
        jemmini.MinMem16k = GetReasonableFixedMemoryAmount() >> 4;
        if (startup_verbose)
            printf("default preallocated memory=%u\n", jemmini.MinMem16k);
    }

	ScanSystemMemory();			/* build up system memory map */

    if (jemmini.NoEMS)
        jemmini.NoFrame = 1;

	if (!jemmini.NoFrame)
		jemmini.Frame = LocatePageFrame();	/* find a contiguos area of 64 KB */


	/*
			allocate from XMS the memory we need
    		this is memory for UMBs, including FF00 if ALTBOOT=0

			+ 20kB for the monitor code, GDT, IDT, stack
			+ 12kB for page tables
			+ 12kB for TSS + IO-Bitmap (includes 3 kB reserve for rounding)
			+ 64kB +-X for DMA buffering
			+ room for other control structures made inside function
								
			+ what the user wants for EMS
	*/

    i = UMBpageswanted() * 4;
    j = 20 + 12 + 12 + (jemmini.AltBoot ? 0 : 1) * 4;

    if (startup_verbose)
        printf("Needed: %u kB for monitor, %u kB for UMBs, %u kB for DMA buffer\n",
               j, i, jemmini.DMABufferSize);

  /* even with NOEMS alloc space for EMS memory management for 8192 kb */
  /* else some DOS extenders (among them DOS4G) won't work (wastes 2,5 kB) */
    if (jemmini.NoEMS)
        jemmini.MaxEMSPages = 512;
    else
        jemmini.MaxEMSPages = MAX_EMS_PAGES_ALLOWED;

    if (!AllocAndInitMem(j + jemmini.DMABufferSize + i)) {
#if INTEGRATED==0
        xmscall(6); /* local disable A20 */
#else
        DisableA20();
#endif
		return 1;
	}

	jemmini.MonitorEnd   = jemmini.MonitorStart + dwAllocatedBytes;

    jemmini.pPageMap = SystemMemory;

	if (startup_verbose)	
        printf("XMS memory block for monitor: %lx-%lx, XMS highest=%lx\n",
			jemmini.MonitorStart, jemmini.MonitorEnd, jemmini.TotalMemory);

/*	printf("xms locked memory at %lx, top of mem %lx(%luMB) alloc bytes %lx(%lu kB)\n",
				jemmini.MonitorStart,
				jemmini.TotalMemory, jemmini.TotalMemory/(1024*1024l),
				dwAllocatedBytes, dwAllocatedBytes/(1024));
*/

    ulFirstPage = InitJemm();

    if (startup_verbose) {
        printf("Physical start address of EMS pages: %lX\n", ulFirstPage);
        if (!emmcall(0x42))	{ /* kickstart dynamic count of available EMS pages */
            printf("Total/available EMS pages: %d/%d (= %lu(%lu) kB)\n",
               emmreg16.dx, emmreg16.bx, (ulong)emmreg16.dx*16, (ulong)emmreg16.bx*16);
        }
	}

#if INTEGRATED==0
    xmscall(6); /* local disable A20 */
#endif

	return 0;
}

/* EOF */
