/* $Id: dat.c,v 1.2 1998/06/12 04:31:15 jhall Exp $ */

/* Functions for reading and dealing with the DAT file. */

/* Copyright (C) 1998 Jim Hall, jhall1@isd.net */

/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <stdio.h>
#include <string.h>				/* for strtok */
#include <stdlib.h>				/* for malloc */
#include "dat_t.h"


/* Symbolic constants */

#define STRLEN 256				/* size of a string */

#define DAT_ARRAY_LEN 80			/* init size of the array */


/* Globals */

int dat_ary_size;				/* size of the DAT array */


/* Functions. */

dat_t *
dat_fread (FILE *stream)
{
  /* Reads a DAT file from a stream into memory.  Returns the array,
     or NULL if an error.  The calling program must free the array
     after it is done with it, or you will leak memory. */

  char ch[2];					/* for reading rank */
  char s[STRLEN];				/* for reading the line */
  int n;					/* counter for ary size */
  int size;					/* the malloc'ed ary size */
  static dat_t *dat_ary;			/* the array to return */

  /* Allocate the array */

  size = DAT_ARRAY_LEN;
  if (!(dat_ary = malloc (sizeof (dat_t) * size))) {
    /* Failed */

    return (NULL);
  }

  /* Read the file until end of file */

  n = 0;

  while (fgets (s, STRLEN, stream) != NULL) {
    /* Break up the string into tokens */

    strncpy (dat_ary[n].fname, strtok (s, " :"), DAT_FNAME_LEN);
    strncpy (ch, strtok (NULL, " :"), 1);

    dat_ary[n].rank = ch[0];

    /* Check on the length of the array */

    if (++n >= size) {
      /* The array is maxed out - give up.  Later versions of this
         function should continue to allocate memory for the array */

      dat_ary_size = n;
      return (dat_ary);
    }
  } /* while */

  /* Return */

  dat_ary_size = n;
  return (dat_ary);
}

dat_t *
dat_read (const char *filename)
{
  /* Reads a DAT file from a file into memory.  Returns the array, or
     NULL if an error.  The calling program must free the array after
     it is done with it, or you will leak memory. */

  FILE *stream;
  static dat_t *dat_ary;

  /* Open the file */

  if (!(stream = fopen (filename, "r"))) {
    /* Failed */

    return (NULL);
  }

  /* Read the file - this sets the dat_ary_size global variable */

  dat_ary = dat_fread (stream);

  /* Close the file, and quit */

  fclose (stream);
  return (dat_ary);
}
