/* $Id: install.c,v 1.16 1998/08/18 03:15:44 jhall Exp $ */

/* main.c - The guts of the Free-DOS Install program. */

/* Copyright (C) 1998 Jim Hall, jhall1@isd.net */

/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <stdio.h>
#include <stdlib.h>				/* for free() */
#include <string.h>				/* for strcpy() */

#if defined(unix)
#include <unistd.h>				/* for getopt() */
#else
#include "getopt.h"				/* for getopt() */
#endif /* UNIX getopt */

#include "cat.h"
#include "dat.h"
#include "getkey.h"
#include "inst.h"
#include "pause.h"
#include "showdesc.h"


/* Symbolic constants */

#define STRLEN 80

#define DAT_FILE  "INSTALL.DAT"			/* disk sets to install */
#define OEM_FILE  "OEM.TXT"			/* the vendor file */


/* Globals */

extern unsigned g_errs;				/* inst.c */
extern unsigned g_warns;			/* inst.c */


/* Functions */

void usage (void);


/* The program starts here */

int
main(int argc, char *argv[])
{
  char from[STRLEN];				/* where to install from */
  char exdir[STRLEN];				/* where to install to */
  char descfile[STRLEN];			/* series description file */
  int i;

  int dsk_ser_n;				/* Size of disk series ary */
  dat_t *dsk_ser;				/* The disk series array */


  /* Evaluate the command line. */

  while ((i = getopt (argc, argv, "hH")) != EOF) {
    switch (i) {
    case 'h':
#if !defined(unix)
    case 'H':					/* case insensitive on DOS */
#endif /* unix */
      usage();
      exit (0);

    default:
      fprintf (stderr, "install: unrecognized option (%c)\n", i);
      usage();
      exit (1);
    } /* switch */
  } /* while */

  if (optind < argc) {
    /* There are options left, but there shouldn't be! */

    fprintf (stderr, "install: too many options on command line\n");
    usage();
    exit (1);
  }


  /* Say hello.  OEM_FILE is a convenience for the vendor, so I really
     don't care if 'cat' succeeds or not. */

  printf ("This program comes with ABSOLUTELY NO WARRANTY\n");
  printf ("This is free software, and you are welcome to redistribute it\n");
  printf ("under certain conditions; see the file `COPYING\' for details.\n");
  cat(OEM_FILE);

  printf ("\n-- FreeDOS Install --\n");
  printf ("Copyright (C) 1997-1998, James Hall <jhall1@isd.net>\n");

  /* Allow the user to bail out */

  if (!pause_yn (1)) {
    /* Quit */

    exit (0);
  }

  /* Read the package list */

  if (!(dsk_ser = dat_read (DAT_FILE))) {
    /* Failed */

    fprintf (stderr, "install: Can't find %s\n", DAT_FILE);
    exit (1);
  }

  dsk_ser_n = dat_ary_size;

  /* Find out where to install from/to */

  printf ("\nInstall from what path? ");
  if (gets (from) == NULL) {
    strcpy (from, "A:\\");
  }

  printf ("Install files to what path? ");
  if (gets (exdir) == NULL) {
    strcpy (exdir, "C:\\");
  }

  /* Print the package list */

  printf ("\n\n");
  printf ("You are about to install these disk series:\n");

  for (i = 0; i < dsk_ser_n; i++) {
    printf ("%s ", dsk_ser[i].fname);
  }
  printf ("\n\n");

  printf ("The install program will now let you install these disk series.  With\n");
  printf ("each series, you will be able to choose which to install and which to\n");
  printf ("skip.  Each disk series consists of several package files.  You can\n");
  printf ("also select which individual packages to install and which to skip.\n");
  printf ("Note that some disk series and package files may already be selected\n");
  printf ("for you.\n\n");

  /* Allow the user to bail out */

  if (!pause_yn (1)) {
    /* Quit */

    exit (0);
  }

  /* Ask which series to install */

  for (i = 0; i < dsk_ser_n; i++) {
    /* Show the description */

    sprintf (descfile, "%s.TXT", dsk_ser[i].fname);

    printf ("\n\nSeries: %s\n", dsk_ser[i].fname);
    cat (descfile);

    /* Find out which ones the user wants to install */

    switch (dsk_ser[i].rank) {
    case 'n':
    case 'N':
      /* Do not install */

      printf ("%s [SKIPPED]\n", dsk_ser[i].fname);
      dsk_ser[i].resp = 0;
      break;

    case 'y':
    case 'Y':
      /* Always install */

      printf ("%s [REQUIRED]\n", dsk_ser[i].fname);
      dsk_ser[i].resp = 1;
      break;

    default:
      /* Optional */

      printf ("%s [OPTIONAL]\n", dsk_ser[i].fname);

      /* Ask the user if you want to install it */

      printf ("Install series %s? [yn] ", dsk_ser[i].fname);
      dsk_ser[i].resp = pause_yn (0);
      break;

    } /* switch */
  } /* for */

  /* Run the install */

  printf ("\n\n");
  printf ("Installing the software ...\n");

  for (i = 0; i < dsk_ser_n; i++) {
    if (dsk_ser[i].resp) {
      instser (dsk_ser[i].fname, from, exdir);
    } /* if */
  } /* for */

  /* Done */

  free (dsk_ser);

  /* Print errors and warnings, if any */

  printf ("\n\n");;

  if ((g_errs > 0) || (g_warns > 0)) {
    printf ("There were %u errors and %u warnings.\n", g_errs, g_warns);
  }

  else {
    printf ("The install program completed successfully.\n");
  }

  exit (0);
}

void
usage (void)
{
  fprintf (stderr, "install: allows you to install software from a distribution.\n");
#if defined(unix)
  fprintf (stderr, "Usage: install [-h]\n");
#else
  fprintf (stderr, "Usage: install [/H]\n");
#endif /* unix */
}
