{ --------------------------------------------------------------- }
{ Dieser Quelltext ist urheberrechtlich geschuetzt.               }
{ (c) 1991-1999 Peter Mandrella                                   }
{ (c) 2000-2001 OpenXP-Team                                       }
{ (c) 2002-2005 FreeXP, http://www.freexp.de                      }
{ CrossPoint ist eine eingetragene Marke von Peter Mandrella.     }
{                                                                 }
{ Die Nutzungsbedingungen fuer diesen Quelltext finden Sie in der }
{ Datei SLIZENZ.TXT oder auf www.crosspoint.de/oldlicense.html.   }
{ --------------------------------------------------------------- }
{ $Id: mimedec.pas,v 1.8 2005/01/01 11:16:28 mw Exp $ }

{ CrossPoint - Zeichensatzdecodierung und -konvertierung }

{$I XPDEFINE.INC }
{$O+,F+}

unit mimedec;

interface

uses xpglobal,typeform;

const
  cs_us_ascii   =    0;
  cs_cp_437     =  437;
  cs_cp_850     =  850;
  cs_cp_858     =  858;
  cs_iso8859_1  =    1;
  cs_iso8859_2  =    2;
  cs_iso8859_9  =    9;
  cs_iso8859_15 =   15;
  cs_win1252    = 1252;


procedure IBM2ISO(var s:string);
procedure IBMToIso1(var data; size:word);
procedure ISO2IBM(var s:string; const charset: word);
procedure Iso1ToIBM(var data; size:word);
procedure Mac2IBM(var data; size:word);
procedure UTF8ToIBM(var s:string);
procedure UTF7ToIBM(var s:string);
function  ibm_charset(s:string):boolean;
function  iso_charset(s:string):word;
function  supported_charset(s:string):boolean;
procedure CharsetToIBM(charset:string; var s:string);

procedure DecodeBase64(var s:string);

procedure UnQuotePrintable(var s:string; qprint,b64,add_cr_lf:boolean);
procedure MimeIsoDecode(var ss:string; maxlen:integer);


implementation  {----------------------------------------------------}

uses xpovl;


{ ausgehend: #46 (".") = unkonvertierbar }

const IBM2ISOtab : array[0..255] of byte =  {EUR #238 siehe set_IBM2ISOtab}
{000} (  32, 46, 46, 46, 46, 46, 46,183, 46,  9, 10, 46, 12, 13, 46, 42,
{016}    62, 60, 46, 33,182,167, 95, 46, 46, 46, 62, 60, 45, 46, 46, 46,
{032}    32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
{048}    48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
{064}    64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
{080}    80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95,
{096}    96, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,
{112}   112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,

{128}   199,252,233,226,228,224,229,231,234,235,232,239,238,236,196,197,
{144}   201,230,198,244,246,242,251,249,255,214,220,162,163,165, 46, 46,
{160}   225,237,243,250,241,209,170,186,191, 45,172,189,188,161,171,187,
{176}    35, 35, 35,124, 43, 43, 43, 43, 43, 43,124, 43, 43, 43, 43, 43,
{192}    43, 43, 43, 43, 45, 43, 43, 43, 43, 43, 43, 43, 43, 45, 43, 43,
{208}    43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 35, 35, 35, 35, 35,
{224}    97,223, 71,112, 83,115,181,116, 80, 84, 79,100, 46,102,101, 46,
{240}    61,177, 62, 60,124,124,247, 61,176,183,183, 46,110,178,183, 32);


{ eingehend: #177 ("") = unkonvertierbar }

      ISO1_2IBMtab : array[128..255] of byte =      { oft: #128 = Euro }
{128} (euro,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
{144}   144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
{160}    32,173,155,156,177,157,124, 21, 34, 67,166,174,170, 45, 82,196,
{176}   248,241,253, 51, 39,230, 20,250, 44, 49,167,175,172,171,177,168,
{192}    65, 65, 65, 65,142,143,146,128, 69,144, 69, 69, 73, 73, 73, 73,
{208}    84,165, 79, 79, 79, 79,153,120,153, 85, 85, 85,154, 89, 84,225,
{224}   133,160,131, 97,132,134,145,135,138,130,136,137,141,161,140,139,
{240}   116,164,149,162,147,111,148,246,148,151,163,150,129,121,116,152);

      ISO2_2IBMtab : array[128..255] of byte =             { kein Euro }
{128}  (128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
{144}   144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
{160}    32, 65,177, 76,177, 76, 83, 21, 34, 83, 83, 84, 90, 45, 90, 90,
{176}   248, 97,177,108, 39,108,115,177, 44,115,115,116,122, 34,122,122,
{192}    82, 65, 65, 65,142, 76, 67,128, 67,144, 69, 69, 69, 73, 73, 68,
{208}    84, 78, 78, 79, 79,153,153,120, 82, 85, 85,154,154, 89, 84,225,
{224}   114,160,131, 97,132,108, 99,135, 99,130,101,137,101,161,140,100,
{240}   100,110,110,162,147,148,148,246,114,117,163,129,129,121,116,177);

      ISO9_2IBMtab : array[128..255] of byte =             { kein Euro }
{128}  (128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
{144}   144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
{160}    32,173,155,156,177,157,124, 21, 34, 67,166,174,170, 45, 82,196,
{176}   248,241,253, 51, 39,230, 20,250, 44, 49,167,175,172,171,177,168,
{192}    65, 65, 65, 65,142,143,146,128, 69,144, 69, 69, 73, 73, 73, 73,
{208}    84,165, 79, 79, 79, 79,153,120,153, 85, 85, 85,154, 89, 83,225,
{224}   133,160,131, 97,132,134,145,135,138,130,136,137,141,161,140,139,
{240}   103,164,149,162,147,111,148,246,148,151,163,150,129,105,116,152);

      ISO15_2IBMtab : array[128..255] of byte =          { #164 = Euro }
{128}  (128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
{144}   144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
{160}    32,173,155,156,euro,157,83, 21,115, 67,166,174,170, 45, 82,196,
{176}   248,241,253, 51, 90,230, 20,250,122, 49,167,175, 79,111, 89,168,
{192}    65, 65, 65, 65,142,143,146,128, 69,144, 69, 69, 73, 73, 73, 73,
{208}    84,165, 79, 79, 79, 79,153,120,153, 85, 85, 85,154, 89, 84,225,
{224}   133,160,131, 97,132,134,145,135,138,130,136,137,141,161,140,139,
{240}   116,164,149,162,147,111,148,246,148,151,163,150,129,121,116,152);

      WIN1252_2IBMtab : array[128..255] of byte =        { #128 = Euro }
{128} (euro,177, 44,159, 34,177,177,177, 94,177, 83, 60, 79,177, 90,177,
{144}   177, 39, 39, 34, 34,254,196,196,126,177,115, 62,111,177,122, 89,
{160}    32,173,155,156,177,157,124, 21, 34, 67,166,174,170, 45, 82,196,
{176}   248,241,253, 51, 39,230, 20,250, 44, 49,167,175,172,171,177,168,
{192}    65, 65, 65, 65,142,143,146,128, 69,144, 69, 69, 73, 73, 73, 73,
{208}    84,165, 79, 79, 79, 79,153,120,153, 85, 85, 85,154, 89, 84,225,
{224}   133,160,131, 97,132,134,145,135,138,130,136,137,141,161,140,139,
{240}   116,164,149,162,147,111,148,246,148,151,163,150,129,121,116,152);

      CP850_2IBMtab : array[128..255] of byte =   { CP858: #213 = Euro }
{128}  (128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
{144}   144,145,146,147,148,149,150,151,152,153,154,148,156,153,120,159,
{160}   160,161,162,163,164,165,166,167,168, 82,170,171,172,173,174,175,
{176}   176,177,178,179,180, 65, 65, 65, 67,185,186,187,188,155,157,191,
{192}   192,193,194,195,196,197, 97, 65,200,201,202,203,204,205,206,177,
{208}   116, 84, 69, 69, 69,105, 73, 73, 73,217,218,219,220,124, 73,223,
{224}    79,225, 79, 79,111, 79,230,116, 84, 85, 85, 85,121, 89,196, 39,
{240}    45,241, 95,177, 20, 21,246, 44,248, 34,250, 49, 51,253,254,255);


     { Mac: a o +RCt'"!O
             ao_  ^__AAOOo --,"`'Y/x<>__
             +,"_AEAEEIIIIOO _OUUUi^~-_,",_
       fehlt: BE, DE, DF }
      Mac2IBMtab : array[128..255] of byte =
      (142,143,128,144,165,153,154,160,133,131,132, 97,134,135,130,138,
       136,137,161,141,140,139,164,162,149,147,148,111,163,151,150,129,
        43,248,155,156, 21,249, 20,225, 82, 67,116, 39, 34, 33,146, 79,
       236,241,243,242,157,230,235,228,227,227,244, 97,111,234, 32,237,
       168,173,170,251,159,247, 94,174,175, 32, 32, 65, 65, 79, 79,111,
        45, 45, 44, 32, 96, 39,246,254,152, 89, 47,120, 60, 62, 32, 32,
        43,250, 44, 32, 32, 65, 69, 65, 69, 69, 73, 73, 73, 73, 79, 79,
        32, 79, 85, 85, 85,105, 94,126, 45, 32,250,248, 44, 34, 44, 32);


procedure IBM2ISO(var s:string); assembler;
asm
     push  es
     cld
     mov   bx,offset IBM2ISOtab
     les   si,s
     segES lodsb                     { Stringlnge }
     mov   cl,al
     xor   ch,ch
     jcxz  @@2
@@1: segES lodsb
     xlat
     mov   es:[si-1],al
     loop  @@1
@@2: pop   es
end;


procedure IBMToIso1(var data; size:word); assembler;
asm
          mov    cx,size
          jcxz   @noconv2
          les    di,data
          mov    bx,offset IBM2ISOtab
          cld
@isolp2:  mov    al,es:[di]
          xlat
          stosb
          loop   @isolp2
@noconv2:
end;


procedure ISO2IBM(var s:string; const charset: word); assembler;
asm
     push  es
     cld
     mov   ax,charset
     cmp   ax,cs_iso8859_15
     jne   @@cs1
     mov   bx,offset ISO15_2IBMtab - 128
     jmp   @@cs99
@@cs1:
     cmp   ax,cs_iso8859_9
     jne   @@cs2
     mov   bx,offset ISO9_2IBMtab - 128
     jmp   @@cs99
@@cs2:
     cmp   ax,cs_iso8859_2
     jne   @@cs3
     mov   bx,offset ISO2_2IBMtab - 128
     jmp   @@cs99
@@cs3:
     cmp   ax,cs_win1252
     jne   @@cs4
     mov   bx,offset WIN1252_2IBMtab - 128
     jmp   @@cs99
@@cs4:
     cmp   ax,cs_cp_850
     jne   @@cs5
     mov   bx,offset CP850_2IBMtab - 128
     jmp   @@cs99
@@cs5:
     mov   bx,offset ISO1_2IBMtab - 128
@@cs99:
     les   si,s
     segES lodsb                     { Stringlnge }
     mov   cl,al
     xor   ch,ch
     jcxz  @@2
@@1: segES lodsb
     cmp   al,127
     jbe   @@3
     xlat
     mov   es:[si-1],al
@@3: loop  @@1
@@2: pop   es
end;


procedure Iso1ToIBM(var data; size:word); assembler;
asm
          mov    cx,size
          jcxz   @noconv1
          les    di,data
          mov    bx,offset ISO1_2IBMtab - 128
          cld
@isolp1:  mov    al,es:[di]
          or     al,al
          jns    @ii1
          xlat
@ii1:     stosb
          loop   @isolp1
@noconv1:
end;


procedure Mac2IBM(var data; size:word); {&uses ebx, esi} assembler;
asm
          mov    bx,offset Mac2IBMtab - 128
          les    si,data
          mov    cx,size
          jcxz   @xende
          jmp    @xloop
@xloop:   mov    al,es:[si]
          inc    si
          cmp    al,127
          ja     @trans
          loop   @xloop
          jmp    @xende
@trans:   xlat
          mov    es:[si-1],al
          loop   @xloop
@xende:
end;


Function ConvertUnicode(ucs:dword):char;assembler;
Const
  Error     = 177;   { unkonvertierbar }
  Anzahl    = 76;
asm
        mov al,error
        mov bx,word ptr ucs
        mov cx,word ptr ucs+2
        or cx,cx               { > $FFFF }
        jne @end
        cmp bx,$80             { < $0080 }
        jb @end
        cmp bh,0               { < $00FF }
        jne @noiso
        mov al,byte ptr ISO1_2IBMTab[bx-128]
        jmp @end
@noiso: mov ax,bx
        mov di,offset @unicode
        mov si,di
        push cs
        pop es
        mov cx,Anzahl
        repne scasw
        mov al,error
        jne @end
        sub di,si
        shr di,1
        mov al,byte ptr cs:[@Asciicode+di-1]
        jmp @end

@Unicode:    dw $20ac,$20a7,$0192,$2310,$2591,$2592,$2593,$2502,$2524,$2561
             dw $2562,$2556,$2555,$2563,$2551,$2557,$255d,$255c,$255b,$2510
             dw $2514,$2534,$252c,$251c,$2500,$253c,$255e,$255f,$255a,$2554
             dw $2569,$2566,$2560,$2550,$256c,$2567,$2568,$2564,$2565,$2559
             dw $2558,$2552,$2553,$256b,$256a,$2518,$250c,$2588,$2584,$258c
             dw $2590,$2580,$03b1,$0393,$03c0,$03a3,$03c3,$03c4,$03a6,$0398
             dw $03a9,$03b4,$221e,$03c6,$03b5,$2229,$2261,$2265,$2264,$2320
             dw $2321,$2248,$2219,$221a,$207f,$25a0

@Asciicode:  db  euro, $9e , $9f , $a9 , $b0 , $b1 , $b2 , $b3 , $b4 , $b5
             db  $b6 , $b7 , $b8 , $b9 , $ba , $bb , $bc , $bd , $be , $bf
             db  $c0 , $c1 , $c2 , $c3 , $c4 , $c5 , $c6 , $c7 , $c8 , $c9
             db  $ca , $cb , $cc , $cd , $ce , $cf , $d0 , $d1 , $d2 , $d3
             db  $d4 , $d5 , $d6 , $d7 , $d8 , $d9 , $da , $db , $dc , $dd
             db  $de , $df , $e0 , $e2 , $e3 , $e4 , $e5 , $e7 , $e8 , $e9
             db  $ea , $eb , $ec , $ed , $ee , $ef , $f0 , $f2 , $f3 , $f4
             db  $f5 , $f7 , $f9 , $fb , $fc , $fe
@end:
end;


procedure UTF8ToIBM(var s:string); { by robo; nach RFC 2279 }
const sc_rest : string[6]='';      { my: erweitert fr Strings > 510 Zeichen }
var     i,j,k : integer;
           sc : record case integer of
                  0 : (s:string[6]);
                  1 : (b:array[0..6] of byte);
                end;
         rest : string[6];
          ucs : longint;
       applen : byte;
begin
  rest:='';
  if sc_rest<>'' then
  begin
    if length(sc_rest)<=255-length(s) then
      s:=sc_rest+s
    else begin
      rest:=right(s,length(sc_rest));
      s:=sc_rest+s
    end;
    sc_rest:='';
  end;
  for i:=1 to length(s) do if byte(s[i]) and $80=$80 then
  begin
    k:=0;
    for j:=0 to 7 do
      if byte(s[i]) and ($80 shr j)=($80 shr j) then inc(k) else break;
    sc.s:=copy(s,i,k);
    if length(sc.s)=k then
    begin
      delete(s,i,k-1);
      if rest<>'' then
      begin
        applen:=min(length(rest),k-1);
        s:=s+left(rest,applen);
        delete(rest,1,applen);
      end;
      for j:=0 to k-1 do sc.b[1]:=sc.b[1] and not ($80 shr j);
      for j:=2 to k do sc.b[j]:=sc.b[j] and $3f;
      ucs:=0;
      for j:=0 to k-1 do ucs:=ucs or (longint(sc.b[k-j]) shl (j*6));
      s[i]:=ConvertUnicode(ucs);
    end
    else begin
      sc_rest:=sc.s;
      delete(s,i,length(sc.s));
      break;
    end;
  end;
end;


procedure UTF7ToIBM(var s:string); { by robo; nach RFC 2152 }
                                   { my: erweitert fr Strings > 255 Zeichen }
const b64alphabet='ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/';
      sc_rest : string='';
var       i,j : integer;
      s1,rest : string;
      linebrk : boolean;
          ucs : smallword;
       applen : byte;

  function has_b64_only:boolean;
  var b64 : boolean;
  begin
    j:=0;
    repeat
      inc(j);
      b64:=cpos(rest[j],b64alphabet)>0;
    until not b64 or (j=length(rest));
    has_b64_only:=b64;
  end;

begin
  rest:=reverse(s);      { checken, ob Linebreak in codiertem Wort vorliegt }
  i:=cpos('+',rest);
  if i>0 then
    rest:=left(rest,i)
  else
    rest:='';
  linebrk:=(rest<>'') and has_b64_only;
  rest:='';
  if sc_rest<>'' then
  begin
    if length(sc_rest)<=255-length(s) then
      s:=sc_rest+s
    else begin
      rest:=right(s,length(sc_rest));
      s:=sc_rest+s
    end;
    sc_rest:='';
  end;
  i:=1;
  j:=posn('+',s,i);
  while j<>0 do
  begin
    i:=j;
    inc(j);
    while (j<=length(s)) and (cpos(s[j],b64alphabet)<>0) do inc(j);
    if (j<=length(s)) and (s[j]='-') then inc(j);
    s1:=copy(s,i,j-i);
    delete(s,i,j-i);
    if rest<>'' then
    begin
      applen:=min(length(rest),254-length(s));  { 254, um 1 Zeichen Reserve }
      s:=s+left(rest,applen);                   { zum spteren Einfgen zu  }
      delete(rest,1,applen);                    { haben                     }
    end;
    if s1='+-' then s1:='+'
    else begin
      if linebrk and (i>length(s)) then  { i>length(s): unvollstndig UTF7- }
      begin                              { codiertes Zeichen erreicht       }
        sc_rest:=s1;
        break;
      end
      else begin
        if firstchar(s1)='+' then delfirst(s1);
        if lastchar(s1)='-' then dellast(s1);
        while (length(s1) mod 4<>0) do s1:=s1+'=';
      end;
      DecodeBase64(s1);
      if odd(length(s1)) then dellast(s1);
      j:=1;
      while length(s1)>j do begin
        ucs:=word(s1[j]) shl 8+word(s1[j+1]);
        if (ucs<$00000080)
          then s1[j]:=char(ucs)
          else s1[j]:=convertUnicode(ucs);
        inc(j);
        delete(s1,j,1);
      end;
    end;
    insert(s1,s,i);
    j:=posn('+',s,i+length(s1));
  end;
end;


{ diese Charsets werden nicht konvertiert }
function ibm_charset(s:string):boolean;
begin
  ibm_charset:=(pos(';'+s+';',';us-ascii;ascii;csascii;ansi_x3.4-1968;'
                             + 'iso-ir-6;ansi_x3.4-1986;iso_646.irv:1991;'
                             + 'iso646-us;us;ibm367;cp367;') > 0) or
               (pos(';'+s+';',';ibm437;cp437;437;cspc8codepage437;') > 0);
end;

{ ISO1 / ISO2 / ISO9 / ISO15 / Win1252 / CP850 / CP858 }
{ diese Charsets drfen und mssen konvertiert werden  }
function iso_charset(s:string):word;
begin
  iso_charset:=0;
  if pos(';'+s+';',';iso-8859-1;iso_8859-1;iso_8859-1:1987;iso-ir-100;latin1;l1;csisolatin1;ibm819;cp819;') > 0 then
    iso_charset:=cs_iso8859_1 else
  if pos(';'+s+';',';iso-8859-2;iso_8859-2;iso_8859-2:1987;iso-ir-101;latin2;l2;csisolatin2;') > 0 then
    iso_charset:=cs_iso8859_2 else
  if pos(';'+s+';',';iso-8859-9;iso_8859-9;iso_8859-9:1989;iso-ir-148;latin5;l5;csisolatin5;') > 0 then
    iso_charset:=cs_iso8859_9 else
  if pos(';'+s+';',';iso-8859-15;iso_8859-15;latin-9;') > 0 then
    iso_charset:=cs_iso8859_15 else
  if s='windows-1252' then
    iso_charset:=cs_win1252 else
  if pos(';'+s+';',';ibm850;cp850;850;cspc850multilingual;') > 0 then
    iso_charset:=cs_cp_850 else
  if pos(';'+s+';',';ibm00858;ccsid00858;cp00858;pc-multilingual-850+euro;') > 0 then
    iso_charset:=cs_cp_858;
end;

function supported_charset(s:string):boolean;
begin
  supported_charset:=(s='') or
                     ibm_charset(s) or (iso_charset(s)>0) or
                     (s='utf-8') or (s='utf-7');
end;

procedure CharsetToIBM(charset:string; var s:string);
var cs : word;
begin
  lostring(charset);
  cs:=iso_charset(charset);
  if cs>0 then
  begin
    if cs=cs_cp_858 then
    begin
      CP850_2IBMtab[213]:=euro;
      cs:=cs_cp_850;
    end else
    if cs=cs_cp_850 then CP850_2IBMtab[213]:=105;
    ISO2IBM(s,cs);
  end else
  if charset='utf-8' then UTF8ToIBM(s) else
  if charset='utf-7' then UTF7ToIBM(s) else
  if charset='' then ISO2IBM(s,cs_iso8859_1);  { OjE-Fix! }
end;


procedure DecodeBase64(var s:string);
const
  b64tab : array[0..127] of byte =
           ( 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,63, 0, 0, 0,64,
            53,54,55,56,57,58,59,60,61,62, 0, 0, 0, 0, 0, 0,
             0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,15,
            16,17,18,19,20,21,22,23,24,25,26, 0, 0, 0, 0, 0,
             0,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,
            42,43,44,45,46,47,48,49,50,51,52, 0, 0, 0, 0, 0);
  b64alphabet='ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=';
var b1,b2,b3,b4 : byte;
    p1,p2,pad   : byte;
    i: integer;

  function nextbyte:byte;
  var p : byte;
  begin
    nextbyte:=0;
    if p1>length(s) then exit;
    repeat
      if s[p1]>#127 then p:=0
      else p:=b64tab[byte(s[p1])];
      inc(p1);
    until (p>0) or (p1>length(s));
    if p>0 then dec(p);
    nextbyte:=p;
  end;

begin
  if length(s)<4 then s:=''
  else begin
    for i:=1 to length(s) do if cpos(s[i],b64alphabet)=0 then exit;
    if s[length(s)]='=' then begin
      if s[length(s)-1]='=' then pad:=2
      else pad:=1;
      if length(s) mod 4<>0 then pad:=3;
    end
    else pad:=0;
    p1:=1; p2:=1;
    while p1<=length(s) do begin
      b1:=nextbyte; b2:=nextbyte; b3:=nextbyte; b4:=nextbyte;
      s[p2]:=chr(b1 shl 2 + b2 shr 4);
      s[p2+1]:=chr((b2 and 15) shl 4 + b3 shr 2);
      s[p2+2]:=chr((b3 and 3) shl 6 + b4);
      inc(p2,3);
    end;
    s[0]:=chr(p2-1-pad);
  end;
end;


procedure UnQuotePrintable(var s:string; qprint,b64,add_cr_lf:boolean);
                                       { MIME-quoted-printable/base64 -> 8bit }
var p,b     : byte;
    softbrk : boolean;

(*
  procedure AddCrlf; assembler; {&uses ebx}  { CR/LF an s anhngen }
  asm
    mov bl,byte ptr s[0]
    mov bh,0
    cmp bx,255
    je  @@1
    inc bx
    mov byte ptr s[bx],13
    cmp bx,255
    je  @@1
    inc bx
    mov byte ptr s[bx],10
@@1:mov byte ptr s[0],bl
  end;
*)

begin
  if qprint then begin
    while (s<>'') and (s[length(s)]=' ') do    { rtrim }
      dec(byte(s[0]));
    softbrk:=(lastchar(s)='=');    { quoted-printable: soft line break }
    if softbrk then dellast(s);
    p:=cpos('=',s);
    if p>0 then
      while p<length(s)-1 do begin
        inc(p);
        b:=hexval(copy(s,p,2));
        if b>0 then begin
          s[p-1]:=chr(b);
          delete(s,p,2);
        end;
        while (p<length(s)) and (s[p]<>'=') do inc(p);
      end;
    if not softbrk then
      if add_cr_lf then {AddCrlf} s:=s+#13#10;
    end
  else if b64 then
    DecodeBase64(s)
  else
    if add_cr_lf then {AddCrlf} s:=s+#13#10;
end;


{ vollstndige RFC-1522-Decodierung }

procedure MimeIsoDecode(var ss:string; maxlen:integer);
var p1,p2,p,i : integer;
    lastEW,
    nextW     : integer;
    code      : char;
    s         : string;
    cset      : string[20];
begin
  for i:=1 to length(ss) do
    if ss[i]=#9 then ss[i]:=' ';

  cset:='';
  p1:=0;
  lastEW:=0;
  repeat
    repeat
      p1:=posn('=?',ss,p1+1);
      if p1>0 then begin
        p2:=p1+2;
        i:=0;
        while (i<3) and (p2<length(ss)) do begin
          if ss[p2]='?' then inc(i)
          else if ss[p2]=' ' then break;
          inc(p2);
        end;
        if (i<3) or (ss[p2]<>'=') then p2:=0 else dec(p2);
      end;
    until (p1=0) or (p2>0);

    if (p1>0) and (p2>0) then begin
      if (lastEW>0) and (lastEW<nextW) and (p1=nextW) then begin
        nextW:=nextW-lastEW;
        delete(ss,lastEW,nextW);
        dec(p1,nextW);
        dec(p2,nextW);
      end;
      s:=copy(ss,p1+2,p2-p1-2);
      delete(ss,p1,p2-p1+2);
      p:=cpos('?',s);
      if p>0 then begin
        cset:=lstr(left(s,p-1));
        delete(s,1,p);
        p:=cpos('?',s);
        if p=2 then begin
          code:=UpCase(s[1]);
          delete(s,1,2);
          case code of
            'Q' : begin
                    for i:=1 to length(s) do
                      if s[i]='_' then s[i]:=' ';
                    s:=s+'=';
                    UnquotePrintable(s,true,false,false);
                  end;
            'B' : UnquotePrintable(s,false,true,false);

          end;
        end;
      end;
      CharsetToIBM(cset,s);
      insert(s,ss,p1);
      lastEW:=p1+length(s);
      nextW:=lastEW;
      while (nextW<length(ss)) and (ss[nextW]=' ') do inc(nextW);
    end;
  until (p1=0) or (p2=0);

  if length(ss)>maxlen then ss[0]:=char(maxlen);
  if cset='' then ISO2IBM(ss,cs_iso8859_1);  { ISO-Decode wenn kein RFC1522 }
  for i:=1 to length(ss) do
    if ss[i]<' ' then ss[i]:=' ';
end;


end.

{
  $Log: mimedec.pas,v $
  Revision 1.8  2005/01/01 11:16:28  mw
  MW: - Willkommen im Jahr 2005

  Revision 1.7  2004/01/09 16:18:57  mw
  MW: - Wir haben jetzt 2004!!

  Revision 1.6  2003/10/13 22:45:12  my
  MY:- Durch Diffen/Patchen verrutschtes 'uses xpovl' an richtige Stelle
       verlagert.

  Revision 1.5  2003/10/13 21:14:29  my
  MY:
  - Decodierung von quoted-printable-, base64-, und/oder UTF-7/8-codierten
    Textteilen in Multipart-Nachrichten korrigiert und deutlich robuster
    gestaltet, speziell im Zusammenhang mit langen Zeilen > 255 Zeichen:
    ----------------------------------------------------------------------
    1. Bei quoted-printable-codierten Zeilen mit mehr als 255 Zeichen
       wurde im Lister nur der Teil bis zu der Stelle angezeigt, die sich
       *vor* der qp-Decodierung an Pos. 255 der Zeile befand. Die bisheri-
       ge Behandlung langer Zeilen ging offenbar davon aus, da sich alle
       Mail- und Newsreader an das RFC-Limit von max. 76 Zeichen bei qp-
       codierten Zeilen halten, was in der Praxis ganz offenbar nicht der
       Fall ist.
       Die neue Routine decodiert jetzt beliebig lange qp-Zeilen korrekt
       und beachtet dabei auch alle theoretisch vorkommenden Sonderflle
       (Pos. 255 bzw. ein beliebiges Vielfaches davon oder ein "Softbreak"
       befinden sich mitten in einem codierten Zeichen, Zeichen "=" befin-
       det sich an Pos. 255, darf aber nicht als Softbreak fehlinterpre-
       tiert werden usw.).
    2. Die UTF-8-Decodierung besa grundstzlich bereits eine Vorkehrung
       gegen solche Sonderflle von "zerrissenen" codierten Zeichen (wie
       sie gerne bei zustzlich base64-codierten Texten entstehen), sie
       funktionierte im Falle langer Zeilen mit mehr als 255 Zeichen
       jedoch nur dann zuverlssig, wenn die Gesamtlnge der Zeile nicht
       mehr als 504 Zeichen betrug (ansonsten entstand Zeichenverlust).
       Auch hier werden jetzt beliebig lange Zeilen korrekt untersttzt.
    3. UTF-8-codierte Texte, die zustzlich qp-codiert waren und bei denen
       sich ein Softbreak "=" mitten in einem UTF-8-codierten Zeichen
       befand (was zulssig ist), wurden nicht korrekt decodiert, weil das
       Softbreak-Zeichen nach der qp-Decodierung und vor der bergabe des
       Strings an die UTF-8-Decodierroutine nicht entfernt wurde.
    4. Die Decodierung von base64-codierten Textteilen erfolgt jetzt immer
       "zeilenweise", d.h. anders als bisher knnen nie CRLFs (oder andere
       Zeilenabschlsse) mitten im zu decodierenden String enthalten sein.
       Dadurch werden jetzt auch UTF-8- und base64-codierte Texte, die
       durch CRLF zerrissene UTF-8-Zeichen enthalten, korrekt decodiert.
    5. Die UTF-7-Decodierung kann jetzt auch mit langen Zeilen > 255
       Zeichen sowie mit durch Softbreaks oder CRLFs zerrissenen codierten
       Zeichen umgehen; es gilt sinngem dasselbe wie bereits oben fr
       die UTF-8-Decodierung ausgefhrt.
    Anmerkung: Diese nderungen beheben noch nicht die gleichartigen
    Probleme, die zum Teil noch beim Beantworten und Weiterleiten von
    Nachrichten bestehen. Dort kann nach wie vor Zeichenverlust entstehen.

  Revision 1.4  2003/08/30 23:14:54  my
  MY+JG:
  - Zeichensatz-Behandlung und -Konvertierung komplett berarbeitet und
    Vorbereitungen fr Euro-Support getroffen:
    ----------------------------------------------------------------------
    1. Fix: Zeichensatz-Konvertiertabellen ISO<=>IBM fr ein- und ausge-
       hende Nachrichten komplett berarbeitet, nderungen und Korrekturen
       bei fast 50 (ISO=>IBM) bzw. 20 (IBM=>ISO) Zeichen vorgenommen
       (siehe Tabellen in MIMEDEC.PAS):
       a) Prinzipiell werden Zeichen, die im Zielzeichensatz nicht exi-
          stieren, statt nach optischen Kriterien jetzt danach konver-
          tiert, wie sie ausgesprochen werden bzw. welche Bedeutung ein
          Symbol hat. So werden z.B. die Zeichen #222 und #254 in ISO1
          (groer und kleiner Buchstabe "Thorn") nicht mehr in ein "P"
          bzw. "p" konvertiert, nur weil das mit viel Phantasie halbwegs
          hnlich aussieht, sondern in ein "T" bzw. "t".
       b) Unkonvertierbare Zeichen, fr die es keine sinnvolle Translite-
          ration im Zielzeichensatz gibt, werden jetzt in das Blockgrafik-
          zeichen #177 (ISO=>IBM) bzw. in einen Punkt (IBM=>ISO) konver-
          tiert, statt als Zeichenwert 1:1 durchgereicht zu werden. So mu
          man bei eingehenden Nachrichten nicht mehr rtseln, ob es sich
          bei dem Zeichen, das man sieht, wirklich um ein korrekt konver-
          tiertes oder doch nur um ein nicht konvertierbares Zeichen
          handelt, und bei ausgehenden Nachrichten werden durch den Punkt
          (statt Leerzeichen) unsinnige Zeilenum-brche vermieden.
       c) Steuerzeichen im Bereich #0-#31 werden bei ausgehenden Nachrich-
          ten (IBM=>ISO) jetzt entweder ebenfalls in einen Punkt oder in
          ein sinnvolles Ersetzungszeichen konvertiert, mit Ausnahme der
          Zeichen #9 (HT), #10 (LF), #12 (FF) und #13 (CR), die unvern-
          dert durchgereicht werden, sowie dem Zeichen #0, das in ein
          Leerzeichen konvertiert wird. (Anmerkung: Es ist zu berlegen,
          Steuerzeichen 1:1 durchzureichen, da sie im Unterschied zu
          ZConnect bei RFC prinzipiell erlaubt sind.)
    2. Unicode-Untersttzung (UTF-7 und UTF-8) fr den von XP lokal ver-
       wendeten Zeichensatz CP437 erweitert: Alle 75 Zeichen, die in
       CP437, nicht aber in ISO-8859-1 existieren (z.B. Block- und Rahmen-
       grafikzeichen), werden jetzt von UTF-7/8 direkt in das entsprechen-
       de Zeichen aus CP437 korrekt konvertiert, statt wie bisher durch
       ein Fragezeichen reprsentiert zu werden.
    3. Untersttzung weiterer Zeichenstze implementiert:
         ISO-8859-2  (osteuropische Variante von ISO-8859-1)
         ISO-8859-9  (trkische Variante von ISO-8859-1)
         ISO-8859-15 (ISO-8859-1 mit Euro-Symbol und einigen anderen
                      Abweichungen)
         CP850       (DOS-Codepage 850)
         CP858       (DOS-Codepage 850 mit Euro-Symbol)
       Es werden alle bei der IANA registrierten Aliasnamen dieser
       Zeichenstze untersttzt.
    4. Einige IANA-Aliasnamen fr die von XP untersttzten Zeichenstze
       ergnzt und auf den neuesten Stand gebracht.
    5. Fix: Nachrichten, die in einem ungltigen oder von XP nicht unter-
       sttzten Zeichensatz vorliegen (ISO-8859-5 Kyrillisch, ISO-8859-6
       Arabisch o.a.), werden jetzt nicht mehr blind auf Basis der in
       diesen Fllen unzutreffenden ISO1-Tabelle "kaputtkonvertiert",
       sondern im Originalzustand belassen. So knnen sie notfalls noch
       manuell dechiffriert werden.
    6. Beim Weiterleiten und Bearbeiten von MIME-Multipart-Nachrichten mit
       Textpart-Anhngen wird jetzt nur noch der Zeichensatz ISO-8859-1
       einer Konvertierung nach CP437 und von dort wieder nach ISO-8859-1
       unterzogen (nicht jedoch mehr andere von XP bei eingehenden Nach-
       richten untersttzte ISO- oder sonstige Zeichenstze). Dasselbe
       gilt fr die DOS-Codepage 850, die bisher flschlicherweise mit
       CP437 gleichgesetzt wurde.
    7. Vorbereitungen fr Euro-Support getroffen:
       a) Bei eingehenden Nachrichten, die ein Euro-Symbol enthalten, wird
          dieses von dem zeichensatzspezifischen Wert in das Zeichen ""
          (#238 in CP437) konvertiert, statt wie bisher als Zeichenwert
          1:1 durchgereicht zu werden (wodurch man je nach Zeichensatz ein
          "", "", "" oder "x" beim Lesen im Lister sah). Dabei werden
          die folgenden Euro-fhigen Zeichenstze bzw. Codierungen unter-
          sttzt:
            ISO-8859-15   (Euro-Symbol auf Pos. #164)
            Windows-1252  (Euro-Symbol auf Pos. #128)
            CP858         (Euro-Symbol auf Pos. #213)
            UTF-7         (Unicode-Zeichen)
            UTF-8         (Unicode-Zeichen)
       b) Manche Mail-/Newsreader (speziell Outlook Express) deklarieren
          mitunter gar keinen oder den Zeichensatz ISO-8859-1, verwenden
          in Wirklichkeit aber den Zeichensatz Windows-1252 und daher den
          Euro dort auch auf Pos. 128. Daher wird auch bei Nachrichten
          ohne Zeichensatzdeklaration oder im Zeichensatz ISO-8859-1 das
          dort eigentlich reservierte Zeichen #128 in das griechische
          Epsilon konvertiert, obwohl dieses Vorgehen strenggenommen nicht
          100%ig korrekt ist. XP hatte aber im Sinne der Fehlertoleranz
          schon immer "OjE-Fixes" dieser Art, insofern ist dieses Vorgehen
          nur konsequent.
       Hinweis: Damit ist noch *keine* vollstndige Untersttzung des
       -------- Euro-Symbols und des Zeichensatzes ISO-8859-15 bei
                *ausgehenden* Nachrichten implementiert - es wird daher
                unbedingt empfohlen, wie bisher beim Quoten und Verfassen
                von Nachrichten die Zeichenkette "EUR" zu verwenden.

  Revision 1.3  2003/07/30 23:09:49  my
  MY:- Source-Header auf "FreeXP" aktualisiert, einige Detailkorrekturen
       an CVS-Logs vorgenommen und hier und da CVS-Loginfos implementiert.

  Revision 1.2  2003/06/25 17:26:42  tw
  auto-de-branching

  Revision 1.1.2.11  2002/04/20 11:35:00  my
  JG+RB+MY:- Routinen fr IANA-Zeichensatz-Aliase erweitert und optimiert.

  Revision 1.1.2.10  2002/04/18 22:16:49  my
  JG+MY:- Untersttzung aller derzeit bei der IANA registrierten Alias-
          Namen fr die von XP bei eingehenden Nachrichten untersttzten
          Zeichenstze (US-ASCII, ISO-8859-x und Windows-1252)
          implementiert.

  Revision 1.1.2.9  2002/04/13 21:27:39  my
  MY:- Letzten Commit wegen IBM-Umlauten erstmal wieder rckgngig gemacht.

  Revision 1.1.2.8  2002/04/13 20:08:48  my
  RB:- Default-Decodierung bei RFC-Nachrichten ohne Zeichensatzdeklaration
       von ISO-8859-1 auf Windows-1252 gendert.

  Revision 1.1.2.7  2002/04/13 10:19:14  my
  RB[+MY]:- Commit mit demselben Zweck wie der vorherige, diesmal mit
            der Pascal-Variante von RB.

  Revision 1.1.2.6  2002/04/13 10:01:20  my
  JG[+MY]:- MIME-Decodierung nach RFC 1522 an RFC 2822/2047 (whitespace
            zwischen "multiple 'encoded word's", auch bei Folding)
            angepat und Erkennung von 'encoded word's verbessert. Korrekt
            decodiert werden jetzt z.B.:

    Subject: =?iso-8859-1?Q?=5BNoten=5D_=5BBuffy=5D_5abb18_=22Der_Zorn_der_G=F6t?=
            =?iso-8859-1?Q?tin=22_=28Intervention=29?=

    Subject: Test =? RFC 1522 =?ISO-8859-1?Q?=E4=F6=FC?= hehe ?=
    Subject: Test ?Q? =?ISO-8859-1?Q?=E4=F6=FC?= hoho

            Der hiermit committete Code ist die ASM-Variante von JG. Der
            Commit dient momentan nur dem Zweck der Archivierung.  Siehe
            auch nchster Commit.

  Revision 1.1.2.5  2002/03/27 19:48:28  my
  RB+MY:- Fehlertolerantere Fassung fr die Decodierung von nach RFC 1522
          codierten Headerzeilen: Bei Leerzeichen im codierten String wird
          nicht mehr abgebrochen (relevant z.B. bei VSOUP).

  Revision 1.1.2.4  2002/03/22 20:14:37  my
  RB:- base64-Decodierung korrigiert und optimiert.

  Revision 1.1.2.3  2002/03/15 12:39:28  my
  MY:- Ungltige base64-Strings werden nicht mehr decodiert, dadurch
       z.B. PktXCode-Meldungen wieder lesbar (Fix re-implementiert).

  Revision 1.1.2.2  2002/03/14 17:01:21  my
  RB:- Fix fr MimeIsoDecode:
       "Test =? RFC 1522 =?ISO-8859-1?Q?=E4=F6=FC?= hehe ?=" wurde nicht
       richtig decodiert. Korrekt decodiert mu das so aussehen:
       "Test =? RFC 1522  hehe ?="

  Revision 1.1.2.1  2002/03/13 23:01:51  my
  RB[+MY]:- Gesamte Zeichensatzdecodierung und -konvertierung entrmpelt,
            von Redundanzen befreit, korrigiert und erweitert:
            - Alle Decodier- und Konvertierroutinen in neue Unit
              MIMEDEC.PAS verlagert.
            - Nach RFC 1522 codierte Dateinamen in Attachments werden
              jetzt decodiert (XPMIME.PAS).
            - 'MimeIsoDecode' kann jetzt auch andere Zeichenstze als
               ISO-8859-1 konvertieren. Daher erfolgt bei nach RFC 1522
               codierten Headerzeilen im Anschlu an die qp- oder base64-
               Decodierung keine starre Konvertierung von ISO-8859-1 mehr,
               sondern es wird der deklarierte Zeichensatz korrekt
               bercksichtigt.
            - Untersttzung fr Zeichenstze ISO-8859-15 und Windows-1252
              implementiert.

}
