/*
 * CMMOVE.C
 *
 *  This file is part of DOSZIP
 *  Copyright (c) 1996 Hjort Nidudsson.
 */

#include <io.h>
#include <errno.h>
#include <string.h>
#include <ifsmgr.h>
#include <dir.h>
#include <progress.h>
#include <dzmain.h>
#include <arch.h>

int __copyfile(long __size, int __date, int __time, char __attrib);
int copy_file(const char *__path, wfblk *__fblock);
int copy_subdir(const char *__path);

static int jump;
static int movefile(FBLK *);
static int dofile(const char *__path, wfblk *__fblock);
static int dodirectory(const char *__path);
static int fbdofile(FBLK *);
static int fbdodirectory(FBLK *);

int movefile(FBLK *b)
{
	int result;

	if (progress_set(__srcfile, __outfile, b->size))
		return _ER_USERABORT;
	if (rename(__srcfile, __outfile)) {
		result = __copyfile(b->size, b->date, b->time, b->flag);
		if (result == 0 && cpy_jump == 0) {
			if (ISRDONLY(b->flag))
				setfattr(__srcfile, 0);
			remove(__srcfile);
		} else if (result == 0) {
			cpy_jump = 0;
			jump++;
		} else {
			return -1;
		}
	}
	return 0;
}

int fbdofile(FBLK *b)
{
	strfcat(__srcfile, __srcpath, b->name);
	strfcat(__outfile, __outpath, b->name);
	return movefile(b);
}

int fbdodirectory(FBLK *b)
{
	int result;
	char subdir[WMAXPATH];

	if (progress_set(b->name, __outpath, 0))
		return _ER_USERABORT;

	strfcat(__srcfile, __srcpath, b->name);
	strfcat(__outfile, __outpath, b->name);
	if (rename(__srcfile, __outfile)) {
		strfcat(subdir, __srcpath, b->name);
		fp_directory = copy_subdir;
		result = scansub(subdir, cp_stdmask, 1);
		if (result == 0 && cpy_jump == 0) {
			fp_directory = dodirectory;
			return scansub(subdir, cp_stdmask, 0);
		} else if (result == 0) {
			cpy_jump = 0;
			jump++;
			return 0;
		}
		return -1;
	}
	return 0;
}

int dofile(const char *p, wfblk *wf)
{
	int result;

	result = copy_file(p, wf);
	if (result == 0 && cpy_jump == 0) {
		if (ISRDONLY(wf->attrib))
			setfattr(__srcfile, 0);
		remove(__srcfile);
	} else if (result == 0) {
		cpy_jump = 0;
		jump++;
	}
	return result;
}

int dodirectory(const char *p)
{
	if (rmdir(p) && jump == 0) {
		ermsg(cp_edel, cp_emdelsub, p, sys_errlist[errno]);
		return -1;
	}
	return 0;
}

int cmmove(void)
{
	FBLK *b;

	if (!(b = cpanel_findfirst()))
		return 0;
	if (ISARCHIVE(b->flag)) {
		notsup();
		return 0;
	}
	if (cpy_initpath(b, 0) == 0)
		return 0;
	if (cpy_flag & CPY_OUTARCH)
		return notsup();
	jump = 0;
	fp_fileblock = dofile;
	progress_open(cp_move, cp_move);
	if (ISSELECTED(b->flag)) {
		do {
			if (ISSUBDIR(b->flag)) {
				if (fbdodirectory(b))
					break;
			} else if (fbdofile(b)) {
				break;
			}
			b->flag &= ~(_A_SELECTED);
		} while ((b = panel_ffselected(cpanel)) != NULL);
	} else {
		if (ISSUBDIR(b->flag))
			fbdodirectory(b);
		else
			movefile(b);
	}
	progress_close();
	if (panel_state(&panela))
		panel_update_fcb(&panela);
	if (panel_state(&panelb))
		panel_update_fcb(&panelb);
	return 1;
}