/*
 *      helpcomp.c
 *
 *      Copyright 2009 Blair <blair@blair-laptop>
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *      MA 02110-1301, USA.
 */

#include "helpcomp.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#define memtell()   ( p - ( char * )hfbuf )

#define TAG_TITLE       0
#define TAG_PAGE        1
#define TAG_STARTPAGE   2
#define TAG_ENDPAGE     3
#define TAG_LINK        4
#define TAG_NOT         -1
#define TAG_MAX         5

#define HLP_DECOMP      0x01

void *hfbuf;
void *binbuf;
char *pos = NULL;
FILE *inf = NULL;
FILE *outf = NULL;
long fsize;

char *tags[] = { "<title>", "<page>", "<startpage>", "<endpage>", "<link>" };

struct help_index       out_header;
struct help_link_tofile*out_links = NULL;
struct help_link       *out_links_real = NULL;
struct help_page_entry *out_pages = NULL;
struct help_link_binary out_blink = { 0, 0, 0 };
int                     out_nlinks;
int                     out_npages;

struct arg_list {
    char *arg_name;
    int   arg_flag;
};

struct arg_list helpcomp_args[] = {
    { "D",      HLP_DECOMP  }, { "H",      -1          },
    { "?",      -1          }, { NULL,     0           }
};

int isarg( int ch )
{
    return( ch == '/' || ch == '-' );
}

/* Parse arguments MS-DOS style */
int parseopts( char *arg, struct arg_list alist[] )
{
    char *a;
    int i;
    int retval = 0;

    if( isarg( *arg ) ) {
        while( *arg ) {
            arg++;
            for( a = arg; *a && !isarg( *a ); a++ );
            for( i = 0; ; i++ ) {
                if( !alist[ i ].arg_name ) {
                    printf( "Illegal argument: %s\n", arg - 1 );
                    return( -1 );
                }
                if( !strncasecmp( alist[ i ].arg_name, arg, a - arg ) ) {
                    retval |= alist[ i ].arg_flag;
                    break;
                }
            }

            arg = a;
        }
    }

    return( retval );
}

char *memreadline( char *buf )
{
    char *p = pos, *b = buf;

    while( p && *p == '\n' ) p++;
    if( pos == NULL || memtell() >= fsize ) {
        pos = hfbuf;
        return( NULL );
    }
    while( memtell() < fsize && *p != '\n' ) *b++ = *p++;
    pos = p;

    *b = '\0';

    return( buf );
}

int name_used( const char *link_name, int type )
{
    int i, r, max = type ? out_npages : out_nlinks;

    for( i = 0; i < max; i++ ) {
        if( type ) r = strcasecmp( link_name, out_pages[ i ].hent_title );
        else       r = strcasecmp( link_name, out_links[ i ].hlink_name );
        if( !r ) return( i + 1 );
    }

    return( 0 );
}

int readtag( const char *tag, char *outbuf )
{
    int i, len;
    char *p;

    if( !( *tag ) ) return( TAG_NOT );
    for( i = 0; i < TAG_MAX; i++ ) {
        p = tags[ i ];
        len = strlen( p );
        if( !strncasecmp( p, tag, len ) ) {
            if( outbuf ) strcpy( outbuf, ( tag + len ) );
            break;
        }
    }

    if( i == TAG_MAX ) i = TAG_NOT;

    return( i );
}

int main( int argc, char **argv )
{
    int i, j, k, l, r = -1, opts = 0;
    char *infile, *outfile;
    int pages, links, endpages, titles, startpages, inpage, error;
    char readbuf[ 160 ], readval[ HELP_MAXNAME ], *m;
    long startpos, bread;

    for( i = 1; i < argc; i++ ) {
        r = parseopts( argv[ i ], helpcomp_args );
        if( !r ) break;
        if( r == -1 ) break;
        opts |= r;
    }

    if( r == -1 || ( argc - i ) % 2 ) {
        printf( "HELPCOMP [/options] filein1 fileout1 [filein2 fileout2 [...]]\n" );
        printf( "  filein:  Uncompiled Help File\n" );
        printf( "  fileout: Output file for Compiled Help File\n" );
        printf( "Options:\n" );
        printf( "  /D       Decompile filein to fileout\n" );
        return( 1 );
    }

    for( ; i < argc; i += 2 ) {
        if( inf ) fclose( inf );
        if( outf ) fclose( outf );
        inf = outf = NULL;
        infile = argv[ i ];
        outfile = argv[ i + 1 ];
        if( !( inf = fopen( infile, "rt" ) ) || !( outf = fopen( outfile, opts & HLP_DECOMP ? "wt" : "wb" ) ) ) {
            printf( "Couldn't open %s and %s for reading and writing\n", infile, outfile );
            continue;
        }

        if( opts & HLP_DECOMP ) { /* Decompiling is the easy part */
            HELP *hlp_input;
            char *pagedata;

            fclose( inf );
            inf = NULL;
            if( ( hlp_input = help_open( infile ) ) == NULL ) {
                printf( "Couldn't open %s for reading\n", infile );
                continue;
            }

            printf( "Decompiling %s\n", infile );
            fprintf( outf, "<title>%s\n\n", help_title( hlp_input ) );
            r = help_pages( hlp_input );
            for( j = 0; j < r; j++ ) {
                fprintf( outf, "<page>%s\n", help_pagetitle( hlp_input, j ) );
                if( help_spage( hlp_input ) == j ) fprintf( outf, "<startpage>\n" );
                if( ( pagedata = m = help_readpage( hlp_input, j ) ) ) for( k = 0; k < help_pagesize( hlp_input, j ); k++ ) {
                    if( *m ) fputc( *m++, outf );
                    else {
                        fprintf( outf, "\n<link>%s\n", help_pagetitle( hlp_input, help_link_to_page( hlp_input, m ) ) );
                        m += 4; /* Skip over the binary link-info structure */
                        k += 3; /* k will be incremented by 1 already */
                    }
                }
                if( pagedata ) free( pagedata );
                fprintf( outf, "\n<endpage>\n\n" );
            }

            continue;
        }

        fseek( inf, 0L, SEEK_END );
        fflush( inf );
        fsize = ftell( inf );
        fseek( inf, 0L, SEEK_SET );

        if( ( hfbuf = malloc( fsize ) ) == NULL ) {
            printf( "Couldn't open a buffer big enough to read %s\n", infile );
            continue;
        }

        bread = fread( hfbuf, 1, fsize, inf );
        if( ftell( inf ) != fsize ) {
            printf( "Couldn't read %s\n", infile );
            continue;
        }
        fsize = bread;
        pos = hfbuf;

        printf( "Compiling %s\n", infile );
        /* Start Pass #1 - Collect number of pages, number of links, and simple errors */
        pages = links = endpages = titles = startpages = inpage = error = l = 0;
        while( l++, memreadline( readbuf ) ) switch( readtag( readbuf, NULL ) ) {
            case TAG_TITLE:
                if( inpage ) {
                    printf( "%s: line %i: <title> cannot be set while in a page\n", infile, l );
                    error++;
                }
                titles++;
                break;
            case TAG_PAGE:
                if( inpage ) {
                    printf( "%s: line %i: <page> cannot be given when already parsing a page\n", infile, l );
                    error++;
                }
                inpage++;
                pages++;
                break;
            case TAG_ENDPAGE:
                if( !inpage ) {
                    printf( "%s: line %i: <endpage> cannot be given when not parsing a page\n", infile, l );
                    error++;
                }
                inpage--;
                break;
            case TAG_STARTPAGE:
                if( !inpage ) {
                    printf( "%s: line %i: <startpage> cannot be set when not parsing a page\n", infile, l );
                    error++;
                }
                startpages++;
                break;
            case TAG_LINK:
                if( !inpage ) {
                    printf( "%s: line %i: <link>s cannot be added when not parsing a page\n", infile, l );
                    error++;
                }
                links++;
                break;
        }

        if( error ) continue;
        if( startpages > 1 ) {
            printf( "%s: Cannot set more than one startpage\n", infile );
            continue;
        }

        if( titles != 1 ) {
            printf( "%s: Title set %i times; it should be set once\n", infile, titles );
            continue;
        }

        if( inpage ) {
            printf( "%s: missing <endpage>\n", infile );
            continue;
        }

        /* Start Pass #2 - Create a table of page names and link names, fill in header components */
        if( out_links ) free( out_links );
        if( out_pages ) free( out_pages );
        if( out_links_real ) free( out_links_real );
        out_links = malloc( links * sizeof( struct help_link_tofile ) );
        out_pages = malloc( pages * sizeof( struct help_page_entry ) );
        out_links_real = malloc( links * sizeof( struct help_link ) );

        if( out_links == NULL || out_pages == NULL || out_links_real == NULL ) {
            printf( "%s: out of memory\n", infile );
            continue;
        }

        memset( out_links, 0, links * sizeof( struct help_link_tofile ) );
        memset( out_links_real, 0, links * sizeof( struct help_link ) );
        memset( out_pages, 0, pages * sizeof( struct help_page_entry ) );
        memset( &out_header, 0, sizeof( struct help_index ) );
        l = 0;

        while( l++, memreadline( readbuf ) ) switch( readtag( readbuf, readval ) ) {
            case TAG_TITLE:
                strcpy( out_header.help_title, readval );
                break;
            case TAG_PAGE:
                if( name_used( readval, 1 ) ) {
                    printf( "%s: line %i: page title \"%s\" already used\n", infile, l, readval );
                    error++;
                }
                inpage++;
                strcpy( out_pages[ out_npages ].hent_title, readval );
                out_pages[ out_npages ].hent_size = 0;
                out_npages++;
                break;
            case TAG_ENDPAGE:
                inpage--;
                break;
            case TAG_STARTPAGE:
                out_header.help_startpage = out_npages - 1;
                break;
            case TAG_LINK:
                out_pages[ out_npages - 1 ].hent_size += 4;
                out_pages[ out_npages - 1 ].hent_nlinks++;
                if( name_used( readval, 0 ) ) break;
                strcpy( out_links[ out_nlinks ].hlink_name, readval );
                out_nlinks++;
                break;
            default:
                if( !inpage ) break;
                out_pages[ out_npages - 1 ].hent_size += strlen( readbuf ) + 1;
        }

        if( error ) continue;
        /* Check for links referencing non-existant pages; fill-in the rest of the link structures */
        for( j = 0; j < out_nlinks; j++ ) {
            m = out_links[ j ].hlink_name;
            if( !( r = name_used( m, 1 ) ) ) {
                printf( "%s: <link> used to reference non-existant page: %s\n", infile, m );
                error++;
            } else out_links_real[ j ].hlink_page = r - 1;
        }
        if( error ) continue;
        /* Fill-in the rest of the page structures */
        startpos = sizeof( struct help_index ) + ( out_nlinks * sizeof( struct help_link ) ) + ( out_npages * sizeof( struct help_page_entry ) );
        for( j = 0; j < out_npages; j++ ) {
            out_pages[ j ].hent_start = startpos;
            startpos += out_pages[ j ].hent_size;
        }
        /* Fill-in the rest of the index structure */
        out_header.help_npages = out_npages;
        out_header.help_nlinks = out_nlinks;
        /* Finally write the binary/compiled file */
        fwrite( &out_header, sizeof( struct help_index ), 1, outf );
        fwrite( out_pages, sizeof( struct help_page_entry ), out_npages, outf );
        fwrite( out_links_real, sizeof( struct help_link ), out_nlinks, outf );

        /* Start Pass #3 - Write the actual text to the output file */
        while( memreadline( readbuf ) ) switch( readtag( readbuf, readval ) ) {
            case TAG_TITLE:
            case TAG_PAGE:
            case TAG_ENDPAGE:
            case TAG_STARTPAGE:
                break;
            case TAG_LINK:
                out_blink.hlink_num = name_used( readval, 0 ) - 1;
                fwrite( &out_blink, sizeof( struct help_link_binary ), 1, outf );
                break;
            default:
                strcat( readbuf, " " );
                fputs( readbuf, outf );
        }

        /* That's it; we're done folks */
        fflush( outf );
    }

    return( 0 );
}
