/* 
 * LTOOLSnet.cs

 Microsoft .NET C# based graphical user interface for the LTOOLS toolkit.

 Copyright (C) 2003-2006 Werner Zimmermann (Werner.Zimmermann@fht-esslingen.de)
 Joerg Holzaepfel  (joerghol@web.de)

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 For further information please see file README.TXT.

 ---History, Changes:-----------------------------------------------------------
 V1.0:  Initial version created by Joerg Holzaepfel as 'Diplomarbeit' at 
 FH Esslingen - University of Applied Sciences, Esslingen, Germany
 under the supervision of Prof. Zimmermann, based on his JAVA version.
 J. Holzaepfel, June 2003
 V1.01: Integrated in the official LTOOLS distribution, renaming everything
 to LTOOLSnet.
 W. Zimmermann, 4 July 2003
 V1.1:  Modifications to integrate ReiserFS driver RFSTOOL.
 W. Zimmermann, 14 Sept. 2003
 V1.2: Copyright update for year 2004, changed call to 'ldir' to 'ldirNT', added 'View' to context menu.
 W. Zimmermann, 20 Dec. 2003
 V1.3: Wait Cursor while executing LTOOLS commands
 W. Zimmermann, 15 Jan. 2006
 */


using System;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Data;
using System.Diagnostics;
using System.Net.Sockets;
using System.Net;
using System.IO;
using System.Text;
using System.Threading;

namespace FHTE.LTOOLS.GUI
{   /// <summary> 
    /// LTOOLSnet is the Main Class of the GUI for the LTools 
    /// </summary> 
    public class LTOOLSnet:System.Windows.Forms.Form
    {
#region Fields
	// ############################################################################################################ 
	// FIELDS 
	// ############################################################################################################ 
	// the components of the GUI 
	private System.Windows.Forms.MainMenu mainMenu;
	private System.Windows.Forms.MenuItem menuFile;
	private System.Windows.Forms.MenuItem menuEdit;
	private System.Windows.Forms.MenuItem menuHelp;
	private System.Windows.Forms.MenuItem menuItemNew;
	private System.Windows.Forms.MenuItem menuItemModify;
	private System.Windows.Forms.MenuItem menuItemDelete;
	private System.Windows.Forms.MenuItem menuItemCopy;
	private System.Windows.Forms.MenuItem menuItemPaste;
	private System.Windows.Forms.MenuItem menuItemPartitions;
	private System.Windows.Forms.MenuItem menuItemDebug;
	private System.Windows.Forms.MenuItem menuItemHelp;
	private System.Windows.Forms.MenuItem menuItemAbout;
	private System.Windows.Forms.MenuItem menuItemConfirmDelete;
	private System.Windows.Forms.MenuItem menuItemExit;
	private System.Windows.Forms.MenuItem menuItemRemote;
	private System.Windows.Forms.MenuItem menuItemReadOnly;
	private System.Windows.Forms.MenuItem menuItemDosPattern;
	private System.Windows.Forms.MenuItem menuItemLinuxPattern;
	private System.Windows.Forms.MenuItem menuItemAddDosDrive;
	private System.Windows.Forms.MenuItem menuItemAddLinuxDrive;
	private System.Windows.Forms.MenuItem menuItemSeparator1;
	private System.Windows.Forms.MenuItem menuItemSeparator2;
	private System.Windows.Forms.MenuItem menuItemSeparator3;
	private System.Windows.Forms.MenuItem menuItemSeparator4;
	private System.Windows.Forms.MenuItem menuItemSeparator5;
	private System.Windows.Forms.MenuItem menuItemSeparator6;
	private System.Windows.Forms.Panel explorerSplitPanel;
	private System.Windows.Forms.Splitter filePanelSplitter;
	private System.Windows.Forms.Panel panelCenter;
	private System.Windows.Forms.Panel panelButtons;
	private System.Windows.Forms.Panel panelToolbar;
	private System.Windows.Forms.ImageList toolbarImageList;
	private System.Windows.Forms.ToolBar toolBar;
	private System.Windows.Forms.ToolBarButton toolBarButtonNew;
	private System.Windows.Forms.ToolBarButton toolBarButtonModify;
	private System.Windows.Forms.ToolBarButton toolBarButtonDelete;
	private System.Windows.Forms.ToolBarButton toolBarButtonCopy;
	private System.Windows.Forms.ToolBarButton toolBarButtonPaste;
	private System.Windows.Forms.ToolBarButton toolBarButtonSeparator1;
	private System.Windows.Forms.ToolBarButton toolBarButtonPartitions;
	private System.Windows.Forms.ToolBarButton toolBarButtonConnect;
	private System.Windows.Forms.ToolBarButton toolBarButtonSeparator2;
	private System.Windows.Forms.ToolBarButton toolBarButtonHelp;
	private System.Windows.Forms.ToolBarButton toolBarButtonAbout;
	private System.Windows.Forms.ToolBarButton toolBarButtonExit;
	private System.Windows.Forms.ToolBarButton toolBarButtonReadOnly;
	private System.Windows.Forms.ToolBarButton toolBarButtonConfirmDelete;
	private System.Windows.Forms.ToolBarButton toolBarButtonSeparator3;
	private System.Windows.Forms.StatusBar statusBar;
	private System.Windows.Forms.StatusBarPanel statusBarPanelLinuxReadOnly;
	private System.Windows.Forms.StatusBarPanel statusBarPanelConfirmDelete;
	private System.Windows.Forms.StatusBarPanel statusBarPanelDosPattern;
	private System.Windows.Forms.StatusBarPanel statusBarPanelLinuxPattern;
	private System.Windows.Forms.StatusBarPanel statusBarPanelFree;
	private FHTE.LTOOLS.GUI.DirFilePanel linuxFilePanel;
	private FHTE.LTOOLS.GUI.DirFilePanel dosFilePanel;
	private System.ComponentModel.IContainer components;

	private static LTOOLSnet myProg = null;			// Reference to the main-class 
	private const string Version = "V1.3";			// Version of the LTOOLSNET C# 
	private const string Caption =				// The caption of the Main-Form 
	"LTOOLSnet-C# " + Version;									//
        private string partitions = "" ;                        // this string saves the output of the partitions-Button
	// in the toolbar, to minimize read operations on the HD 
	private double sizeFactor = 0.5;			// this is a variable to set the size of the 
	// dosFilePanel, when the form was resized 
	private static int splashDuration = 2000;		// duration of the splash-screen to be shown 
	private static string lreadguiDirectory =		// the path of the LTOOLSnet.exe-file 
	AppDomain.CurrentDomain.BaseDirectory;			// "Environment.CurrentDirectory" gets the 
	// Directory, from which the Process started !!! 
	private static bool confirmDelete = true;		// true, if you have to confirm deleting a file 
	private static bool isLinuxReadOnly = true;		// true, if the linux-partition is set to read only 
	public static int BUFSIZE = 32768;			// BUFSIZE for socket-connection 
	private static Socket socket = null;			// the socket to the remote-server 
	private static bool connected = false;			// connection flag, true if connected to remote-server 
	private static RemoteHost remoteHost =			// RemoteHost reference for current selected host 
	new RemoteHost("localhost", 1605);			// to connect in RemoteBoxDialog 
	public static MyDebug debugDialog;			// Dialog for debug-output 
	private static bool debug = false;			// true, if the Debug-mode is on (DebugDialog Window) 
	private static string dosFilePattern = "*";		// the filepattern for the DOS-files 
	private static string linuxFilePattern = "*";		// the filepattern for the Linux-files 
	// the variables for the copy and paste action 
	private static ArrayList copyFilePath = null;		// full path of the files to be copied 
	private static ArrayList copyFileName = null;		// just the name of the file/directory 
	private static bool copyFileIsLinux = false;		// the file is a linux/dos file 
	private static bool copyFileIsValid = false;		// true, if there is a file to copy 
	private static HelpBox helpBox = null;			// reference to a HelpBox 
	private static RemoteBoxDialog remoteDialog = null;	// reference to a RemoteBoxDialog 
	private static Color connectedColor = Color.AliceBlue;	// the backgroundcolor of the linuxFilePanel if 
	// connected to a remote server        
	private const int IndexIconReadOnly = 11;		// Indices of icon in the toolbarImageList 
	private const int IndexIconReadWrite = 12;
	private const int IndexIconConfirm = 13;
	private const int IndexIconNoConfirm = 14;

#endregion

#region Properties
	// ############################################################################################################ 
	// PROPERTIES 
	// ############################################################################################################ 
	/// <summary> 
	/// returns a reference to the application 
	/// </summary> 
	public static LTOOLSnet MyProg
        {   get
            {   return myProg;
	    }
	}
	/// <summary> 
	/// returns the ArrayList with the full filepaths for copy and paste action 
	/// </summary> 
	public static ArrayList CopyFilePath
        {   get
            {   return copyFilePath;
	    }
	}
	/// <summary> 
	/// returns the ArrayList with the filenames for copy and paste action 
	/// </summary> 
	public static ArrayList CopyFileName
        {   get
            {   return copyFileName;
	    }
	}
	/// <summary> 
	/// returns a bool-valus if the files in ArrayList CopyFilePath are linuxfiles or not 
	/// </summary> 
	public static bool CopyFileIsLinux
        {   get
            {   return copyFileIsLinux;
	    }
	}
	/// <summary> 
	/// returns or sets a bool-value if the files to copy are valid (1 or more files in ArrayList) 
	/// </summary> 
	public static bool CopyFileIsValid
        {   get
            {   return copyFileIsValid;
	    }
	    set
            {   copyFileIsValid = value;
	    }
	}
	/// <summary> 
	/// returns the number of the files in ArrayList CopyFilePath 
	/// </summary> 
	public static int CopyFileCount
        {   get
            {   if (copyFilePath == null)
                {   return 0;
		} else
                {   return copyFilePath.Count;
		}
	    }
	}
	/// <summary> 
	/// returns a bool-value if a MessageBox should be shown to confirm deleting files 
	/// </summary> 
	public static bool ConfirmDelete
        {   get
            {   return confirmDelete;
	    }
	}
	/// <summary> 
	/// returns or sets a bool-value if the Linux partitions are set to readonly 
	/// </summary> 
	public static bool IsLinuxReadOnly
        {   get
            {   return isLinuxReadOnly;
	    }
	    set
            {   isLinuxReadOnly = value;
	    }
	}
	/// <summary> 
	/// returns the socket to connect to remote server 
	/// </summary> 
	public static Socket NetSocket
        {   get
            {   return socket;
	    }
	}
	/// <summary> 
	/// returns a bool-value if the application is connected to a remote server or not 
	/// </summary> 
	public static bool Connected
        {   get
            {   return connected;
	    }
	    // set { connected = value ; } 
	}
	/// <summary> 
	/// returns or sets the DebugMode 
	/// </summary> 
	public static bool Debug
        {   get
            {   return debug;
	    }
	    set
            {   debug = value;
	    }
	}
	/// <summary> 
	/// returns the string with the partitions on the local machine or on the remote server 
	/// </summary> 
	public string Partitions
        {   get
            {   return partitions;
	    }
	}
	/// <summary> 
	/// returns the search pattern for DOS files 
	/// </summary> 
	public static string DOSFilePattern
        {   get
            {   return dosFilePattern;
	    }
	}
	/// <summary> 
	/// returns the search pattern for Linux files 
	/// </summary> 
	public static string LinuxFilePattern
        {   get
            {   return linuxFilePattern;
	    }
	}
	/// <summary> 
	/// returns the directory which contains the LTollsGui.exe 
	/// </summary> 
	public static string GUIDirectory
        {   get
            {   return lreadguiDirectory;
	    }
	}
#endregion

#region Constructors
	// ############################################################################################################ 
	// CONSTRUCTORS 
	// ############################################################################################################ 
	/// <summary> 
	/// Constructor of the MainClass. Builds up the GUI 
	/// </summary> 
	public LTOOLSnet()
        {               //
			// Required for Windows Form Designer support
			//
			InitializeComponent();

			//
	    // TODO: Add any constructor code after InitializeComponent call 
			//
			if (Debug == true)
            {   debugDialog = new MyDebug() ;
                menuItemDebug.Visible = true ;
                //debugDialog.Visible = false ;
	    if (debugDialog.Visible == true)
            {   menuItemDebug.Checked = true;
	    }
	}
	LTOOLSnet.myProg = this;
	this.Text = Caption;
	DebugOutput("TempPathLinuxFiles: " + DirFilePanel.tempPathLinuxFiles);
	DebugOutput("Current Directory : " + Environment.CurrentDirectory);
	menuItemReadOnly.Checked = isLinuxReadOnly;
	toolBarButtonReadOnly.Pushed = isLinuxReadOnly;
	menuItemConfirmDelete.Checked = confirmDelete;
	toolBarButtonConfirmDelete.Pushed = confirmDelete;
	UpdateStatusToolBar();

	dosFilePanel.SetDebugDialog(debugDialog);
	dosFilePanel.FillDosDirectoryTree();

	linuxFilePanel.SetDebugDialog(debugDialog);
	linuxFilePanel.FillLinuxDirectoryTree();
    }
#endregion

#region Methods
    // ############################################################################################################ 
    // METHODS 
    // ############################################################################################################ 
    /// <summary> 
    /// Clean up any resources being used. 
    /// </summary> 
    /// <param name="disposing"></param> 
    protected override void Dispose(bool disposing)
    {   if (disposing)
        {   if (components != null)
            {   components.Dispose();
	    }
	}
	base.Dispose(disposing);
    }
    /// <summary> 
    /// the LTOOLSnet_Closing() method aborts the closing event of the form to ask if the user 
    /// really wants to quit the application in a Yes/No MessageBox. 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void LTOOLSnet_Closing(object sender, System.ComponentModel.CancelEventArgs e)
    {   // The Closing event occurs as the form is being closed. 
	// When a form is closed, all resources created within 
	// the object are released and the form is disposed. 
	// If you cancel this event, the form remains opened. 
	// To cancel the closure of a form, set the Cancel property 
	// of the CancelEventArgs passed to your event handler to 
	// true. 
	e.Cancel = true;
	// then forward the event to the menuitem exit 
	MenuItemExit_Click(sender, e);
    }
    /// <summary> 
    /// the LTOOLSnet_Resize() method sets the size of the dosFilePanel. 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void LTOOLSnet_Resize(object sender, System.EventArgs e)
    {   int height = (int) (sizeFactor * (double) explorerSplitPanel.Height);
	 dosFilePanel.Size = new Size(dosFilePanel.Size.Width, height);
    }
    /// <summary> 
    /// the FilePanelSplitter_SplitterMoved() method sets the sizefactor of the dosFilePanel, after moving 
    /// the splitter between Dos and Linux FilePanel 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void FilePanelSplitter_SplitterMoved(object sender, System.Windows.Forms.SplitterEventArgs e)
    {   // save the prozentual size of the dosFilePanel 
	// if you change the size of the main-Form, the size of the panels 
	// are changed in the prozentual way 
	sizeFactor = (double) dosFilePanel.Height / (double) explorerSplitPanel.Height;
    }
    /// <summary> 
    /// the DebugOutput() appends the string to the Text in the DebugDialog 
    /// </summary> 
    /// <param name="str">string to show in DebugOutput</param> 
    private static void DebugOutput(string str)
    {   if (Debug && debugDialog != null)
        {   debugDialog.AppendText(str + "\r\n");
	}
    }

    /// <summary> 
    /// the SetCopyFilePath() method sets the variables for copy and paste action 
    /// </summary> 
    /// <param name="path">ArrayList with the full paths of the files to be copied</param> 
    /// <param name="name">ArrayList with the names of the files to be copied</param> 
    /// <param name="isLinux">bool flag to show if the files are from a Linux-partition</param> 
    public static void SetCopyFilePath(ArrayList path, ArrayList name, bool isLinux)
    {   copyFileIsLinux = isLinux;
	copyFilePath = null;
	copyFilePath = path;
	copyFileName = null;
	copyFileName = name;
	if (CopyFileCount > 0)
        {   copyFileIsValid = true;
	}
    }

    /// <summary> 
    /// the ListViewItemFilesDelete()-method is called from the Menuitem 'Edit-Delete', the contextmenu of the 
    /// ListView in the DirFilePane and from the KeyUp-event of the ListView. 
    /// It deletes all selected items in the ListView. 
    /// </summary> 
    public void ListViewItemFilesDelete()
    {   DebugOutput("DeleteListViewItemFiles()");

	string message = "Select file or directory to delete in the listview.";
	string caption = "LTOOLS";

	if (this.ActiveControl == dosFilePanel)			// the DOSFilePanel is selected 
        {   DebugOutput("The DOSPanel is active.");
	    ListView.SelectedListViewItemCollection items = dosFilePanel.GetSelectedListViewItemFile();
	    if (items != null)
            {   dosFilePanel.DeleteDosFile(items, confirmDelete);
	    } else
            {   MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Information);
	    }
	} else if (this.ActiveControl == linuxFilePanel)	// the LinuxFilePanel is selected 
        {   DebugOutput("The LINUXPanel is active.");
	    ListView.SelectedListViewItemCollection items = linuxFilePanel.GetSelectedListViewItemFile();
	    if (items != null)
            {   linuxFilePanel.DeleteLinuxFile(items, confirmDelete);
	    } else
            {   MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Information);
	    }
	} else
        {   DebugOutput("NO Panel is active.");
	    MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Information);
	}
	confirmDelete = true;
	UpdateStatusToolBar();
    }
    /// <summary> 
    /// the ListViewItemFilesModify()-method is called from the Menuitem 'Edit-Modify', the contextmenu of the 
    /// ListView in the DirFilePane. 
    /// It modifies all selected items in the ListView. 
    /// </summary> 
    public void ListViewItemFilesModify()
    {   DebugOutput("ListViewItemFilesModify");
	string message = "Select file or directory to modify in the listview.";
	string caption = "LTOOLS";

	if (this.ActiveControl == dosFilePanel)			// the DOSFilePanel is selected 
        {   ListView.SelectedListViewItemCollection items = dosFilePanel.GetSelectedListViewItemFile();
	    if (items != null)
            {   dosFilePanel.ModifyDosFile(items);
	    } else
            {   MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Information);
	    }
	} else if (this.ActiveControl == linuxFilePanel)	// the LinuxFilePanel is selected 
        {   ListView.SelectedListViewItemCollection items = linuxFilePanel.GetSelectedListViewItemFile();
	    if (items != null)
            {   linuxFilePanel.ModifyLinuxFile(items);
	    } else
            {   MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Information);
	    }
	} else
        {   MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Information);
	}
    }
    /// <summary> 
    /// the ListViewItemFilesCopy()-method is called from the Menuitem 'Edit-Copy', the contextmenu of the 
    /// ListView in the DirFilePane. 
    /// It copies all selected items in the ListView into the ArrayLists() of the LTOOLSnet-class. 
    /// </summary> 
    public void ListViewItemFilesCopy()
    {   DebugOutput("ListViewItemFilesCopy()");

	string caption = "LTOOLS";
	// reset the variables for copy 
	 LTOOLSnet.CopyFileIsValid = false;
	 LTOOLSnet.copyFileIsLinux = false;
	 LTOOLSnet.copyFileName = null;
	 LTOOLSnet.copyFileName = new ArrayList();
	 LTOOLSnet.copyFilePath = null;
	 LTOOLSnet.copyFilePath = new ArrayList();

	 ListView.SelectedListViewItemCollection items = null;

	if (this.ActiveControl == dosFilePanel)			// the DOSFilePanel is selected 
        {   items = dosFilePanel.GetSelectedListViewItemFile();
	    copyFileIsLinux = false;
	} else if (this.ActiveControl == linuxFilePanel)	// the LinuxFilePanel is selected 
        {   items = linuxFilePanel.GetSelectedListViewItemFile();
	    copyFileIsLinux = true;
	}
	if (items == null)
        {   string message = "Select file(s) to copy.";
	    MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Information);
	    return;
	}
	foreach(ListViewItemFile item in items)
        {   if (item.IsError)					// it's an error-message 
            {   copyFileIsValid = false;
		return;
	    }
	    if (!item.FullFilePath.Equals(""))
            {   if (item.IsDirectory)				// selected ListViewItem is a directory 
                {   string message = "Sorry, directories cannot be copied.";
		    MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Error);
		    copyFileIsValid = false;
		    return;
		} else if (item.IsLink)				// selected ListViewItem is a symbolic link (linux) 
                {   DebugOutput("Link: " + item.FullFilePath + " ---- " + item.FileName);
		    copyFilePath.Add(linuxFilePanel.GetFilePathFromLink(item.FullFilePath));
		    copyFileName.Add(linuxFilePanel.GetFileNameFromLink(item.FullFilePath));
		} else
		    // selected ListViewItem is a ordinary file 
                {   copyFilePath.Add(item.FullFilePath);
		    copyFileName.Add(item.FileName);
		}
	    }
	}
	if (copyFilePath.Count > 0)
        {   copyFileIsValid = true;
	} else
        {   copyFileIsValid = false;
	}
    }
    /// <summary> 
    /// the ListViewItemFilesPaste()-method is called from the Menuitem 'Edit-Paste', the contextmenu of the 
    /// ListView in the DirFilePane. 
    /// It pastes all items in the ArrayLists() of the LTOOLSnet-class into the ListView. 
    /// </summary> 
    public void ListViewItemFilesPaste()
    {   DebugOutput("ListViewItemFilesPaste()");
	string caption = "LTOOLS";
	string targetFilePath;
	if (CopyFileIsValid == false)
        {   string message = "You must copy a file before you can paste it.";
	     MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Information);
	     return;
	}
	if ((this.ActiveControl == dosFilePanel) &&
	     (dosFilePanel.GetSelectedDirectoryNode() != null))
        {   targetFilePath = (dosFilePanel.GetSelectedDirectoryNode()).Path;
	    dosFilePanel.PasteToDos(targetFilePath, true);
	} else if ((this.ActiveControl == linuxFilePanel) &&
		   (linuxFilePanel.GetSelectedDirectoryNode() != null))
        {   targetFilePath = (linuxFilePanel.GetSelectedDirectoryNode()).Path;
	    linuxFilePanel.PasteToLinux(targetFilePath, true);
	} else
        {   string message = "Select directory to paste in.";
	    MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Information);
	}
    }

    public void ListViewItemFilesView()
    {   DebugOutput("ListViewItemFilesView");
	string message = "Select file to view";
	string caption = "LTOOLS";

	if (this.ActiveControl == dosFilePanel)			// the DOSFilePanel is selected 
	{
	    ListView.SelectedListViewItemCollection items = dosFilePanel.GetSelectedListViewItemFile();
	    if (items != null)
	    {	    foreach(ListViewItemFile item in items)
			dosFilePanel.StartDosApplication(item, false);
	    } 
	    else
	    {
		    MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Information);
	    }
	} 
	else if (this.ActiveControl == linuxFilePanel)	// the LinuxFilePanel is selected 
	{
		ListView.SelectedListViewItemCollection items = linuxFilePanel.GetSelectedListViewItemFile();
	    if (items != null)
	    {	foreach(ListViewItemFile item in items)
		    linuxFilePanel.StartLinuxApplication(item, false);
	    } 
	    else
	    {
		    MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Information);
	    }
	} 
	else
	{
		MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Information);
	}
    }

#region Windows Form Designer generated code
    /// <summary> 
    /// Required method for Designer support - do not modify 
    /// the contents of this method with the code editor. 
    /// </summary> 
    private void InitializeComponent()
    {   this.components = new System.ComponentModel.Container();
	System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(LTOOLSnet));
	this.mainMenu = new System.Windows.Forms.MainMenu();
	this.menuFile = new System.Windows.Forms.MenuItem();
	this.menuItemRemote = new System.Windows.Forms.MenuItem();
	this.menuItemSeparator3 = new System.Windows.Forms.MenuItem();
	this.menuItemReadOnly = new System.Windows.Forms.MenuItem();
	this.menuItemConfirmDelete = new System.Windows.Forms.MenuItem();
	this.menuItemSeparator4 = new System.Windows.Forms.MenuItem();
	this.menuItemDosPattern = new System.Windows.Forms.MenuItem();
	this.menuItemLinuxPattern = new System.Windows.Forms.MenuItem();
	this.menuItemSeparator6 = new System.Windows.Forms.MenuItem();
	this.menuItemAddDosDrive = new System.Windows.Forms.MenuItem();
	this.menuItemAddLinuxDrive = new System.Windows.Forms.MenuItem();
	this.menuItemSeparator5 = new System.Windows.Forms.MenuItem();
	this.menuItemExit = new System.Windows.Forms.MenuItem();
	this.menuEdit = new System.Windows.Forms.MenuItem();
	this.menuItemNew = new System.Windows.Forms.MenuItem();
	this.menuItemModify = new System.Windows.Forms.MenuItem();
	this.menuItemDelete = new System.Windows.Forms.MenuItem();
	this.menuItemCopy = new System.Windows.Forms.MenuItem();
	this.menuItemPaste = new System.Windows.Forms.MenuItem();
	this.menuHelp = new System.Windows.Forms.MenuItem();
	this.menuItemPartitions = new System.Windows.Forms.MenuItem();
	this.menuItemSeparator1 = new System.Windows.Forms.MenuItem();
	this.menuItemDebug = new System.Windows.Forms.MenuItem();
	this.menuItemHelp = new System.Windows.Forms.MenuItem();
	this.menuItemSeparator2 = new System.Windows.Forms.MenuItem();
	this.menuItemAbout = new System.Windows.Forms.MenuItem();
	this.toolbarImageList = new System.Windows.Forms.ImageList(this.components);
	this.panelToolbar = new System.Windows.Forms.Panel();
	this.toolBar = new System.Windows.Forms.ToolBar();
	this.toolBarButtonNew = new System.Windows.Forms.ToolBarButton();
	this.toolBarButtonModify = new System.Windows.Forms.ToolBarButton();
	this.toolBarButtonDelete = new System.Windows.Forms.ToolBarButton();
	this.toolBarButtonCopy = new System.Windows.Forms.ToolBarButton();
	this.toolBarButtonPaste = new System.Windows.Forms.ToolBarButton();
	this.toolBarButtonSeparator1 = new System.Windows.Forms.ToolBarButton();
	this.toolBarButtonPartitions = new System.Windows.Forms.ToolBarButton();
	this.toolBarButtonConnect = new System.Windows.Forms.ToolBarButton();
	this.toolBarButtonSeparator2 = new System.Windows.Forms.ToolBarButton();
	this.toolBarButtonReadOnly = new System.Windows.Forms.ToolBarButton();
	this.toolBarButtonConfirmDelete = new System.Windows.Forms.ToolBarButton();
	this.toolBarButtonSeparator3 = new System.Windows.Forms.ToolBarButton();
	this.toolBarButtonHelp = new System.Windows.Forms.ToolBarButton();
	this.toolBarButtonAbout = new System.Windows.Forms.ToolBarButton();
	this.toolBarButtonExit = new System.Windows.Forms.ToolBarButton();
	this.panelCenter = new System.Windows.Forms.Panel();
	this.explorerSplitPanel = new System.Windows.Forms.Panel();
	this.filePanelSplitter = new System.Windows.Forms.Splitter();
	this.linuxFilePanel = new FHTE.LTOOLS.GUI.DirFilePanel();
	this.dosFilePanel = new FHTE.LTOOLS.GUI.DirFilePanel();
	this.panelButtons = new System.Windows.Forms.Panel();
	this.statusBar = new System.Windows.Forms.StatusBar();
	this.statusBarPanelFree = new System.Windows.Forms.StatusBarPanel();
	this.statusBarPanelLinuxReadOnly = new System.Windows.Forms.StatusBarPanel();
	this.statusBarPanelConfirmDelete = new System.Windows.Forms.StatusBarPanel();
	this.statusBarPanelDosPattern = new System.Windows.Forms.StatusBarPanel();
	this.statusBarPanelLinuxPattern = new System.Windows.Forms.StatusBarPanel();
	this.panelToolbar.SuspendLayout();
	this.panelCenter.SuspendLayout();
	this.explorerSplitPanel.SuspendLayout();
	this.panelButtons.SuspendLayout();
	((System.ComponentModel.ISupportInitialize) (this.statusBarPanelFree)).BeginInit();
	((System.ComponentModel.ISupportInitialize) (this.statusBarPanelLinuxReadOnly)).BeginInit();
	((System.ComponentModel.ISupportInitialize) (this.statusBarPanelConfirmDelete)).BeginInit();
	((System.ComponentModel.ISupportInitialize) (this.statusBarPanelDosPattern)).BeginInit();
	((System.ComponentModel.ISupportInitialize) (this.statusBarPanelLinuxPattern)).BeginInit();
	this.SuspendLayout();
	// 
	// mainMenu 
	// 
	this.mainMenu.MenuItems.AddRange(new System.Windows.Forms.MenuItem[]
                                         {
                                         this.menuFile,
					 this.menuEdit,
					 this.menuHelp
					 }
	);
	// 
	// menuFile 
	// 
	this.menuFile.Index = 0;
	this.menuFile.MenuItems.AddRange(new System.Windows.Forms.MenuItem[]
                                         {
                                         this.menuItemRemote,
					 this.menuItemSeparator3,
					 this.menuItemReadOnly,
					 this.menuItemConfirmDelete,
					 this.menuItemSeparator4,
					 this.menuItemDosPattern,
					 this.menuItemLinuxPattern,
					 this.menuItemSeparator6,
					 this.menuItemAddDosDrive,
					 this.menuItemAddLinuxDrive,
					 this.menuItemSeparator5,
					 this.menuItemExit
					 }
	);
	this.menuFile.Text = "&File";
	this.menuFile.Select += new System.EventHandler(this.MenuFile_Select);
	// 
	// menuItemRemote 
	// 
	this.menuItemRemote.Index = 0;
	this.menuItemRemote.Text = "Remote Connect ...";
	this.menuItemRemote.Click += new System.EventHandler(this.MenuItemRemote_Click);
	// 
	// menuItemSeparator3 
	// 
	this.menuItemSeparator3.Index = 1;
	this.menuItemSeparator3.Text = "-";
	// 
	// menuItemReadOnly 
	// 
	this.menuItemReadOnly.Checked = true;
	this.menuItemReadOnly.Index = 2;
	this.menuItemReadOnly.Text = "Linux read only";
	this.menuItemReadOnly.Click += new System.EventHandler(this.MenuItemReadOnly_Click);
	// 
	// menuItemConfirmDelete 
	// 
	this.menuItemConfirmDelete.Checked = true;
	this.menuItemConfirmDelete.Index = 3;
	this.menuItemConfirmDelete.Text = "Confirm Delete";
	this.menuItemConfirmDelete.Click += new System.EventHandler(this.MenuItemConfirmDelete_Click);
	// 
	// menuItemSeparator4 
	// 
	this.menuItemSeparator4.Index = 4;
	this.menuItemSeparator4.Text = "-";
	// 
	// menuItemDosPattern 
	// 
	this.menuItemDosPattern.Index = 5;
	this.menuItemDosPattern.Text = "DOS Pattern ...";
	this.menuItemDosPattern.Click += new System.EventHandler(this.MenuItemDosPattern_Click);
	// 
	// menuItemLinuxPattern 
	// 
	this.menuItemLinuxPattern.Index = 6;
	this.menuItemLinuxPattern.Text = "Linux Pattern ...";
	this.menuItemLinuxPattern.Click += new System.EventHandler(this.MenuItemLinuxPattern_Click);
	// 
	// menuItemSeparator6 
	// 
	this.menuItemSeparator6.Index = 7;
	this.menuItemSeparator6.Text = "-";
	// 
	// menuItemAddDosDrive 
	// 
	this.menuItemAddDosDrive.Index = 8;
	this.menuItemAddDosDrive.Text = "Add DOS Drive ...";
	this.menuItemAddDosDrive.Click += new System.EventHandler(this.MenuItemAddDosDrive_Click);
	// 
	// menuItemAddLinuxDrive 
	// 
	this.menuItemAddLinuxDrive.Index = 9;
	this.menuItemAddLinuxDrive.Text = "Add Linux Drive ...";
	this.menuItemAddLinuxDrive.Click += new System.EventHandler(this.MenuItemAddLinuxDrive_Click);
	// 
	// menuItemSeparator5 
	// 
	this.menuItemSeparator5.Index = 10;
	this.menuItemSeparator5.Text = "-";
	// 
	// menuItemExit 
	// 
	this.menuItemExit.Index = 11;
	this.menuItemExit.Text = "Exit";
	this.menuItemExit.Click += new System.EventHandler(this.MenuItemExit_Click);
	// 
	// menuEdit 
	// 
	this.menuEdit.Index = 1;
	this.menuEdit.MenuItems.AddRange(new System.Windows.Forms.MenuItem[]
                                         {
                                         this.menuItemNew,
					 this.menuItemModify,
					 this.menuItemDelete,
					 this.menuItemCopy,
					 this.menuItemPaste
					 }
	);
	this.menuEdit.Text = "&Edit";
	// 
	// menuItemNew 
	// 
	this.menuItemNew.Index = 0;
	this.menuItemNew.Text = "New";
	this.menuItemNew.Click += new System.EventHandler(this.MenuItemNew_Click);
	// 
	// menuItemModify 
	// 
	this.menuItemModify.Index = 1;
	this.menuItemModify.Text = "Modify";
	this.menuItemModify.Click += new System.EventHandler(this.MenuItemModify_Click);
	// 
	// menuItemDelete 
	// 
	this.menuItemDelete.Index = 2;
	this.menuItemDelete.Text = "Delete";
	this.menuItemDelete.Click += new System.EventHandler(this.MenuItemDelete_Click);
	// 
	// menuItemCopy 
	// 
	this.menuItemCopy.Index = 3;
	this.menuItemCopy.Shortcut = System.Windows.Forms.Shortcut.CtrlC;
	this.menuItemCopy.Text = "Copy";
	this.menuItemCopy.Click += new System.EventHandler(this.MenuItemCopy_Click);
	// 
	// menuItemPaste 
	// 
	this.menuItemPaste.Index = 4;
	this.menuItemPaste.Shortcut = System.Windows.Forms.Shortcut.CtrlV;
	this.menuItemPaste.Text = "Paste";
	this.menuItemPaste.Click += new System.EventHandler(this.MenuItemPaste_Click);
	// 
	// menuHelp 
	// 
	this.menuHelp.Index = 2;
	this.menuHelp.MenuItems.AddRange(new System.Windows.Forms.MenuItem[]
                                         {
                                         this.menuItemPartitions,
					 this.menuItemSeparator1,
					 this.menuItemDebug,
					 this.menuItemHelp,
					 this.menuItemSeparator2,
					 this.menuItemAbout
					 }
	);
	this.menuHelp.Text = "&Help";
	this.menuHelp.Select += new System.EventHandler(this.MenuHelp_Select);
	// 
	// menuItemPartitions 
	// 
	this.menuItemPartitions.Index = 0;
	this.menuItemPartitions.Text = "Show Harddisk Partitions ...";
	this.menuItemPartitions.Click += new System.EventHandler(this.MenuItemPartitions_Click);
	// 
	// menuItemSeparator1 
	// 
	this.menuItemSeparator1.Index = 1;
	this.menuItemSeparator1.Text = "-";
	// 
	// menuItemDebug 
	// 
	this.menuItemDebug.Index = 2;
	this.menuItemDebug.Text = "DebugDialog ...";
	this.menuItemDebug.Visible = false;
	this.menuItemDebug.Click += new System.EventHandler(this.MenuItemDebug_Click);
	// 
	// menuItemHelp 
	// 
	this.menuItemHelp.Index = 3;
	this.menuItemHelp.Text = "Help ...";
	this.menuItemHelp.Click += new System.EventHandler(this.MenuItemHelp_Click);
	// 
	// menuItemSeparator2 
	// 
	this.menuItemSeparator2.Index = 4;
	this.menuItemSeparator2.Text = "-";
	// 
	// menuItemAbout 
	// 
	this.menuItemAbout.Index = 5;
	this.menuItemAbout.Text = "About ...";
	this.menuItemAbout.Click += new System.EventHandler(this.MenuItemAbout_Click);
	// 
	// toolbarImageList 
	// 
	this.toolbarImageList.ColorDepth = System.Windows.Forms.ColorDepth.Depth32Bit;
	this.toolbarImageList.ImageSize = new System.Drawing.Size(16, 16);
	this.toolbarImageList.ImageStream = ((System.Windows.Forms.ImageListStreamer) (resources.GetObject("toolbarImageList.ImageStream")));
	this.toolbarImageList.TransparentColor = System.Drawing.Color.Transparent;
	// 
	// panelToolbar 
	// 
	this.panelToolbar.Controls.AddRange(new System.Windows.Forms.Control[]
                                            {
                                            this.toolBar
					    }
	);
	this.panelToolbar.Dock = System.Windows.Forms.DockStyle.Top;
	this.panelToolbar.Name = "panelToolbar";
	this.panelToolbar.Size = new System.Drawing.Size(792, 39);
	this.panelToolbar.TabIndex = 1;
	// 
	// toolBar 
	// 
	this.toolBar.Buttons.AddRange(new System.Windows.Forms.ToolBarButton[]
                                      {
                                      this.toolBarButtonNew,
				      this.toolBarButtonModify,
				      this.toolBarButtonDelete,
				      this.toolBarButtonCopy,
				      this.toolBarButtonPaste,
				      this.toolBarButtonSeparator1,
				      this.toolBarButtonPartitions,
				      this.toolBarButtonConnect,
				      this.toolBarButtonSeparator2,
				      this.toolBarButtonReadOnly,
				      this.toolBarButtonConfirmDelete,
				      this.toolBarButtonSeparator3,
				      this.toolBarButtonHelp,
				      this.toolBarButtonAbout,
				      this.toolBarButtonExit
				      }
	);
	this.toolBar.ButtonSize = new System.Drawing.Size(60, 36);
	this.toolBar.Dock = System.Windows.Forms.DockStyle.Fill;
	this.toolBar.DropDownArrows = true;
	this.toolBar.ImageList = this.toolbarImageList;
	this.toolBar.Name = "toolBar";
	this.toolBar.ShowToolTips = true;
	this.toolBar.Size = new System.Drawing.Size(792, 39);
	this.toolBar.TabIndex = 1;
	this.toolBar.ButtonClick += new System.Windows.Forms.ToolBarButtonClickEventHandler(this.ToolBar_ButtonClick);
	// 
	// toolBarButtonNew 
	// 
	this.toolBarButtonNew.ImageIndex = 0;
	this.toolBarButtonNew.Text = "New";
	this.toolBarButtonNew.ToolTipText = "New";
	// 
	// toolBarButtonModify 
	// 
	this.toolBarButtonModify.ImageIndex = 1;
	this.toolBarButtonModify.Text = "Modify";
	this.toolBarButtonModify.ToolTipText = "Modify";
	// 
	// toolBarButtonDelete 
	// 
	this.toolBarButtonDelete.ImageIndex = 2;
	this.toolBarButtonDelete.Text = "Delete";
	this.toolBarButtonDelete.ToolTipText = "Delete";
	// 
	// toolBarButtonCopy 
	// 
	this.toolBarButtonCopy.ImageIndex = 3;
	this.toolBarButtonCopy.Text = "Copy";
	this.toolBarButtonCopy.ToolTipText = "Copy";
	// 
	// toolBarButtonPaste 
	// 
	this.toolBarButtonPaste.ImageIndex = 4;
	this.toolBarButtonPaste.Text = "Paste";
	this.toolBarButtonPaste.ToolTipText = "Paste";
	// 
	// toolBarButtonSeparator1 
	// 
	this.toolBarButtonSeparator1.Style = System.Windows.Forms.ToolBarButtonStyle.Separator;
	// 
	// toolBarButtonPartitions 
	// 
	this.toolBarButtonPartitions.ImageIndex = 5;
	this.toolBarButtonPartitions.Text = "Partitions";
	this.toolBarButtonPartitions.ToolTipText = "Partitions";
	// 
	// toolBarButtonConnect 
	// 
	this.toolBarButtonConnect.ImageIndex = 8;
	this.toolBarButtonConnect.Text = "Connect";
	this.toolBarButtonConnect.ToolTipText = "Connect";
	// 
	// toolBarButtonSeparator2 
	// 
	this.toolBarButtonSeparator2.Style = System.Windows.Forms.ToolBarButtonStyle.Separator;
	// 
	// toolBarButtonReadOnly 
	// 
	this.toolBarButtonReadOnly.ImageIndex = 11;
	this.toolBarButtonReadOnly.Style = System.Windows.Forms.ToolBarButtonStyle.ToggleButton;
	this.toolBarButtonReadOnly.Text = "Read Only";
	this.toolBarButtonReadOnly.ToolTipText = "Linux Read Only";
	// 
	// toolBarButtonConfirmDelete 
	// 
	this.toolBarButtonConfirmDelete.ImageIndex = 13;
	this.toolBarButtonConfirmDelete.Style = System.Windows.Forms.ToolBarButtonStyle.ToggleButton;
	this.toolBarButtonConfirmDelete.Text = "Confirm";
	this.toolBarButtonConfirmDelete.ToolTipText = "Confirm Delete";
	// 
	// toolBarButtonSeparator3 
	// 
	this.toolBarButtonSeparator3.Style = System.Windows.Forms.ToolBarButtonStyle.Separator;
	// 
	// toolBarButtonHelp 
	// 
	this.toolBarButtonHelp.ImageIndex = 6;
	this.toolBarButtonHelp.Text = "Help";
	this.toolBarButtonHelp.ToolTipText = "Help";
	// 
	// toolBarButtonAbout 
	// 
	this.toolBarButtonAbout.ImageIndex = 7;
	this.toolBarButtonAbout.Text = "About";
	this.toolBarButtonAbout.ToolTipText = "About";
	// 
	// toolBarButtonExit 
	// 
	this.toolBarButtonExit.ImageIndex = 10;
	this.toolBarButtonExit.Text = "Exit";
	this.toolBarButtonExit.ToolTipText = "Exit";
	// 
	// panelCenter 
	// 
	this.panelCenter.Controls.AddRange(new System.Windows.Forms.Control[]
                                           {
                                           this.explorerSplitPanel,
					   this.panelButtons
					   }
	);
	this.panelCenter.Dock = System.Windows.Forms.DockStyle.Fill;
	this.panelCenter.Location = new System.Drawing.Point(0, 39);
	this.panelCenter.Name = "panelCenter";
	this.panelCenter.Size = new System.Drawing.Size(792, 485);
	this.panelCenter.TabIndex = 2;
	// 
	// explorerSplitPanel 
	// 
	this.explorerSplitPanel.Controls.AddRange(new System.Windows.Forms.Control[]
                                                  {
                                                  this.filePanelSplitter,
						  this.linuxFilePanel,
						  this.dosFilePanel
						  }
	);
	this.explorerSplitPanel.Dock = System.Windows.Forms.DockStyle.Fill;
	this.explorerSplitPanel.Name = "explorerSplitPanel";
	this.explorerSplitPanel.Size = new System.Drawing.Size(792, 463);
	this.explorerSplitPanel.TabIndex = 2;
	// 
	// filePanelSplitter 
	// 
	this.filePanelSplitter.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
	this.filePanelSplitter.Dock = System.Windows.Forms.DockStyle.Top;
	this.filePanelSplitter.Location = new System.Drawing.Point(0, 220);
	this.filePanelSplitter.Name = "filePanelSplitter";
	this.filePanelSplitter.Size = new System.Drawing.Size(792, 6);
	this.filePanelSplitter.TabIndex = 5;
	this.filePanelSplitter.TabStop = false;
	this.filePanelSplitter.SplitterMoved += new System.Windows.Forms.SplitterEventHandler(this.FilePanelSplitter_SplitterMoved);
	// 
	// linuxFilePanel 
	// 
	this.linuxFilePanel.AllowDrop = true;
	this.linuxFilePanel.Dock = System.Windows.Forms.DockStyle.Fill;
	this.linuxFilePanel.IsLinux = false;
	this.linuxFilePanel.Location = new System.Drawing.Point(0, 220);
	this.linuxFilePanel.Name = "linuxFilePanel";
	this.linuxFilePanel.Size = new System.Drawing.Size(792, 243);
	this.linuxFilePanel.TabIndex = 4;
	this.linuxFilePanel.ViewListView = System.Windows.Forms.View.Details;
	// 
	// dosFilePanel 
	// 
	this.dosFilePanel.AllowDrop = true;
	this.dosFilePanel.Dock = System.Windows.Forms.DockStyle.Top;
	this.dosFilePanel.IsLinux = false;
	this.dosFilePanel.Name = "dosFilePanel";
	this.dosFilePanel.Size = new System.Drawing.Size(792, 220);
	this.dosFilePanel.TabIndex = 3;
	this.dosFilePanel.ViewListView = System.Windows.Forms.View.Details;
	// 
	// panelButtons 
	// 
	this.panelButtons.BackColor = System.Drawing.SystemColors.Control;
	this.panelButtons.Controls.AddRange(new System.Windows.Forms.Control[]
                                            {
                                            this.statusBar
					    }
	);
	this.panelButtons.Dock = System.Windows.Forms.DockStyle.Bottom;
	this.panelButtons.Location = new System.Drawing.Point(0, 463);
	this.panelButtons.Name = "panelButtons";
	this.panelButtons.Size = new System.Drawing.Size(792, 22);
	this.panelButtons.TabIndex = 0;
	// 
	// statusBar 
	// 
	this.statusBar.Anchor = (System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right);
	this.statusBar.Dock = System.Windows.Forms.DockStyle.None;
	this.statusBar.Name = "statusBar";
	this.statusBar.Panels.AddRange(new System.Windows.Forms.StatusBarPanel[]
                                       {
                                       this.statusBarPanelFree,
				       this.statusBarPanelLinuxReadOnly,
				       this.statusBarPanelConfirmDelete,
				       this.statusBarPanelDosPattern,
				       this.statusBarPanelLinuxPattern
				       }
	);
	this.statusBar.RightToLeft = System.Windows.Forms.RightToLeft.No;
	this.statusBar.ShowPanels = true;
	this.statusBar.Size = new System.Drawing.Size(792, 22);
	this.statusBar.TabIndex = 0;
	this.statusBar.PanelClick += new System.Windows.Forms.StatusBarPanelClickEventHandler(this.StatusBar_PanelClick);
	// 
	// statusBarPanelFree 
	// 
	this.statusBarPanelFree.BorderStyle = System.Windows.Forms.StatusBarPanelBorderStyle.None;
	this.statusBarPanelFree.Width = 192;
	// 
	// statusBarPanelLinuxReadOnly 
	// 
	this.statusBarPanelLinuxReadOnly.MinWidth = 150;
	this.statusBarPanelLinuxReadOnly.Text = "LinuxReadOnly";
	this.statusBarPanelLinuxReadOnly.Width = 150;
	// 
	// statusBarPanelConfirmDelete 
	// 
	this.statusBarPanelConfirmDelete.MinWidth = 150;
	this.statusBarPanelConfirmDelete.Text = "ConfirmDelete";
	this.statusBarPanelConfirmDelete.Width = 150;
	// 
	// statusBarPanelDosPattern 
	// 
	this.statusBarPanelDosPattern.MinWidth = 150;
	this.statusBarPanelDosPattern.Text = "DosPattern";
	this.statusBarPanelDosPattern.Width = 150;
	// 
	// statusBarPanelLinuxPattern 
	// 
	this.statusBarPanelLinuxPattern.MinWidth = 150;
	this.statusBarPanelLinuxPattern.Text = "LinuxPattern";
	this.statusBarPanelLinuxPattern.Width = 150;
	// 
	// LTOOLSnet 
	// 
	this.AllowDrop = true;
	this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
	this.ClientSize = new System.Drawing.Size(792, 524);
	this.Controls.AddRange(new System.Windows.Forms.Control[]
                               {
                               this.panelCenter,
			       this.panelToolbar
			       }
	);
	this.Icon = ((System.Drawing.Icon) (resources.GetObject("$this.Icon")));
	this.Menu = this.mainMenu;
	this.MinimumSize = new System.Drawing.Size(630, 510);
	this.Name = "LTOOLSnet";
	this.SizeGripStyle = System.Windows.Forms.SizeGripStyle.Show;
	this.StartPosition = System.Windows.Forms.FormStartPosition.CenterScreen;
	this.Text = "LTOOLSnet-C#";
	this.Resize += new System.EventHandler(this.LTOOLSnet_Resize);
	this.Closing += new System.ComponentModel.CancelEventHandler(this.LTOOLSnet_Closing);
	this.panelToolbar.ResumeLayout(false);
	this.panelCenter.ResumeLayout(false);
	this.explorerSplitPanel.ResumeLayout(false);
	this.panelButtons.ResumeLayout(false);
	((System.ComponentModel.ISupportInitialize) (this.statusBarPanelFree)).EndInit();
	((System.ComponentModel.ISupportInitialize) (this.statusBarPanelLinuxReadOnly)).EndInit();
	((System.ComponentModel.ISupportInitialize) (this.statusBarPanelConfirmDelete)).EndInit();
	((System.ComponentModel.ISupportInitialize) (this.statusBarPanelDosPattern)).EndInit();
	((System.ComponentModel.ISupportInitialize) (this.statusBarPanelLinuxPattern)).EndInit();
	this.ResumeLayout(false);

    }
#endregion

#region Menu and MenuItem Event-Handler
    /// <summary> 
    /// the MenuHelp_Select() method sets the menuitem DebugDialog checked or unchecked 
    /// if you are in the debug mode and the dialog is visible or not. 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuHelp_Select(object sender, System.EventArgs e)
    {   if (Debug)
        {   if (debugDialog.Visible)
            {   menuItemDebug.Checked = true;
	    } else
            {   menuItemDebug.Checked = false;
	    }
	}
    }
    /// <summary> 
    /// the MenuFile_Select() method sets the menuItems "Linux Read Only" and "Confirm delete" checked or 
    /// unchecked. 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuFile_Select(object sender, System.EventArgs e)
    {   menuItemReadOnly.Checked = IsLinuxReadOnly;
	menuItemConfirmDelete.Checked = ConfirmDelete;
    }
    /// <summary> 
    /// this method opens or closes a connection to a remote server. 
    /// you can specify the remote server in a dialog-box (hostname/port) 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuItemRemote_Click(object sender, System.EventArgs e)
    {   if (connected)
        {   ConnectionClose();
	    if (connected == false)
            {   this.Text = Caption;
		menuItemRemote.Text = "Remote Connect ...";
		toolBarButtonConnect.Text = "Connect";
		toolBarButtonConnect.ToolTipText = "Connect";
		toolBarButtonConnect.ImageIndex -= 1;
		linuxFilePanel.FillLinuxDirectoryTree();
		linuxFilePanel.SetBackground(dosFilePanel.GetBackColorTreeView());
		partitions = "";
	    }
	} else
        {   if (remoteDialog == null)
            {   remoteDialog = new RemoteBoxDialog(remoteHost);
	    }
	    if (remoteDialog.ShowDialog() == DialogResult.Cancel)
            {   return;
	    }
	    remoteHost = remoteDialog.RemoteServer;

	    ConnectionOpen();
	    connected = socket.Connected;

	    DebugOutput("connected: " + connected.ToString());
	    if (connected == true)
            {   this.Text = Caption + " connected to " + remoteHost.Host;
		menuItemRemote.Text = "Remote disconnect";
		toolBarButtonConnect.Text = "Disconnect";
		toolBarButtonConnect.ToolTipText = "Disconnect";
		toolBarButtonConnect.ImageIndex += 1;
		linuxFilePanel.FillLinuxDirectoryTree();
		linuxFilePanel.SetBackground(connectedColor);
		partitions = "";
	    }
	}
    }

    /// <summary> 
    /// this method sets the IsLinuxReadOnly property and updates the MenuItem and the statusbar 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuItemReadOnly_Click(object sender, System.EventArgs e)
    {   menuItemReadOnly.Checked = !menuItemReadOnly.Checked;
	IsLinuxReadOnly = menuItemReadOnly.Checked;
	UpdateStatusToolBar();
    }

    /// <summary> 
    /// the MenuItemConfirmDelete_Click() method sets the menuitem checked or unchecked 
    /// and the confirmDelete varibale to show or not a confirmation dialog when you delete a file. 
    /// At last update the statusbar. 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuItemConfirmDelete_Click(object sender, System.EventArgs e)
    {   menuItemConfirmDelete.Checked = !menuItemConfirmDelete.Checked;
	confirmDelete = menuItemConfirmDelete.Checked;
	UpdateStatusToolBar();
    }

    /// <summary> 
    /// the MenuItemDosPattern_Click() shows a dialog, where you can modify the searchpattern for 
    /// files in the Dos ListView 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuItemDosPattern_Click(object sender, System.EventArgs e)
    {   FilePatternDialog patternDialog = new FilePatternDialog(DOSFilePattern, false);
	if (patternDialog.ShowDialog() != DialogResult.OK)
        {   return;
	}
	dosFilePattern = patternDialog.FilePattern;
	UpdateStatusToolBar();
	dosFilePanel.UpdateDosListView(dosFilePanel.GetSelectedDirectoryNode());
    }

    /// <summary> 
    /// the MenuItemLinuxPattern_Click() shows a dialog, where you can modify the searchpattern for 
    /// files in the Linux ListView 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuItemLinuxPattern_Click(object sender, System.EventArgs e)
    {   FilePatternDialog patternDialog = new FilePatternDialog(LinuxFilePattern, true);
	if (patternDialog.ShowDialog() != DialogResult.OK)
        {   return;
	}
	linuxFilePattern = patternDialog.FilePattern;
	UpdateStatusToolBar();
	DirectoryNode dirNode = linuxFilePanel.GetSelectedDirectoryNode();
	if (dirNode != null)
        {   linuxFilePanel.UpdateLinuxListView(dirNode);
	}
    }

    /// <summary> 
    /// the MenuItemAddDosDrive_Click() method shows a Dialog to add a drive to the DOS tree. 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuItemAddDosDrive_Click(object sender, System.EventArgs e)
    {   dosFilePanel.MenuItemTreeViewAddDrive_Click(sender, e);
    }

    /// <summary> 
    /// the MenuItemAddLinuxDrive_Click() method shows a Dialog to add a drive to the Linux tree. 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuItemAddLinuxDrive_Click(object sender, System.EventArgs e)
    {   linuxFilePanel.MenuItemTreeViewAddDrive_Click(sender, e);
    }

    /// <summary> 
    /// aks if the user really wants to exit, if yes, then free resources and 
    /// delete temporary files, 
    /// if not, then do nothing and return 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuItemExit_Click(object sender, System.EventArgs e)
    {   if (MessageBox.Show("Do you really want to exit?",
			    "LTOOLSnet-C# - Exit",
			    MessageBoxButtons.YesNo,
			    MessageBoxIcon.Question)
	    == DialogResult.Yes)				// show a confirmation dialog 
        {   string tempfile = "zzz.tmp";			// delete the files: zzz.tmp 
	    if (File.Exists(tempfile))
            {   File.Delete(tempfile);
	    }
	    if (Connected)					// close the connection to the remote server 
            {   ConnectionClose();
	    }
	    if (Directory.Exists(DirFilePanel.tempPathLinuxFiles))
	    {							// delete files from the ltoolsTemp-directory 
		string[]files = Directory.GetFiles(DirFilePanel.tempPathLinuxFiles);
		foreach(string file in files)
                {   DebugOutput("delete file: " + file);
		    File.Delete(file);
		}
	    }
	    this.Dispose();
	} else
        {   return;
	}
    }

    /// <summary> 
    /// the MenuItemNew_Click() method delegates the work to the active DirFilePanel (Dos or Linux) 
    /// and opens a ModifyDosDialog or ModifyLinuxDialog. If there is no DirFilePanel active, a MessageBox 
    /// gives you a hint. 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuItemNew_Click(object sender, System.EventArgs e)
    {   try
        {   string message = "Select directory where to add the new item.";
	    string caption = "LTOOLS";
	    if (this.ActiveControl == dosFilePanel)		// the dosFilePanel is selected 
            {   DirectoryNode tempNode;
		if ((tempNode = dosFilePanel.GetSelectedDirectoryNode()) != null)
                {   dosFilePanel.NewDosDirectory(tempNode);
		} else
                {   MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Information);
		}
	    } else if (this.ActiveControl == linuxFilePanel)	// the LinuxFilePanel is selected 
            {   DirectoryNode tempNode;
		if ((tempNode = linuxFilePanel.GetSelectedDirectoryNode()) != null)
                {   linuxFilePanel.NewLinuxDirectory(tempNode);
		} else
                {   MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Information);
		}
	    } else
            {   MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Information);
	    }
	}
	catch(Exception ex)
        {   DebugOutput(ex.Message);
	}
    }

    /// <summary> 
    /// the MenuItemModify_Click() method delegates the work to the active DirFilePanel (DOS or Linux) 
    /// and opens a ModifyDosDialog or ModifyLinuxDialog. If there is no DirFilePanel active, a MessageBox 
    /// gives you a hint. 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuItemModify_Click(object sender, System.EventArgs e)
    {   ListViewItemFilesModify();
    }

    /// <summary> 
    /// the MenuItemDelete_Click() method delegates the work to the active DirFilePanel (DOS or Linux) 
    /// If there is no DirFilePanel active, a MessageBox gives you a hint. 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuItemDelete_Click(object sender, System.EventArgs e)
    {   ListViewItemFilesDelete();
    }

    /// <summary> 
    /// the MenuItemCopy_Click() method puts the selected Files in a DirFilePanel (DOS or Linux) 
    /// in the ArrayLists for the copy and paste action. If there is no file selected a MessageBox 
    /// gives you a hint. If there is a directory selected the copy and paste action is not valid 
    /// because directories cannot be copied. 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuItemCopy_Click(object sender, System.EventArgs e)
    {   ListViewItemFilesCopy();
    }

    /// <summary> 
    /// the MenuItemPaste_Click() method delegates the work to a DirFilePanel (Dos or Linux) 
    /// if there are valid files saved in the ArrayLists for copy and paste action else a 
    /// MessageBox gives you a hint. 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuItemPaste_Click(object sender, System.EventArgs e)
    {   ListViewItemFilesPaste();
    }

    /// <summary> 
    /// the MenuItemPartitions_Click() method shows a MessageBox with the partions on the local machine 
    /// or if the LTOOLSnet is connected to a remote server the partions on the server. 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuItemPartitions_Click(object sender, System.EventArgs e)
    {   if (Partitions.Equals(""))
        {   if (connected)
            {   LDir ldir = new LDir("ldirNT", "-x -part", NetSocket);
		 partitions = ldir.GetPartitions();
	    } else
            {   LDir ldir = new LDir("ldirNT", "-part");
		partitions = ldir.GetPartitions();
	    }
	}
	StringBuilder caption = new StringBuilder("LTOOLS Partitions");
	if (connected)
        {   caption.Append(" - " + remoteHost.Host);
	} else
        {   caption.Append(" - " + Environment.MachineName);
	}
	// MessageBox.Show(Partitions, caption.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Information) ; 
	PartitionsDialog partDialog = new PartitionsDialog(Partitions, caption.ToString());
	partDialog.ShowDialog();
    }

    /// <summary> 
    /// shows the help dialog 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuItemHelp_Click(object sender, System.EventArgs e)
    {   if (helpBox == null)
        {   helpBox = new HelpBox(false, lreadguiDirectory);
	} else
        {   helpBox.Visible = true;
	}
    }

    /// <summary> 
    /// shows the About Dialog 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuItemAbout_Click(object sender, System.EventArgs e)
    {   LDir ldir = new LDir("ldirNT", "-v");
	string message =
	"LTOOLSnet-C# " + Version + "\n(C) 2003 Joerg Holzaepfel - joerghol@web.de\n" +
	"(C) 1999-2006 Werner.Zimmermann@fht-esslingen.de\n" +
	"Operating System: " + Environment.OSVersion + "\n" +
	".NET Framework version " + Environment.Version + "\n\n" +
	"Using" + ldir.Response;
	 DebugOutput("Message:" + ldir.Response);
	 MessageBox.Show(message, "LTOOLS", MessageBoxButtons.OK, MessageBoxIcon.Information);
    }

    /// <summary> 
    /// the MenuItemDebug_Click() method shows the debugDialog or hides the debugDialog 
    /// and sets the Menuitem checked if necessary 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void MenuItemDebug_Click(object sender, System.EventArgs e)
    {   debugDialog.Visible = !debugDialog.Visible;
	menuItemDebug.Checked = !menuItemDebug.Checked;
    }
#endregion

#region ToolBar Event-Handler
    /// <summary> 
    /// a Button in the toolbar was clicked, jump to the right function 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void ToolBar_ButtonClick(object sender, System.Windows.Forms.ToolBarButtonClickEventArgs e)
    {   if (e.Button.Equals(toolBarButtonNew))
        {   MenuItemNew_Click(sender, e);
	} else if (e.Button.Equals(toolBarButtonModify))
        {   MenuItemModify_Click(sender, e);
	} else if (e.Button.Equals(toolBarButtonDelete))
        {   MenuItemDelete_Click(sender, e);
	} else if (e.Button.Equals(toolBarButtonCopy))
        {   MenuItemCopy_Click(sender, e);
	} else if (e.Button.Equals(toolBarButtonPaste))
        {   MenuItemPaste_Click(sender, e);
	} else if (e.Button.Equals(toolBarButtonPartitions))
        {   MenuItemPartitions_Click(sender, e);
	} else if (e.Button.Equals(toolBarButtonHelp))
        {   MenuItemHelp_Click(sender, e);
	} else if (e.Button.Equals(toolBarButtonAbout))
        {   MenuItemAbout_Click(sender, e);
	} else if (e.Button.Equals(toolBarButtonConnect))
        {   MenuItemRemote_Click(sender, e);
	} else if (e.Button.Equals(toolBarButtonExit))
        {   MenuItemExit_Click(sender, e);
	} else if (e.Button.Equals(toolBarButtonConfirmDelete))
        {   confirmDelete = !confirmDelete;
	    UpdateStatusToolBar();
	} else if (e.Button.Equals(toolBarButtonReadOnly))
        {   isLinuxReadOnly = !isLinuxReadOnly;
	    UpdateStatusToolBar();
	}
    }
#endregion

#region Methods to connect and disconnect to remote server
    /// <summary> 
    /// Open socket connection to remote server 
    /// </summary> 
    private static void ConnectionOpen()
    {   string caption = "LTOOLS";
	 try
        {   string server = remoteHost.Host;
	    int port = remoteHost.Port;

	    // open connection 
	    // IPAddress and IPEndPoint represent the endpoint that will receive the request. 
	    // Get the first IPAddress in the list using DNS. 
	    IPAddress hostadd = Dns.Resolve(server).AddressList[0];
	    IPEndPoint EPhost = new IPEndPoint(hostadd, port);

	    //Creates the Socket for sending data over TCP. 
	     socket = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);

	    // Connects to the host using IPEndPoint. 
	     socket.Connect(EPhost);
	    if (!socket.Connected)
            {   string message = "Unable to connect to host: " + remoteHost.Host;
		 MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Error);
		 return;
	    }
	    DebugOutput("Client connected to server: " + remoteHost.Host + ":" + remoteHost.Port);
	    connected = true;
	}
	catch(Exception e)
        {   connected = false;
	    string message = "Client connect to server failed:\n" + e.Message;
	    MessageBox.Show(message, caption, MessageBoxButtons.OK, MessageBoxIcon.Error);
	    DebugOutput(message);
	}
    }

    /// <summary> 
    /// the CloseConnection() method closes the connection to remote server. 
    /// </summary> 
    private static void ConnectionClose()
    {   try
        {   Encoding ASCII = Encoding.ASCII;
	    String Get = "ESC";
	     Byte[] ByteGet = ASCII.GetBytes(Get);
	     socket.Send(ByteGet, ByteGet.Length, SocketFlags.None);
	     socket.Close();
	     connected = false;
	}
	catch(Exception ex)
        {   MessageBox.Show(ex.Message, "LTOOLS", MessageBoxButtons.OK, MessageBoxIcon.Error);
	    connected = false;
	}
    }
#endregion

#region StatusBar
    /// <summary> 
    /// the UpdateStatusToolBar() method updates the texts in the panels of the statusbar. 
    /// There are panels for LinuxReadOnly, ConfirmDelete, DosFilePattern and LinuxFilePattern. 
    /// </summary> 
    private void UpdateStatusToolBar()
    {   // statusBarPanelLinuxReadOnly.Text  = "Linux ReadOnly: " + isLinuxReadOnly.ToString() ; 
	// statusBarPanelConfirmDelete.Text  = "Confirm Delete: " + confirmDelete.ToString() ; 
	statusBarPanelDosPattern.Text = "DOS: " + DOSFilePattern;
	statusBarPanelLinuxPattern.Text = "Linux: " + LinuxFilePattern;
	toolBarButtonConfirmDelete.Pushed = confirmDelete;
	toolBarButtonReadOnly.Pushed = isLinuxReadOnly;
	if (isLinuxReadOnly)
        {   toolBarButtonReadOnly.ImageIndex = IndexIconReadOnly;
	    toolBarButtonReadOnly.Text = "Read Only";
	    statusBarPanelLinuxReadOnly.Text = "Linux Read Only";
	    // statusBarPanelLinuxReadOnly.Icon = notifyIconReadOnly.Icon ; 
	} else
        {   toolBarButtonReadOnly.ImageIndex = IndexIconReadWrite;
	    toolBarButtonReadOnly.Text = "R/W";
	    statusBarPanelLinuxReadOnly.Text = "Linux Read/Write";
	    // statusBarPanelLinuxReadOnly.Icon = notifyIconReadWrite.Icon ; 
	}
	if (confirmDelete)
        {   toolBarButtonConfirmDelete.ImageIndex = IndexIconConfirm;
	    toolBarButtonConfirmDelete.Text = "Confirm";
	    statusBarPanelConfirmDelete.Text = "Confirm Delete";
	    // statusBarPanelConfirmDelete.Icon = notifyIconConfirm.Icon ; 
	} else
        {   toolBarButtonConfirmDelete.ImageIndex = IndexIconNoConfirm;
	    toolBarButtonConfirmDelete.Text = "No Confirm";
	    statusBarPanelConfirmDelete.Text = "Don't Confirm Delete";
	    // statusBarPanelConfirmDelete.Icon = notifyIconConfirmNot.Icon ; 
	}
    }

    /// <summary> 
    /// the StatusBar_PanelClick() method reacts on doubleclicks in the statusbar. There are four panels 
    /// in the statusbar for LinuxReadOnly, ConfirmDelete, DosPattern, LinuxPattern. A doubleclick in the 
    /// LinuxReadOnly and ConfirmDelete sets the flags on true or false. A doubleclick in the DosPattern or 
    /// LinuxPattern shows the FilePattern dialog. 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private void StatusBar_PanelClick(object sender, System.Windows.Forms.StatusBarPanelClickEventArgs e)
    {   if (e.Clicks >= 2)
        {   if (e.StatusBarPanel == statusBarPanelLinuxReadOnly)
            {   isLinuxReadOnly = !isLinuxReadOnly;
	    } else if (e.StatusBarPanel == statusBarPanelConfirmDelete)
            {   confirmDelete = !confirmDelete;
	    } else if (e.StatusBarPanel == statusBarPanelDosPattern)
            {   MenuItemDosPattern_Click(sender, e);
	    } else if (e.StatusBarPanel == statusBarPanelLinuxPattern)
            {   MenuItemLinuxPattern_Click(sender, e);
	    }
	    UpdateStatusToolBar();
	}
    }
#endregion

    /// <summary> 
    /// Shows a SplashScreen (duration milliseconds) before the application starts. 
    /// </summary> 
    /// <param name="duration">time to show the SplashScreen</param> 
    public static void ShowSplashScreen(int duration)
    {   try
        {   // MessageBox.Show(lreadguiDirectory + @"SplashScreen.bmp") ; 
	    int splashScreenWidth = 320;
	    int splashScreenHeight = 240;
	    int borderWidth = 2;
	    string pathSplashScreen = lreadguiDirectory + @"SplashScreen.bmp";

	    Form splashScreen = new Form();
	     splashScreen.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
	     splashScreen.ClientSize = new System.Drawing.Size(splashScreenWidth, splashScreenHeight);
	     splashScreen.FormBorderStyle = System.Windows.Forms.FormBorderStyle.None;
	     splashScreen.BackColor = Color.Black;
	     splashScreen.Name = "SplashScreen";
	     splashScreen.ShowInTaskbar = false;
	     splashScreen.StartPosition = System.Windows.Forms.FormStartPosition.CenterScreen;
	     splashScreen.Text = "SplashScreen";
	     splashScreen.TopMost = true;

	     splashScreen.SuspendLayout();

	    Label labelInfo = new Label();
	     labelInfo.Font = new Font(FontFamily.GenericSansSerif.ToString(), (float) 10.0);
	     labelInfo.Text = "LTOOLSnet-C# " + LTOOLSnet.Version + " " + " 2003-2006";
	     labelInfo.TextAlign = ContentAlignment.MiddleCenter;
	     labelInfo.Size = new Size(splashScreenWidth - (2 * borderWidth), 20);
	     labelInfo.Location = new Point(borderWidth, splashScreenHeight - borderWidth - 20);
	     labelInfo.BackColor = Color.Transparent;

	    Panel backgroundImage = new Panel();
	     backgroundImage.Size = new Size(splashScreenWidth - (borderWidth * 2), splashScreenHeight - (borderWidth * 2));
	     backgroundImage.Location = new Point(borderWidth, borderWidth);
	    if (File.Exists(pathSplashScreen))
            {   backgroundImage.BackgroundImage = new Bitmap(pathSplashScreen);
	    } else
            {   Label labelLTools = new Label();
		labelLTools.Font = new Font(FontFamily.GenericSansSerif.ToString(), (float) 48.0);
		labelLTools.Text = "LTOOLS";
		labelLTools.TextAlign = ContentAlignment.MiddleCenter;
		labelLTools.Size = new Size(splashScreenWidth - (2 * borderWidth), 60);
		labelLTools.Location = new Point(borderWidth, splashScreenHeight - borderWidth - 160);
		labelLTools.ForeColor = Color.White;
		backgroundImage.BackColor = Color.CornflowerBlue;
		backgroundImage.Controls.Add(labelLTools);
		labelInfo.ForeColor = Color.White;
	    }

	    backgroundImage.Controls.Add(labelInfo);
	    splashScreen.Controls.Add(backgroundImage);
	    splashScreen.ResumeLayout(false);

	    splashScreen.Update();
	    splashScreen.Show();
	    splashScreen.Update();
	    Thread.Sleep(duration);
	    splashScreen.Visible = false;
	}
	catch(Exception e)
        {   DebugOutput(e.Message + "\n\n" + e.StackTrace);
	    // MessageBox.Show(e.Message + "\n\n" + e.StackTrace) ; 
	}
    }
    /// <summary> 
    /// Closes the SplashScreen, this method is just for testing. 
    /// </summary> 
    /// <param name="sender">sender object</param> 
    /// <param name="e">event data</param> 
    private static void SplashScreen_Click(object sender, System.EventArgs e)
    {   ((Form) sender).Dispose();
    }


    /// <summary> 
    /// The main entry point for the application. 
    /// </summary> 
    [STAThread]
    static void Main(string[]args)
    {   ShowSplashScreen(LTOOLSnet.splashDuration);
	// examine the commandline arguments 
	int length = args.Length;
	for (int i = 0; i < length; i++)
        {   if (args[i].Substring(0, 2).ToLower() == "-d")
            {   LTOOLSnet.Debug = true;
	    }
	    /*
	       if (args[i] == "-host")
               {
               if (i+1 < length)
               {
               remoteHost.Host = args[i+1] ;
	       }
	       }
	     */
	}
	Application.Run(new LTOOLSnet());
    }
#endregion

}
}

