/* 
 * LTOOLSgui.java

 Java based graphical user interface for the LTOOLS toolkit.

 Requires Java 1.1 or newer. Does run as a standalone application, not as an applet.

 Copyright (C) 1999-2006 Werner Zimmermann(Werner.Zimmermann@fht-esslingen.de)

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 For further information please see file README.TXT.

 ---History, Changes:-----------------------------------------------------------
 V2.0: First release of a Swing based version. Versions before V2.0 were based on AWT
 and released under the name LREADgui. The conversion to Swing is based on pre-
 works by Sebastian Schmar. Many thanks!
 This program requires JDK 1.2 or newer
 W. Zimmermann, 15 March 2000

 V2.1: Include a dialog box showing partition table of all harddisks.
 Add menu item to toggle the Linux filesystem between read only (default) and read write.
 W. Zimmermann, 03 Nov. 2000

 V2.2: Allow multiple selection of files for copy and delete
 W. Zimmermann, 24 Nov. 2000

 V2.3: Make program also run as applet. Needs policy set.
 W. Zimmermann, 15 Oct. 2001

 V2.4: Changed parser for ldir temp files from fixed to variable field length
 to allow 32bit uid/gid etc.
 W. Zimmermann, 01 Nov. 2001

 V2.5: Alphabetic sorting of files and directories
 W. Zimmermann, 25 Feb. 2002

 V2.6: Speeded up alphabetic sorting.
 W. Zimmermann, 07 Mar. 2002

 V2.7: Size of main window changed
 W. Zimmermann, 14 Mar. 2003

 V2.8: Support for file names with ' '. Bug fix for drag and drop error with read only file system.
 W. Zimmermann, 6 Apr 2003

 V3.0: GUI changed by J. Holzaepfel, with minor modifications by W. Zimmermann
 J. Holzaepfel, June 2003

 V3.1: Changed error handling, when ldir issues an error. Parsing response of execOperatingSystemCommand()
 changed to find errors, when showing directory listings. Now show the error message in a message box instead 
 of the file tree. 
 Modified parsing of 'ldir -part' to adapt it to ReiserFS RFSTOOL.
 W. Zimmermann, Aug. 2003

 V3.2: Copyright update for 2004. View menu item in context menu.
 W. Zimmermann, 20 Dec. 2003
 
 V3.3: Use Eclipse for GUI design.
 W. Zimmermann, 28 Feb. 2004
 
 V3.4: Change cursor to WAIT_CURSOR while executing LTOOLS commands
 W. Zimmermann, 15 Jan. 2006
 */

import java.io. *;
import java.net. *;
import java.util. *;
import java.awt. *;
//import java.awt.event. *;
import javax.swing. *;
//import javax.swing.event. *;
import javax.swing.tree. *;

/** The Main-Class of the LTOOLSgui
 * @author Werner Zimmermann
 */
public class LTOOLSgui extends javax.swing.JFrame
{   // ################################################################################################################ 
    // FIELDS 
    // ################################################################################################################ 
    static final String VERSION = "V3.4";
    static boolean DEBUG = false;				// set to 'true', if you want to get all these debug messages 
    static boolean isApplet = false;
    static LTOOLSgui myProg;					// reference to the application 
    static String codeBase = "";				// 
    static String lreadguiDirectory;				// the directory of the application 
    static String tempPathLinuxFiles;				// the directory to save temporary linux files 
    static Vector copyListItemFiles = new Vector();		// Vector with all files to copy 
    static boolean copyFileIsLinux = false;			// true if files are linux-files 
    static boolean copyFileIsValid = false;			// true if files are valid to copy 
    static TreePath copyToDirectory = null;			// The Directory(TreePath) to which the files should be 
    // copied by Drag and Drop or by buttons 
    static Socket qSocket = null;				// Socket for remote connection 
    static BufferedInputStream bReceive = null;			// Receive stream 
    static BufferedOutputStream bSend = null;			// Transmit stream 
    static BufferedReader qReceive = null;			// Receive stream 
    static BufferedWriter qSend = null;				// Transmit stream 
    static String hostname = "localhost";			// hostname of remote-server 
    static int port = 1605;					// Server port number 
    static boolean connected = false;				// true if connected to a remote server 
    static boolean isLinuxReadOnly;				// true if linux-drives set to read only 
    static boolean confirmDelete;				// true if you want confirm-dialogs by deleting files 
    private static String dosFilePattern = "*";			// the filepattern for the DOS-files 
    private static String linuxFilePattern = "*";		// the filepattern for the Linux-files 
    private static DirFilePane activePane = null;		// ref to the active DirFilePane 
    private HelpBox helpBox = null;				// ref to the HelpBox 
    ImageIcon confirmIcon;					// Icon for the Buttons in the "statusbar" 
    ImageIcon confirmNotIcon;					// Icon for the Buttons in the "statusbar" 
    ImageIcon readOnlyIcon;					// Icon for the Buttons in the "statusbar" 
    ImageIcon readWriteIcon;					// Icon for the Buttons in the "statusbar" 
    ImageIcon connectIcon;					// Icon for the Buttons in the "statusbar" 
    ImageIcon disconnectIcon;					// Icon for the Buttons in the "statusbar" 
    // ################################################################################################################ 
    // CONSTRUCTORS 
    // ################################################################################################################ 
	/**
	 * Constructor
	 */
    public LTOOLSgui()
    {   //{{INIT_CONTROLS 
	getContentPane().setLayout(new BorderLayout(0, 0));
	setSize(800, 600);
	setVisible(false);
	toolbarJPanel.setLayout(new GridLayout(1, 1, 0, 0));
	getContentPane().add(BorderLayout.NORTH, toolbarJPanel);
	toolbarJPanel.setBounds(0, 0, 800, 29);
	toolbarJPanel.add(mainJToolBar);
	mainJToolBar.setBounds(0, 0, 800, 29);
	newJButton.setDefaultCapable(false);
	newJButton.setToolTipText("Create a new directory");
	newJButton.setText("New");
	newJButton.setActionCommand("New");
	mainJToolBar.add(newJButton);
	newJButton.setBounds(16, 2, 59, 25);
	modifyJButton.setDefaultCapable(false);
	modifyJButton.setToolTipText("Change filenames and attributes");
	modifyJButton.setText("Modify");
	modifyJButton.setActionCommand("Modify");
	mainJToolBar.add(modifyJButton);
	modifyJButton.setBounds(75, 2, 71, 25);
	deleteJButton.setDefaultCapable(false);
	deleteJButton.setToolTipText("Delete a file or directory");
	deleteJButton.setText("Delete");
	deleteJButton.setActionCommand("Delete");
	mainJToolBar.add(deleteJButton);
	deleteJButton.setBounds(146, 2, 71, 25);
	copyJButton.setDefaultCapable(false);
	copyJButton.setToolTipText("Copy a file");
	copyJButton.setText("Copy");
	copyJButton.setActionCommand("Copy");
	mainJToolBar.add(copyJButton);
	copyJButton.setBounds(217, 2, 63, 25);
	pasteJButton.setDefaultCapable(false);
	pasteJButton.setToolTipText("Insert a file");
	pasteJButton.setText("Paste");
	pasteJButton.setActionCommand("Paste");
	mainJToolBar.add(pasteJButton);
	pasteJButton.setBounds(280, 2, 67, 25);
	partitionsJButton.setDefaultCapable(false);
	partitionsJButton.setToolTipText("Show Harddisk Partitions");
	partitionsJButton.setText("Partitions");
	partitionsJButton.setActionCommand("Partitions");
	mainJToolBar.add(partitionsJButton);
	partitionsJButton.setBounds(347, 2, 89, 25);
	helpJButton.setDefaultCapable(false);
	helpJButton.setToolTipText("Help");
	helpJButton.setText("Help");
	helpJButton.setActionCommand("Help");
	mainJToolBar.add(helpJButton);
	helpJButton.setBounds(436, 2, 59, 25);
	aboutJButton.setDefaultCapable(false);
	aboutJButton.setToolTipText("About");
	aboutJButton.setText("About");
	aboutJButton.setActionCommand("About");
	mainJToolBar.add(aboutJButton);
	aboutJButton.setBounds(495, 2, 67, 25);
	centerJPanel.setLayout(new BorderLayout(0, 0));
	getContentPane().add(BorderLayout.CENTER, centerJPanel);
	centerJPanel.setBounds(0, 29, 800, 546);
	explorerJSplitPane.setOrientation(javax.swing.JSplitPane.VERTICAL_SPLIT);
	centerJPanel.add(BorderLayout.CENTER, explorerJSplitPane);
	explorerJSplitPane.setBounds(0, 0, 800, 546);
	statusJPanel.setLayout(new BorderLayout(0, 0));
	getContentPane().add(BorderLayout.SOUTH, statusJPanel);
	statusJPanel.setBounds(0, 575, 800, 25);
	JPanel1.setLayout(new GridLayout(1, 4, 0, 0));
	statusJPanel.add(BorderLayout.CENTER, JPanel1);
	JPanel1.setBounds(0, 0, 800, 25);
	remoteJButton.setToolTipText("Connect to a remote host (host must be running LREADjav)");
	remoteJButton.setText("Connect");
	remoteJButton.setActionCommand("Connect");
	JPanel1.add(remoteJButton);
	remoteJButton.setBounds(0, 0, 200, 25);
	readOnlyJToggleButton.setToolTipText("Sets the Linux-partitions to \"read only\" or \"writeable\"");
	readOnlyJToggleButton.setText("Read Only");
	readOnlyJToggleButton.setActionCommand("Read Only");
	JPanel1.add(readOnlyJToggleButton);
	readOnlyJToggleButton.setBounds(200, 0, 200, 25);
	confirmDeleteJToggleButton.setToolTipText("Show a confirm-dialog or not");
	confirmDeleteJToggleButton.setText("Confirm Delete");
	confirmDeleteJToggleButton.setActionCommand("Confirm Delete");
	JPanel1.add(confirmDeleteJToggleButton);
	confirmDeleteJToggleButton.setBounds(400, 0, 200, 25);
	exitJButton.setToolTipText("Press to close LTOOLSgui");
	exitJButton.setText("Exit");
	exitJButton.setActionCommand("Exit");
	JPanel1.add(exitJButton);
	exitJButton.setBounds(600, 0, 200, 25);
	//$$ mainJMenuBar.move(0,601); 
	fileJMenu.setText("File");
	fileJMenu.setActionCommand("File");
	mainJMenuBar.add(fileJMenu);
	remoteJMenuItem.setText("Remote Connect");
	remoteJMenuItem.setActionCommand("Remote Connect");
	fileJMenu.add(remoteJMenuItem);
	fileJMenu.add(JSeparator1);
	linuxReadOnlyJCheckBoxMenuItem.setSelected(true);
	linuxReadOnlyJCheckBoxMenuItem.setText("Linux Read Only");
	linuxReadOnlyJCheckBoxMenuItem.setActionCommand("Linux Read Only");
	fileJMenu.add(linuxReadOnlyJCheckBoxMenuItem);
	confirmDeleteJCheckBoxMenuItem.setSelected(true);
	confirmDeleteJCheckBoxMenuItem.setText("Confirm Delete");
	confirmDeleteJCheckBoxMenuItem.setActionCommand("Confirm Delete");
	fileJMenu.add(confirmDeleteJCheckBoxMenuItem);
	fileJMenu.add(JSeparator2);
	dosPatternJMenuItem.setText("DOS Pattern ...");
	dosPatternJMenuItem.setActionCommand("DOS Pattern ...");
	fileJMenu.add(dosPatternJMenuItem);
	linuxPatternJMenuItem.setText("Linux Pattern ...");
	linuxPatternJMenuItem.setActionCommand("Linux Pattern ...");
	fileJMenu.add(linuxPatternJMenuItem);
	fileJMenu.add(JSeparator3);
	addDOSDriveJMenuItem.setText("Add DOS Drive ...");
	addDOSDriveJMenuItem.setActionCommand("Add DOS Drive ...");
	fileJMenu.add(addDOSDriveJMenuItem);
	addLinuxDriveJMenuItem.setText("Add Linux Drive ...");
	addLinuxDriveJMenuItem.setActionCommand("Add Linux Drive ...");
	fileJMenu.add(addLinuxDriveJMenuItem);
	fileJMenu.add(JSeparator4);
	exitJMenuItem.setText("Exit");
	exitJMenuItem.setActionCommand("Exit");
	fileJMenu.add(exitJMenuItem);
	editJMenu.setText("Edit");
	editJMenu.setActionCommand("Edit");
	mainJMenuBar.add(editJMenu);
	newJMenuItem.setText("New");
	newJMenuItem.setActionCommand("New");
	editJMenu.add(newJMenuItem);
	modifyJMenuItem.setText("Modify");
	modifyJMenuItem.setActionCommand("Modify");
	editJMenu.add(modifyJMenuItem);
	deleteJMenuItem.setText("Delete");
	deleteJMenuItem.setActionCommand("Delete");
	editJMenu.add(deleteJMenuItem);
	copyJMenuItem.setText("Copy");
	copyJMenuItem.setActionCommand("Copy");
	editJMenu.add(copyJMenuItem);
	pasteJMenuItem.setText("Paste");
	pasteJMenuItem.setActionCommand("Paste");
	editJMenu.add(pasteJMenuItem);
	viewJMenuItem.setText("View");
	viewJMenuItem.setActionCommand("View");
	editJMenu.add(viewJMenuItem);
	helpJMenu.setText("Help");
	helpJMenu.setActionCommand("Help");
	mainJMenuBar.add(helpJMenu);
	partitionsJMenuItem.setText("Show Harddisk Partitions");
	partitionsJMenuItem.setActionCommand("Show Harddisk Partitions");
	helpJMenu.add(partitionsJMenuItem);
	helpJMenu.add(JSeparator5);
	helpJMenuItem.setText("Help ...");
	helpJMenuItem.setActionCommand("Help ...");
	helpJMenu.add(helpJMenuItem);
	helpJMenu.add(JSeparator6);
	aboutJMenuItem.setText("About");
	aboutJMenuItem.setActionCommand("About");
	helpJMenu.add(aboutJMenuItem);
	//$$ JMenuBar1.move(0,601); 
	setTitle("A Simple Frame");
	setJMenuBar(mainJMenuBar);
	//}} 

	//{{REGISTER_LISTENERS 
	SymWindow aSymWindow = new SymWindow();
	 this.addWindowListener(aSymWindow);
	SymAction lSymAction = new SymAction();
	 newJButton.addActionListener(lSymAction);
	 modifyJButton.addActionListener(lSymAction);
	 deleteJButton.addActionListener(lSymAction);
	 copyJButton.addActionListener(lSymAction);
	 pasteJButton.addActionListener(lSymAction);
	 partitionsJButton.addActionListener(lSymAction);
	 helpJButton.addActionListener(lSymAction);
	 aboutJButton.addActionListener(lSymAction);
	 remoteJMenuItem.addActionListener(lSymAction);
	 linuxReadOnlyJCheckBoxMenuItem.addActionListener(lSymAction);
	 confirmDeleteJCheckBoxMenuItem.addActionListener(lSymAction);
	 dosPatternJMenuItem.addActionListener(lSymAction);
	 linuxPatternJMenuItem.addActionListener(lSymAction);
	 addDOSDriveJMenuItem.addActionListener(lSymAction);
	 addLinuxDriveJMenuItem.addActionListener(lSymAction);
	 exitJMenuItem.addActionListener(lSymAction);
	 newJMenuItem.addActionListener(lSymAction);
	 modifyJMenuItem.addActionListener(lSymAction);
	 deleteJMenuItem.addActionListener(lSymAction);
	 copyJMenuItem.addActionListener(lSymAction);
	 pasteJMenuItem.addActionListener(lSymAction);
	 viewJMenuItem.addActionListener(lSymAction);
	 partitionsJMenuItem.addActionListener(lSymAction);
	 helpJMenuItem.addActionListener(lSymAction);
	 aboutJMenuItem.addActionListener(lSymAction);
	 remoteJButton.addActionListener(lSymAction);
	 readOnlyJToggleButton.addActionListener(lSymAction);
	 confirmDeleteJToggleButton.addActionListener(lSymAction);
	 exitJButton.addActionListener(lSymAction);
	//}} 

	 mainJToolBar.setLayout(new GridLayout(1, 8, 0, 0));
	 setTitle("LTOOLSgui " + myProg.VERSION + " Werner.Zimmermann@fht-esslingen.de");
	 newJButton.setIcon(new ImageIcon("images/new.gif"));
	 modifyJButton.setIcon(new ImageIcon("images/modify.gif"));
	 deleteJButton.setIcon(new ImageIcon("images/cut.gif"));
	 copyJButton.setIcon(new ImageIcon("images/copy.gif"));
	 pasteJButton.setIcon(new ImageIcon("images/paste.gif"));
	 helpJButton.setIcon(new ImageIcon("images/help16x16.gif"));
	 partitionsJButton.setIcon(new ImageIcon("images/partitions16x16.gif"));
	 aboutJButton.setIcon(new ImageIcon("images/info16x16.gif"));
	 dosPane = new DirFilePane(false);
	 linuxPane = new DirFilePane(true);
	 explorerJSplitPane.setTopComponent(dosPane);
	 explorerJSplitPane.setBottomComponent(linuxPane);
	 explorerJSplitPane.setDividerLocation(0.5);

	 isLinuxReadOnly = linuxReadOnlyJCheckBoxMenuItem.isSelected();
	 confirmDelete = confirmDeleteJCheckBoxMenuItem.isSelected();

	if (isApplet)
        {   exitJButton.setEnabled(false);
	    exitJMenuItem.setEnabled(false);
	}
	setIconImage(getToolkit().getImage("./images/app.gif"));

	 readOnlyJToggleButton.setSelected(isLinuxReadOnly);

	 confirmIcon = new ImageIcon("images/confirm16x16.gif");
	 confirmNotIcon = new ImageIcon("images/confirmnot16x16.gif");
	 setConfirmDeleteView();

	 readOnlyIcon = new ImageIcon("images/readonly16x16.gif");
	 readWriteIcon = new ImageIcon("images/readwrite16x16.gif");
	 setReadOnlyView();

	 connectIcon = new ImageIcon("images/connect16x16.gif");
	 disconnectIcon = new ImageIcon("images/disconnect16x16.gif");
	 remoteJButton.setIcon(connectIcon);

	 exitJButton.setIcon(new ImageIcon("images/exit16x16.gif"));

	//{{INIT_MENUS 
	//}} 

    }

	/** Constructor
     * @param title the caption of the window
     */
    public LTOOLSgui(String title)
    {   this();
	setTitle(title);
    }

    // ################################################################################################################ 
    // METHODS 
    // ################################################################################################################ 
	/** The entry point for this application.
     * Sets the Look and Feel to the System Look and Feel.
     * Creates a new LTOOLSgui and makes it visible.
     * @param args the commandline-arguments
     */
    static public void main(String args[])
    {   showSplash(2000);
	int i = 0;
	 try
        {   if (args.length > 0)
            {   if ((args[0].startsWith("-h")) || (args[0].startsWith("-?")))
                {   System.out.println("LTOOLS Graphical User Interface " + myProg.VERSION + "\n(C) 1999-2006 Werner.Zimmermann@fht-esslingen.de\n\n"
				       + "Usage:  LTOOLSgui [-d] [Linux_drive [DOS_drive]]\n"
				       + "Note:   Java 1.2 or newer needed\n"
				       + "        set DOS_drive to \"\" if running LTOOLSgui on a UNIX host\n\n");
		    System.exit(0);				// close the application 
		}
		if (args[0].startsWith("-d"))
                {   DEBUG = true;
		    i = 1;
		}
	    }
	    if (isApplet)
		lreadguiDirectory = LTOOLSapplet.ltoolappletCodeBase;
	    else
		lreadguiDirectory = System.getProperty("user.dir") + File.separator;
	    tempPathLinuxFiles = lreadguiDirectory + "temp" + File.separator;

	    File tempDir = new File(tempPathLinuxFiles);
	    if (!tempDir.exists())				// create Temp-Directory if it doesn't exist. 
            {   tempDir.mkdir();
	    }
	    try							// Change the look and feel to platform specific 
            {   UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
                System.out.println("Using "+UIManager.getSystemLookAndFeelClassName()+"\n");
	    }
	    catch(Exception e)
            {
            }

	    System.out.println("LTOOLSgui " + VERSION + " (C) 1999-2006 Werner.Zimmermann@fht-esslingen.de\n" +
		    "Operating System: " + System.getProperty("os.name") + " " + System.getProperty("os.arch") + " " +
			       System.getProperty("os.version") + "\n" +
			       "Java VM         : " + System.getProperty("java.vendor") + " " + System.getProperty("java.version") + "\n");

	    debugOutput("\n--------------------------------------------------------------------------------------------");
	    debugOutput("tempPathLinuxFiles: " + tempPathLinuxFiles);
	    debugOutput("\n--------------------------------------------------------------------------------------------");

	    //Create a new instance of our application's frame, and make it visible. 
	    Dimension screen = Toolkit.getDefaultToolkit().getScreenSize();
	    myProg = new LTOOLSgui();
	    int width = myProg.getWidth();
	    int height = myProg.getHeight();
	    int x = (screen.width - width) / 2;
	    int y = (screen.height - height) / 2;
	    myProg.setBounds(x, y, width, height);
	    myProg.setVisible(true);
	}
	catch(Throwable t)
        {   t.printStackTrace();
	    //Ensure the application exits with an error condition. 
	    System.exit(1);
	}
    }
    /** sets the frame visible or invisible
     * @param b true to set the frame visible
     */
    public void setVisible(boolean b)
    {   if (b)
        {   setLocation(50, 50);
	}
	super.setVisible(b);
    }
//* calls super.addNotify() 
    public void addNotify()
    {   // Record the size of the window prior to calling parents addNotify. 
	Dimension d = getSize();

	 super.addNotify();

	if (fComponentsAdjusted)
	     return;

	// Adjust components according to the insets 
	Insets ins = getInsets();
	 setSize(ins.left + ins.right + d.width, ins.top + ins.bottom + d.height);
	Component components[] = getContentPane().getComponents();
	for (int i = 0; i < components.length; i++)
        {   Point p = components[i].getLocation();
	     p.translate(ins.left, ins.top);
	     components[i].setLocation(p);
	}
	fComponentsAdjusted = true;
    }

    // Used for addNotify check. 
    boolean fComponentsAdjusted = false;

//* WindowAdapter-class 
    class SymWindow extends java.awt.event.WindowAdapter
    {   /** the windowClosing()-method of the WindowAdapter-class
         * @param event the event data
         */
	public void windowClosing(java.awt.event.WindowEvent event)
        {   Object object = event.getSource();
	    if (object == LTOOLSgui.this)
		 LTOOLSGui_WindowClosing(event);
	}
    }
    /** invoked before closing the frame
     * @param event the event data
     */
    void LTOOLSGui_WindowClosing(java.awt.event.WindowEvent event)
    {   this.exitApplication();
    }
    //{{DECLARE_CONTROLS 
    javax.swing.JPanel toolbarJPanel = new javax.swing.JPanel();
    javax.swing.JToolBar mainJToolBar = new javax.swing.JToolBar();
    javax.swing.JButton newJButton = new javax.swing.JButton();
    javax.swing.JButton modifyJButton = new javax.swing.JButton();
    javax.swing.JButton deleteJButton = new javax.swing.JButton();
    javax.swing.JButton copyJButton = new javax.swing.JButton();
    javax.swing.JButton pasteJButton = new javax.swing.JButton();
    javax.swing.JButton partitionsJButton = new javax.swing.JButton();
    javax.swing.JButton helpJButton = new javax.swing.JButton();
    javax.swing.JButton aboutJButton = new javax.swing.JButton();
    javax.swing.JPanel centerJPanel = new javax.swing.JPanel();
    javax.swing.JSplitPane explorerJSplitPane = new javax.swing.JSplitPane();
    javax.swing.JPanel statusJPanel = new javax.swing.JPanel();
    javax.swing.JPanel JPanel1 = new javax.swing.JPanel();
    javax.swing.JButton remoteJButton = new javax.swing.JButton();
    javax.swing.JToggleButton readOnlyJToggleButton = new javax.swing.JToggleButton();
    javax.swing.JToggleButton confirmDeleteJToggleButton = new javax.swing.JToggleButton();
    javax.swing.JButton exitJButton = new javax.swing.JButton();
    javax.swing.JMenuBar mainJMenuBar = new javax.swing.JMenuBar();
    javax.swing.JMenu fileJMenu = new javax.swing.JMenu();
    javax.swing.JMenuItem remoteJMenuItem = new javax.swing.JMenuItem();
    javax.swing.JSeparator JSeparator1 = new javax.swing.JSeparator();
    javax.swing.JCheckBoxMenuItem linuxReadOnlyJCheckBoxMenuItem = new javax.swing.JCheckBoxMenuItem();
    javax.swing.JCheckBoxMenuItem confirmDeleteJCheckBoxMenuItem = new javax.swing.JCheckBoxMenuItem();
    javax.swing.JSeparator JSeparator2 = new javax.swing.JSeparator();
    javax.swing.JMenuItem dosPatternJMenuItem = new javax.swing.JMenuItem();
    javax.swing.JMenuItem linuxPatternJMenuItem = new javax.swing.JMenuItem();
    javax.swing.JSeparator JSeparator3 = new javax.swing.JSeparator();
    javax.swing.JMenuItem addDOSDriveJMenuItem = new javax.swing.JMenuItem();
    javax.swing.JMenuItem addLinuxDriveJMenuItem = new javax.swing.JMenuItem();
    javax.swing.JSeparator JSeparator4 = new javax.swing.JSeparator();
    javax.swing.JMenuItem exitJMenuItem = new javax.swing.JMenuItem();
    javax.swing.JMenu editJMenu = new javax.swing.JMenu();
    javax.swing.JMenuItem newJMenuItem = new javax.swing.JMenuItem();
    javax.swing.JMenuItem modifyJMenuItem = new javax.swing.JMenuItem();
    javax.swing.JMenuItem deleteJMenuItem = new javax.swing.JMenuItem();
    javax.swing.JMenuItem copyJMenuItem = new javax.swing.JMenuItem();
    javax.swing.JMenuItem pasteJMenuItem = new javax.swing.JMenuItem();
    javax.swing.JMenuItem viewJMenuItem = new javax.swing.JMenuItem();
    javax.swing.JMenu helpJMenu = new javax.swing.JMenu();
    javax.swing.JMenuItem partitionsJMenuItem = new javax.swing.JMenuItem();
    javax.swing.JSeparator JSeparator5 = new javax.swing.JSeparator();
    javax.swing.JMenuItem helpJMenuItem = new javax.swing.JMenuItem();
    javax.swing.JSeparator JSeparator6 = new javax.swing.JSeparator();
    javax.swing.JMenuItem aboutJMenuItem = new javax.swing.JMenuItem();
    //}} 

    //{{DECLARE_MENUS 
    //}} 

    DirFilePane dosPane;					// the DirFilePane for the DOS-Filesystem 
    DirFilePane linuxPane;					// the DirFilePane for the Linux-Filesystem 

//* ActionListener-class 
    class SymAction implements java.awt.event.ActionListener
    {   /** the actionPerformed()-method of the ActionListener
         * @param event the event data
         */
	public void actionPerformed(java.awt.event.ActionEvent event)
        {   Object object = event.getSource();
	    if (object == newJButton)
		 newJButton_actionPerformed(event);
	    else if (object == modifyJButton)
		 modifyJButton_actionPerformed(event);
	    else if (object == deleteJButton)
		 deleteJButton_actionPerformed(event);
	    else if (object == copyJButton)
		 copyJButton_actionPerformed(event);
	    else if (object == pasteJButton)
		 pasteJButton_actionPerformed(event);
	    else if (object == partitionsJButton)
		 partitionsJButton_actionPerformed(event);
	    else if (object == helpJButton)
		 helpJButton_actionPerformed(event);
	    else if (object == aboutJButton)
		 aboutJButton_actionPerformed(event);
	    else if (object == remoteJMenuItem)
		 remoteJMenuItem_actionPerformed(event);
	    else if (object == linuxReadOnlyJCheckBoxMenuItem)
		 linuxReadOnlyJCheckBoxMenuItem_actionPerformed(event);
	    else if (object == confirmDeleteJCheckBoxMenuItem)
		 confirmDeleteJCheckBoxMenuItem_actionPerformed(event);
	    else if (object == dosPatternJMenuItem)
		 dosPatternJMenuItem_actionPerformed(event);
	    else if (object == linuxPatternJMenuItem)
		 linuxPatternJMenuItem_actionPerformed(event);
	    else if (object == addDOSDriveJMenuItem)
		 addDOSDriveJMenuItem_actionPerformed(event);
	    else if (object == addLinuxDriveJMenuItem)
		 addLinuxDriveJMenuItem_actionPerformed(event);
	    else if (object == exitJMenuItem)
		 exitJMenuItem_actionPerformed(event);
	    else if (object == newJMenuItem)
		 newJMenuItem_actionPerformed(event);
	    else if (object == modifyJMenuItem)
		 modifyJMenuItem_actionPerformed(event);
	    else if (object == deleteJMenuItem)
		 deleteJMenuItem_actionPerformed(event);
	    else if (object == copyJMenuItem)
		 copyJMenuItem_actionPerformed(event);
	    else if (object == pasteJMenuItem)
		 pasteJMenuItem_actionPerformed(event);
	    else if (object == viewJMenuItem)
		 viewJMenuItem_actionPerformed(event);
	    else if (object == partitionsJMenuItem)
		 partitionsJMenuItem_actionPerformed(event);
	    else if (object == helpJMenuItem)
		 helpJMenuItem_actionPerformed(event);
	    else if (object == aboutJMenuItem)
		 aboutJMenuItem_actionPerformed(event);
	    else if (object == remoteJButton)
		 remoteJButton_actionPerformed(event);
	    else if (object == readOnlyJToggleButton)
		 readOnlyJToggleButton_actionPerformed(event);
	    else if (object == confirmDeleteJToggleButton)
		 confirmDeleteJToggleButton_actionPerformed(event);
	    else if (object == exitJButton)
		 exitJButton_actionPerformed(event);
	}
    }
    /** Invoked when the newJButton has been clicked
     * @param event the event data
     */
    void newJButton_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("newButtonAction");
	newJMenuItem_actionPerformed(event);
    }
    /** Invoked when the modifyJButton has been clicked
     * @param event the event data
     */
    void modifyJButton_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("ModifyButtonAction");
	modifyJMenuItem_actionPerformed(event);
    }
    /** Invoked when the deleteJButton has been clicked
     * @param event the event data
     */
    void deleteJButton_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("delButtonAction");
	deleteJMenuItem_actionPerformed(event);
    }
    /** Invoked when the copyJButton has been clicked
     * @param event the event data
     */
    void copyJButton_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("copyButtonAction");
	copyJMenuItem_actionPerformed(event);

    }
    /** Invoked when the pasteJButton has been clicked
     * @param event the event data
     */
    void pasteJButton_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("pasteButtonAction");
	pasteJMenuItem_actionPerformed(event);

    }
    /** Invoked when the partitionsJButton has been clicked
     * @param event the event data
     */
    void partitionsJButton_actionPerformed(java.awt.event.ActionEvent event)
    {   // to do: code goes here. 
	partitionsJMenuItem_actionPerformed(event);
    }
    /** Invoked when the helpJButton has been clicked
     * @param event the event data
     */
    void helpJButton_actionPerformed(java.awt.event.ActionEvent event)
    {   helpJMenuItem_actionPerformed(event);
    }
    /** Invoked when the remoteJButton has been clicked
     * @param event the event data
     */
    void remoteJButton_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("RemoteButton_Action");
	remoteJMenuItem_actionPerformed(event);
    }
    /** Invoked when the aboutJButton has been clicked
     * @param event the event data
     */
    void aboutJButton_actionPerformed(java.awt.event.ActionEvent event)
    {   aboutJMenuItem_actionPerformed(event);
    }
    /** Invoked when the remoteJMenuItem has been clicked
     * @param event the event data
     */
    void remoteJMenuItem_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("RemoteMenuItem_Action");
	if (LTOOLSgui.isConnected())
        {   connectionClose();
	    if (LTOOLSgui.isConnected() == false)
            {   remoteJMenuItem.setText("Remote connect");
		remoteJButton.setIcon(connectIcon);
		remoteJButton.setText("Connect");
		linuxPane.fillLinuxDirectoryTree();
		linuxPane.setBackgroundTreeTable(dosPane.getJTable().getBackground());
	    }
	} else
        {   RemoteBoxDialog myRemote = new RemoteBoxDialog(this, hostname, "" + port);
	    myRemote.setVisible(true);
	    if (myRemote.getStatus() == RemoteBoxDialog.CANCEL)
		return;
	    hostname = myRemote.getHostName();
	    port = myRemote.getPortNumber();
	    connectionOpen();
	    if (LTOOLSgui.isConnected() == true)
            {   remoteJMenuItem.setText("Remote disconnect");
		remoteJButton.setIcon(disconnectIcon);
		remoteJButton.setText("Disconnect");
		linuxPane.fillLinuxDirectoryTree();
		linuxPane.setBackgroundTreeTable(new Color(240, 248, 255));	// getHSBColor((float)0.94, (float)0.97, (float)1.0)) ; // (240 , 248, 255)) ; 
	    }
	}
    }
    /** Invoked when the linuxReadOnlyJCheckBoxMenuItem has been clicked
     * @param event the event data
     */
    void linuxReadOnlyJCheckBoxMenuItem_actionPerformed(java.awt.event.ActionEvent event)
    {   isLinuxReadOnly = linuxReadOnlyJCheckBoxMenuItem.isSelected();
	setReadOnlyView();
    }
    /** Invoked when the confirmDeleteJCheckBoxMenuItem has been clicked
     * @param event the event data
     */
    void confirmDeleteJCheckBoxMenuItem_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("confirmDelete - MenuItem");
	confirmDelete = confirmDeleteJCheckBoxMenuItem.isSelected();
	setConfirmDeleteView();
    }
    /** Invoked when the dosPatternJMenuItem has been clicked
     * @param event the event data
     */
    void dosPatternJMenuItem_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("DOS Pattern MenuItem");
	FilePatternDialog dialog = new FilePatternDialog(this, true, dosFilePattern, false);
	 dialog.show();
	 debugOutput("Status: " + dialog.getStatus() + "\nPattern: " + dialog.getFilePattern());
	if (dialog.getStatus())
        {   dosFilePattern = dialog.getFilePattern();
	    dosPane.updateDOSJTable(dosPane.getTreeNode(dosPane.getTreeSelectionPath()));
	}
    }
    /** Invoked when the linuxPatternJMenuItem has been clicked
     * @param event the event data
     */
    void linuxPatternJMenuItem_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("Linux Pattern MenuItem");
	FilePatternDialog dialog = new FilePatternDialog(this, true, linuxFilePattern, true);
	 dialog.show();
	 debugOutput("Status: " + dialog.getStatus() + "\nPattern: " + dialog.getFilePattern());
	if (dialog.getStatus())
        {   linuxFilePattern = dialog.getFilePattern();
	    linuxPane.updateLinuxJTable(linuxPane.getTreeNode(linuxPane.getTreeSelectionPath()));
	}
    }
    /** Invoked when the addDOSDriveJMenuItem has been clicked
     * @param event the event data
     */
    void addDOSDriveJMenuItem_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("add DOS Drive");
	dosPane.addDrive();
    }
    /** Invoked when the addLinuxDriveJMenuItem has been clicked
     * @param event the event data
     */
    void addLinuxDriveJMenuItem_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("add Linux Drive");
	linuxPane.addDrive();
    }
    /** Invoked when the exitJMenuItem has been clicked
     * @param event the event data
     */
    void exitJMenuItem_actionPerformed(java.awt.event.ActionEvent event)
    {   this.exitApplication();
    }
    /** Invoked when the newJMenuItem has been clicked
     * @param event the event data
     */
    void newJMenuItem_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("newItemAction");
	String caption = "LTOOLS";
	String msg = "Select directory where to add the new item.";

	if (activePane == dosPane)
        {   if (dosPane.getTreeSelectionPath() != null)
            {   dosPane.newDosDirectory(dosPane.getTreeSelectionPath());
	    } else
            {   LTOOLSgui.beepSound();
		JOptionPane.showMessageDialog(this, msg, caption, JOptionPane.ERROR_MESSAGE);
	    }
	} else if (activePane == linuxPane)
        {   if (linuxPane.getTreeSelectionPath() != null)
            {   linuxPane.newLinuxDirectory(linuxPane.getTreeSelectionPath());
	    } else
            {   LTOOLSgui.beepSound();
		JOptionPane.showMessageDialog(this, msg, caption, JOptionPane.ERROR_MESSAGE);
	    }
	} else
        {   LTOOLSgui.beepSound();
	    JOptionPane.showMessageDialog(this, msg, caption, JOptionPane.ERROR_MESSAGE);
	}
    }
    /** Invoked when the modifyJMenuItem has been clicked
     * @param event the event data
     */
    void modifyJMenuItem_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("modifyItemAction");
	modifyListItemFiles();
    }
    /** the modifyListItemFiles()-method calls the modify-method of the active
     * DirFilePane in the DirFilePane-class for all selected ListItemFiles in the
     * JTable.
     */
    public void modifyListItemFiles()
    {   String caption = "LTOOLS";
	if (getActiveDirFilePane() == dosPane)
        {   Vector items = dosPane.getSelectedListItemFile();
	    for (int i = 0; i < items.size(); i++)
            {   ListItemFile item = (ListItemFile) items.elementAt(i);
		if (!item.getFileName().equals(".."))
                {   dosPane.modifyDosFile(item);
		}
	    }
	    dosPane.updateDOSJTable(dosPane.getTreeNode(dosPane.getJTree().getSelectionPath()));
	} else if (getActiveDirFilePane() == linuxPane)
        {   Vector items = linuxPane.getSelectedListItemFile();
	    for (int i = 0; i < items.size(); i++)
            {   ListItemFile item = (ListItemFile) items.elementAt(i);
		if (!item.getFileName().equals(".."))
                {   linuxPane.modifyLinuxFile(item);
		}
	    }
	    linuxPane.updateLinuxJTable(linuxPane.getTreeNode(linuxPane.getJTree().getSelectionPath()));
	} else
        {   LTOOLSgui.beepSound();
	    String message = "Select directory or file to modify.";
	    JOptionPane.showMessageDialog(this, message, caption, JOptionPane.ERROR_MESSAGE);
	}
    }

    public void viewListItemFiles()
    {   String caption = "LTOOLS";
    	if (getActiveDirFilePane() == dosPane)
        {   Vector items = dosPane.getSelectedListItemFile();
	    for (int i = 0; i < items.size(); i++)
            {   ListItemFile item = (ListItemFile) items.elementAt(i);
		if (!item.getFileName().equals(".."))
                {   dosPane.startDosApplication(item, false);
		}
	    }
//	    dosPane.updateDOSJTable(dosPane.getTreeNode(dosPane.getJTree().getSelectionPath()));
	} else if (getActiveDirFilePane() == linuxPane)
        {   Vector items = linuxPane.getSelectedListItemFile();
	    for (int i = 0; i < items.size(); i++)
            {   ListItemFile item = (ListItemFile) items.elementAt(i);
		if (!item.getFileName().equals(".."))
                {   linuxPane.startLinuxApplication(item, false);
		}
	    }
//	    linuxPane.updateLinuxJTable(linuxPane.getTreeNode(linuxPane.getJTree().getSelectionPath()));
	} else
        {   LTOOLSgui.beepSound();
	    String message = "Select directory or file to modify.";
	    JOptionPane.showMessageDialog(this, message, caption, JOptionPane.ERROR_MESSAGE);
	}
    }

    /** Invoked when the deleteJMenuItem has been clicked
     * @param event the event data
     */
    void deleteJMenuItem_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("delItemAction");
	deleteListItemFiles();
    }
    /** Invoked when the copyJMenuItem has been clicked
     * @param event the event data
     */
    void copyJMenuItem_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("copyItemAction");
	copyListItemFiles();
    }
    /** Invoked when the pasteJMenuItem has been clicked
     * @param event the event data
     */
    void pasteJMenuItem_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("pasteItemAction");
	pasteListItemFiles();
    }
    /** Invoked when the viewJMenuItem has been clicked
     * @param event the event data
     */
    void viewJMenuItem_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("pasteItemAction");
	viewListItemFiles();
    }
    /** Invoked when the partitionsJMenuItem has been clicked
     * @param event the event data
     */
    void partitionsJMenuItem_actionPerformed(java.awt.event.ActionEvent event)
    {   String info = linuxPane.showPartitions();

	String caption;
	if (LTOOLSgui.isConnected())
        {   caption = "LTOOLS Partitions - " + hostname;
	} else
        {   try
            {   if (!File.separator.equals("\\"))		// Linux 
                {   info = info + "\n\n\nSorry, 'ldir -part' only works under DOS/Windows - use 'fdisk' instead";
		}
		caption = "LTOOLS Partitions - " + InetAddress.getLocalHost();
	    }
	    catch(Exception e)
            {   e.printStackTrace();
		caption = "LTOOLS Partitions";
	    }
	}

	PartitionsFrame dialog = new PartitionsFrame(this, true, info, caption);
	dialog.show();
	// JOptionPane.showMessageDialog(this, info, "LTOOLS", JOptionPane.INFORMATION_MESSAGE); 
    }
    /** Invoked when the helpJMenuItem has been clicked
     * @param event the event data
     */
    void helpJMenuItem_actionPerformed(java.awt.event.ActionEvent event)
    {   if (helpBox == null)
        {   helpBox = new HelpBox(this, false, "file:" + lreadguiDirectory);
	    helpBox.setVisible(true);
	} else
        {   helpBox.setVisible(true);
	}
    }
    /** Invoked when the aboutJMenuItem has been clicked
     * @param event the event data
     */
    void aboutJMenuItem_actionPerformed(java.awt.event.ActionEvent event)
    {   JOptionPane.showMessageDialog(this, "LTOOLSgui  " + LTOOLSgui.myProg.VERSION
				      + " 1999-2006 Werner.Zimmermann@fht-esslingen.de\nOperating System: "
				      + System.getProperty("os.name") + " " + System.getProperty("os.arch")
				      + " " + System.getProperty("os.version") + "\n"
			  + "Java VM: " + System.getProperty("java.vendor") + " " + System.getProperty("java.version")
				      + "\n\n" + "Using\n"
			  + linuxPane.execOperatingSystemCommand("ldir -x -v", false).replace('\r', ' ').substring(2),
				      "LTOOLS", JOptionPane.INFORMATION_MESSAGE);
    }
    /** Invoked when the readOnlyJToggleButton has been clicked
     * @param event the event data
     */
    void readOnlyJToggleButton_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("ReadOnly ToggleButton");
	isLinuxReadOnly = readOnlyJToggleButton.isSelected();
	setReadOnlyView();
    }
    /** Invoked when the confirmDeleteJToggleButton has been clicked
     * @param event the event data
     */
    void confirmDeleteJToggleButton_actionPerformed(java.awt.event.ActionEvent event)
    {   debugOutput("ConfirmDelete ToggleButton");
	confirmDelete = !confirmDelete;
	setConfirmDeleteView();
    }
    /** Invoked when the exitJButton has been clicked
     * @param event the event data
     */
    void exitJButton_actionPerformed(java.awt.event.ActionEvent event)
    {   this.exitApplication();
    }
	/** the exitApplication() closes the connection to a remote server, if connected and
     * deletes all temporary files created and exits the application.
     */
    void exitApplication()
    {   try
        {   LTOOLSgui.beepSound();
	    // Show a confirmation dialog 
	    int reply = JOptionPane.showConfirmDialog(this,
						      "Do you really want to exit?",
						      "LTOOLSgui - Exit",
						      JOptionPane.YES_NO_OPTION,
						      JOptionPane.QUESTION_MESSAGE);
	    // If the confirmation was affirmative, handle exiting. 
	    if (reply == JOptionPane.YES_OPTION)
            {   File file1 = new File("ZZZ.ZWZ");		// Delete the Dos file again 
		if (file1.exists())
		     file1.delete();
		File file2 = new File("zzz.tmp");		// Delete the Dos file again 
		if (file1.exists())
		     file2.delete();

		// Delete the Files in the tempPathLinuxFiles-directory 
		File tempDir = new File(tempPathLinuxFiles);
		if (tempDir.exists())
                {   File[]files = tempDir.listFiles();
		    for (int k = 0; k < files.length; k++)
                    {   if (files[k].delete())
                        {   debugOutput("File \"" + files[k].toString() + "\" deleted.");
			} else
                        {   debugOutput("ERROR deleting File \"" + files[k].toString() + "\".");
			}
		    }
		}
		if (LTOOLSgui.isConnected())
		    connectionClose();

		this.setVisible(false);				// hide the Frame 
		this.dispose();					// free the system resources 
		System.exit(0);					// close the application 
	    }
	}
	catch(Exception e)
        {
        }
    }

    //###### Debugging ################################################################################# 
    /** the debugOutput()-method shows debugging-information on the stdout
     * @param text the text to be shown on the stdout
     */
    private static void debugOutput(String text)		//Debug output to console 
    {   if (DEBUG == true)
	    System.out.println(text);
    }
    /** The getDebug()-method returns the value of the DEBUG-field
     * @return the value of the DEBUG-field
     */
    public static boolean getDebug()
    {   return DEBUG;
    }
    /**
     * Open connection to remote server
     */
    static void connectionOpen()
    {   debugOutput("Started openConnection()");
	try
        {   qSocket = new Socket(hostname, port);		// Open a socket to the server. 
	    qReceive = new BufferedReader(new InputStreamReader(qSocket.getInputStream()));
	    bReceive = new BufferedInputStream(qSocket.getInputStream());
	    qSend = new BufferedWriter(new OutputStreamWriter(qSocket.getOutputStream()));
	    bSend = new BufferedOutputStream(qSocket.getOutputStream());

	    // debugOutput("qSocket.isConnected(): " + qSocket.isConnected()) ;     // isConnected() since SunJava1.4 
	    if ((qSocket != null) && (qReceive != null) && (qSend != null))
            {   connected = true;
		debugOutput("Client connected to server");
	    } else
            {   debugOutput("Client connect to server failed");
		connected = false;
		LTOOLSgui.beepSound();
		JOptionPane.showMessageDialog(myProg, "Client connect to server failed", "LTOOLS", JOptionPane.ERROR_MESSAGE);
	    }
	}
	catch(UnknownHostException excpt)
        {   System.err.println("Unknown host " + excpt);
	    LTOOLSgui.beepSound();
	    JOptionPane.showMessageDialog(myProg, "Unknown hostname \"" + hostname, "LTOOLS", JOptionPane.ERROR_MESSAGE);
	}
	catch(IOException excpt)
        {   System.err.println("Failed I/O to " + excpt);
	    LTOOLSgui.beepSound();
	    JOptionPane.showMessageDialog(myProg, "Client connect to server failed: " + excpt, "LTOOLS", JOptionPane.ERROR_MESSAGE);
	}
    }
    /**
     * Close connection to remote server
     */
    static void connectionClose()
    {   debugOutput("Started closeConnection()");

	if (LTOOLSgui.isConnected() == false)
	    return;

	try
        {   if (qSend != null)
            {   qSend.write("ESC", 0, 3);			// Send close command (ESC) to server 
		qSend.close();
		bSend.close();
		qSend = null;					// Close transmit stream 
	    }
	    if (qReceive != null)
            {   qReceive.close();
		bReceive.close();
		qReceive = null;				// Close receive stream 
	    }
	    if (qSocket != null)
            {   qSocket.close();
		qSocket = null;					// Close socket 
	    }
	    connected = false;
	    debugOutput("Connection closed");
	}
	catch(IOException excpt)
        {   System.err.println("Failed I/O to " + excpt);
	    connected = false;
	}
    }
    /** the isConnected()-method returns the value of the connected-field
     * @return the value of the connected-field
     */
    public static boolean isConnected()
    {   return connected;
    }
    /** the showSplash()-method shows a splashscreen while starting the application
     * @param duration the duration of showing the splashscreen in milliseconds
     */
    // A simple little method to show a title screen in the 
    // center of the screen for a given amount of time. 
    public static void showSplash(int duration)
    {   JWindow splash = new JWindow();
	JPanel content = (JPanel) splash.getContentPane();

	// set the window's bounds, centering the window 
	int width = 320;
	int height = 240;
	Dimension screen = Toolkit.getDefaultToolkit().getScreenSize();
	int x = (screen.width - width) / 2;
	int y = (screen.height - height) / 2;
	 splash.setBounds(x, y, width, height);

	// build the splash screen 
	JLabel label = new JLabel(new ImageIcon("images/SplashScreen.gif"));
	JLabel copyrt = new JLabel("LTOOLSgui " + LTOOLSgui.VERSION + " (C) 1999-2006", JLabel.CENTER);
	 copyrt.setFont(new Font("Sans-Serif", Font.BOLD, 12));
	 content.add(label, BorderLayout.CENTER);
	 content.add(copyrt, BorderLayout.SOUTH);
	 content.setBorder(BorderFactory.createLineBorder(Color.black, 1));

	// display it 
	 splash.setVisible(true);

	// Wait a little while, maybe while loading resources 
	 try
        {   Thread.sleep(duration);
	}
	catch(Exception e)
        {
        }
	splash.setVisible(false);
    }
    /** the getDosFilePattern()-method returns the current search-pattern for DOS-files
     * @return the pattern for the dos-files
     */
    public static String getDosFilePattern()
    {   return dosFilePattern;
    }
    /** sets the File-pattern for the Dos-DirFilePane
     * @param pattern new search-pattern for DOS-files
     */
    public static void setDosFilePattern(String pattern)
    {   dosFilePattern = pattern;
    }
    /** the getLinuxFilePattern()-method returns the current search-pattern for
     * Linux-files
     * @return the pattern for the linux-files
     */
    public static String getLinuxFilePattern()
    {   return linuxFilePattern;
    }
    /** sets the File-pattern for the Dos-DirFilePane
     * @param pattern new search-pattern for DOS-files
     */
    public static void setLinuxFilePattern(String pattern)
    {   linuxFilePattern = pattern;
    }
    /** the setActiveDirFilePane()-method sets the activePane-field to the current
     * active DirFilePane
     * @param temp temp - sets the active DirFilePane of the application (Linux or Dos)
     */
    public static void setActiveDirFilePane(DirFilePane temp)
    {   activePane = temp;
    }
    /** the getActiveDirFilePane()-method returns the value of the activePane-field
     * @return the current active DirFilePane (Linux or Dos)
     */
    public static DirFilePane getActiveDirFilePane()
    {   return activePane;
    }
    /** the getDOSFilePane() returns the dosPane-field
     * @return the DOS-DirFilePane
     */
    public DirFilePane getDOSFilePane()
    {   return dosPane;
    }
    /** the getDOSFilePane() returns the linuxPane-field
     * @return the Linux-DirFilePane
     */
    public DirFilePane getLinuxFilePane()
    {   return linuxPane;
    }
//* the deleteListItemFiles() deletes the selected files of the active DirFilePane 
    public void deleteListItemFiles()
    {   String caption = "LTOOLS";
	if (getActiveDirFilePane() == dosPane)
        {   Vector items = dosPane.getSelectedListItemFile();
	    for (int i = 0; i < items.size(); i++)
            {   ListItemFile item = (ListItemFile) items.elementAt(i);
		if (!item.getFileName().equals(".."))
                {   dosPane.deleteDosFile(item);
		}
	    }
	    dosPane.updateDOSJTable(dosPane.getTreeNode(dosPane.getJTree().getSelectionPath()));
	    dosPane.updateTreeNode(dosPane.getJTree().getSelectionPath());
	} else if (getActiveDirFilePane() == linuxPane)
        {   if (LTOOLSgui.isLinuxReadOnly)
            {   LTOOLSgui.beepSound();
		String msg = "Your Linux filesystem is set to read only - change in menu 'File - Linux read only' first.";
		JOptionPane.showMessageDialog(LTOOLSgui.myProg, msg, "LTOOLS", JOptionPane.ERROR_MESSAGE);
		return;
	    }
	    Vector items = linuxPane.getSelectedListItemFile();
	    for (int i = 0; i < items.size(); i++)
            {   ListItemFile item = (ListItemFile) items.elementAt(i);
		if (!item.getFileName().equals(".."))
                {   linuxPane.deleteLinuxFile(item);
		}
	    }
	    linuxPane.updateLinuxJTable(linuxPane.getTreeNode(linuxPane.getJTree().getSelectionPath()));
	    linuxPane.updateTreeNode(linuxPane.getJTree().getSelectionPath());
	} else
        {   LTOOLSgui.beepSound();
	    String message = "Select directory or file to delete.";
	    JOptionPane.showMessageDialog(this, message, caption, JOptionPane.ERROR_MESSAGE);
	}
	// confirmDelCheckBox.setSelected(true); 
	confirmDelete = true;
	setConfirmDeleteView();
    }
    /** used in dragOver()-method and drop()-method of the DirFilePane to set a TreePath
     * and copyListItemFiles()-method and pasteListItemFiles of LTOOLSgui to reset the copToDirectory to null
     * the copyToDirectory should be checked in the pasteToDos() and pasteToLinux()-methods of DirFilePand
     * if the copyToDirectory is null the copy-action is by buttons or from jTable to jTable
     * else it's a copyAction from a jTable to a jTree-Directory
     * @param dir the TreePath of the directory to copy the files to
     */
    static public void setCopyToDirectory(TreePath dir)
    {   LTOOLSgui.copyToDirectory = dir;
    }
    /** the getCopyToDirectory() returns the TreePath of the copyToDirectory-field
     * @return returns the value of the copyToDirectory-field
     */
    static public TreePath getCopyToDirectory()
    {   return LTOOLSgui.copyToDirectory;
    }
    /** the copyListItemFiles()-method puts the selected ListItemFiles of the active
     * DirFilePane in the copyListItemFiles-Vector.
     */
    public void copyListItemFiles()
    {   String caption = "LTOOLS";
	 LTOOLSgui.setCopyToDirectory(null);			// reset the copyToDirectory 
	if (getActiveDirFilePane() == dosPane)
        {   copyListItemFiles.removeAllElements();
	    copyListItemFiles = dosPane.getSelectedListItemFile();
	    copyFileIsLinux = false;				// it's a Dos-file 
	    copyFileIsValid = true;
	    // if it is not a directory but a file, it is valid 
	    for (int i = 0; i < copyListItemFiles.size(); i++)
            {   if (((ListItemFile) copyListItemFiles.elementAt(i)).isDirectory())
                {   LTOOLSgui.beepSound();
		    copyFileIsValid = false;
		    String msg = "Sorry, directories cannot be copied.";
		     JOptionPane.showMessageDialog(LTOOLSgui.myProg, msg, caption, JOptionPane.ERROR_MESSAGE);
		} else if (((ListItemFile) copyListItemFiles.elementAt(i)).isError())
                {   LTOOLSgui.beepSound();
		    copyFileIsValid = false;
		    String msg = "Sorry, errormessages cannot be copied.";
		    JOptionPane.showMessageDialog(LTOOLSgui.myProg, msg, caption, JOptionPane.ERROR_MESSAGE);
		}
		debugOutput("item to copy: " + ((ListItemFile) copyListItemFiles.elementAt(i)).getFullFilePath());
		if (copyFileIsValid == false)
                {   copyListItemFiles.removeAllElements();
		    return;
		}
	    }
	} else if (getActiveDirFilePane() == linuxPane)
        {   copyListItemFiles.removeAllElements();
	    copyListItemFiles = linuxPane.getSelectedListItemFile();
	    copyFileIsLinux = true;				// it's a linux-file 
	    copyFileIsValid = true;
	    // if it is not a directory but a file, it is valid 
	    for (int i = 0; i < copyListItemFiles.size(); i++)
            {   if (((ListItemFile) copyListItemFiles.elementAt(i)).isDirectory())
                {   LTOOLSgui.beepSound();
		    copyFileIsValid = false;
		    String msg = "Sorry, directories cannot be copied";
		    JOptionPane.showMessageDialog(LTOOLSgui.myProg, msg, caption, JOptionPane.ERROR_MESSAGE);
		} else if (((ListItemFile) copyListItemFiles.elementAt(i)).isError())
                {   LTOOLSgui.beepSound();
		    copyFileIsValid = false;
		    String msg = "Sorry, errormessages cannot be copied";
		    JOptionPane.showMessageDialog(LTOOLSgui.myProg, msg, caption, JOptionPane.ERROR_MESSAGE);
		} else if (((ListItemFile) copyListItemFiles.elementAt(i)).isLink())
                {   LTOOLSgui.beepSound();
		    copyFileIsValid = false;
		    String msg = "Sorry, links cannot be copied.";
		    JOptionPane.showMessageDialog(LTOOLSgui.myProg, msg, caption, JOptionPane.ERROR_MESSAGE);
		}
		debugOutput("item to copy: " + ((ListItemFile) copyListItemFiles.elementAt(i)).getFullFilePath());
		if (copyFileIsValid == false)
                {   copyListItemFiles.removeAllElements();
		    return;
		}
	    }
	} else
        {   LTOOLSgui.beepSound();
	    JOptionPane.showMessageDialog(this, "Select file to copy.", caption, JOptionPane.ERROR_MESSAGE);
	}
    }
    /** the pasteListItemFiles()-method calls the method of the active DirFilePane to
     * paste in the DirFilePane() the files of the copyListItemFiles-Vector.
     */
    public void pasteListItemFiles()
    {   debugOutput("pasteListItemFiles()");
	String caption = "LTOOLS";
	if (copyFileIsValid == false)
        {   LTOOLSgui.beepSound();
	    String msg = "You must copy a file before you can paste it.";
	     JOptionPane.showMessageDialog(this, msg, caption, JOptionPane.ERROR_MESSAGE);
	     return;
	}
	if (getActiveDirFilePane() == dosPane)
        {   debugOutput("paste files to dos.");
	    dosPane.pasteToDos(true);
	} else if (getActiveDirFilePane() == linuxPane)
        {   debugOutput("paste files to linux.");
	    linuxPane.pasteToLinux(true);
	} else
        {   LTOOLSgui.beepSound();
	    String msg = "Select directory to paste in.";	// "(in right directory tree only)" ; 
	    JOptionPane.showMessageDialog(this, msg, caption, JOptionPane.ERROR_MESSAGE);
	}
	LTOOLSgui.setCopyToDirectory(null);			// reset the copyToDirectory 
    }
    /** the setConfirmDeleteView()-method sets the icon and the text of the
     * confirmDeleteToggleButton and sets the togglebutton and the menuItem selected
     * or not.
     */
    private void setConfirmDeleteView()
    {   if (confirmDelete)
        {   confirmDeleteJToggleButton.setIcon(confirmIcon);
	    confirmDeleteJToggleButton.setText("Confirm");
	} else
        {   confirmDeleteJToggleButton.setIcon(confirmNotIcon);
	    confirmDeleteJToggleButton.setText("No Confirm");
	}
	confirmDeleteJToggleButton.setSelected(confirmDelete);
	confirmDeleteJCheckBoxMenuItem.setSelected(confirmDelete);
    }
    /** the setReadOnlyView()-method sets the icon and the text of the
     * readOnlyToggleButton and sets the togglebutton and the menuItem selected or not.
     */
    private void setReadOnlyView()
    {   if (isLinuxReadOnly)
        {   readOnlyJToggleButton.setIcon(readOnlyIcon);
	    readOnlyJToggleButton.setText("Linux Read Only");
	} else
        {   readOnlyJToggleButton.setIcon(readWriteIcon);
	    readOnlyJToggleButton.setText("ReadWrite");
	}
	linuxReadOnlyJCheckBoxMenuItem.setSelected(isLinuxReadOnly);
	readOnlyJToggleButton.setSelected(isLinuxReadOnly);
    }
    /** The beepSound()-method makes a short beep for e.g. JOptionPane's showing an
     * error-message
     */
    static public void beepSound()
    {   Toolkit.getDefaultToolkit().beep();			// Beep 
    }
}

