/*
 * ListItemFile.java
 */
//import java.util.*;
import javax.swing.*;

/** The ListItemFile-class represents a file/directory in the JTable of the
 * DirFilePane-class. It contains all necessary information of a file.
 * @author Joerg Holzaepfel
 */
public class ListItemFile implements Comparable
{
    public static ImageIcon [] icons = {                        // Icons to represent a filetype in the JTable
        new ImageIcon("images/FOLDER16x16.gif"),                // 0
        new ImageIcon("images/FILE16x16.gif"),                  // 1
        new ImageIcon("images/FOLDERUP16x16.gif"),              // 2
        new ImageIcon("images/Word16x16.gif"),                  // 3
        new ImageIcon("images/Access16x16.gif"),                // 4
        new ImageIcon("images/Excel16x16.gif"),                 // 5
        new ImageIcon("images/Powerpoint16x16.gif"),            // 6
        new ImageIcon("images/ExeCom16x16.gif"),                // 7
        new ImageIcon("images/java16x16.gif"),                  // 8
        new ImageIcon("images/winzip16x16.gif"),                // 9
        new ImageIcon("images/error16x16.gif"),                 // 10
        new ImageIcon("images/html16x16.gif"),                  // 11
        new ImageIcon("images/Link16x16.gif"),                  // 12
        new ImageIcon("images/mp316x16.gif"),                   // 13
        new ImageIcon("images/pdf16x16.gif"),                   // 14
        new ImageIcon("images/avi16x16.gif"),                   // 15
        new ImageIcon("images/Bitmap16x16.gif")                 // 16
    }; 
    public static int ICON_FOLDER     = 0 ;                     // the index of the ImageIcon in the array icons
    public static int ICON_FILE       = 1 ;
    public static int ICON_FOLDERUP   = 2 ;
    public static int ICON_WORD       = 3 ;
    public static int ICON_ACCESS     = 4 ;
    public static int ICON_EXCEL      = 5 ;
    public static int ICON_POWERPOINT = 6 ;
    public static int ICON_EXE        = 7 ;
    public static int ICON_JAVA       = 8 ;
    public static int ICON_ZIP        = 9 ;
    public static int ICON_ERROR      = 10 ;
    public static int ICON_HTML       = 11 ;
    public static int ICON_LINK       = 12 ;
    public static int ICON_MP3        = 13 ;
    public static int ICON_PDF        = 14 ;
    public static int ICON_AVI        = 15 ;
    public static int ICON_BITMAP     = 16 ;

    private String fileName;                                    // the name of the file/directory (without full path)
    private LToolGuiTableIconData tableIconData ;               // IconData for the TableCellRenderer (filename, icon)
    private boolean isDirectory ;                               // true if the file is a directory
    private boolean isLink ;                                    // true if the file is a symbolic link (just linux)
    private boolean isError ;                                   // true if the ListItemFile represents an error-message
    private String directory;                                   // the parent-directory containing this file/directory
    private String fullFilePath;                                // the full path of this file
    private String fileSize;                                    // the size of the file in bytes
    private String fileLastModified;                            // the last-modified date of this file
    private int fileIconIndex ;                                 // the index of the icon for this filetype
    private String linuxFileInfoString;                         // the full linuxString of the ldir-command for this 
                                                                // file (just Linux, DOS == "")
    // private boolean isReadOnly;
    
    /** this ListItemFile()-constructor is for showing errormessage int the JTable. It
     * sets just the fileName attribute of the class, the other attributes are
     * set with default-values for error-messages.
     * @param error the errormessage to show in the JTable
     */
    public ListItemFile(String error)
    {
        this.fileName            = error ;
        if (this.fileName == null)
        {
            this.fileName = "ERROR" ;
        }
        this.isDirectory         = false ;
        this.isError             = true ;
        this.directory           = "" ;
        this.fullFilePath        = "" ;
        this.fileSize            = "" ;
        this.fileLastModified    = "" ;
        this.linuxFileInfoString = "" ;
        this.fileIconIndex       = ICON_ERROR ;
        this.tableIconData       = new LToolGuiTableIconData(icons[this.fileIconIndex], this.fileName) ;
    }
    /** Creates a new instance of ListItemFile
     * @param fileName the filename of this file
     * @param isDirectory true if the file is a directory
     * @param directory the parent directory containing this file/directory
     * @param fullFilePath the full path of this file
     * @param fileSize the filesize of this file in Bytes
     * @param fileLastModified the last modified date of this file
     * @param linuxFileInfoString the linuxFileInfoString of the ldir-command with all infos of this file.
     * @param fileIconIndex the fileIconIndex of the icon to show in the JTable.
     */
    public ListItemFile(String fileName, boolean isDirectory, String directory, String fullFilePath, 
        String fileSize, String fileLastModified, String linuxFileInfoString, int fileIconIndex)
    {
        this.fileName           = fileName ;
        this.isDirectory        = isDirectory ;
        this.isError            = false ;
        this.directory          = directory ;
        this.fullFilePath       = fullFilePath ;
        this.fileSize           = fileSize ;
        this.fileLastModified   = fileLastModified ;
        this.linuxFileInfoString = linuxFileInfoString ;
        isLink = false ;
        if (linuxFileInfoString != "")
        {
            if (linuxFileInfoString.charAt(0) == 'l')
            {
                isLink = true ;
            }
        }
        this.fileIconIndex      = fileIconIndex ;
        if (isDirectory)
        {
            if (!this.fileName.equals(".."))
            {
                this.fileIconIndex = ICON_FOLDER ;
            }
        }
        else if (isLink())
        {
            this.fileIconIndex = ICON_LINK ;
        }
        else
        {
            if (this.fileName.toLowerCase().endsWith(".doc"))
            {
                this.fileIconIndex = ICON_WORD ;
            }
            else if (this.fileName.toLowerCase().endsWith(".mdb"))
            {
                this.fileIconIndex = ICON_ACCESS ;
            }
            else if (this.fileName.toLowerCase().endsWith(".xls"))
            {
                this.fileIconIndex = ICON_EXCEL ;
            }
            else if (this.fileName.toLowerCase().endsWith(".ppt"))
            {
                this.fileIconIndex = ICON_POWERPOINT ;
            }
            else if (this.fileName.toLowerCase().endsWith(".exe") ||
                     this.fileName.toLowerCase().endsWith(".com"))
            {
                this.fileIconIndex = ICON_EXE ;
            }
            else if (this.fileName.toLowerCase().endsWith(".java"))
            {
                this.fileIconIndex = ICON_JAVA ;
            }
            else if (this.fileName.toLowerCase().endsWith(".zip"))
            {
                this.fileIconIndex = ICON_ZIP ;
            }
            else if (this.fileName.toLowerCase().endsWith(".html") ||
                     this.fileName.toLowerCase().endsWith(".htm"))
            {
                this.fileIconIndex = ICON_HTML ;
            }
            else if (this.fileName.toLowerCase().endsWith(".mp3") ||
                     this.fileName.toLowerCase().endsWith(".wav"))
            {
                this.fileIconIndex = ICON_MP3 ;
            }
            else if (this.fileName.toLowerCase().endsWith(".pdf"))
            {
                this.fileIconIndex = ICON_PDF ;
            }
            else if (this.fileName.toLowerCase().endsWith(".avi") ||
                     this.fileName.toLowerCase().endsWith(".mpg"))
            {
                this.fileIconIndex = ICON_AVI ;
            }
            else if (this.fileName.toLowerCase().endsWith(".bmp") ||
                     this.fileName.toLowerCase().endsWith(".jpg") ||
                     this.fileName.toLowerCase().endsWith(".gif") ||
                     this.fileName.toLowerCase().endsWith(".png"))
            {
                this.fileIconIndex = ICON_BITMAP ;
            }
            else
            {
                this.fileIconIndex = ICON_FILE ;
            }
        }
        this.tableIconData = new LToolGuiTableIconData(icons[this.fileIconIndex], this.fileName) ;
    }
    /** the getFileName()-method returns the name of the file
     * @return the name of the file, like "test.txt"
     */
    public String getFileName()
    {
        return fileName;
    }
    /** the getFileIconIndex()-method returns the index of the icon to show in the
     * JTable
     * @return the index of the icon in the icons-array
     */
    public int getFileIconIndex()
    {
        return fileIconIndex ;
    }
    /** the getImage()-method returns the ImageIcon with the given index of the
     * icons-array
     * @param index the index of the icon in the icons-array
     * @return the ImageIcon with the given index
     */
    public ImageIcon getImage(int index)
    {
        return icons[index] ;
    }
    /** the getTableIconData()-method returns the LToolGuiTableIconData of this file to
     * be shown in the JTable.
     * @return the LToolGuiTableIconData of this file
     */
    public LToolGuiTableIconData getTableIconData()
    {
        return tableIconData ;
    }
    /** the getFullFilePath()-method returns the full path of the file.
     * @return the full path of the file, like "/dev/hda6 /home/test/test.txt" or
     * "C:\temp\test.txt"
     */
    public String getFullFilePath()
    {
        return fullFilePath ;
    }
    /** the getFileSize()-method returns a string with the filesize in Bytes
     * @return the size of this file in Bytes (as a string)
     */
    public String getFileSize()
    {
        return fileSize ;
    }
    /** the getFileLastModified()-method gets the lastModified date of this file
     * @return the last modified date of this file
     */
    public String getFileLastModified()
    {
        return fileLastModified ;
    }
    /** the getAccess()-method gets the access-rights of the linux-file (not for
     * DOS-files)
     * @return the accessrights of a linuxfile
     */
    public String getAccess()
    {
        
        if (linuxFileInfoString.equals(""))                     // it's a dos ListItemFile
        {
            return "" ;
        }
        return DirFilePane.getLdirField(linuxFileInfoString, 0) ;
    }
    /** the getUserID()-method gets the UserID of the linux-file (not for DOS-files)
     * @return the UserID of the linuxfile (not for DOS-files)
     */
    public String getUserID()
    {
        if (linuxFileInfoString.equals(""))                     // it's a dos ListItemFile
        {
            return "" ;
        }
        return DirFilePane.getLdirField(linuxFileInfoString, 1) ;
    }
    /** the getGroupD()-method gets the GrouID of the linux-file (not for DOS-files)
     * @return the GroupID of the linuxfile (not for DOS-files)
     */
    public String getGroupID()
    {
        if (linuxFileInfoString.equals(""))                     // it's a dos ListItemFile
        {
            return "" ;
        }
        return DirFilePane.getLdirField(linuxFileInfoString, 2) ;
    }
    /** the getFileTyp()-method returns the extension of this file(not for directories)
     * to show this extension in the JTable as the "Type" coloumn.
     * @return the extension of this file (if it's not a directory or a file without extension)
     */
    public String getFileTyp()
    {
        if (!isDirectory())
        {
            String typ = "" ;
            if (getFileName().lastIndexOf(".") > -1)
            {
                typ = getFileName().substring(getFileName().lastIndexOf(".")) ;
            }
            return typ ;
        }
        return "" ;
    }
    /** the isDirectory()-method returns true if the file is directory
     * @return true if the file is a directory
     */
    public boolean isDirectory()
    {
        return isDirectory ;
    }    
    /** the isLink()-method returns true if the file is symbolic link (just for linux)
     * @return true if the linux-file is a symbolic link
     */
    public boolean isLink()
    {
        return isLink ;
    }
    /** the isError()-method returns true if this ListItemFile represents an
     * error-message.
     * @return true if this ListItemFile represents an error-message
     */
    public boolean isError()
    {
        return isError ;
    }
    /** the setFullFilePath() sets the full path of this file
     * @param path the full path of this file
     */
    public void setFullFilePath(String path)
    {
        fullFilePath = path ;
    }

    /** Compares this object with the specified object for order.  Returns a
     * negative integer, zero, or a positive integer as this object is less
     * than, equal to, or greater than the specified object.<p>
     *
     * In the foregoing description, the notation
     * <tt>sgn(</tt><i>expression</i><tt>)</tt> designates the mathematical
     * <i>signum</i> function, which is defined to return one of <tt>-1</tt>,
     * <tt>0</tt>, or <tt>1</tt> according to whether the value of <i>expression</i>
     * is negative, zero or positive.
     *
     * The implementor must ensure <tt>sgn(x.compareTo(y)) ==
     * -sgn(y.compareTo(x))</tt> for all <tt>x</tt> and <tt>y</tt>.  (This
     * implies that <tt>x.compareTo(y)</tt> must throw an exception iff
     * <tt>y.compareTo(x)</tt> throws an exception.)<p>
     *
     * The implementor must also ensure that the relation is transitive:
     * <tt>(x.compareTo(y)&gt;0 &amp;&amp; y.compareTo(z)&gt;0)</tt> implies
     * <tt>x.compareTo(z)&gt;0</tt>.<p>
     *
     * Finally, the implementer must ensure that <tt>x.compareTo(y)==0</tt>
     * implies that <tt>sgn(x.compareTo(z)) == sgn(y.compareTo(z))</tt>, for
     * all <tt>z</tt>.<p>
     *
     * It is strongly recommended, but <i>not</i> strictly required that
     * <tt>(x.compareTo(y)==0) == (x.equals(y))</tt>.  Generally speaking, any
     * class that implements the <tt>Comparable</tt> interface and violates
     * this condition should clearly indicate this fact.  The recommended
     * language is "Note: this class has a natural ordering that is
     * inconsistent with equals."
     *
     * @param   o the Object to be compared.
     * @return  a negative integer, zero, or a positive integer as this object
     * 		is less than, equal to, or greater than the specified object.
     *
     * @throws ClassCastException if the specified object's type prevents it
     *         from being compared to this Object.
     *
     */
    public int compareTo(Object o) 
    {
        if (o == null)
        {
            return 1 ;
        }
        if (o.getClass() != this.getClass())
        {
            throw new ClassCastException() ;
        }
        ListItemFile item = (ListItemFile) o ;
        if (this.linuxFileInfoString.equals(""))
        {
            // System.out.println("ListItemFile-compareTo: DOS") ;
            return this.getFileName().toLowerCase().compareTo(item.getFileName().toLowerCase()) ;
        }
        // System.out.println("ListItemFile-compareTo: LINUX") ;
        return this.getFileName().compareTo(item.getFileName()) ;
    }
    /** the printItem()-method is just for debugging. */
    public void printItem()
    {
        System.out.println("############################################################################") ;
        System.out.println("linuxFileInfoString : " + this.linuxFileInfoString) ;
        System.out.println("FileName            : " + this.getFileName()) ;
        System.out.println("FullFilePath        : " + this.getFullFilePath()) ;
        System.out.println("FileSize            : " + this.getFileSize()) ;
        System.out.println("FileTyp             : " + this.getFileTyp()) ;
        System.out.println("isDirectory         : " + this.isDirectory()) ;
        System.out.println("isLink              : " + this.isLink()) ;
        System.out.println("Access              : " + this.getAccess()) ;
        System.out.println("FileIconIndex       : " + this.getFileIconIndex()) ;
        System.out.println("FileLastModified    : " + this.getFileLastModified()) ;
        System.out.println("GroupID             : " + this.getGroupID()) ;
        System.out.println("UserID              : " + this.getUserID()) ;
        System.out.println("############################################################################") ;
    }
}
