/* ar.c - the archiver

   AUTHOR: Gregory Pietsch
   
   DESCRIPTION:
   
   The ar utility can be used to create and maintain groups of files combined
   into an archive. Once an archive has been created, new files can be added,
   and existing files can be extracted, deleted, or replaced. If an archive
   consists entirely of printable files, the entire archive shall be printable.

   COPYRIGHT NOTICE AND DISCLAIMER:

   Copyright (C) 2020 Gregory Pietsch

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 3
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.

*/

/* includes */
#include "config.h"
#include <ctype.h>
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#include <stdio.h>
#if defined (STDC_HEADERS) || defined (HAVE_STDLIB_H)
#include <stdlib.h>
#endif
#if defined (STDC_HEADERS) || defined (HAVE_STRING_H)
#include <string.h>
#endif
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#include <time.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>	/* need access */
#endif
#include <sys/stat.h>
#include "defines.h"
#include "dynstr.h"
#include "getopt.h"

/* macros */
#if !defined (HAVE_REMOVE) && defined (HAVE_UNLINK)
#define remove(x)	unlink(x)
#endif
#if !defined (HAVE_RENAME) && defined (HAVE_UNLINK) && defined (HAVE_LINK)
#define rename(x, y)	(link(x,y)?(-1):(unlink(x)))
#endif

/* typedefs */

typedef enum RELATIVE_POSITION_T
{
  NO_RELATIVE_POS,
  BEFORE,
  AFTER
} RELATIVE_POSITION_T;

typedef struct AR_HEADER_T
{
  STRING_T *name;
  time_t mtime;
  uid_t uid;
  gid_t gid;
  mode_t mode;
  off_t size;
} AR_HEADER_T;

/* static data */
static int f_create = 0, f_overwrite = 1, f_symtab = 0, f_truncate = 0,
  f_update = 0, f_verbose = 0;
static char f_command = '\0';
static char *progname = 0;
static char *posname = 0;
static char *archive_name = 0;
static DAS_ARRAY_T *filenames = 0;
static char tmp_filename[L_tmpnam + 1];
static RELATIVE_POSITION_T relpos = NO_RELATIVE_POS;
static int show_help = 0, show_version = 0;
static char shortopts[] = "abcCdimpqrstTuvx";
static struct option longopts[] = {
  {"help", 0, &show_help, 1},
  {"version", 0, &show_version, 1},
  {0, 0, 0, 0}
};

/* functions */

#ifndef HAVE_MEMCMP
/* No memcmp, have to roll our own.  */
/* Nonzero if either X or Y is not aligned on a "long" boundary.  */
#ifdef _ALIGN
#define UNALIGNED(X, Y) \
  (((long)X & (sizeof (long) - 1)) | ((long)Y & (sizeof (long) - 1)))
#else
#define UNALIGNED(X, Y) (0)
#endif

/* How many bytes are copied each iteration of the word copy loop.  */
#define LBLOCKSIZE (sizeof (long))

/* Threshhold for punting to the byte copier.  */
#define TOO_SMALL(LEN)  ((LEN) < LBLOCKSIZE)

static int (memcmp) (const void *m1, const void *m2, size_t n)
{
  const unsigned char *s1 = (const unsigned char *) m1;
  const unsigned char *s2 = (const unsigned char *) m2;
#ifndef _OPTIMIZED_FOR_SIZE
  const unsigned long *a1, *a2;

  /* If the size is too small or either pointer is unaligned, punt into the
     byte compare loop.  Hopefully, this will not turn up in inner loops.  */
  if (!TOO_SMALL (n) && !UNALIGNED (s1, s2))
    {
      a1 = (unsigned long *) s1;
      a2 = (unsigned long *) s2;
      while (n >= LBLOCKSIZE)
	{
	  if (*a1 != *a2)
	    break;
	  ++a1;
	  ++a2;
	  n -= LBLOCKSIZE;
	}
      /* Check n mod LBLOCKSIZE remaining characters */
      s1 = (unsigned char *) a1;
      s2 = (unsigned char *) a2;
    }
#endif /* _OPTIMIZED_FOR_SIZE */
  while (n--)
    {
      if (*s1 != *s2)
	return (*s1 < *s2) ? -1 : +1;
      ++s1;
      ++s2;
    }
  return 0;
}

#undef LBLOCKSIZE
#undef TOO_SMALL
#undef UNALIGNED
#endif /* !HAVE_MEMCMP */

/* ARHctor: constructor for AR_HEADER_T.  */
void
ARHctor (AR_HEADER_T * this)
{
  this->name = DScreate ();
  this->mtime = 0;
  this->uid = 0;
  this->gid = 0;
  this->mode = 0;
  this->size = 0;
}

/* ARHdtor: destructor for AR_HEADER_T.  */
void
ARHdtor (AR_HEADER_T * x)
{
  DSdestroy (x->name);
  x->name = 0;
  x->mtime = 0;
  x->uid = 0;
  x->gid = 0;
  x->mode = 0;
  x->size = 0;
}

/* ARHcreate: Create an AR_HEADER_T out of thin air.  */
AR_HEADER_T *
ARHcreate (void)
{
  AR_HEADER_T *x = malloc (sizeof (AR_HEADER_T));

  if (x == 0)
    Nomemory ();
  ARHctor (x);
  return x;
}

/* ARHdestroy: Destroy an AR_HEADER_T.  */
void
ARHdestroy (AR_HEADER_T * x)
{
  ARHdtor (x);
  free (x);
}

/* file_exists: Return 1 iff a file exists, else 0.  */
int
file_exists (char *name)
{
#ifdef HAVE_ACCESS
  return access (name, F_OK) == 0;
#else
  FILE *f;

  f = fopen (name, "rb");
  if (f)
    {
      fclose (f);
      return 1;
    }
  else
    return 0;
#endif    
}

/* read_line: read a line from a file.  Return the line.  */
STRING_T *
read_line (FILE * f, STRING_T * s)
{
  char b[BUFSIZ];

  if (s == 0)
    s = DScreate ();
  else
    DSresize (s, 0, 0);
  while (fgets (b, BUFSIZ, f) != 0)
    {
      DSappendcstr (s, b, NPOS);
      if (DSget_at (s, DSlength (s) - 1) == '\n')
	break;
    }
  return s;
}

/* get_numeric: Parse a number from a string.  */
unsigned long long
get_numeric (STRING_T * s, size_t p, size_t l, int b)
{
  unsigned long long x = 0ULL;
  size_t i;
  char c;

  for (i = 0; i < l; ++i)
    {
      c = DSget_at (s, p + i);
      if (isdigit (c) && (c - '0') < b)
	x = x * b + c - '0';
      else
	break;
    }
  return x;
}

/* read_header: The string read from the file shall be in the format:
   char name[16];
   char mtime[12];
   char uid[6];
   char gid[6];
   char mode[8];
   char size[10];
   char trailer[2]; -- should be "`<newline>"

   Returns: 1 if there was a header in the file, 0 if not.
*/
int
read_header (FILE * f, AR_HEADER_T * hdr)
{
  STRING_T *s = 0;
  size_t n = 0, i;
  int c;

  s = read_line (f, s);
  if (DSlength (s) == 0)
    {
      DSdestroy (s);
      return 0;
    }
  hdr->mtime = (time_t) get_numeric (s, 16, 12, 10);
  hdr->uid = (uid_t) get_numeric (s, 28, 6, 10);
  hdr->gid = (gid_t) get_numeric (s, 34, 6, 10);
  hdr->mode = (mode_t) get_numeric (s, 40, 8, 8);
  hdr->size = (off_t) get_numeric (s, 48, 10, 10);
  if (hdr->name == 0)
    hdr->name = DScreate ();
  else
    DSresize (hdr->name, 0, 0);
  if (memcmp (DScstr (s), "#1/", 3) == 0)
    {
      n = (size_t) get_numeric (s, 3, 13, 10);
      hdr->size -= n;
      while (n--)
	{
	  c = getc (f);
	  DSappendchar (hdr->name, c, 1);
	}
    }
  else
    {
      n = 16;
      for (i = 0; i < n; ++i)
	{
	  c = DSget_at (s, i);
	  if (isspace (c))
	    break;
	  else
	    DSappendchar (hdr->name, c, 1);
	}
    }
  DSdestroy (s);
  return 1;
}

/* long_filename: Is a filename too long to fit in 16 characters or
   does it have a space? */
int
long_filename (STRING_T * name)
{
  return (DSlength (name) >= 16
	  || DSfind_first_of (name, " \f\n\r\t\v", 0, NPOS) != NPOS);
}

/* write_header: Write an AR_HEADER_T to a file.  */
void
write_header (FILE * f, AR_HEADER_T * hdr)
{
  char buf[61], x[17];
  size_t i, l;

  l = DSlength (hdr->name);
  if (!long_filename (hdr->name))
    {
      /* normal */
      sprintf (buf, "%-16s%-12llu%-6llu%-6llu%-8o%-10llu`\n",
	       DScstr (hdr->name), (unsigned long long) hdr->mtime,
	       (unsigned long long) hdr->uid,
	       (unsigned long long) hdr->gid,
	       (unsigned) hdr->mode, (unsigned long long) hdr->size);
      fputs (buf, f);
    }
  else
    {
      sprintf (x, "#1/%u", (unsigned) l);
      sprintf (buf, "%-16s%-12llu%-6llu%-6llu%-8o%-10llu`\n",
	       x, (unsigned long long) hdr->mtime,
	       (unsigned long long) hdr->uid,
	       (unsigned long long) hdr->gid,
	       (unsigned) hdr->mode, (unsigned long long) hdr->size + l);
      fputs (buf, f);
      for (i = 0; i < l; ++i)
	fputc (DSget_at (hdr->name, i), f);
    }
}

/* open_tmpfile: Open a temporary filename for writing.  */
FILE *
open_tmpfile (void)
{
  FILE *f = 0;

  tmpnam (tmp_filename);
  f = fopen (tmp_filename, "wb");
  if (f == 0)
    {
      fprintf (stderr, "%s: Could not open \'%s\' for writing.\n", progname,
	       tmp_filename);
      exit (EXIT_FAILURE);
    }
  fputs ("!<arch>\n", f);
  return f;
}

/* open_archive: Open an archive for reading.  */
FILE *
open_archive (char *name)
{
  FILE *f = 0;
  STRING_T *s = 0;

  if (file_exists (name))
    {
      f = fopen (name, "rb");
      if (f == 0)
	{
	  fprintf (stderr, "%s: Could not open archive \'%s\' for reading.\n",
		   progname, name);
	  exit (EXIT_FAILURE);
	}
      s = read_line (f, s);
      if (DScomparecstr (s, "!<arch>\n", 0, NPOS) != 0)
	{
	  fprintf (stderr, "%s: File \'%s\' is not an archive.\n", progname,
		   name);
	  exit (EXIT_FAILURE);
	}
      DSdestroy (s);
      return f;
    }
  else
    {
      fprintf (stderr, "%s: Cannot open file \'%s\'.\n", progname, name);
      exit (EXIT_FAILURE);
    }
  return 0;			/* Should never get here ... */
}

/* copy_file: copy the contents of one file to another.
   out can also be a null pointer.  */
void
copy_file (FILE * in, FILE * out, size_t n)
{
  unsigned char buf[BUFSIZ];
  size_t x;

  while (n >= BUFSIZ)
    {
      x = fread (buf, BUFSIZ, 1, in);
      if (x && out)
	fwrite (buf, BUFSIZ, 1, out);
      n -= BUFSIZ;
    }
  x = fread (buf, n, 1, in);
  if (x && out)
    fwrite (buf, n, 1, out);
}

/* delete_files: Delete files from an archive. */
void
delete_files (void)
{
  FILE *arf = 0, *outf = 0;
  AR_HEADER_T *hdr = ARHcreate ();
  size_t i, n;
  STRING_T *s = 0;

  arf = open_archive (archive_name);
  outf = open_tmpfile ();
  while (!feof (arf))
    {
      if (read_header (arf, hdr) == 0)
	break;
      for (i = 0; i < DAS_length (filenames); ++i)
	{
	  s = DAS_get_at (filenames, i);
	  if (DScompare (s, hdr->name, 0, NPOS) == 0)
	    break;
	}
      n = hdr->size;
      if ((long_filename (hdr->name) ? n : n + DSlength (hdr->name)) & 1)
	++n;
      if (i < DAS_length (filenames))
	{
	  /* skip file */
	  if (f_verbose)
	    printf ("d - %s\n", DScstr (hdr->name));
	  copy_file (arf, 0, n);
	  DAS_remove (filenames, i, 1);
	}
      else
	{
	  /* write file */
	  write_header (outf, hdr);
	  copy_file (arf, outf, n);
	}
    }
  ARHdestroy (hdr);
  fclose (arf);
  fclose (outf);
  remove (archive_name);
  rename (tmp_filename, archive_name);
}

/* move_files: Move files around in an archive.  */
void
move_files (void)
{
  FILE *arf = 0, *outf = 0, *tmpf = 0, *f;
  char outf_name[L_tmpnam + 1], tmpf_name[L_tmpnam + 1];
  AR_HEADER_T *hdr = ARHcreate ();
  size_t i, l, n;
  STRING_T *s = 0;
  struct stat tmpf_stat;

  /* First, extract the files to be moved into a temporary file.  */
  l = DAS_length (filenames);
  if (l == 0)
    return;
  arf = open_archive (archive_name);
  outf = open_tmpfile ();
  strcpy (outf_name, tmp_filename);
  tmpnam (tmpf_name);
  tmpf = fopen (tmpf_name, "wb");
  if (tmpf == 0)
    {
      fprintf (stderr, "%s: Cannot open temporary file\n", progname);
      exit (EXIT_FAILURE);
    }
  while (!feof (arf))
    {
      if (read_header (arf, hdr) == 0)
	break;
      for (i = 0; i < DAS_length (filenames); ++i)
	{
	  s = DAS_get_at (filenames, i);
	  if (DScompare (s, hdr->name, 0, NPOS) == 0)
	    break;
	}
      n = hdr->size;
      f = (i < DAS_length (filenames)) ? tmpf : outf;
      /* extract file */
      write_header (f, hdr);
      copy_file (arf, f, n);
      if ((long_filename (hdr->name) ? n : n + DSlength (hdr->name)) & 1)
	{
	  getc (arf);
	  putc ('\n', f);
	}
      DAS_remove (filenames, i, 1);
    }
  fclose (arf);
  fclose (outf);
  fclose (tmpf);
  /* now insert files at the right spot */
  outf = open_archive (outf_name);
  f = open_tmpfile ();
  if (stat (tmpf_name, &tmpf_stat))
    {
      fprintf (stderr, "%s: temporary file does not exist. Strange...\n",
	       progname);
      exit (EXIT_FAILURE);
    }
  while (!feof (outf))
    {
      if (read_header (outf, hdr) == 0)
	break;
      if (relpos == BEFORE
	  && DScomparecstr (hdr->name, posname, 0, NPOS) == 0)
	{
	  tmpf = fopen (tmpf_name, "rb");
	  copy_file (tmpf, f, tmpf_stat.st_size);
	  fclose (tmpf);
	  remove (tmpf_name);
	}
      write_header (f, hdr);
      n = hdr->size;
      copy_file (outf, f, n);
      if ((long_filename (hdr->name) ? n : n + DSlength (hdr->name)) & 1)
	{
	  getc (outf);
	  putc ('\n', f);
	}
      if (relpos == AFTER && DScomparecstr (hdr->name, posname, 0, NPOS) == 0)
	{
	  tmpf = fopen (tmpf_name, "rb");
	  copy_file (tmpf, f, tmpf_stat.st_size);
	  fclose (tmpf);
	  remove (tmpf_name);
	}
    }
  if (file_exists (tmpf_name))
    {
      /* Move files to the end of the archive.  */
      tmpf = fopen (tmpf_name, "rb");
      copy_file (tmpf, f, tmpf_stat.st_size);
      fclose (tmpf);
      remove (tmpf_name);
    }
  ARHdestroy (hdr);
  fclose (f);
  fclose (outf);
  remove (archive_name);
  remove (outf_name);
  remove (tmpf_name);
  rename (tmp_filename, archive_name);
}

/* print_files: Output an archive's contents. */
void
print_files (void)
{
  FILE *arf = 0;
  AR_HEADER_T *hdr = ARHcreate ();
  size_t i, l, n;
  STRING_T *s = 0;

  arf = open_archive (archive_name);
  l = DAS_length (filenames);
  while (!feof (arf))
    {
      if (read_header (arf, hdr) == 0)
	break;
      for (i = 0; i < l; ++i)
	{
	  s = DAS_get_at (filenames, i);
	  if (DScompare (s, hdr->name, 0, NPOS) == 0)
	    break;
	}
      n = hdr->size;
      if ((long_filename (hdr->name) ? n : n + DSlength (hdr->name)) & 1)
	++n;
      if (l == 0 || i < l)
	{
	  if (f_verbose)
	    printf ("\n%s\n\n", DScstr (hdr->name));
	  else
	    puts (DScstr (hdr->name));
	  copy_file (arf, stdout, n);
	}
      else
	copy_file (arf, 0, n);
    }
  ARHdestroy (hdr);
  fclose (arf);
}

/* append_files: Append files to an existing archive, or create a new 
   archive. */
void
append_files (void)
{
  FILE *inf = 0, *outf = 0;
  int archive_existed = 0;
  AR_HEADER_T *hdr = ARHcreate ();
  struct stat st;
  size_t i, l;

  archive_existed = file_exists (archive_name);
  if (archive_existed)
    {
      outf = fopen (archive_name, "ab");
      if (outf == 0)
	{
	  fprintf (stderr, "%s: Cannot open archive \'%s\'.\n", progname,
		   archive_name);
	  exit (EXIT_FAILURE);
	}
    }
  else
    {
      outf = open_tmpfile ();
      if (f_create == 0)
	fprintf (stderr, "%s: WARNING: Creating archive \'%s\'.", progname,
		 archive_name);
    }
  hdr->name = DScreate ();
  for (i = 0, l = DAS_length (filenames); i < l; ++i)
    {
      DSassign (hdr->name, DAS_get_at (filenames, i), 0, NPOS);
      if (!file_exists (DScstr (hdr->name)))
	continue;
      if (stat (DScstr (hdr->name), &st) != 0)
	{
	  fprintf (stderr, "%s: stat failed\n", progname);
	  exit (EXIT_FAILURE);
	}
      hdr->mtime = st.st_mtime;
      hdr->uid = st.st_uid;
      hdr->gid = st.st_gid;
      hdr->mode = st.st_mode;
      hdr->size = st.st_size;
      write_header (outf, hdr);
      inf = fopen (DScstr (hdr->name), "rb");
      if (inf == 0)
	{
	  fprintf (stderr, "%s: Cannot open file \'%s\'.", progname,
		   DScstr (hdr->name));
	  exit (EXIT_FAILURE);
	}
      copy_file (inf, outf, st.st_size);
      if ((long_filename (hdr->name) ? hdr->size : hdr->size +
	   DSlength (hdr->name)) & 1)
	putc ('\n', outf);
      fclose (inf);
    }
  ARHdestroy (hdr);
  fclose (outf);
  if (!archive_existed)
    rename (tmp_filename, archive_name);
}

/* add_files: Add files to an archive.  */
void
add_files (void)
{
  FILE *inf = 0, *outf = 0, *arf = 0, *tmpf = 0, *f = 0;
  int archive_existed = 0;
  AR_HEADER_T *hdr = ARHcreate ();
  struct stat st, tmpf_stat;
  size_t i, l, n;
  char outf_name[L_tmpnam + 1], tmpf_name[L_tmpnam + 1];

  /* If the archive file doesn't exist, we can just append.  */
  archive_existed = file_exists (archive_name);
  if (!archive_existed)
    {
      append_files ();
      return;
    }
  /* The standard says that if no files are specified and the archive exists,
     the results are undefined.  Obviously, in this case the program shouldn't
     have to do anything.  */
  l = DAS_length (filenames);
  if (l == 0)
    return;
  /* Update the existing files in the archive.  */
  arf = open_archive (archive_name);
  outf = open_tmpfile ();
  strcpy (outf_name, tmp_filename);
  while (!feof (arf))
    {
      if (read_header (arf, hdr) == 0)
	break;
      for (i = 0; i < l; ++i)
	{
	  if (DScompare (hdr->name, DAS_get_at (filenames, i), 0, NPOS) == 0)
	    break;
	}
      stat (DScstr (hdr->name), &st);
      if (i >= l || f_update == 0 || hdr->mtime >= st.st_mtime)
	{
	  /* normal */
	  write_header (outf, hdr);
	  n = hdr->size;
	  if ((long_filename (hdr->name) ? n : n + DSlength (hdr->name)) & 1)
	    ++n;
	  copy_file (arf, outf, n);
	}
      else
	{
	  /* update */
	  hdr->mtime = st.st_mtime;
	  hdr->uid = st.st_uid;
	  hdr->gid = st.st_gid;
	  hdr->mode = st.st_mode;
	  hdr->size = st.st_size;
	  write_header (outf, hdr);
	  inf = fopen (DScstr (hdr->name), "rb");
	  if (inf == 0)
	    {
	      fprintf (stderr, "%s: Cannot open file \'%s\'.", progname,
		       DScstr (hdr->name));
	      exit (EXIT_FAILURE);
	    }
	  copy_file (inf, outf, st.st_size);
	  if ((long_filename (hdr->name) ? hdr->size : hdr->size +
	       DSlength (hdr->name)) & 1)
	    putc ('\n', outf);
	  fclose (inf);
	  DAS_remove (filenames, i, 1);
	}
    }
  fclose (arf);
  fclose (outf);
  /* Put the remainder of the files in the right spot.  */
  tmpnam (tmpf_name);
  tmpf = fopen (tmpf_name, "wb");
  if (tmpf == 0)
    {
      fprintf (stderr, "%s: Couldn't open temp file.\n", progname);
      exit (EXIT_FAILURE);
    }
  for (i = 0, l = DAS_length (filenames); i < l; ++i)
    {
      DSassign (hdr->name, DAS_get_at (filenames, i), 0, NPOS);
      if (!file_exists (DScstr (hdr->name)))
	continue;
      if (stat (DScstr (hdr->name), &st) != 0)
	{
	  fprintf (stderr, "%s: stat failed\n", progname);
	  exit (EXIT_FAILURE);
	}
      hdr->mtime = st.st_mtime;
      hdr->uid = st.st_uid;
      hdr->gid = st.st_gid;
      hdr->mode = st.st_mode;
      hdr->size = st.st_size;
      write_header (tmpf, hdr);
      inf = fopen (DScstr (hdr->name), "rb");
      if (inf == 0)
	{
	  fprintf (stderr, "%s: Cannot open file \'%s\'.", progname,
		   DScstr (hdr->name));
	  exit (EXIT_FAILURE);
	}
      copy_file (inf, tmpf, st.st_size);
      if ((long_filename (hdr->name) ? hdr->size : hdr->size +
	   DSlength (hdr->name)) & 1)
	putc ('\n', tmpf);
      fclose (inf);
    }
  fclose (tmpf);
  if (stat (tmpf_name, &tmpf_stat))
    {
      fprintf (stderr, "%s: temporary file does not exist. Strange...\n",
	       progname);
      exit (EXIT_FAILURE);
    }
  outf = open_archive (outf_name);
  f = open_tmpfile ();
  while (!feof (outf))
    {
      if (read_header (outf, hdr) == 0)
	break;
      if (relpos == BEFORE
	  && DScomparecstr (hdr->name, posname, 0, NPOS) == 0)
	{
	  tmpf = fopen (tmpf_name, "rb");
	  copy_file (tmpf, f, tmpf_stat.st_size);
	  fclose (tmpf);
	  remove (tmpf_name);
	}
      write_header (f, hdr);
      n = hdr->size;
      copy_file (outf, f, n);
      if ((long_filename (hdr->name) ? n : n + DSlength (hdr->name)) & 1)
	{
	  getc (outf);
	  putc ('\n', f);
	}
      if (relpos == AFTER && DScomparecstr (hdr->name, posname, 0, NPOS) == 0)
	{
	  tmpf = fopen (tmpf_name, "rb");
	  copy_file (tmpf, f, tmpf_stat.st_size);
	  fclose (tmpf);
	  remove (tmpf_name);
	}
    }
  if (file_exists (tmpf_name))
    {
      /* Move files to the end of the archive.  */
      tmpf = fopen (tmpf_name, "rb");
      copy_file (tmpf, f, tmpf_stat.st_size);
      fclose (tmpf);
      remove (tmpf_name);
    }
  ARHdestroy (hdr);
  fclose (f);
  fclose (outf);
  remove (archive_name);
  remove (outf_name);
  remove (tmpf_name);
  rename (tmp_filename, archive_name);
}

/* toc: Output an archive's table of contents. */
void
toc (void)
{
  FILE *arf = 0;
  AR_HEADER_T *hdr = ARHcreate ();
  size_t i, l, n, x;
  STRING_T *s = 0;
  char buftm[BUFSIZ], t[32], *q;
  const char *p;
  static const char rwx[] = "rwxrwxrwx";
  struct tm *ptm;

  arf = open_archive (archive_name);
  l = DAS_length (filenames);
  while (!feof (arf))
    {
      if (read_header (arf, hdr) == 0)
	break;
      for (i = 0; i < l; ++i)
	{
	  s = DAS_get_at (filenames, i);
	  if (DScompare (s, hdr->name, 0, NPOS) == 0)
	    break;
	}
      n = hdr->size;
      if ((long_filename (hdr->name) ? n : n + DSlength (hdr->name)) & 1)
	++n;
      if (l == 0 || i < l)
	{
	  if (f_verbose)
	    {
	      /* assemble verbose filename */
	      for (p = rwx, q = t, x = 0400; *p; ++p, ++q, x >>= 1)
		*q = (hdr->mode & x) ? *p : '-';
	      *q = '\0';
	      ptm = localtime (&(hdr->mtime));
	      strftime (buftm, BUFSIZ, "%b %d %H:%M %Y", ptm);
	      printf ("%s %u/%u %llu %s %s\n",
		      t, (unsigned) hdr->uid, (unsigned) hdr->gid,
		      (unsigned long long) hdr->size, buftm,
		      DScstr (hdr->name));
	    }
	  else
	    puts (DScstr (hdr->name));
	}
      copy_file (arf, 0, n);
    }
  ARHdestroy (hdr);
  fclose (arf);
}

/* extract_files: Extract files from an archive. */
void
extract_files (void)
{
  FILE *arf = 0, *outf = 0;
  AR_HEADER_T *hdr = ARHcreate ();
  size_t i, l, n;
  STRING_T *s = 0;

  arf = open_archive (archive_name);
  l = DAS_length (filenames);
  while (!feof (arf))
    {
      if (read_header (arf, hdr) == 0)
	break;
      for (i = 0; i < DAS_length (filenames); ++i)
	{
	  s = DAS_get_at (filenames, i);
	  if (DScompare (s, hdr->name, 0, NPOS) == 0)
	    break;
	}
      n = hdr->size;
      if ((l == 0 || i < DAS_length (filenames))
	  && (!file_exists (DScstr (hdr->name)) || f_overwrite))
	{
	  /* extract file */
	  if (f_verbose)
	    printf ("x - %s\n", DScstr (hdr->name));
	  outf = fopen (DScstr (hdr->name), "wb");
	  copy_file (arf, outf, n);
	  fclose (outf);
	  if ((long_filename (hdr->name) ? n : n + DSlength (hdr->name)) & 1)
	    getc (arf);
	  DAS_remove (filenames, i, 1);
	}
      else
	{
	  /* skip over file */
	  if ((long_filename (hdr->name) ? n : n + DSlength (hdr->name)) & 1)
	    ++n;
	  copy_file (arf, 0, n);
	}
    }
  ARHdestroy (hdr);
  fclose (arf);
}

void
version (void)
{
  puts (PACKAGE_STRING);
}

void
help (void)
{
#include "help.h"
}

void
parse_args (int argc, char *argv[])
{
  int opt;
  STRING_T *s = 0;

  progname = argv[0];
  while ((opt = getopt_long (argc, argv, shortopts, longopts, 0)) != EOF)
    {
      switch (opt)
	{
	case 'a':
	case 'b':
	case 'i':		/* relative position */
	  if (relpos != NO_RELATIVE_POS)
	    {
	      fprintf (stderr, "%s: Multiple relative positions\n", argv[0]);
	      exit (EXIT_FAILURE);
	    }
	  relpos = (opt == 'a') ? AFTER : BEFORE;
	  break;
	case 'c':		/* suppress diagnostic on create */
	  f_create = 1;
	  break;
	case 'C':		/* prevent extracted files from replacing 
				   like-named existing files */
	  f_overwrite = 0;
	  break;
	case 'd':		/* delete */
	case 'm':		/* move */
	case 'p':		/* print */
	case 'q':		/* append */
	case 'r':		/* add */
	case 't':		/* table of contents */
	case 'x':		/* extract */
	  if (f_command)
	    {
	      fprintf (stderr, "%s: Multiple commands given\n", argv[0]);
	      exit (EXIT_FAILURE);
	    }
	  f_command = opt;
	  break;
	case 's':		/* force regeneration of the symbol table */
	  f_symtab = 1;
	  break;
	case 'T':		/* Allow filename truncation of extracted files */
	  f_truncate = 1;
	  break;
	case 'u':		/* update older files in archive */
	  f_update = 1;
	  break;
	case 'v':		/* Give verbose output */
	  f_verbose = 1;
	  break;
	case 0:		/* --help and --version */
	  if (show_help)
	    help ();
	  if (show_version)
	    version ();
	  exit (EXIT_SUCCESS);
	  break;
	default:
	  fprintf (stderr, "%s: Unrecognized command -%c\n", argv[0], opt);
	  exit (EXIT_FAILURE);
	}
    }
  if (relpos)
    {
      if (argv[optind] == 0)
	{
	  fprintf (stderr, "%s: Relative position unspecified\n", argv[0]);
	  exit (EXIT_FAILURE);
	}
      posname = argv[optind++];
    }
  if (argv[optind] == 0)
    {
      fprintf (stderr, "%s: Archive file unspecified\n", argv[0]);
      exit (EXIT_FAILURE);
    }
  archive_name = argv[optind++];
  filenames = DAS_create ();
  s = DScreate ();
  for (; argv[optind] != 0; ++optind)
    {
      DSassigncstr (s, argv[optind], NPOS);
      DAS_append (filenames, s, 1, 0);
    }
  DSdestroy (s);
}

int
main (int argc, char *argv[])
{
  parse_args (argc, argv);
  switch (f_command)
    {
    case 'd':			/* delete */
      delete_files ();
      break;
    case 'm':			/* move */
      move_files ();
      break;
    case 'p':			/* print */
      print_files ();
      break;
    case 'q':			/* append */
      append_files ();
      break;
    case 'r':			/* add */
      add_files ();
      break;
    case 't':			/* table of contents */
      toc ();
      break;
    case 'x':			/* extract */
      extract_files ();
      break;
    default:
      fprintf (stderr, "%s: Command not specified\n", argv[0]);
      exit (EXIT_FAILURE);
    }
  return 0;
}

/* END OF FILE */
