/*
 *  split - split a file into pieces
 *  Copyright (C) 2016-2017 David McMackins II
 *
 *  Copying and distribution of this file, with or without modification,
 *  are permitted in any medium without royalty provided the copyright
 *  notice and this notice are preserved. This file is offered as-is,
 *  without any warranty.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "switch.h"

enum jobtype
{
	LINES = 'L',
	CHARS = 'B',
	CHUNKS = 'N'
};

static long
flength(FILE *f)
{
	int rc;
	long len;

	rc = fseek(f, 0, SEEK_END);
	if (rc)
		return -1;

	len = ftell(f);
	rewind(f);

	return len;
}

static long
get_line_count(FILE *f)
{
	int c;
	long count = 1;

	while ((c = fgetc(f)) != EOF)
		if ('\n' == c)
			++count;

	rewind(f);
	return count;
}

int
main(int argc, char *argv[])
{
	char *optarg, *prefix = "X", *name = NULL;
	enum jobtype jobtype = LINES;
	FILE *in = NULL;
	int rc = 0, c, extra = 0, optind = 1;
	long i, in_len, num_parts, size_each = 1000, written = 0;

	reorder_argv(argc, argv);
	while ((c = getswitch(argc, argv, "BLNVH?", &optind, &optarg))
		!= SWITCH_END)
	{
		switch (c)
		{
		case 'B':
		case 'L':
		case 'N':
			if (!optarg)
			{
				fprintf(stderr,
					"SPLIT: switch /%c requires an argument\r\n",
					c);
				return 1;
			}

			jobtype = c;

			rc = sscanf(optarg, "%ld", &size_each);
			if (rc != 1 || size_each < 0)
			{
				fprintf(stderr,
					"SPLIT: %s is an invalid number\r\n",
					optarg);
				return 1;
			}
			break;

		case 'V':
			printf("SPLIT for DOS v0.0.1\r\nCopyright (C) 2016-2017 David McMackins II\r\n");
			return 0;

		case 'H':
		case '?':
			printf("Usage: SPLIT [/B:BYTES | /L:LINES | /N:CHUNKS] [/V] [/H | /?] FILE [PREFIX]");
			return 0;

		case SWITCH_UNKNOWN:
			fprintf(stderr,
				"SPLIT: unknown switch '/%c'. See 'SPLIT /?' for usage.\r\n",
				c);
			return 1;
		}
	}

	switch (argc - optind)
	{
	case 2:
		prefix = argv[optind + 1];
		// SPILLS OVER!
	case 1:
		in = fopen(argv[optind], "rb");
		if (!in)
		{
			fprintf(stderr, "SPLIT: Error opening %s\r\n",
				argv[optind]);
			return 1;
		}
		break;

	default:
		if (argc == optind)
			fprintf(stderr,
				"SPLIT: Must provide an input file\r\n");
		else
			fprintf(stderr, "SPLIT: Too many arguments\r\n");

		return 1;
	}

	in_len = flength (in);
	if (in_len < 0)
	{
		fprintf(stderr,
			"SPLIT: Error getting input length. Too big?\r\n");
		rc = 1;
		goto end;
	}

	switch (jobtype)
	{
	case LINES:
		num_parts = get_line_count(in) / size_each + 1;
		break;

	case CHARS:
		num_parts = in_len / size_each + 1;
		break;

	case CHUNKS:
		num_parts = size_each;
		size_each = in_len / num_parts;
		extra = in_len % num_parts;
		break;
	}

	if (num_parts > 999)
	{
		fprintf(stderr, "SPLIT: Cannot have more than 999 parts\r\n");
		rc = 1;
		goto end;
	}

	name = malloc((strlen(prefix) + 5) * sizeof(char));
	if (!name)
	{
		fprintf(stderr, "SPLIT: Out of memory\r\n");
		rc = 1;
		goto end;
	}

	for (i = 1; i <= num_parts && written < in_len; ++i)
	{
		long to_write = size_each;
		FILE *out;

		if (extra)
		{
			++to_write;
			--extra;
		}

		sprintf(name, "%s.%ld", prefix, i);

		out = fopen(name, "wb");
		if (!out)
		{
			fprintf(stderr, "SPLIT: Failed to open %s\r\n", name);
			rc = 1;
			goto end;
		}

		while (to_write)
		{
			c = fgetc(in);
			if (EOF == c)
				break;

			rc = fputc(c, out);
			if (EOF == rc)
				break;

			if ('\n' == c || jobtype != LINES)
				--to_write;

			++written;
		}

		fflush(out);
		fclose(out);

		if (EOF == rc)
			goto end;
	}

end:
	if (in)
		fclose(in);

	if (name)
		free(name);

	return !!rc;
}

