/*
 *  wc - count lines, words, and characters of files
 *  Copyright (C) 2016-2017 David McMackins II
 *
 *  Copying and distribution of this file, with or without modification,
 *  are permitted in any medium without royalty provided the copyright
 *  notice and this notice are preserved. This file is offered as-is,
 *  without any warranty.
 */

#include <ctype.h>
#include <stdbool.h>
#include <stdio.h>

#include "switch.h"

enum modes
{
	LINES,
	WORDS,
	CHARS,
	NUM_MODES
};

static void
count(FILE *f, unsigned long long *counts)
{
	int c, last = ' ';

	for (c = 0; c < NUM_MODES; ++c)
		counts[c] = 0;

	while ((c = fgetc(f)) != EOF)
	{
		if ('\n' == c)
			++counts[LINES];

		if (isspace(c) && !isspace(last))
			++counts[WORDS];

		++counts[CHARS];
		last = c;
	}
}

static void
print_counts(const bool *wants, const unsigned long long *counts,
	const char *name, bool format)
{
	int i, count = 0;

	for (i = 0; i < NUM_MODES; ++i)
		if (wants[i])
			++count;

	format = format || (count > 1);

	for (i = 0; i < NUM_MODES; ++i)
	{
		if (wants[i])
		{
			const char *fmt = format ? "%9llu " : "%llu ";
			printf(fmt, counts[i]);
		}
	}

	printf("%s\r\n", name);
}

int
main(int argc, char *argv[])
{
	bool want_all = true, wants[NUM_MODES];
	char *optarg;
	int rc, c, optind = 1;
	unsigned long long counts[NUM_MODES], totals[NUM_MODES];

	for (c = 0; c < NUM_MODES; ++c)
	{
		wants[c] = false;
		counts[c] = 0;
		totals[c] = 0;
	}

	reorder_argv(argc, argv);
	while ((c = getswitch(argc, argv, "CLWVH?", &optind, &optarg))
		!= SWITCH_END)
	{
		switch (c)
		{
		case 'C':
			wants[CHARS] = true;
			want_all = false;
			break;

		case 'L':
			wants[LINES] = true;
			want_all = false;
			break;

		case 'W':
			wants[WORDS] = true;
			want_all = false;
			break;

		case 'V':
			printf("WC for DOS v0.1.1\r\nCopyright (C) 2016-2017 David McMackins II\r\n");
			return 0;

		case 'H':
		case '?':
			printf("Usage: wc [/L] [/W] [/C] [/H | /?] [FILE]...\r\n");
			return 0;

		case SWITCH_UNKNOWN:
			fprintf(stderr,
				"WC: unknown switch '/%c'. See 'WC /?' for usage.\r\n",
				c);
			return 1;
		}
	}

	if (want_all)
		for (c = 0; c < NUM_MODES; ++c)
			wants[c] = true;

	if (argc == optind)
	{
		count(stdin, counts);
		rc = 0;
		print_counts(wants, counts, "", false);
	}
	else
	{
		bool want_totals = (argc - optind) > 1;

		for (; optind < argc; ++optind)
		{
			FILE *f = fopen(argv[optind], "rb");
			if (!f)
			{
				fprintf(stderr, "WC: failed to open %s\r\n",
					argv[optind]);
				rc = 1;
				continue;
			}

			count(f, counts);
			print_counts(wants, counts, argv[optind], want_totals);

			for (c = 0; c < NUM_MODES; ++c)
				totals[c] += counts[c];
		}

		if (want_totals)
			print_counts(wants, totals, "total", true);
	}

	return !!rc;
}
