/* $Id: input.c,v 1.2 2002/07/29 17:28:51 cegger Exp $
******************************************************************************

   Input-KII: Input for KII
      
   Copyright (C) 2002 Paul Redmond

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdlib.h>
#include "config.h"
#include <ggi/internal/gii-dl.h>
#include <ggi/internal/gii_debug.h>

#include <kii/kii.h>

typedef struct {
	kii_context_t *ctx;
} kii_priv;

#define KII_PRIV(inp) ((kii_priv *)inp->priv)
#define KII_CTX(inp)  (KII_PRIV(inp)->ctx)

static gii_cmddata_getdevinfo kii_devinfo = {
    "Kernel Input Interface",
    "kii",
    emAll,
    GII_NUM_UNKNOWN,
    GII_NUM_UNKNOWN
};

static gii_event_mask GII_kii_handle_key(gii_input *inp, gii_event *ge,
	kii_event_t *ke)
{
	_giiEventBlank(ge, sizeof(gii_key_event));
	ge->any.size = sizeof(gii_key_event);
	ge->any.origin = inp->origin;
	switch (K_TYPE(ke->key.sym)) {
	
	case K_TYPE_META:
	case K_TYPE_FUNCTION:
	case K_TYPE_NUMPAD:
		return emZero;
	case K_TYPE_SPECIAL:
		switch (ke->key.sym) {
		case K_ENTER: ge->key.sym = GIIK_Enter; break;
		}
		break;
	case K_TYPE_CURSOR:
		switch (ke->key.sym) {
		
		case K_UP:    ge->key.sym = GIIK_Up; break;
		case K_DOWN:  ge->key.sym = GIIK_Down;break;
		case K_LEFT:  ge->key.sym = GIIK_Left;break;
		case K_RIGHT: ge->key.sym = GIIK_Right;break;
		}
		break;
	case K_TYPE_SHIFT:
	case K_TYPE_CONSOLE:
	case K_TYPE_ASCII:
	case K_TYPE_DEAD:
		return emZero;
	default:
		if (ke->key.sym < 0x80)
			ge->key.sym = ke->key.sym;
		else
			return emZero;
	}
	ge->any.type = ke->any.type;
	ge->key.label = ge->key.button = ge->key.sym;
	_giiEvQueueAdd(inp, ge);
	
	return (1 << ge->any.type);
}

static gii_event_mask GII_kii_handle_ptr(gii_input *inp, gii_event *ge,
	kii_event_t *ke)
{

	if (ke->any.type == KII_EV_PTR_STATE)
		return emZero;

	if (ke->any.type < KII_EV_PTR_BUTTON_PRESS) {

		/* pointer move event */
    		_giiEventBlank(ge, sizeof(gii_pmove_event));
		switch (ke->any.type) {
		
		case KII_EV_PTR_RELATIVE:
			ge->any.type = evPtrRelative; break;
			
		case KII_EV_PTR_ABSOLUTE:
			ge->any.type = evPtrAbsolute; break;
		
		default:
			return emZero;
		}
		ge->any.origin = inp->origin;
		ge->any.size = sizeof(gii_pmove_event);
		ge->pmove.x = ke->pmove.x;
		ge->pmove.y = ke->pmove.y;
		
		_giiEvQueueAdd(inp, ge);
	}
	else {
		int i;
		kii_u_t mask = 1;

    		_giiEventBlank(ge, sizeof(gii_pbutton_event));
		switch (ke->any.type) {
		
		case KII_EV_PTR_BUTTON_PRESS:
			ge->any.type = evPtrButtonPress; break;
			
		case KII_EV_PTR_BUTTON_RELEASE:
			ge->any.type = evPtrButtonRelease; break;
		
		default:
			return emZero;
		}
		ge->any.origin = inp->origin;
		ge->any.size = sizeof(gii_pbutton_event);
		for (i = 0; i < 32; i++) {
		
			if (ke->pbutton.button & mask) {
			
				ge->pbutton.button = i; /* zero based? +1? */
				_giiEvQueueAdd(inp, ge);
			}
			mask <<= 1;
		}
	}
	
	return (1 << ge->any.type);
}

static gii_event_mask GII_kii_poll(gii_input *inp, void *arg)
{
	kii_event_t *ke;
	gii_event_mask em = emZero;
	gii_event ge;

	if (!kiiEventAvailable(KII_CTX(inp)))
		return emZero;
		
	for (ke=kiiNextEvent(KII_CTX(inp));ke;ke=kiiNextEvent(KII_CTX(inp))) {

		if ((1 << ke->any.type) & KII_EM_KEYBOARD)
			em |= GII_kii_handle_key(inp, &ge, ke);
		else if ((1 << ke->any.type) & KII_EM_POINTER)
			em |= GII_kii_handle_ptr(inp, &ge, ke);
	}

	return em;
}

static int GII_kii_send_event(gii_input *inp, gii_event *ev)
{
	return 0;
}

static int GII_kii_close(gii_input *inp) 
{
    kii_priv *priv = KII_PRIV(inp);
        
    free(priv);

    GIIDPRINT_MISC("kii: exit OK.\n");

    return 0;
}


int GIIdlinit(gii_input *inp, const char *args, void *argptr)
{
	kii_priv *priv;
	int ret;

	GIIDPRINT_MISC("kii starting. (args=%s,argptr=%p)\n",
			args, argptr);
    
	if ((priv = inp->priv = malloc(sizeof(kii_priv))) == NULL) {

		return GGI_ENOMEM;
	}

	if(_giiRegisterDevice(inp,&kii_devinfo,NULL)==0) {
		
		free(priv);
		return GGI_ENOMEM;
	}

	if (KII_EOK != kiiInit(&KII_CTX(inp))) {
	
		free(priv);
		return GGI_ENODEVICE;
	}

	if (KII_EOK != kiiMapDevice(KII_CTX(inp))) {
#if 0
		kiiShutdown(KII_CTX(inp));
#endif
		free(priv);	
		return GGI_ENODEVICE;
	}

	inp->GIIsendevent = GII_kii_send_event;
	inp->GIIeventpoll = GII_kii_poll;
	inp->GIIclose     = GII_kii_close;
   
#if 0
	inp->targetcan = emPointer;
#endif
	inp->GIIseteventmask(inp, emAll);

	inp->targetcan = emAll;
	inp->curreventmask = emAll;
    
	inp->maxfd = kiiEventDeviceFD(KII_CTX(inp)) + 1;

	GIIDPRINT_MISC("kii fully up\n");

	return 0;
}
