/*
******************************************************************************

   LibGAlloc: extension initialization.

   Copyright (C) 2001 Christoph Egger   [Christoph_Egger@t-online.de]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdio.h>
#include <string.h>

#include "config.h"
#include <ggi/internal/galloc.h>
#include <ggi/internal/galloc_debug.h>

/* We use LibGG to manage config files */
#include <ggi/gg.h>


/* Static variables 
 */
static int 	 _gallocLibIsUp       = 0;
static void	*_gallocConfigHandle  = NULL;
static char      _gallocconfstub[512] = GALLOCCONFFILE;
static char     *_gallocconffile      = _gallocconfstub+GALLOCTAGLEN;

int _gallocDebugSync = 0;
uint32 _gallocDebugState = 0;

extern GAanprintf GALLOC_Stubs_anprintf;


/* Extension ID. Defaulting to -1 should make segfault on abuse more likely...
 * This is exported so that sublibs can use it.
 */
EXPORTVAR ggi_extid _gallocID = -1;


#define SUBLIB_PREFIX	"GALLOCdl_"


/* Dummy function which returns -1
   We use this to reset the function pointers */
static int dummyfunc(void)
{
	GALLOCDPRINT_CORE("dummyfunc() of LibGAlloc called\n");
	return GALLOC_EUNAVAILABLE;
}	/* dummyfunc */


/* Reset all function pointers to dummyfunc/default function. */
static void clearfuncs(gallocpriv *priv)
{
	priv->check = (void *) dummyfunc;
	priv->set = (void *) dummyfunc;
	priv->release = (void *) dummyfunc;
	priv->anprintf = (void *) dummyfunc;
	priv->_mode = (void *) dummyfunc;
	priv->checkifshareable = (void *) dummyfunc;
	priv->default_callbacks[0] = (void *) dummyfunc;
	priv->default_callbacks[1] = (void *) dummyfunc;
	priv->default_callbacks[2] = (void *) dummyfunc;
	priv->default_callbacks[3] = (void *) dummyfunc;
	priv->default_callbacks[4] = (void *) dummyfunc;
	priv->default_callbacks[5] = (void *) dummyfunc;
	priv->default_callbacks[6] = (void *) dummyfunc;
	priv->default_callbacks[7] = (void *) dummyfunc;
	priv->default_callbacks[8] = (void *) dummyfunc;

	/* Not a function, but must be cleared. */
	priv->haslist = (void *)dummyfunc;

}	/* clearfuncs */


/* This function is called by LibGGI when the API-list changes */
static int changed(ggi_visual_t vis, int whatchanged)
{
	GALLOCDPRINT_LIBS("changed(%p, 0x%x) of LibGAlloc called\n",
	       vis, whatchanged);

	switch (whatchanged) {
	case GGI_CHG_APILIST: {
		int i;
		const char *fname;
		ggi_dlhandle *lib;
		char api[GGI_MAX_APILEN], args[GGI_MAX_APILEN];

		clearfuncs(LIBGGI_GALLOCEXT(vis));

		for(i = 0; ggiGetAPI(vis, i, api, args) == 0; i++) {

		        strcat(api,"-galloc");
			fname = ggMatchConfig(_gallocConfigHandle, api, NULL);
			if (fname == NULL) {
				/* No special implementation for this sublib */
				continue;
			}
			lib = ggiExtensionLoadDL(vis, fname, args, NULL,
						 SUBLIB_PREFIX);
		}
	}
	break;
	}
                                
	return 0;
}	/* changed */



/* Initialize the extension
 */
int ggiGAInit(void)
{
	int err;
	char *str;

	_gallocLibIsUp++;
	if (_gallocLibIsUp > 1) return 0; /* Initialize only at first call. */

	str = getenv("GA_DEBUG");
	if (str == NULL) str = getenv("GALLOC_DEBUG");
	if (str != NULL) {
		_gallocDebugState = atoi(str);
		GALLOCDPRINT_CORE("Debugging=%d\n", _gallocDebugState);
	}	/* if */

	str = getenv("GA_DEBUGSYNC");
	if (str == NULL) str = getenv("GALLOC_DEBUGSYNC");
	if (str != NULL) {
		_gallocDebugSync = 1;
	}	/* if */


	err = ggLoadConfig(_gallocconffile, &_gallocConfigHandle);
	if (err != GGI_OK) {
		fprintf(stderr,"LibGAlloc: couldn't open %s\n",
			_gallocconffile);
		_gallocLibIsUp--;
		return err;
	}	/* if */

	_gallocID = ggiExtensionRegister("GAlloc", sizeof(gallocpriv), changed);

	if (_gallocID < 0) {
		fprintf(stderr,
			"LibGAlloc: failed to register as extension.\n");
		_gallocLibIsUp--;
		ggFreeConfig(_gallocConfigHandle);
		return _gallocID;
	}	/* if */

	return 0;
}	/* ggiGAInit */


/* Deinitialize the extension
 */
int ggiGAExit(void)
{
	int rc;
    
	if (!_gallocLibIsUp) return -1;

	if (_gallocLibIsUp > 1) {
		/* Exit only at last call */
		_gallocLibIsUp--;
		return 0;
	}	/* if */
	
	rc = ggiExtensionUnregister(_gallocID);
	ggFreeConfig(_gallocConfigHandle);

	_gallocLibIsUp = 0;

	return rc;
}	/* ggiGAExit */


/* Attach the extension to a visual
 */
int ggiGAAttach(ggi_visual_t vis)
{
	int rc;

	GALLOCDPRINT_CORE("ggiGAAttach(%p) called\n", vis);

	rc = ggiExtensionAttach(vis, _gallocID);
	if (rc < 0) {
		GALLOCDPRINT("ggiExtensionAttach() failed with error: %i!\n", rc);
		goto err0;
	}	/* if */

	if (rc == 0) {
		/* We are actually creating the primary instance. */
		memset(LIBGGI_GALLOCEXT(vis), 0, sizeof(gallocpriv));
		LIBGGI_GALLOCEXT(vis)->dummy = (void *)dummyfunc;
		LIBGGI_GALLOCEXT(vis)->priv = NULL;
		GA_RESLIST(vis) = NULL;

		/* Now fake an "API change" so the right libs get loaded */
		changed(vis, GGI_CHG_APILIST);
	}	/* if */

err0:
	return rc;
}	/* ggiGAAttach */


/* Detach the extension from a visual
 */
int ggiGADetach(ggi_visual_t vis)
{
	int rc;
	ggiGA_resource_list cur_reslist = NULL;
	void *priv = NULL;

	GALLOCDPRINT_CORE("ggiGADetach(%p) called\n", vis);

	cur_reslist = GA_RESLIST(vis);
	priv = LIBGGI_GALLOCEXT(vis)->priv;

	rc = ggiExtensionDetach(vis, _gallocID);
	if (rc != 0) return rc;


	/* Free the currently active request structure
	 * if not already.
	 */
	if (cur_reslist != NULL) {
		ggiGAEmptyList(&cur_reslist);
	}	/* if */

	/* Free what's on the the priv sublib hook
	 * (should be free already!)
	 */
	if (priv != NULL) {
		free(priv);
	}	/* if */

	return rc;
}	/* ggiGADetach */

#define COMPILING_GALLOC_C
#include <ggi/ga_prop.h>


/* This should be included by exactly _one_ file in each extension library. */
#include <ggi/internal/ggilibinit.h>

