/* $Id: input.c,v 1.4 2002/11/29 22:11:55 cegger Exp $
******************************************************************************

   Input-KII: Input for KII
      
   Copyright (C) 2002 Paul Redmond

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdlib.h>
#include "config.h"
#include <ggi/internal/gii-dl.h>
#include <ggi/internal/gii_debug.h>


#define KII_NEED_MODIFIER_KEYSYMS
#include <kgi/config.h>
#include <kgi/system.h>
#include <kii/kii.h>


typedef struct {
	kii_context_t *ctx;
} kii_priv;

#define KII_PRIV(inp) ((kii_priv *)inp->priv)
#define KII_CTX(inp)  (KII_PRIV(inp)->ctx)

static gii_cmddata_getdevinfo devinfo =
{
	"Kernel Input Interface",
	"kii",
	emAll,
	GII_NUM_UNKNOWN,
	GII_NUM_UNKNOWN
};

static gii_event_mask GII_kii_handle_key(gii_input *inp, gii_event *ge,
	const kii_event_t *ke)
{
	_giiEventBlank(ge, sizeof(gii_key_event));
	ge->any.size = sizeof(gii_key_event);
	ge->any.origin = inp->origin;
	
	GIIDPRINT_LIBS("sym: 0x%.8x, code: 0x%.8x, effect: 0x%.4x, "
		"normal: 0x%.4x, locked: 0x%.4x, sticky: 0x%.4x\n",
		ke->key.sym, ke->key.code, ke->key.effect,
		ke->key.normal, ke->key.locked, ke->key.sticky);
	
	/* FIXME: What about shift L/R ctrl L/R */
	if (ke->key.effect & KII_MM_SHIFT)
		ge->key.modifiers |= GII_MOD_SHIFT;
	if (ke->key.effect & KII_MM_CTRL)
		ge->key.modifiers |= GII_MOD_CTRL;
	if (ke->key.effect & KII_MM_ALT)
		ge->key.modifiers |= GII_MOD_ALT;
	if (ke->key.effect & KII_MM_ALTGR)
		ge->key.modifiers |= GII_MOD_ALTGR;
	
	switch (ke->key.sym) {
	
	case K_UP:
		ge->key.sym = GIIK_Up;
		break;
	case K_DOWN:
		ge->key.sym = GIIK_Down;
		break;
	case K_LEFT:
		ge->key.sym = GIIK_Left;
		break;
	case K_RIGHT:
		ge->key.sym = GIIK_Right;
		break;

	case K_ENTER:
		ge->key.sym = GIIK_Enter;
		break;

	case K_NORMAL_SHIFT:
		ge->key.sym = GIIK_Shift;
		break;
	case K_NORMAL_SHIFTL:
		ge->key.sym = GIIK_ShiftL;
		break;
	case K_NORMAL_SHIFTR:
		ge->key.sym = GIIK_ShiftR;
		break;
	case K_NORMAL_CTRL:
		ge->key.sym = GIIK_Ctrl;
		break;
	case K_NORMAL_CTRLL:
		ge->key.sym = GIIK_CtrlL;
		break;
	case K_NORMAL_CTRLR:
		ge->key.sym = GIIK_CtrlR;
		break;
	case K_NORMAL_ALT:
		ge->key.sym = GIIK_AltL;
		break;
	case K_NORMAL_ALTGR:
		ge->key.sym = GIIK_AltR;
		break;

	default:
		if (ke->key.sym < 0x80)
			ge->key.sym = ke->key.sym;
		else
			return emZero;
	}
	/* Event types are the same */
	ge->any.type = ke->any.type;
	/* Button number is the scancode */
	ge->key.button = ke->key.code;
	/* FIXME: Do labels */
	ge->key.label = ge->key.sym;
	
	_giiEvQueueAdd(inp, ge);
	
	return (1 << ge->any.type);
}

static gii_event_mask GII_kii_handle_ptr(gii_input *inp, gii_event *ge,
	const kii_event_t *ke)
{

	if (ke->any.type == KII_EV_PTR_STATE) {
	
		return emZero;
	}

	if (ke->any.type < KII_EV_PTR_BUTTON_PRESS) {

		/* pointer move event */
    		_giiEventBlank(ge, sizeof(gii_pmove_event));
		
		switch (ke->any.type) {
		
		case KII_EV_PTR_RELATIVE:
			ge->any.type = evPtrRelative;
			break;
			
		case KII_EV_PTR_ABSOLUTE:
			ge->any.type = evPtrAbsolute;
			break;
		
		default:
			return emZero;
		}
		ge->any.origin = inp->origin;
		ge->any.size = sizeof(gii_pmove_event);
		ge->pmove.x = ke->pmove.x;
		ge->pmove.y = ke->pmove.y;
		
		_giiEvQueueAdd(inp, ge);
	}
	else {
		int i;
		kii_u_t mask = 1;

    		_giiEventBlank(ge, sizeof(gii_pbutton_event));
		switch (ke->any.type) {
		
		case KII_EV_PTR_BUTTON_PRESS:
			ge->any.type = evPtrButtonPress; break;
			
		case KII_EV_PTR_BUTTON_RELEASE:
			ge->any.type = evPtrButtonRelease; break;
		
		default:
			return emZero;
		}
		ge->any.origin = inp->origin;
		ge->any.size = sizeof(gii_pbutton_event);
		for (i = 0; i < 32; i++) {
		
			if (ke->pbutton.button & mask) {
			
				ge->pbutton.button = i; /* zero based? +1? */
				_giiEvQueueAdd(inp, ge);
			}
			mask <<= 1;
		}
	}
	
	return (1 << ge->any.type);
}

static gii_event_mask GII_kii_poll(gii_input *inp, void *arg)
{
	const kii_event_t *ke;
	gii_event_mask em = emZero;
	gii_event ge;

	if (!kiiEventAvailable(KII_CTX(inp))) {
	
		return emZero;
	}
		
	for (ke=kiiNextEvent(KII_CTX(inp));ke;ke=kiiNextEvent(KII_CTX(inp))) {

		if ((1 << ke->any.type) & KII_EM_KEYBOARD) {
		
			em |= GII_kii_handle_key(inp, &ge, ke);
		}
		else if ((1 << ke->any.type) & KII_EM_POINTER) {
		
			em |= GII_kii_handle_ptr(inp, &ge, ke);
		}
	}

	return em;
}

static void send_devinfo(gii_input *inp)
{
	gii_event ev;
	gii_cmddata_getdevinfo *dinfo;
	int size = sizeof(gii_cmd_nodata_event)+sizeof(gii_cmddata_getdevinfo);
	
	_giiEventBlank(&ev, size);
	
	ev.any.size = size;
	ev.any.type = evCommand;
	ev.any.origin = inp->origin;
	ev.cmd.code = GII_CMDCODE_GETDEVINFO;
	
	dinfo = (gii_cmddata_getdevinfo *)&ev.cmd.data;
	*dinfo = devinfo;
	
	_giiEvQueueAdd(inp, &ev);
}

static int GII_kii_send_event(gii_input *inp, gii_event *ev)
{
	if (ev->any.target != inp->origin &&
	    ev->any.target != GII_EV_TARGET_ALL) {
	
		return GGI_EEVNOTARGET;
	}
	
	if (ev->any.type != evCommand) {
	
		return GGI_EEVUNKNOWN;
	}
	
	if (ev->cmd.code == GII_CMDCODE_GETDEVINFO) {
	
		send_devinfo(inp);
		return 0;
	}
	
	return GGI_EEVUNKNOWN;
}

static int GII_kii_close(gii_input *inp) 
{
	kii_priv *priv = KII_PRIV(inp);
        
	free(priv);

	GIIDPRINT_MISC("kii: exit OK.\n");

	return 0;
}


int GIIdlinit(gii_input *inp, const char *args, void *argptr)
{
	kii_priv *priv;

	GIIDPRINT_MISC("kii starting. (args=%s,argptr=%p)\n",
			args, argptr);
    
	if ((priv = inp->priv = malloc(sizeof(kii_priv))) == NULL) {

		return GGI_ENOMEM;
	}

	if(_giiRegisterDevice(inp, &devinfo, NULL) == 0) {
		
		free(priv);
		return GGI_ENOMEM;
	}

	if (KII_EOK != kiiInit(&KII_CTX(inp))) {
	
		free(priv);
		return GGI_ENODEVICE;
	}

	if (KII_EOK != kiiMapDevice(KII_CTX(inp))) {
	
		/* FIXME: kiiInit now leaks memory */
		free(priv);	
		return GGI_ENODEVICE;
	}

	inp->GIIsendevent = GII_kii_send_event;
	inp->GIIeventpoll = GII_kii_poll;
	inp->GIIclose     = GII_kii_close;
   
	inp->GIIseteventmask(inp, emAll);

	inp->targetcan = emAll;
	inp->curreventmask = emAll;
    
	inp->maxfd = kiiEventDeviceFD(KII_CTX(inp)) + 1;

	GIIDPRINT_MISC("kii fully up\n");

	return 0;
}
