/*
******************************************************************************

   LibGalloc: plain functions

   Copyright (C) 2001 Christoph Egger   [Christoph_Egger@t-online.de]
   Copyright (C) 2001 Brian S. Julin    [bri@calyx.com]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <ggi/internal/galloc.h>
#include <ggi/internal/galloc_debug.h>

#include <string.h>

/* "external" API */

int ggiGAAdd(ggiGA_resource_list *reqlist, 
	     struct ggiGA_resource_props *props, 
	     enum ggiGA_resource_type rt, ggiGA_resource_handle *handle)
{
	int rc = GALLOC_OK;
	ggiGA_resource_handle result;

	if (reqlist == NULL) return GGI_EARGINVAL;
	if (props == NULL) return GGI_EARGINVAL;

	result = ggiGACopyIntoResource(props, NULL, 0);
	if (result == NULL) goto err1;

	result->res_type = rt;	

	result->next = NULL;
	rc = ggiGAAppendList(result, reqlist);

	if (handle != NULL) handle[0] = result;

	LIB_ASSERT(rc == GALLOC_OK);

	return GALLOC_OK;

 err1:
	if (handle != NULL) handle[0] = NULL;
	return GGI_ENOMEM;
}	/* ggiGAAdd */


int ggiGAAddMode(ggi_visual_t vis, ggiGA_resource_list *reqlist, 
		 ggi_mode *mode, ggi_directbuffer *db,
		 ggiGA_resource_handle *handle)
{
	int rc = GALLOC_OK;
	ggiGA_resource_handle result;

	if (reqlist == NULL) return GGI_EARGINVAL;
	if (mode == NULL) return GGI_EARGINVAL;

	result = calloc(1, sizeof(struct ggiGA_resource));
	if (result == NULL) goto err0;

	result->priv = calloc(1, sizeof(struct ggiGA_mode));
	if (result == NULL) goto err1;

	result->priv_size = sizeof(struct ggiGA_mode);
	result->next = NULL;

	/* GA_RT_FRAME resources don't have a properties structure. */
	result->props = NULL;

	memcpy(&(((struct ggiGA_mode *)result->priv)->mode), mode, 
		sizeof(ggi_mode));

	if (db != NULL) {
		memcpy(&(((struct ggiGA_mode *)result->priv)->db), db, 
			sizeof(ggi_directbuffer));
	}	/* if */

	ggiGA_Mode(vis, &result);

	rc = ggiGAAppendList(result, reqlist);
	LIB_ASSERT(rc == GALLOC_OK);

	if (handle != NULL) handle[0] = result;

	return rc;

err1:
	free(result);
err0:
	if (handle != NULL) handle[0] = NULL;
	return GGI_ENOMEM;
}	/* ggiGAAddMode */



/* These are just rips from the corresponding GGI core Set* functions. */
int ggiGAAddTextMode(ggi_visual_t vis, ggiGA_resource_list *reqlist,
		     int cols,int rows,
		     int vcols,int vrows,
		     int fontsizex,int fontsizey,
		     ggi_graphtype type,
		     ggiGA_resource_handle *handle) 
{
	ggi_mode mode;

	mode.frames	= GGI_AUTO;
	mode.visible.x	= cols;
	mode.visible.y	= rows;
	mode.virt.x	= vcols;
	mode.virt.y	= vrows;
	mode.size.x	= mode.size.y = GGI_AUTO;
	mode.graphtype	= type;
	mode.dpp.x	= fontsizex;
	mode.dpp.y	= fontsizey;
        
	return ggiGAAddMode(vis, reqlist, &mode, NULL, handle);
}	/* ggiGAAddTextMode */


int ggiGAAddGraphMode(ggi_visual_t vis, ggiGA_resource_list *reqlist,
		     int xsize,int ysize,
		     int xvirtual,int yvirtual,ggi_graphtype type,
		     ggiGA_resource_handle *handle)
{
	ggi_mode mode;
        
	mode.frames	= GGI_AUTO;
	mode.visible.x	= xsize;
	mode.visible.y	= ysize;
	mode.virt.x	= xvirtual;
	mode.virt.y	= yvirtual;
	mode.size.x	= mode.size.y = GGI_AUTO;
	mode.graphtype	= type;
	mode.dpp.x	= mode.dpp.y = GGI_AUTO;

	return ggiGAAddMode(vis, reqlist, &mode, NULL, handle);
}	/* ggiGAAddGraphMode */


int ggiGAAddSimpleMode(ggi_visual_t vis, ggiGA_resource_list *reqlist,
		       int xsize, int ysize, 
		       int frames,
		       ggi_graphtype type,
		       ggiGA_resource_handle *handle) 
{
	ggi_mode mode;
        
	mode.frames	= frames;
	mode.visible.x	= xsize;
	mode.visible.y	= ysize;
	mode.virt.x	= mode.virt.y = GGI_AUTO;
	mode.size.x	= mode.size.y = GGI_AUTO;
	mode.graphtype	= type;
	mode.dpp.x	= mode.dpp.y = GGI_AUTO;

	return ggiGAAddMode(vis, reqlist, &mode, NULL, handle);
}	/* ggiGAAddSimpleMode */


int ggiGAClearMotorProperties(struct ggiGA_resource_props *prop)
{
	if (prop == NULL) return GGI_EARGINVAL;

	memset(prop, 0, sizeof(*prop));

	/* Set useful default flags here */
	GA_FLAG(prop) = (LL_COORDBASE_UNITS & GA_FLAG_UNIT_MASK);
	GA_FLAG(prop) |= (LL_COORDBASE_PIXEL & GA_FLAG_COORDBASE_MASK);

	/* Set useful default flags here */
	prop->qty = 1;
	prop->sub.motor.mul_max.x = 1;
	prop->sub.motor.mul_min.x = 1;
	prop->sub.motor.div_max.x = 1;
	prop->sub.motor.div_min.x = 1;
	prop->sub.motor.mul_max.y = 1;
	prop->sub.motor.mul_min.y = 1;
	prop->sub.motor.div_max.y = 1;
	prop->sub.motor.div_min.y = 1;

	return GALLOC_OK;
}	/* ggiGAClearMotorProperties */


int ggiGAClearCarbProperties(struct ggiGA_resource_props *prop)
{
	if (prop == NULL) return GGI_EARGINVAL;

	memset(prop, 0, sizeof(*prop));

	/* Set useful default flags here */
	prop->qty = 1;

	return GALLOC_OK;
}	/* ggiGAClearCarbProperties */


int ggiGAClearTankProperties(struct ggiGA_resource_props *prop)
{
	if (prop == NULL) return GGI_EARGINVAL;

	memset(prop, 0, sizeof(*prop));

	/* Set useful default flags here */
	GA_FLAG(prop) = (GA_FLAG_STEREO_NOT);

	prop->qty = 1;

	prop->storage_share |= GA_SHARE_COPYONWRITE;

	return GALLOC_OK;
}	/* ggiGAClearTankProperties */


struct ggiGA_resource_props *ggiGAGetProperties(ggiGA_resource_handle handle)
{
	if (handle == NULL) {
		return NULL;
	}	/* if */

	return handle->props;
}	/* ggiGAGetProperties */


int ggiGAIsFailed(ggiGA_resource_handle handle)
{
	int state;

	if (handle == NULL) return GGI_EARGINVAL;

	state = (handle->res_state & GA_STATE_FAILED);

	if (state == GA_STATE_FAILED) {
		return 1;
	} else {
		return 0;
	}	/* if */
}	/* ggiGAIsFailed */


int ggiGAIsModified(ggiGA_resource_handle handle)
{
	int state;

	if (handle == NULL) return GGI_EARGINVAL;

	state = (handle->res_state & GA_STATE_MODIFIED);

	if (state == GA_STATE_MODIFIED) {
		return 1;
	} else {
		return 0;
	}	/* if */
}	/* ggiGAIsModified */


int ggiGAIsMotor(ggiGA_resource_handle handle)
{
	unsigned int type;

	if (handle == NULL) return GGI_EARGINVAL;

	type = (handle->res_type & GA_RT_MOTOR);

	if (type == GA_RT_MOTOR) {
		return 1;
	} else {
		return 0;
	}	/* if */
}	/* ggiGAIsMotor */


int ggiGAIsCarb(ggiGA_resource_handle handle)
{
	unsigned int type;

	if (handle == NULL) return GGI_EARGINVAL;

	type = (handle->res_type & GA_RT_CARB);

	if (type == GA_RT_CARB) {
		return 1;
	} else {
		return 0;
	}	/* if */
}	/* ggiGAIsCarb */


int ggiGAIsTank(ggiGA_resource_handle handle)
{
	unsigned int type;

	if (handle == NULL) return GGI_EARGINVAL;

	type = (handle->res_type & (GA_RT_MOTOR | GA_RT_CARB));

	if (type == 0) {
		return 1;
	} else {
		return 0;
	}	/* if */
}	/* ggiGAIsTank */


ggi_visual_t ggiGAGetVisual(ggiGA_resource_handle handle)
{
	if (handle == NULL) {
		return NULL;
	}	/* if */

	/* When this assertion fails, then
	 * there is either a bug or that resource failed
	 * or whatever...
	 */
	LIB_ASSERT(handle->vis != NULL);
	return handle->vis;
}	/* ggiGAGetVisual */




ggiGA_resource_handle ggiGAMotorGetTank(ggiGA_resource_list list,
				ggiGA_resource_handle motor,
				ggiGA_resource_handle tank)
{
	int tag_m, tag_c;
	ggiGA_resource_handle current = NULL;
	ggiGA_resource_handle carb = NULL;


	LIB_ASSERT(list != NULL);
	LIB_ASSERT(motor != NULL);

	tag_m = ggiGAGetTag(motor);
	/* if this fails, then the motor isn't tagged, which don't have to be */
	LIB_ASSERT(tag_m != 0);

	/* A motor is always before the tank */
	current = motor->next;
	while (current != NULL) {

		LIB_ASSERT(ggiGAIsTank(current));

		carb = ggiGATankGetCarb(list, current, NULL);
		if (carb != NULL) {
			tag_c = ggiGAGetTag(carb);

			if (tag_c == tag_m) return current;
		}	/* if */

		current = current->next;
	}	/* while */


	/* motor has no tanks */

	return NULL;
}	/* ggiGAMotorGetTank */



ggiGA_resource_handle ggiGATankGetCarb(ggiGA_resource_list list,
				ggiGA_resource_handle tank,
				ggiGA_resource_handle carb)
{
	ggiGA_resource_handle tmp = NULL;

	LIB_ASSERT(list != NULL);
	LIB_ASSERT(tank != NULL);

	if (carb == NULL) {
		tmp = tank->next;
	
		if (ggiGAIsCap(tmp)) {
			tmp = tmp->next;
		}	/* if */
	} else {
		LIB_ASSERT(ggiGAIsCarb(carb));

		tmp = carb->next;
		if (ggiGAIsCarb(tmp)) {
			return tmp;
		}	/* if */
	}	/* if */

	return NULL;
}	/* ggiGATankGetCarb */


ggi_mode *ggiGAGetGGIMode(ggiGA_resource_handle handle)
{
	if (handle == NULL) {
		/* handle not valid! */
		return NULL;
	}	/* if */

	if (ggiGA_TYPE(handle) != GA_RT_FRAME) {
		return NULL;
	}	/* if */

	LIB_ASSERT (handle->priv);
	LIB_ASSERT (handle->priv_size == sizeof(struct ggiGA_mode));

	return (&(((struct ggiGA_mode *)(handle->priv))->mode));
}	/* ggiGAGetGGIMode */


ggi_directbuffer *ggiGAGetGGIDB(ggiGA_resource_handle handle)
{
	if (handle == NULL) {
		/* handle not valid! */
		return NULL;
	}	/* if */

	if (ggiGA_TYPE(handle) != GA_RT_FRAME) {
		return NULL;
	}	/* if */

	LIB_ASSERT (handle->priv);
	LIB_ASSERT (handle->priv_size == sizeof(struct ggiGA_mode));

	return (&(((struct ggiGA_mode *)(handle->priv))->db));
}	/* ggiGAGetGGIDB */


ggiGA_resource_handle ggiGAHandle(ggiGA_resource_list reslist,
				  ggiGA_resource_list reqlist,
				  ggiGA_resource_handle reqhandle)
{
	ggiGA_resource_handle reqcurrent;
	ggiGA_resource_handle rescurrent;

	if (reqlist == NULL) return NULL;
	if (reslist == NULL) return NULL;
	if (reqhandle == NULL) return NULL;


	reqcurrent = reqlist;
	rescurrent = reslist;
	while (reqcurrent != NULL) {
		if (reqcurrent == reqhandle) return rescurrent;

		reqcurrent = reqcurrent->next;
		rescurrent = rescurrent->next;
		if (rescurrent == NULL && reqcurrent != NULL) {
			return NULL;
		}	/* if */
	}	/* while */

	/* There was no such resource */
	return NULL;
}	/* ggiGAHandle */


int ggiGAReleaseVis(ggi_visual_t vis,
		ggiGA_resource_list *list,
		ggiGA_resource_handle *handle)
{
	if (vis == NULL) return GGI_EARGINVAL;
	if (list == NULL) return GGI_EARGINVAL;
	if (handle == NULL) return GGI_EARGINVAL;


	if (vis != ggiGAGetVisual(handle[0])) return GALLOC_EFAILED;

	return ggiGARelease(vis, list, handle);
}	/* ggiGAReleaseVis */


int ggiGAReleaseList(ggi_visual_t vis, ggiGA_resource_list *list)
{
	int rc;
	int ret = GALLOC_OK;
	ggiGA_resource_handle lastmode = NULL;
	ggiGA_resource_handle current = NULL;
	ggiGA_resource_handle next_res = NULL;
	char *str;

	if (list == NULL) return GGI_EARGINVAL;

	if (list[0] == NULL) {
		/* obviously, we cannot release an empty list */
		return GALLOC_OK;
	}	/* if */

	current = list[0];
	while (current != NULL) {
		next_res = current->next;

		GALLOCDPRINT("%s:%s:%i: current: %p, next_res: %p\n",
			     DEBUG_INFO, current, next_res);

		while ((next_res != NULL) && (!ggiGAIsCompoundHead(next_res)))
		  	next_res = next_res->next;



		GALLOCDPRINT("%s:%s:%i: current: %p, next_res: %p\n",
			      DEBUG_INFO, current, next_res);

		ggiGAanprintf(vis, list[0], 8092, NULL, &str);
		GALLOCDPRINT("list before release: \n%s\n", str);
		free(str);

		rc = ggiGARelease(vis, list, &current);
		LIB_ASSERT(rc == GALLOC_OK);

		ggiGAanprintf(vis, list[0], 8092, NULL, &str);
		GALLOCDPRINT("list after release: \n%s\n", str);
		free(str);


		current = next_res;
	}	/* while */


	/* The list should now be clean, with the only time a resource
	 * was not freed when asked being GA_RT_FRAMEs, but those should
	 * have been silently ignored without errors from ggiGARelease.
	 * Now by hand we Remove all the GA_RT_FRAME resources except
	 * lastmode and its cap.
	 */

	lastmode = ggiGAFindLastMode(list[0], NULL);

	current = list[0];
	while (current != NULL) {
		next_res = current->next;

		LIB_ASSERT(ggiGA_TYPE(current) == GA_RT_FRAME);
		if (current == lastmode) break;

		rc = ggiGARemove(list, &current);
		LIB_ASSERT(rc == GALLOC_OK);

		current = next_res;
	}	/* while */



	/* There should be one resource remaining, that being the 
	 * main video mode.
	 */
	LIB_ASSERT(list != NULL);
	LIB_ASSERT(list[0]->res_type == GA_RT_FRAME);
	LIB_ASSERT( (list[0]->next != NULL
		 && list[0]->next->res_type == GA_RT_FRAME)
		 || (list[0]->next == NULL));

	if (list[0]->res_type != GA_RT_FRAME) ret = GALLOC_EFAILED;

	return ret;

}	/* ggiGAReleaseList */




/*******************************************************
 * "gray area" API
 */

int ggiGASubUnit2Unit(ggiGA_resource_handle handle,
                      ggi_coord subunit, ggi_coord *unit)
{
	ggi_coord div_max;

	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(unit != NULL);

	if (!ggiGAIsMotor(handle)) return GALLOC_EFAILED;

	LIB_ASSERT(handle->props->sub.motor.div_max.x != 0);
	LIB_ASSERT(handle->props->sub.motor.div_max.y != 0);

	div_max = handle->props->sub.motor.div_max;

	unit->x = subunit.x / div_max.x;
	unit->y = subunit.y / div_max.y;

	return GALLOC_OK;
}	/* ggiGASubUnit2Unit */


int ggiGAUnit2SubUnit(ggiGA_resource_handle handle,
                      ggi_coord unit, ggi_coord *subunit)
{
	ggi_coord div_max;

	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(subunit != NULL);

	if (!ggiGAIsMotor(handle)) return GALLOC_EFAILED;

	LIB_ASSERT(handle->props->sub.motor.div_max.x != 0);
	LIB_ASSERT(handle->props->sub.motor.div_max.y != 0);

	div_max = handle->props->sub.motor.div_max;

	subunit->x = unit.x * div_max.x;
	subunit->y = unit.y * div_max.y;

	return GALLOC_OK;
}	/* ggiGAUnit2SubUnit */



int ggiGAPixel2Dot(ggi_visual_t vis, ggi_coord pixel, ggi_coord *dot)
{
	LIB_ASSERT(vis != NULL);
	LIB_ASSERT(dot != NULL);

	LIB_ASSERT(LIBGGI_MODE(vis)->dpp.x != 0);
	LIB_ASSERT(LIBGGI_MODE(vis)->dpp.y != 0);

	dot->x = pixel.x / LIBGGI_MODE(vis)->dpp.x;
	dot->y = pixel.y / LIBGGI_MODE(vis)->dpp.y;

	return GALLOC_OK;
}	/* ggiGAPixel2Dot */


int ggiGADot2Pixel(ggi_visual_t vis, ggi_coord dot, ggi_coord *pixel)
{
	LIB_ASSERT(vis != NULL);
	LIB_ASSERT(pixel != NULL);

	LIB_ASSERT(LIBGGI_MODE(vis)->dpp.x != 0);
	LIB_ASSERT(LIBGGI_MODE(vis)->dpp.y != 0);

	pixel->x = LIBGGI_MODE(vis)->dpp.x * dot.x;
	pixel->y = LIBGGI_MODE(vis)->dpp.y * dot.y;

	return GALLOC_OK;
}	/* ggiGADot2Pixel */


/* Convert any nature measure to pixels and vice versa.
 */
int ggiGAConv2Pixel(ggi_visual_t vis, enum ggi_ll_coordbase cb,
		    ggi_coord coord, ggi_coord *pixel)
{
	ggi_float mmpp_x;	/* mm per pixel */
	ggi_float mmpp_y;	/* mm per pixel */
	ggi_coord size;

	LIB_ASSERT(vis != NULL);
	LIB_ASSERT(pixel != NULL);

	/* We know, this size is always given in mm */
	size = LIBGGI_MODE(vis)->size;

	if ((size.x == 0) || (size.y == 0)) {
		/* Ooops! Unknown or invalid physical size!
		 * Can't support natural maesures. We fail!
		 */

		return GALLOC_EFAILED;
	}	/* if */

	mmpp_x = (ggi_float)(size.x / LIBGGI_MODE(vis)->visible.x);
	mmpp_y = (ggi_float)(size.y / LIBGGI_MODE(vis)->visible.y);


	switch (cb) {
	case LL_COORDBASE_MILLIMETER:
		pixel->x = (int)(mmpp_x * coord.x);
		pixel->y = (int)(mmpp_y * coord.y);
		break;

	case LL_COORDBASE_MICROMETER:
		{
		ggi_float tmp_x, tmp_y;

		/* convert um to mm */
		tmp_x = (ggi_float)(coord.x / 1000);
		tmp_y = (ggi_float)(coord.y / 1000);

		pixel->x = (int)(mmpp_x * tmp_x);
		pixel->y = (int)(mmpp_y * tmp_y);
		break;
		}

	case LL_COORDBASE_1_64_INCH:
		{
		ggi_float tmp_x, tmp_y;

		/* convert 1/64th inch to mm */
		tmp_x = (ggi_float)(coord.x / 0.396875);
		tmp_y = (ggi_float)(coord.y / 0.396875);

		pixel->x = (int)(mmpp_x * tmp_x);
		pixel->y = (int)(mmpp_y * tmp_y);
		break;
		}

	case LL_COORDBASE_DOT:

		return ggiGADot2Pixel(vis, coord, pixel);

	case LL_COORDBASE_PIXEL:

		pixel[0] = coord;
		break;

	default:
		/* Sorry, no supported meassure */
		return GALLOC_EFAILED;
	}	/* switch */

	return GALLOC_OK;
}	/* ggiGAConv2Pixel */


int ggiGAConvPixel2(ggi_visual_t vis, enum ggi_ll_coordbase cb,
		    ggi_coord pixel, ggi_coord *coord)
{
	ggi_float mmpp_x;	/* mm per pixel */
	ggi_float mmpp_y;	/* mm per pixel */
	ggi_coord size;

	LIB_ASSERT(vis != NULL);
	LIB_ASSERT(coord != NULL);

	/* We know, this size is always given in mm */
	size = LIBGGI_MODE(vis)->size;

	if ((size.x == 0) || (size.y == 0)) {
		/* Ooops! Unknown or invalid physical size!
		 * Can't support natural maesures. We fail!
		 */

		return GALLOC_EFAILED;
	}	/* if */

	mmpp_x = (ggi_float)(LIBGGI_MODE(vis)->visible.x / size.x);
	mmpp_y = (ggi_float)(LIBGGI_MODE(vis)->visible.y / size.y);


	switch (cb) {
	case LL_COORDBASE_MILLIMETER:
		coord->x = (int)(mmpp_x * pixel.x);
		coord->y = (int)(mmpp_y * pixel.y);
		break;

	case LL_COORDBASE_MICROMETER:
		coord->x = (int)(mmpp_x * pixel.x);
		coord->y = (int)(mmpp_y * pixel.y);

		/* convert mm to um */
		coord->x *= 1000;
		coord->y *= 1000;
		break;

	case LL_COORDBASE_1_64_INCH:
		{
		ggi_float tmp_x, tmp_y;

		/* convert mm to 1/64th inch */
		tmp_x = (ggi_float)(pixel.x * 0.396875);
		tmp_y = (ggi_float)(pixel.y * 0.396875);

		coord->x = (int)(mmpp_x * tmp_x);
		coord->y = (int)(mmpp_y * tmp_y);
		break;
		}

	case LL_COORDBASE_DOT:

		return ggiGAPixel2Dot(vis, pixel, coord);

	case LL_COORDBASE_PIXEL:

		coord[0] = pixel;
		break;

	default:
		/* Sorry, no natural meassure */
		return GALLOC_EFAILED;
	}	/* switch */

	return GALLOC_OK;
}	/* ggiGAConvPixel2 */


/* Convert any nature measure to pixels and vice versa.
 */
int ggiGAConv2Dot(ggi_visual_t vis, enum ggi_ll_coordbase cb,
		  ggi_coord coord, ggi_coord *dot)
{
	int rc;
	ggi_coord pixel;

	LIB_ASSERT(vis != NULL);
	LIB_ASSERT(dot != NULL);

	rc = ggiGAConv2Pixel(vis, cb, coord, &pixel);
	LIB_ASSERT(rc != GALLOC_OK);
	if (rc != GALLOC_OK) goto exit;

	rc = ggiGAPixel2Dot(vis, pixel, dot);
	LIB_ASSERT(rc != GALLOC_OK);

exit:
	return rc;
}	/* ggiGAConv2Dot */


int ggiGAConvDot2(ggi_visual_t vis, enum ggi_ll_coordbase cb,
		  ggi_coord dot, ggi_coord *coord)
{
	int rc;
	ggi_coord pixel;

	LIB_ASSERT(vis != NULL);
	LIB_ASSERT(coord != NULL);

	rc = ggiGADot2Pixel(vis, dot, &pixel);
	LIB_ASSERT(rc != GALLOC_OK);

	rc = ggiGAConvPixel2(vis, cb, pixel, coord);
	LIB_ASSERT(rc != GALLOC_OK);

	return rc;
}	/* ggiGAConvDot2 */





ggiGA_resource_handle ggiGACopyIntoResource(struct ggiGA_resource_props *props,
					void *priv, size_t privlen)
{
	ggiGA_resource_handle handle;

	LIB_ASSERT(props != NULL);

	handle = calloc(1, sizeof(handle[0]));
	if (handle == NULL) goto err0;

	handle->props = calloc(1, sizeof(struct ggiGA_resource_props));
	if (handle->props == NULL) goto err1;
	memcpy(handle->props, props, sizeof(struct ggiGA_resource_props));

	handle->storage_share = (props->storage_share & GA_SHARE_REQUEST_MASK);
	handle->storage_ok = props->storage_ok;

	handle->priv = handle->next = NULL;
	return(handle);

 err1:
	free(handle);
 err0:
	return NULL;
}	/* ggiGACopyIntoResource */


ggiGA_resource_handle ggiGAWrapIntoResource(struct ggiGA_resource_props *props,
					void *priv, size_t privlen) 
{
	ggiGA_resource_handle handle;


	LIB_ASSERT(priv != NULL);
	LIB_ASSERT(props != NULL);


	handle = calloc(1, sizeof(handle[0]));
	if (handle == NULL) goto err0;

	handle->props = props;
	handle->priv = priv;
	handle->priv_size = privlen;
	handle->next = NULL;

	handle->storage_share = (props->storage_share & GA_SHARE_REQUEST_MASK);
	handle->storage_ok = props->storage_ok;

	return handle;

 err0:
	return NULL;
}	/* ggiGAWrapIntoResource */


void *ggiGAGetPriv(ggiGA_resource_handle handle)
{
	if (handle == NULL) {
		GALLOCDPRINT("NULL passed to GetPriv\n"); 
		return(NULL);
	}	/* if */

	return handle->priv;
}	/* ggiGAGetPriv */


/* Returns the pointer of the next member of the list
 */
ggiGA_resource_list ggiGAGetNext(ggiGA_resource_list list)
{
	if (list == NULL) return NULL;

	return list->next;
}	/* ggiGAGetNext */


int ggiGASetPriv(ggiGA_resource_handle handle, void *priv, size_t size)
{
	if (handle == NULL) return GGI_EARGINVAL;
	if (priv == NULL && size == 0) {
		if (handle->priv != NULL) free (handle->priv);
		handle->priv = NULL;
		return GALLOC_OK;
	}	/* if */

	if (priv == NULL) return GGI_EARGINVAL;
	if (size == 0) return GGI_EARGINVAL;

	if (handle->priv != NULL) free(handle->priv);
	handle->priv_size = 0;

	handle->priv = malloc(size);
	if (handle->priv == NULL) return GGI_ENOMEM;

	handle->priv_size = size;
	memcpy(handle->priv, priv, size);

	return GALLOC_OK;
}	/* ggiGASetPriv */


enum ggiGA_resource_type ggiGAGetType(ggiGA_resource_handle handle) 
{
	return (handle->res_type & GA_RT_FULLTYPE_MASK);
}	/* ggiGAGetType */


enum ggiGA_resource_type ggiGASetType(ggiGA_resource_handle handle, 
				   enum ggiGA_resource_type rt)
{
	enum ggiGA_resource_type tmp;

	tmp = handle->res_type;
	handle->res_type &= ~GA_RT_FULLTYPE_MASK;
	handle->res_type |= rt;
	return (tmp & GA_RT_FULLTYPE_MASK);
}	/* ggiGASetType */


enum ggiGA_resource_state ggiGAGetState(ggiGA_resource_handle handle) 
{
	return (handle->res_state & GA_STATE_RESPONSE_MASK);
}	/* ggiGAGetState */


enum ggiGA_resource_state ggiGASetState(ggiGA_resource_handle handle, 
				enum ggiGA_resource_state rs)
{
	enum ggiGA_resource_state tmp;

	tmp = (handle->res_state & GA_STATE_RESPONSE_MASK);
	rs &= GA_STATE_REQUEST_MASK;
	rs |= (handle->res_state & ~GA_STATE_REQUEST_MASK);

	handle->res_state = rs;
	return tmp;
}	/* ggiGASetState */


unsigned int ggiGAGetCoordBase(ggiGA_resource_handle handle)
{
	return handle->cb;
}	/* ggiGAGetCoordBase */


enum ggiGA_storage_type ggiGAGetStorage(ggiGA_resource_handle handle)
{
	return (handle->storage_ok);
}	/* ggiGAGetStorage */


enum ggiGA_storage_share ggiGAGetStorageShareState(ggiGA_resource_handle handle)
{
	return (handle->storage_share & GA_SHARE_RESPONSE_MASK);
}	/* ggiGAGetStorageShareState */


enum ggiGA_storage_share ggiGASetStorageShareState(ggiGA_resource_handle handle,
						enum ggiGA_storage_share ss)
{
	enum ggiGA_storage_share tmp;

	tmp = (handle->storage_share & GA_SHARE_RESPONSE_MASK);
	ss &= GA_SHARE_REQUEST_MASK;
	ss |= (handle->storage_share & ~GA_SHARE_REQUEST_MASK);

	handle->storage_share = ss;
	return tmp;
}	/* ggiGASetStorageShareState */


int ggiGAIsShared(ggiGA_resource_handle handle)
{
	return ((handle->storage_share & GA_SHARE_SHARED) ? 1 : 0);
}	/* ggiGAIsShared */

int ggiGAIsShareable(ggiGA_resource_handle handle)
{
	return ((handle->storage_share & GA_SHARE_SHAREABLE) ? 1 : 0);
}	/* ggiGAIsShareable */

int ggiGAHasCopyOnWrite(ggiGA_resource_handle handle)
{
	return ((handle->storage_share & GA_SHARE_COPYONWRITE) ? 1 : 0);
}	/* ggiGAHasCopyOnWrite */

int ggiGAIsReadOnly(ggiGA_resource_handle handle)
{
	return ((handle->storage_ok & GA_STORAGE_WRITE) ? 0 : 1);
}	/* ggiGAIsReadOnly */


#if 0

int ggiGACopy2Write(ggiGA_resource_handle *handle)
{
	ggiGA_resource_list list = NULL;
	ggiGA_resource_list list2 = NULL;
	ggiGA_resource_handle handle2 = NULL;
	void *handle_data = NULL;

	int rc = GALLOC_OK;


	GALLOCDPRINT_SHARE("%s:%s:%i: entered\n", DEBUG_INFO);

	if (handle == NULL) return GGI_EARGINVAL;
	if (handle[0] == NULL) return GGI_EARGINVAL;

	if (!ggiGAIsShared(handle[0]) || !ggiGAHasCopyOnWrite(handle[0])) {
		GALLOCDPRINT_SHARE("%s:%s:%i: handle not shared or no copy2write-flag\n",
				DEBUG_INFO);
		rc = GALLOC_EFAILED;
		goto err0;
	}	/* if */

	GALLOCDPRINT_SHARE("%s:%s:%i: make backup\n", DEBUG_INFO);

	/* Backup handle in case of failure */
	handle_data = malloc(sizeof(handle[0]));
	if (!handle_data) {
		rc = GGI_ENOMEM;
		goto err0;
	}	/* if */
	memcpy(handle_data, handle[0], sizeof(handle[0]));


	rc = ggiGAGet(handle[0]->vis, &list);
	LIB_ASSERT(rc == GALLOC_OK);
	if (rc != GALLOC_OK) goto err0;

	LIB_ASSERT(ggiGAFind(list, handle[0]));

	GALLOCDPRINT_SHARE("%s:%s:%i: unshare\n", DEBUG_INFO);

	rc = ggiGAUndoShare(list, handle[0]);
	LIB_ASSERT(rc == GALLOC_OK);
	if (rc != GALLOC_OK) goto err1;

	LIB_ASSERT(handle[0]->priv == NULL);

	/* We assume here, that all list-members are flagged
	 * with (GA_STATE_NOCHANGE | GA_STATE_NORESET)
	 * for speed. If this is not the case, then there is
	 * a bug anywhere.
	 */

	handle[0]->res_state &= ~GA_STATE_NORESET;

	GALLOCDPRINT_SHARE("%s:%s:%i: recheck\n", DEBUG_INFO);

	rc = ggiGACheck(handle[0]->vis, list, NULL);
	LIB_ASSERT(rc == GALLOC_OK);
	if (rc != GALLOC_OK) goto err2;

	LIB_ASSERT(handle[0]->priv != NULL);

	GALLOCDPRINT_SHARE("%s:%s:%i: reset\n", DEBUG_INFO);

	rc = ggiGASet(handle[0]->vis, list, &list2);
	LIB_ASSERT(rc == GALLOC_OK);
	if (rc != GALLOC_OK) goto err2;

	LIB_ASSERT(handle[0]->priv != NULL);

	GALLOCDPRINT_SHARE("%s:%s:%i: update handle\n", DEBUG_INFO);

	handle2 = ggiGAHandle(list2, list, handle[0]);
	LIB_ASSERT(handle2 != NULL);
	if (handle2 == NULL) goto err2;

	handle[0] = handle2;
	rc = GALLOC_OK;

	ggiGAEmptyList(&list2);
	ggiGAEmptyList(&list);
	free(handle_data);
	GALLOCDPRINT_SHARE("%s:%s:%i: exit successful\n", DEBUG_INFO);
	return rc;

err2:
	ggiGAEmptyList(&list2);
err1:
	ggiGAEmptyList(&list);
	memcpy(handle[0], handle_data, sizeof(handle[0]));
	free(handle_data);
err0:
	GALLOCDPRINT_SHARE("%s:%s:%i: exit with error: %i\n",
				DEBUG_INFO, rc);
	return rc;
}	/* ggiGACopy2Write */

#endif
