/*
******************************************************************************

   LibGAlloc: Templates for use by simpler extensions.

   Copyright (C) 2001 Christoph Egger   [Christoph_Egger@t-online.de]
   Copyright (C) 2001 Brian S. Julin    [bri@calyx.com]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <ggi/internal/galloc.h>
#include <ggi/internal/galloc_debug.h>
#include <ggi/ga_prop.h>
#include <stdio.h>
#include <string.h>

/* This function parses through a list, linking resources to one-another
 * in a manner that represents the tag groups found.
 */
int ggiGA_basic_parse_list(ggiGA_resource_list list) {
	ggiGA_resource_handle cur, last, grep;
	
	cur = list;
	last = NULL;
	while (cur) {
		int variant_cur;
		/* Initialize resource by making it a 1-member list */
		cur->left = cur->right = cur;
		/* Also initialize second link in Tag Group Heads */
		if (ggiGA_FULLTYPE(cur) == GA_RT_RESLIST_TAGHEAD) {
			((ggiGA_taghead_t)cur)->left2 =
			  ((ggiGA_taghead_t)cur)->right2 = cur;
		}	/* if */
		/* Caps get pointed to their owners. */
		if (ggiGAIsCap(cur)) {
		  cur->left = cur->right = last;
		  goto nextres1;
		}	/* if */
		variant_cur = 
			ggiGAIsMotor(cur) ? 1 : (ggiGAIsCarb(cur) ? 2 : 0);
		/* Pass # 1: Make double linked lists,
		 * one per variant, per tag, e.g. all motors
		 * with tag == 2 are put in a list, all carbs
		 * with tag == 2 are put in another list, etc.
		 */
		if (!ggiGA_TAG(cur)) goto taghead;
		grep = list;
		while ((grep != NULL) && (grep != cur)) {
			int variant_grep;

			if (ggiGAIsCap(grep)) goto nextgrep1;
			variant_grep = ggiGAIsMotor(grep) ? 1 : 
			  (ggiGAIsCarb(grep) ? 2 : 0);
			/* Tag group heads link into the doubly linked list
			 * via their second set of pointers if their second
			 * tag matches.
			 */
			if ((variant_cur == variant_grep) &&
			    (ggiGA_FULLTYPE(grep) == GA_RT_RESLIST_TAGHEAD) &&
			    (((ggiGA_taghead_t)grep)->tag2 == ggiGA_TAG(cur)))
			{
				ggiGA_taghead_t th;
				th = (ggiGA_taghead_t)grep;
				cur->left = grep;
				cur->right = th->right2;
				th->right2->left = cur;
				th->right2 = cur;
				break;
			}	/* if */
			if ((variant_cur == variant_grep) &&
			    ggiGA_TAG(cur) == ggiGA_TAG(grep)) {
				cur->left = grep;
				cur->right = grep->right;
				grep->right->left = cur;
				grep->right = cur;
				break;
			}	/* if */
		nextgrep1:
				grep = grep->next;
		}	/* while */
	taghead:
		/* More or less repeat of the above code using the
		 * second link in the tag head as the starting point.
		 */
		if (ggiGA_FULLTYPE(cur) == GA_RT_RESLIST_TAGHEAD) {
			ggiGA_taghead_t th;
		  
			th = (ggiGA_taghead_t)cur;
			if (!th->tag2) break;

			grep = list;
		
			while (grep && (grep != cur)) {
				int variant_grep;

				if (ggiGAIsCap(cur)) goto nextgrep2;
				variant_grep = ggiGAIsMotor(grep) ? 1 : 
				  (ggiGAIsCarb(grep) ? 2 : 0);

				if ((variant_cur == variant_grep) &&
				    th->tag2 == ggiGA_TAG(grep)) {
					th->left2 = grep;
					th->right2 = grep->right;
					grep->right->left = cur;
					grep->right = cur;
					break;
				}	/* if */
			nextgrep2:
				grep = grep->next;
			}	/* while */
		}	/* if */
	nextres1:
		last = cur;
		cur = cur->next;
	}	/* while */


	/* Pass 2: Find rings of carbs which have no Tag Group Head
	 * in them.  Link them to the ring of motors with the same 
	 * tag.  If the motor ring contains a tag group head, use that as
	 * the linking point.
	 */
	cur = list;
	while (cur) {
		int found;
		if (!ggiGAIsCarb(cur)) goto nextres2;
		/* Check if this carb has already been unlinked */
		if (!ggiGAIsCarb(cur->left)) goto nextres2;
		/* Check if there is a tag group head for this
		 * ring (Note: "for this ring", not "in this ring").
		 * What we are looking for is a tag group head 
		 * which is linked into the ring via it's second set of 
		 * pointers. Those linked in by the first set are just
		 * regular members of this tag group which happen to have
		 * a subordinate tag group, so they don't count.
		 */
		grep = cur;
		found = 0;
		do {
			if (grep->right->left != grep) {
				ggiGA_taghead_t th;

				th = (ggiGA_taghead_t)grep->right;
			 	LIB_ASSERT(ggiGA_FULLTYPE(th) == GA_RT_RESLIST_TAGHEAD);
				LIB_ASSERT(th->left2 == grep);
				LIB_ASSERT(th->tag2 == ggiGA_TAG(cur));
				LIB_ASSERT(ggiGA_TAG(grep) != ggiGA_TAG(cur));

				found = 1;
				break;
			}
			grep = grep->right;
		} while (grep != cur);
		/* A tag group head was found for this ring, so we leave
		 * it alone. 
		 */
		if (found) goto nextres2;
		/* Now look for a matching motor ring */
		grep = list;
		found = 0;
		while (grep) {
			if (ggiGAIsMotor(grep) && 
			    (ggiGA_TAG(grep) == ggiGA_TAG(cur))) {
				found = 1;
				break;
			}
			grep = grep->next;
		}
		if (!found) goto nextres2;
		/* Look for a tag group head in the motor ring */
		found = 0;
		last = grep;
		do {
			if (grep->right->left != grep) {
				ggiGA_taghead_t th;

				th = (ggiGA_taghead_t)grep->right;
			 	LIB_ASSERT(ggiGA_FULLTYPE(th) == GA_RT_RESLIST_TAGHEAD);
				LIB_ASSERT(th->left2 == grep);
				LIB_ASSERT(th->tag2 == ggiGA_TAG(last));
				LIB_ASSERT(ggiGA_TAG(grep) != ggiGA_TAG(last));

				found = 1;
				break;
			}
			grep = grep->right;
		} while (grep != last);
		last = grep;
		/* Point all the left hand pointers in the carb ring to
		 * the motor ring.  We know there are no tag group heads
		 * to worry about.
		 */
		grep = cur;
		do {
			grep->left = last;
			grep = grep->right;
		} while (grep != cur);

	nextres2:
		cur = cur->next;
	}
	return 0;
}



/* When we need to "start from the top" we copy the master haslist
 * back into a *pre-existing* copy of the haslist, without altering
 * the parsed list structure and keeping the pointers to the master
 * resources intact.
 */
static inline void reset_haslist(ggi_visual_t vis, 
				 ggiGA_resource_list haslist) {
	struct ggiGA_resource res;
	ggiGA_resource_handle master;

	master = GA_HASLIST(vis);
	while (master) {
		memcpy(&res, haslist, sizeof(struct ggiGA_resource));
		if (haslist->props != NULL)
			memcpy(haslist->props, master->props, 
			       sizeof(struct ggiGA_resource_props));
		if (haslist->priv != NULL)
			memcpy(haslist->priv, master->priv, 
			       haslist->priv_size);
		memcpy(haslist, master, sizeof(struct ggiGA_resource));
		haslist->master = master;
		haslist->props  = res.props;
		haslist->priv   = res.priv;
		haslist->left   = res.left;
		haslist->right  = res.right;
		if ((res.res_type & GA_RT_FULLTYPE_MASK)
		    == GA_RT_RESLIST_TAGHEAD) {
			((ggiGA_taghead_t)haslist)->left2 =
			  ((ggiGA_taghead_t)&res)->left2;
			((ggiGA_taghead_t)haslist)->right2 =
			  ((ggiGA_taghead_t)&res)->right2;
		}
		haslist->next   = res.next;

		master = master->next;
		haslist = haslist->next;
	}	/* while */
}

/* Approximately as above, for the reslist.
 */
static inline void reset_reqlist(ggiGA_resource_list origreq, 
				 ggiGA_resource_list reqlist) {
	struct ggiGA_resource res;

	while (origreq) {
		memcpy(&res, reqlist, sizeof(struct ggiGA_resource));
		if (reqlist->props != NULL)
			memcpy(reqlist->props, origreq->props, 
			       sizeof(struct ggiGA_resource_props));
		if (reqlist->priv != NULL)
			memcpy(reqlist->priv, origreq->priv, 
			       reqlist->priv_size);
		memcpy(reqlist, origreq, sizeof(struct ggiGA_resource));
		reqlist->master = res.master;
		reqlist->props  = res.props;
		reqlist->priv   = res.priv;
		reqlist->left   = res.left;
		reqlist->right  = res.right;
		if ((res.res_type & GA_RT_FULLTYPE_MASK)
		    == GA_RT_RESLIST_TAGHEAD) {
			((ggiGA_taghead_t)reqlist)->left2 =
			  ((ggiGA_taghead_t)&res)->left2;
			((ggiGA_taghead_t)reqlist)->right2 =
			  ((ggiGA_taghead_t)&res)->right2;
		}
		reqlist->next   = res.next;
		reqlist->res_state |= GA_STATE_FAILED;

		origreq = origreq->next;
		reqlist = reqlist->next;
	}	/* while */
}

static void reset_state(ggi_visual_t vis, struct ggiGA_template_state *state,
                        void *privstate) {
	if (LIBGGI_GALLOCEXT(vis)->reset_privstate &&
	    !(LIBGGI_GALLOCEXT(vis)->reset_privstate(vis, state, privstate)))
		return;
	state->mode = NULL;
}	/* reset_state */


/* Finds the taghead of res, (but only if it occurs before res 
 * in the list).  Returns NULL when nothing found.
 */
static inline ggiGA_taghead_t find_taghead(ggiGA_resource_list list,
					   ggiGA_resource_handle this) {
	ggiGA_resource_handle tmp2;
	ggiGA_taghead_t taghead;

	if (!ggiGA_TAG(this)) return NULL;

	tmp2 = list;
	taghead = NULL;

	while (tmp2 != this) {
		if (ggiGA_FULLTYPE(tmp2) != GA_RT_RESLIST_TAGHEAD) 
		  goto skip;
		taghead = (ggiGA_taghead_t)tmp2;
		if (ggiGA_TAG(this) == taghead->tag2) break;
		taghead = NULL;
	skip: 
		tmp2 = tmp2->next;
	}
	return taghead;
}

#define find_compound(list,res) ggiGAFindCompoundHead(list, res)

/* This function checks to see if has in haslist, which is a proposed 
 * master resource for res in reslist, would place res in the "correct"
 * haslist tag group/compound as determined by resources which res 
 * is attached to.  The rules defining "correct" are a bit complex:
 */
static inline int do_fit(ggiGA_resource_list reqlist,
			 ggiGA_resource_handle req,
			 ggiGA_resource_list haslist,
			 ggiGA_resource_handle has) {
	ggiGA_resource_handle has_tmp, req_tmp;

	/* All resources before req (e.g. resources which
	 * have already been checked) which are in req's 
	 * compound must have masters in the corresponding
         * compound in haslist (that is, the compound which
	 * has belongs to.)
	 *
	 * Since resources prior to this have succeeded
	 * this test already, it is only necessary to 
	 * do the following check:
	 */
	req_tmp = find_compound(reqlist, req);
	has_tmp = find_compound(haslist, has);
	if ((req_tmp != req) &&
	    (find_compound(haslist, req_tmp->master) != has_tmp)) return -1;

	/* All resources before and including req which are in 
	 * the same ring as req must have masters in the corresponding
         * ring in haslist (that is, the ring that has belongs to.)
	 *
	 * Since resources prior to this have succeeded
	 * this test already, it is only necessary to 
	 * check one such resource.  But first we must find it.
	 * (It cannot be the tag group head.)
	 */
	req_tmp = req;
	do {
		/* Skip past the tag group head of the ring. */
		if (ggiGA_TAG(req_tmp) != ggiGA_TAG(req)) {
		  LIB_ASSERT(ggiGA_FULLTYPE(req_tmp) == GA_RT_RESLIST_TAGHEAD);
			req_tmp = ((ggiGA_taghead_t)req_tmp)->right2;
			continue;
		}	/* if */
		req_tmp = req_tmp->right;
		if (!ggiGAIsFailed(req_tmp)) break;
	} while (req_tmp != req);
	if (req_tmp == req) goto alone;
	has_tmp = has;
	do {
		if (req_tmp->master == has_tmp) break; 
		/* Skip past the tag group head of the ring. */
		if (ggiGA_TAG(has_tmp) != ggiGA_TAG(has)) {
		  LIB_ASSERT(ggiGA_FULLTYPE(has_tmp) == GA_RT_RESLIST_TAGHEAD);
			has_tmp = ((ggiGA_taghead_t)has_tmp)->right2;
			continue;
		}	/* if */
		has_tmp = has_tmp->right;
	} while (has_tmp != has);
	if (req_tmp->master != has_tmp) return -1;

	/* If the req's left pointer is pointing out if its ring, 
	 * we check to make sure that ring it is hooked to corresponds
	 * to the ring that has's left pointer points to.  (That is, 
	 * all the resources in that ring have masters which are in the 
	 * corresponding ring in haslist ring.
	 *
	 * Again, this test only applies to resources which have
	 * already passed a check; unchecked resources are ignored.
	 */
 alone:
	req_tmp = req->left;
	do {
		if (req_tmp == req) return GALLOC_OK; /* ring is 2-way */
		/* Skip past the tag group head of the ring. */
		if (ggiGA_TAG(req_tmp) != ggiGA_TAG(req)) {
		  LIB_ASSERT(ggiGA_FULLTYPE(req_tmp) == GA_RT_RESLIST_TAGHEAD);
			req_tmp = ((ggiGA_taghead_t)req_tmp)->right2;
			continue;
		}	/* if */
		req_tmp = req_tmp->right;
	} while (req_tmp != req->left);
	/* Find a pointer in the ring that has already been checked */
	if (!ggiGAIsFailed(req->left)) goto skip;
	req_tmp = req->left;
	do {
		if (!ggiGAIsFailed(req_tmp)) break;
		/* Skip past the tag group head of the ring. */
		if (ggiGA_TAG(req_tmp) != ggiGA_TAG(req)) {
		  LIB_ASSERT(ggiGA_FULLTYPE(req_tmp) == GA_RT_RESLIST_TAGHEAD);
			req_tmp = ((ggiGA_taghead_t)req_tmp)->right2;
			continue;
		}	/* if */
		req_tmp = req_tmp->right;
	} while (req_tmp != req->left);
	/* If there are no successful resources yet in that ring, we will
	 * give the requested resource the benefit of the doubt.
	 *
	 * However, right now this case is not handled elsewhere in the code, 
	 * so we complain. 
	 */
	LIB_ASSERT(req_tmp != req->left /* beware: unchecked ring! */);
	if (req_tmp == req->left) return GALLOC_OK;
 skip:
	/* Finally, we look for its master in the haslist */
	has_tmp = has->left;
	do {
		if (req_tmp->master == has_tmp) break; 
		/* Simply traverse through the tag group heads of the ring. */
		if (ggiGA_TAG(has_tmp) != ggiGA_TAG(has)) {
		  LIB_ASSERT(ggiGA_FULLTYPE(has_tmp) == GA_RT_RESLIST_TAGHEAD);
			has_tmp = ((ggiGA_taghead_t)has_tmp)->right2;
			continue;
		}	/* if */
		has_tmp = has_tmp->right;
	} while (has_tmp != has->left);
	if (req_tmp->master != has_tmp) return -1;
	
	return GALLOC_OK;
}

static inline int do_callback(ggi_visual_t vis,
			      ggiGA_resource_list reslist,
			      enum ggiGA_callback_command command,
			      ggiGA_resource_handle res,
			      ggiGA_resource_handle head,
			      ggiGA_taghead_t taghead,
			      ggiGA_resource_handle has,
			      struct ggiGA_template_state *state,
			      void *privstate);


int ggiGACheck_template (ggi_visual_t vis, 
			 ggiGA_resource_list request,
			 ggiGA_resource_list *result, 
			 struct ggiGA_template_state *state,
			 void *privstate)
{
	ggiGA_resource_list haslist = NULL;
	ggiGA_resource_list reqlist = NULL;

	ggiGA_resource_handle tmp1 = NULL;
	ggiGA_resource_handle tmp2 = NULL;
	ggiGA_resource_list reslist = NULL;
	ggiGA_resource_handle farthest = NULL;
	int at_farthest = 1;
	int need_reset = 0;

	int rc = GALLOC_OK;

	if (request == NULL) {

		/* ggiGAGet -- begin */

		if (result == NULL) return GGI_EARGINVAL;

		if (GA_RESLIST(vis) == NULL) {
			/* Visual's mode either unset or ggiSetMode used */
			ggi_mode mode;

			GALLOCDPRINT("Adding implicite GA_RT_FRAME\n");
		
			/* This is where the mode resource magically appears */
			rc = ggiGetMode(vis, &mode);
			LIB_ASSERT(rc == GALLOC_OK);

			tmp1 = NULL;
			rc = ggiGAAddMode(vis, &(GA_RESLIST(vis)),
					  &mode, NULL, &tmp1);
			LIB_ASSERT(rc == GALLOC_OK);
			if (rc != GALLOC_OK) return GALLOC_EFAILED;

			ggiGAFlagActive(tmp1);
			tmp1->vis = vis;

		}	/* if */

		rc = ggiGACopyList(GA_RESLIST(vis), result);
		LIB_ASSERT(rc == GALLOC_OK);
		return rc;

		/* ggiGAGet -- end */

	}	/* if */

	/* Make a copy of the haslist, with links back to the master copy. */
	rc = ggiGACopyList(GA_HASLIST(vis), &haslist);
	LIB_ASSERT(rc == GALLOC_OK);
	if (rc != GALLOC_OK) return rc;
	ggiGA_basic_parse_list(haslist);
	reset_haslist(vis, haslist);
	reset_state(vis, state, privstate);

	state->haslist = haslist;

	/* Make a copy of the request, with cleared master links. */
	rc = ggiGACopyList(request, &reqlist);
	LIB_ASSERT(rc == GALLOC_OK);
	if (rc != GALLOC_OK) return rc;
	ggiGA_basic_parse_list(reqlist);
	tmp1 = reqlist;
	while (tmp1 != NULL) {
		tmp1->master = NULL;
		tmp1 = tmp1->next;
	}	/* while */

#if 0
	reset_reqlist(request, reqlist);
	do {
	  char *foo = NULL;
	  ggiGAanprintf(vis, haslist, 9000, "", &foo); 
	  fprintf(stderr, "haslist = \n%s\n", foo);
	  free(foo);
	  foo = NULL;
	  ggiGAanprintf(vis, reqlist, 9000, "", &foo); 
	  fprintf(stderr, "reqlist = \n%s\n", foo);
	  free(foo);
	} while (0);
#endif

#define CALLBACK_REQ(_cbflag) do_callback(vis, reqlist, _cbflag, \
tmp2, compound, taghead, tmp2->master, state, privstate)

#define CALLBACK_HAS(_cbflag) do_callback(vis, reqlist, _cbflag, \
tmp2, compound, taghead, tmp2, state, privstate)

	tmp1 = reqlist;
	reset_reqlist(request, reqlist);  
	reset_haslist(vis, haslist);
	while (tmp1 != NULL) {
		ggiGA_taghead_t taghead = NULL;
		ggiGA_resource_handle compound = NULL;
		if (need_reset > 1) {
			reset_reqlist(request, reqlist);
			reset_haslist(vis, haslist);
			reset_state(vis, state, privstate);
			tmp2 = reqlist;
			while (tmp2 != tmp1) {
				taghead = find_taghead(reqlist, tmp2);
				compound = find_compound(reqlist, tmp2);
				if (tmp2->res_state & GA_STATE_NORESET) 
					goto nocheck1;
#warning why no CB_MATCH here I ask myself
				rc = CALLBACK_REQ(GA_CB_PRECHECK);
				LIB_ASSERT(rc == GALLOC_OK);
				rc = CALLBACK_REQ(GA_CB_CHECK);
				LIB_ASSERT(rc == GALLOC_OK);
			nocheck1:
				ggiGAFlagSuccess(tmp2);
				if (ggiGAIsCap(tmp2->next)) 
					ggiGAFlagSuccess(tmp2);
				rc = CALLBACK_REQ(GA_CB_POSTCHECK);
				LIB_ASSERT(rc == GALLOC_OK);
				tmp2 = tmp2->next;
			}
			need_reset = 0;
		}
		if (need_reset) {
			tmp2 = reqlist;
			while (tmp2 != tmp1) {
				taghead = find_taghead(reqlist, tmp2);
				compound = find_compound(reqlist, tmp2);
				rc = CALLBACK_REQ(GA_CB_RECHECK);
				LIB_ASSERT(rc == GALLOC_OK);
				tmp2 = tmp2->next;
			}
			need_reset = 0;
		}
		taghead = find_taghead(reqlist, tmp1);
		compound = find_compound(reqlist, tmp1);
		tmp2 = (tmp1->master == NULL) ? haslist : tmp1->master->next;
		while (tmp2 != NULL) {
			if (CALLBACK_HAS(GA_CB_MATCH)) goto nexthas;
			if (tmp2->res_state & GA_STATE_NORESET) 
				goto nocheck2;
			if (CALLBACK_HAS(GA_CB_PRECHECK)) goto nexthas;
			if (do_fit(reqlist, tmp1, haslist, tmp2))
				goto nexthas;
			if (CALLBACK_HAS(GA_CB_CHECK)) goto nexthas;
		nocheck2:
			if (CALLBACK_HAS(GA_CB_POSTCHECK)) goto nexthas;
			ggiGAFlagSuccess(tmp1);
			if (ggiGAIsCap(tmp1->next)) ggiGAFlagSuccess(tmp1);
			tmp1->master = tmp2;
			break;
		nexthas:
			tmp2 = tmp2->next;
		}	/* while */
		if (tmp2 == NULL) { /* No match */
			if (tmp1 != reqlist) {
				tmp1->master = NULL;
				tmp2 = reqlist;
				while (tmp2->next != tmp1) tmp2 = tmp2->next;
			}	/* if */
			fprintf(stderr, "rolling back to %p\n", tmp2);
			if (at_farthest) 
				/* Take a snapshot for the result */
				ggiGACopyList(reqlist, &reslist);
			tmp1 = tmp2;
			at_farthest = 0;
			if (tmp2 != NULL) {
				ggiGAFlagFailed(tmp2);
				taghead = find_taghead(reqlist, tmp1);
				compound = find_compound(reqlist, tmp1);
				tmp2 = tmp2->next;
				while (tmp2 != NULL) {
					tmp2->master = NULL;
					tmp2 = tmp2->next;
				}	/* while */
				need_reset = CALLBACK_REQ(GA_CB_UNCHECK);
			}	/* if */
		}	/* if */
		else { 
			if (at_farthest) {
				farthest = tmp1;
				/* Take a snapshot for the result */
				ggiGACopyList(reqlist, &reslist);
			}
			else if (tmp1 == farthest) at_farthest = 1;
			tmp1 = tmp1->next; 
		}	/* if-else */
	}	/* while */
	if (tmp2 == NULL) rc = GALLOC_EUNAVAILABLE;

	/* Done with the copy of the request */
	ggiGAEmptyList(&reqlist);

	/* Now we build the result: reslist contains the list
	 * at the point that the most resources were successful.
	 * but all the resources in it have been altered, even if
	 * they had the GGI_GA_NOCHANGE flag set.  We copy the
	 * original reqlist and then copy the contents of the
	 * resources that should change into this list, and return
	 * it.
	 */
	if (result == NULL) goto out_noresult;

	/* Check for the handle-preserving special case. */
	if (request != result[0]) {
		ggiGACopyList(request, result);
	}	/* if */
	tmp1 = reslist;
	tmp2 = result[0];
	while (tmp1) {
		struct ggiGA_resource_props *props;
		ggiGA_resource_handle next;
		void *priv;

		/* The resource part is always copied */
		props = tmp2->props;
		priv  = tmp2->priv;
		next  = tmp2->next;
		memcpy (tmp2, tmp1, sizeof(struct ggiGA_resource));
		tmp2->props  = props;
		tmp2->priv   = priv;
		tmp2->next   = next;
		tmp2->right  = NULL;
		tmp2->left   = NULL;
		/* Master links provided for ggiGASet -- could be cleared 
		 * by target before handing back to user if we want to 
		 * be more opaque.
		 */
		if (tmp2->master) tmp2->master = tmp2->master->master;
		if (ggiGA_FULLTYPE(tmp2) == GA_RT_RESLIST_TAGHEAD) {
			((ggiGA_taghead_t)tmp2)->left2 = NULL;
			((ggiGA_taghead_t)tmp2)->right2 = NULL;
		}
		if (tmp2->res_state & GA_STATE_NOCHANGE) goto nochange;
		if (props) memcpy(tmp2->props, tmp1->props, 
				  sizeof(struct ggiGA_resource_props));
		if (priv) memcpy(tmp2->priv, tmp1->priv, tmp2->priv_size);
	nochange:
		if (ggiGAIsFailed(tmp1)) break; /* Quit after first failure */
		tmp1 = tmp1->next;
		tmp2 = tmp2->next;
	}

 out_noresult:

	do {
	  char *foo = NULL;
	  ggiGAanprintf(vis, reslist, 9000, "", &foo);
	  fprintf(stderr, "reslist = \n%s\n", foo);
	  free(foo);
	} while (0);
	if (result && result[0]) {
	  do {
	    char *foo = NULL;
	    ggiGAanprintf(vis, result[0], 9000, "", &foo);
	    fprintf(stderr, "result = \n%s\n", foo);
	    free(foo);
	  } while (0);
	}

	ggiGAEmptyList(&reslist);
	return rc;
}	/* ggiGACheck_template */


int ggiGASet_template(ggi_visual_t vis, 
		      ggiGA_resource_list request,
		      ggiGA_resource_list *result, 
		      struct ggiGA_template_state *state,
		      void *privstate)
{
	ggiGA_resource_list reqlist = NULL;
	ggiGA_resource_list haslist = NULL;
	int rc = GALLOC_OK;
	ggiGA_resource_handle tmp, compound, active, appendat;
	ggiGA_taghead_t taghead;

	LIB_ASSERT(request != NULL);
	if (request == NULL) return GALLOC_ESPECIFY;

	/* Decide if handle-preserving special case or not */
	if ((result != NULL) && (request == result[0])) {
		reqlist = request; /* Work directly with user handles. */
	}
	else {
		ggiGACopyList(request, &reqlist); /* Work with a copy */  
	}

	/* Ignore/turn off GA_STATE_NOCHANGE_FLAGS */
	tmp = reqlist;
	while (tmp != NULL) {
		tmp->res_state &= ~GA_STATE_NOCHANGE;
		tmp = tmp->next;
	}

	rc = ggiGACheck(vis, reqlist, result);

	/* Don't need this anymore (but don't empty result in special case.) */
	if (reqlist != request) ggiGAEmptyList(&reqlist);

	/* Just returning here gives identical behavior to ggiGACheck,
	 * except for handling of immutable resources.  No fuss, no muss.
	 */
	if (rc != GALLOC_OK) return rc;

	/* ggiGACheck will have deleted our parsed state, so recreate it. */
	ggiGA_basic_parse_list(result[0]);

	haslist = state->haslist;

	/* Nuance: currently when adding resources to the active
	 * list, the new resources must be at the end of the list,
	 * and the list must contain a copy of the active resources
	 * up to the point where the new resources begin.
	 *
	 * Even with these restrictions it is a bit tricky to
	 * figure out where the old resources end and the new
	 * ones begin, because new mutual resources will be flagged
	 * active also.  They can be told apart because their properties
	 * are at the same memory location.
	 */
	active = appendat = GA_RESLIST(vis);
	tmp = result[0];
	while ((tmp != NULL) && (appendat != NULL)) {
		ggiGA_resource_handle mutual;

		if (!(tmp->res_state & GA_STATE_NORESET)) break;
		mutual = ggiGAPropsHandle(result[0], tmp);
		if ((mutual != NULL) && (mutual != tmp) &&
		    (ggiGAPropsHandle(active, appendat)
		     != ggiGAHandle(active, result[0], mutual))) break;
		tmp = tmp->next;
		appendat = appendat->next;
	};
#warning release instead of just truncate here.
	if (appendat != NULL) ggiGATruncateList(GA_RESLIST(vis), &appendat);

#if 0
	do {
	  char *foo = NULL;
	  ggiGAanprintf(vis, GA_RESLIST(vis), 9000, "", &foo);
	  fprintf(stderr, "posttruncate = \n%s\n", foo);
	  free(foo);
	} while (0);

	do {
	  char *foo = NULL;
	  ggiGAanprintf(vis, result[0], 9000, "", &foo);
	  fprintf(stderr, "preset = \n%s\n", foo);
	  free(foo);
	} while (0);
#endif

	tmp = result[0];
	appendat = active = GA_RESLIST(vis);
	while (tmp != NULL) {
		if (ggiGAIsCap(tmp)) goto setact;

		taghead = find_taghead(result[0], tmp);
		compound = find_compound(result[0], tmp);

		rc = (do_callback(vis, result[0], GA_CB_SET,
				  tmp, compound, taghead,
				  tmp->master, state, privstate));
		LIB_ASSERT(rc == GALLOC_OK);
		if (rc != GALLOC_OK) {
			ggiGAFlagFailed(tmp);
			break;
		}
setact:
		tmp->vis = vis;
		if (appendat == NULL) {		/* New resource. */
			ggiGA_resource_handle tmp2;
			ggiGAFlagActive(tmp);

			tmp2 = ggiGAPropsHandle(result[0], tmp);
			if (tmp2 == tmp)
				ggiGAAppend(tmp, &active);
			else {
				tmp2 = ggiGAHandle(active, result[0], tmp2);
				ggiGAAddMutual(active, &tmp2);
				/* The new mutual is flagged failed. */
				ggiGAFlagSuccess(tmp2);
			}
		}

		tmp = tmp->next;
		if (appendat) appendat = appendat->next;
	}	/* while */
	GA_RESLIST(vis) = active;

	do {
	  char *foo = NULL;
	  ggiGAanprintf(vis, GA_RESLIST(vis), 9000, "", &foo);
	  fprintf(stderr, "setresult = \n%s\n", foo);
	  free(foo);
	} while (0);


	/* Clean up the result list for use by the caller, by removing parsed 
	 * state and master links which they have no business touching.
	 */
	tmp = result[0];
	while (tmp) {
		struct ggiGA_resource_props *props;

		props = tmp->props;
		tmp->right  = NULL;
		tmp->left   = NULL;
		tmp->master = NULL;
		if (ggiGA_FULLTYPE(tmp) == GA_RT_RESLIST_TAGHEAD) {
			((ggiGA_taghead_t)tmp)->left2 = NULL;
			((ggiGA_taghead_t)tmp)->right2 = NULL;
		}
		tmp = tmp->next;
	}

	LIB_ASSERT(rc == GALLOC_OK);
	if (rc != GALLOC_OK) return rc;

	return rc;
}	/* ggiGASet_template */



static inline int do_callback(ggi_visual_t vis,
			      ggiGA_resource_list reslist,
			      enum ggiGA_callback_command command,
			      ggiGA_resource_handle res,
			      ggiGA_resource_handle head,
			      ggiGA_taghead_t taghead,
			      ggiGA_resource_handle has,
			      struct ggiGA_template_state *state,
			      void *privstate) 
{
	ggiGA_reslist_callback *cb;
	cb = NULL;
	if (has->cbflags1 & command) cb = has->cb1;
	else if (has->cbflags2 & command) cb = has->cb2;
	else {
		int i = 0;
		do {
			if ((command >> i) & 1) break;
		} while (++i < 32);
		if (i >= 32) return GGI_EARGINVAL;
		cb = LIBGGI_GALLOCEXT(vis)->default_callbacks[i];
	}
	if (cb == NULL) return GGI_EARGINVAL;
	return cb(vis, reslist, command, res, head, taghead, has,
		  state, privstate);
}



/* Default functions for callbacks implementing lowest common denominator */

/* The match function performs a basic type/bounds checking of the resource,
 * to see if it is just plain incompatible.  It does not alter the resource,
 * so it does not check any feilds that feed back suggestions.
 */
int ggiGA_default_match(ggi_visual_t vis,
			ggiGA_resource_list reslist,
			enum ggiGA_callback_command command,
			ggiGA_resource_handle res,
			ggiGA_resource_handle head,
			ggiGA_taghead_t taghead,
			ggiGA_resource_handle has,
			struct ggiGA_template_state *state,
			void *privstate) {

	/* Don't process NULLs or Caps */
	if (res == NULL || has == NULL) return -1;
	if (ggiGAIsCap(res) || ggiGAIsCap(has)) return -1;

	/* Check resource variant */
	if (((ggiGAIsCarb(res) << 1) | ggiGAIsMotor(res)) !=
	    ((ggiGAIsCarb(has) << 1) | ggiGAIsMotor(has))) return -1;

	/* Check resource type */
	if (ggiGA_TYPE(res) != ggiGA_TYPE(has)) return -1;
	if (ggiGA_SUBTYPE(res) && 
	    (ggiGA_SUBTYPE(res) != ggiGA_SUBTYPE(has))) return -1;

	/* Anything else to check at this stage? */

	return 0;
}


/* The precheck callback checks values for basic compatibility,
 * and does send back suggestions... so, it picks up where the
 * match callback left off.  We know that any resource that gets 
 * to the precheck callback has succeeded the match callback.
 * However, we are still only checking for basic incompatibility,
 * so the availability of storage (or other allocatable quantities) 
 * are not checked.
 */

int ggiGA_default_motor_precheck (ggi_visual_t vis,
				  ggiGA_resource_list reslist,
				  enum ggiGA_callback_command command,
				  ggiGA_resource_handle res,
				  ggiGA_resource_handle head,
				  ggiGA_taghead_t taghead,
				  ggiGA_resource_handle has,
				  struct ggiGA_template_state *state,
				  void *privstate) {
	return 0;
}

int ggiGA_default_carb_precheck(ggi_visual_t vis,
				ggiGA_resource_list reslist,
				enum ggiGA_callback_command command,
				ggiGA_resource_handle res,
				ggiGA_resource_handle head,
				ggiGA_taghead_t taghead,
				ggiGA_resource_handle has,
				struct ggiGA_template_state *state,
				void *privstate) {
	return 0;
}

int ggiGA_default_tank_precheck (ggi_visual_t vis,
				 ggiGA_resource_list reslist,
				 enum ggiGA_callback_command command,
				 ggiGA_resource_handle res,
				 ggiGA_resource_handle head,
				 ggiGA_taghead_t taghead,
				 ggiGA_resource_handle has,
				 struct ggiGA_template_state *state,
				 void *privstate) {
	ggiGA_resource_handle res_c = NULL;
	ggiGA_resource_handle has_c = NULL;
	struct ggiGA_resource_props *res_p, *has_p, *res_c_p, *has_c_p;
	struct ggiGA_tank_props *res_sp, *has_sp, *res_c_sp, *has_c_sp;
	int rc;
	enum ggiGA_storage_type storage_mask_res, storage_mask_has;
	enum ggiGA_storage_type storage_mask;

	res_p = res->props; res_sp = &(res_p->sub.tank);
	has_p = has->props; has_sp = &(has_p->sub.tank);

	if (ggiGAIsCap(res->next)) {
		if (ggiGATankCapSanity(res) != GALLOC_OK) 
			return GALLOC_EFAILED;
		res_c = res->next;
		res_c_p = res_c->props; res_c_sp = &(res_c_p->sub.tank);
	}

	if (ggiGAIsCap(has->next)) {
		has_c = has->next;
		has_c_p = has_c->props; has_c_sp = &(has_c_p->sub.tank);
	}
	
	rc = 0;
	if ((res_c == NULL) && (has_c == NULL)) goto nocaps;
	if ((res_c != NULL) && (has_c != NULL)) goto bothcaps;
	if ((res_c != NULL) && (has_c == NULL)) goto rescap;
	if ((res_c == NULL) && (has_c != NULL)) goto hascap;

 nocaps:
	/* Compare graphtype with haslist */
	if (!(res_sp->gt & GT_DEPTH_MASK)) 
		res_sp->gt |= has_sp->gt & GT_DEPTH_MASK;
	rc |= (has_sp->gt & GT_DEPTH_MASK) &&
		((res_sp->gt & GT_DEPTH_MASK) != 
		 (has_sp->gt & GT_DEPTH_MASK));
	if (rc == 1) { 
		res_sp->gt &= ~GT_DEPTH_MASK;
		res_sp->gt |= has_sp->gt & GT_DEPTH_MASK;
		rc |= 2;
	}
	if (!(res_sp->gt & GT_SCHEME_MASK)) 
		res_sp->gt |= has_sp->gt & GT_SCHEME_MASK;
	rc |= (has_sp->gt & GT_SCHEME_MASK) &&
		((res_sp->gt & GT_SCHEME_MASK) != 
		 (has_sp->gt & GT_SCHEME_MASK));
	if (rc == 1) { 
		res_sp->gt &= ~GT_SCHEME_MASK;
		res_sp->gt |= has_sp->gt & GT_SCHEME_MASK;
		rc |= 2;
	}
	if (!(res_sp->gt & GT_SUBSCHEME_MASK))
		res_sp->gt |= has_sp->gt & GT_SUBSCHEME_MASK;
	rc |= (has_sp->gt & GT_SUBSCHEME_MASK) &&
		((res_sp->gt & GT_SUBSCHEME_MASK) != 
		 (has_sp->gt & GT_SUBSCHEME_MASK));
	if (rc == 1) { 
		res_sp->gt &= ~GT_SUBSCHEME_MASK;
		res_sp->gt |= has_sp->gt & GT_SUBSCHEME_MASK;
		rc |= 2;
	}
	if (!(res_sp->gt & GT_SIZE_MASK)) 
		res_sp->gt |= has_sp->gt & GT_SIZE_MASK;
	rc |= (has_sp->gt & GT_SIZE_MASK) &&
		((res_sp->gt & GT_SIZE_MASK) != 
		 (has_sp->gt & GT_SIZE_MASK));
	if (rc == 1) { 
		res_sp->gt &= ~GT_SIZE_MASK;
		res_sp->gt |= has_sp->gt & GT_SIZE_MASK;
		rc |= 2;
	}


	/* Compare quantity with haslist */
	if (res_p->qty == GGI_AUTO) res_p->qty = has_p->qty;
	rc |=  	(has_p->qty != GGI_AUTO) &&
		(res_p->qty != has_p->qty);
	if (rc == 1) { 
		res_p->qty = has_p->qty;
		rc |= 2;
	}

	/* Compare geometry with haslist */
	if (res_p->size.area.x == GGI_AUTO) 
		res_p->size.area.x = has_p->size.area.x;
	rc |=	(has_p->size.area.x != GGI_AUTO) && 
		(res_p->size.area.x != has_p->size.area.x);
	if (rc == 1) { 
		res_p->size.area.x = has_p->size.area.x;
		rc |= 2;
	}
	if (res_p->size.area.y == GGI_AUTO) 
		res_p->size.area.y = has_p->size.area.y;
	rc |=	(has_p->size.area.y != GGI_AUTO) &&
		(res_p->size.area.y != has_p->size.area.y);
	if (rc == 1) { 
		res_p->size.area.y = has_p->size.area.y;
		rc |= 2;
	}

	/* Compare storage type with haslist */
	storage_mask_has = ~(has_p->storage_ok ^ has_p->storage_need);
	if (res_p->storage_ok == GA_STORAGE_DONTCARE) {
		/* Nuance: In has, storage_ok:n = 0, storage_need:n = 1 never 
		 * happens (forbidden combination.)
		 */
		res_p->storage_ok    = has_p->storage_ok;
		res_p->storage_need &= ~storage_mask_has;
		res_p->storage_need |= storage_mask_has & has_p->storage_need;
		res_p->storage_ok   ^= res_p->storage_need & ~storage_mask_has;
	}
	storage_mask_res = ~(res_p->storage_ok ^ res_p->storage_need);
	storage_mask = storage_mask_res & storage_mask_has;
	if ((storage_mask & has_p->storage_ok) !=
	    (storage_mask & res_p->storage_ok)) {
		if (rc == 1) {
			res_p->storage_ok   = has_p->storage_ok;
			res_p->storage_need = has_p->storage_need;
		}
		rc |= 2;
	} else {
		storage_mask = ~storage_mask_res & storage_mask_has;
		res_p->storage_ok   &= ~storage_mask;
		res_p->storage_need &= ~storage_mask;
		res_p->storage_ok   |= storage_mask & has_p->storage_ok;
		res_p->storage_need |= storage_mask & has_p->storage_ok;
	}	/* if-else */


	/* Compare storage snap with haslist. */
	if (res_sp->buf_snap.area.x == GGI_AUTO) 
		res_sp->buf_snap.area.x = has_sp->buf_snap.area.x;
	rc |=	(has_sp->buf_snap.area.x != GGI_AUTO) && 
		(res_sp->buf_snap.area.x != has_sp->buf_snap.area.x);
	if (rc == 1) { 
		res_sp->buf_snap.area.x = has_sp->buf_snap.area.x;
		rc |= 2;
	}
	if (res_sp->buf_snap.area.y == GGI_AUTO) 
		res_sp->buf_snap.area.y = has_sp->buf_snap.area.y;
	rc |=	(has_sp->buf_snap.area.y != GGI_AUTO) &&
		(res_sp->buf_snap.area.y != has_sp->buf_snap.area.y);
	if (rc == 1) { 
		res_sp->buf_snap.area.y = has_sp->buf_snap.area.y;
		rc |= 2;
	}

	if (rc) rc = GALLOC_EFAILED;
	return rc;
 bothcaps:
	fprintf(stderr, "TODO: Caps unimplemented here.\n");
	rc = GALLOC_EFAILED;
	return rc;
 rescap:
	fprintf(stderr, "TODO: Caps unimplemented here.\n");
	rc = GALLOC_EFAILED;
	return rc;
 hascap:
	fprintf(stderr, "TODO: Caps unimplemented here.\n");
	rc = GALLOC_EFAILED;
	return rc;
}

int ggiGA_default_precheck (ggi_visual_t vis,
			 ggiGA_resource_list reslist,
			 enum ggiGA_callback_command command,
			 ggiGA_resource_handle res,
			 ggiGA_resource_handle head,
			 ggiGA_taghead_t taghead,
			 ggiGA_resource_handle has,
			 struct ggiGA_template_state *state,
			 void *privstate) {
	if (ggiGA_FULLTYPE(res) == GA_RT_RESLIST_TAGHEAD) {
	  fprintf(stderr, "Tag group heads not yet implemented\n");
	  return -1;
	}

	if (ggiGAIsMotor(res)) 
		return ggiGA_default_motor_precheck(vis, reslist, command, 
						 res, head, taghead, has,
						 state, privstate);

	if (ggiGAIsCarb(res)) 
		return ggiGA_default_carb_precheck(vis, reslist, command, 
						res, head, taghead, has,
						state, privstate);

	return ggiGA_default_tank_precheck(vis, reslist, command, 
					res, head, taghead, has,
					state, privstate);
}

/* The check callback checks whether storage (or other allocatable 
 * quantities) is actually available, and sends back suggestions.
 * It does not actually mark the checked quantity as allocated.
 * We know that any resource that makes it to the check callback
 * has suceeded the precheck callback.
 *
 * The postcheck callback actually marks regions of storage as 
 * used (for the duration of the call to ggiGACheck, at least.)
 *
 * The recheck callback re-marks regions as used after
 * they have been temporarily "unchecked" by the uncheck callback,
 * or if the whole list is being rechecked from the top to
 * roll back to a previous state.  In the former case, the resource
 * will be marked as failed.
 *
 * These three callbacks will often share the same actual callback 
 * function, as they do here.
 *
 * It is known that the resource has passed precheck, and it has
 * passed the do_fit function before these callbacks are called.
 * 
 * NOTE: In-service resources *will* be passed to these callbacks, 
 * so they should be anticipated (and they can be told apart by
 * the GA_STATE_NORESET state flag).
 */
int ggiGA_default_motor_check (ggi_visual_t vis,
			       ggiGA_resource_list reslist,
			       enum ggiGA_callback_command command,
			       ggiGA_resource_handle res,
			       ggiGA_resource_handle head,
			       ggiGA_taghead_t taghead,
			       ggiGA_resource_handle has,
			       struct ggiGA_template_state *state,
			       void *privstate) {

	/* For now, we should not be getting any rechecks. */
	LIB_ASSERT (command != GA_CB_RECHECK);

	return 0;
}

int ggiGA_default_carb_check(ggi_visual_t vis,
			     ggiGA_resource_list reslist,
			     enum ggiGA_callback_command command,
			     ggiGA_resource_handle res,
			     ggiGA_resource_handle head,
			     ggiGA_taghead_t taghead,
			     ggiGA_resource_handle has,
			     struct ggiGA_template_state *state,
			     void *privstate) {

	/* For now, we should not be getting any rechecks. */
	LIB_ASSERT (command != GA_CB_RECHECK);

	fprintf(stderr, "carb check called\n");

	return 0;
}

int ggiGA_default_tank_check (ggi_visual_t vis,
			      ggiGA_resource_list reslist,
			      enum ggiGA_callback_command command,
			      ggiGA_resource_handle res,
			      ggiGA_resource_handle head,
			      ggiGA_taghead_t taghead,
			      ggiGA_resource_handle has,
			      struct ggiGA_template_state *state,
			      void *privstate) {

	/* For now, we should not be getting any rechecks. */
	LIB_ASSERT (command != GA_CB_RECHECK);

	/* Here we will be checking with the storage handlers.
	 * If the resource in in service, we must choose the
	 * same exact storage location as it is using in the
	 * in-service storage state.
	 */

	if (command != GA_CB_POSTCHECK) return GALLOC_OK;

	/* Here we will be reserving with the storage handlers.
	 * (Not in the actual in-service storage, rather in our
	 * hypothetical storage)
	 */

	return 0;
}

int ggiGA_default_check (ggi_visual_t vis,
			 ggiGA_resource_list reslist,
			 enum ggiGA_callback_command command,
			 ggiGA_resource_handle res,
			 ggiGA_resource_handle head,
			 ggiGA_taghead_t taghead,
			 ggiGA_resource_handle has,
			 struct ggiGA_template_state *state,
			 void *privstate) {

	if (ggiGA_FULLTYPE(res) == GA_RT_RESLIST_TAGHEAD) {
	  fprintf(stderr, "Tag group heads not yet implemented\n");
	  return -1;
	}

	if (ggiGAIsMotor(res)) 
		return ggiGA_default_motor_check(vis, reslist, command, 
						 res, head, taghead, has,
						 state, privstate);

	if (ggiGAIsCarb(res)) 
		return ggiGA_default_carb_check(vis, reslist, command, 
						res, head, taghead, has,
						state, privstate);

	return ggiGA_default_tank_check(vis, reslist, command, 
					res, head, taghead, has,
					state, privstate);
}

/* The uncheck callback removes any pre-allocation of storage that
 * the given resource caused.  It should return 0 if the state of
 * the target is not disturbed by doing so, 1 if the state must be
 * restored by calling the recheck callback on all resources in the
 * list, and 2 if all state hould be wiped clean and the 
 * precheck/check/postcheck callbacks should be called on all 
 * resources again.
 */ 
int ggiGA_default_motor_uncheck (ggi_visual_t vis,
				 ggiGA_resource_list reslist,
				 enum ggiGA_callback_command command,
				 ggiGA_resource_handle res,
				 ggiGA_resource_handle head,
				 ggiGA_taghead_t taghead,
				 ggiGA_resource_handle has,
				 struct ggiGA_template_state *state,
				 void *privstate) {
	return 2; /* Force full rerun */
}

int ggiGA_default_carb_uncheck(ggi_visual_t vis,
			       ggiGA_resource_list reslist,
			       enum ggiGA_callback_command command,
			       ggiGA_resource_handle res,
			       ggiGA_resource_handle head,
			       ggiGA_taghead_t taghead,
			       ggiGA_resource_handle has,
			       struct ggiGA_template_state *state,
			       void *privstate) {


	return 2; /* Force full rerun */
}

int ggiGA_default_tank_uncheck (ggi_visual_t vis,
				ggiGA_resource_list reslist,
				enum ggiGA_callback_command command,
				ggiGA_resource_handle res,
				ggiGA_resource_handle head,
				ggiGA_taghead_t taghead,
				ggiGA_resource_handle has,
				struct ggiGA_template_state *state,
				void *privstate) {
	return 2; /* Force full rerun */
}

int ggiGA_default_uncheck (ggi_visual_t vis,
			   ggiGA_resource_list reslist,
			   enum ggiGA_callback_command command,
			   ggiGA_resource_handle res,
			   ggiGA_resource_handle head,
			   ggiGA_taghead_t taghead,
			   ggiGA_resource_handle has,
			   struct ggiGA_template_state *state,
			   void *privstate) {

	fprintf(stderr, "default uncheck %p\n", res);

	if (ggiGA_FULLTYPE(res) == GA_RT_RESLIST_TAGHEAD) {
	  fprintf(stderr, "Tag group heads not yet implemented\n");
	  return 2;
	}
	
	if (ggiGAIsMotor(res)) 
		return ggiGA_default_motor_uncheck(vis, reslist, command, 
						   res, head, taghead, has,
						   state, privstate);

	if (ggiGAIsCarb(res)) 
		return ggiGA_default_carb_uncheck(vis, reslist, command, 
						  res, head, taghead, has,
						  state, privstate);
	
	return ggiGA_default_tank_uncheck(vis, reslist, command, 
					  res, head, taghead, has,
					  state, privstate);
	return 0;
}


/* The set function cements the allocation of resources, and
 * may alter global hardware state (that is, state which affects
 * the basic hardware layout scheme of resources but is not specific
 * to a certain resource.)
 */
int ggiGA_default_motor_set (ggi_visual_t vis,
			     ggiGA_resource_list reslist,
			     enum ggiGA_callback_command command,
			     ggiGA_resource_handle res,
			     ggiGA_resource_handle head,
			     ggiGA_taghead_t taghead,
			     ggiGA_resource_handle has,
			     struct ggiGA_template_state *state,
			     void *privstate) {
	return 0;
}

int ggiGA_default_carb_set(ggi_visual_t vis,
			   ggiGA_resource_list reslist,
			   enum ggiGA_callback_command command,
			   ggiGA_resource_handle res,
			   ggiGA_resource_handle head,
			   ggiGA_taghead_t taghead,
			   ggiGA_resource_handle has,
			   struct ggiGA_template_state *state,
			   void *privstate) {
	return 0;
}

int ggiGA_default_tank_set (ggi_visual_t vis,
			    ggiGA_resource_list reslist,
			    enum ggiGA_callback_command command,
			    ggiGA_resource_handle res,
			    ggiGA_resource_handle head,
			    ggiGA_taghead_t taghead,
			    ggiGA_resource_handle has,
			    struct ggiGA_template_state *state,
			    void *privstate) {
	struct ggiGA_resource_props *res_p = NULL;
	struct ggiGA_tank_props *res_sp = NULL;

	/* See if the resource has already been set. */
	if (res->res_state & GA_STATE_NORESET) return GALLOC_OK;

	LIB_ASSERT(!ggiGAIsFailed(res));
	if (ggiGAIsFailed(res)) return GALLOC_EFAILED;
	res_p = res->props;
	if (res_p == NULL) return GALLOC_EFAILED;
	res_sp = &(res_p->sub.tank);

	/* Here we would use the storage managers to reserve storage in the
	 * in-service storage state.  Until then, we will handle RAM resources
	 * by simply allocating them.
	 */
	if (res_p->storage_ok != GA_STORAGE_SWAP) return GALLOC_EFAILED;
	res_sp->read = res_sp->write =
          calloc(res_p->size.area.x * res_p->size.area.y,
		 GT_SIZE(res_sp->gt) / 8);

        if (res_sp->read == NULL) return GALLOC_EUNAVAILABLE;

	return GALLOC_OK;
}


int ggiGA_default_set(ggi_visual_t vis,
		      ggiGA_resource_list reslist,
		      enum ggiGA_callback_command command,
		      ggiGA_resource_handle res,
		      ggiGA_resource_handle head,
		      ggiGA_taghead_t taghead,
		      ggiGA_resource_handle has,
		      struct ggiGA_template_state *state,
		      void *privstate) {

	if (ggiGA_FULLTYPE(res) == GA_RT_RESLIST_TAGHEAD) {
	  fprintf(stderr, "Tag group heads not yet implemented\n");
	  return 2;
	}
	
	if (ggiGAIsMotor(res)) 
		return ggiGA_default_motor_set(vis, reslist, command, 
					       res, head, taghead, has,
					       state, privstate);

	if (ggiGAIsCarb(res)) 
		return ggiGA_default_carb_set(vis, reslist, command, 
					      res, head, taghead, has,
					      state, privstate);
	
	return ggiGA_default_tank_set(vis, reslist, command, 
				      res, head, taghead, has,
				      state, privstate);

}

/* The unset callback removes any global hardware state.
 */
int ggiGA_default_unset(ggi_visual_t vis,
			ggiGA_resource_list reslist,
			enum ggiGA_callback_command command,
			ggiGA_resource_handle res,
			ggiGA_resource_handle head,
			ggiGA_taghead_t taghead,
			ggiGA_resource_handle has,
			struct ggiGA_template_state *state,
			void *privstate) {

	fprintf(stderr, "Default unset called\n");
	return 0;
}

/* The release callback deallocates storage.
 */
int ggiGA_default_release(ggi_visual_t vis,
			  ggiGA_resource_list reslist,
			  enum ggiGA_callback_command command,
			  ggiGA_resource_handle res,
			  ggiGA_resource_handle head,
			  ggiGA_taghead_t taghead,
			  ggiGA_resource_handle has,
			  struct ggiGA_template_state *state,
			  void *privstate) {
	fprintf(stderr, "Default unset called\n");
	return 0;
}


/* ---------------------------------------------------------------------- */
/* Old templates, slated for replacement by the above.                    */
/* ---------------------------------------------------------------------- */


int ggiGARelease_template(ggi_visual_t vis, ggiGA_resource_list *list,
			  ggiGA_resource_handle *handle, 
			  struct ggiGA_template_state *state,
			  void *privstate)
{
	ggiGA_resource_list reslist = NULL;
	ggiGA_resource_list reqlist = NULL;
	ggiGA_resource_handle res = NULL;
	ggiGA_resource_handle req = NULL;
	int rc = GALLOC_OK;

	if (list == NULL) return GGI_EARGINVAL;
	if (handle == NULL) return GGI_EARGINVAL;
	if (handle[0] == NULL) return GALLOC_ESPECIFY;

	reqlist = list[0];
	req = handle[0];

	rc = ggiGAGet(vis, &reslist);
	LIB_ASSERT(rc == GALLOC_OK);

	GALLOCDPRINT("reslist = %p\n", reslist);

	rc = ggiGAFind(reqlist, req);
	if (rc != GALLOC_OK) {
		/* not in this list! */
		rc = GALLOC_EUNAVAILABLE;
		goto cleanup;
	}	/* if */

	res = ggiGAHandle(reslist, reqlist, req);

	/* We have option of doing more thorough sanity checking here. */
	if (res == NULL) {
		rc = GALLOC_EFAILED;
		goto cleanup;
	}	/* if */

#if 0
	/* If shared... */
	if (ggiGAIsShared(rescurrent)) {
		/* ... then we unshare it without making a copy,
		 * so that we have a handle with no actual storage.
		 */
		rc = ggiGAUndoShare(reslist, rescurrent);
		LIB_ASSERT(rc == GALLOC_OK);
		if (rc != GALLOC_OK) {
			rc = GALLOC_EFAILED;
			goto install;
		}	/* if */
	}	/* if */
#endif

	/* Caps cannot be freed separately; user must free the one above. */
	if (ggiGAIsCap(res)) goto cleanup;

	/* If we release a compound head, then we release it together
	 * with all attached seeaboves (and CAPs).  We release from the
	 * bottom of the compound upwards.
	 */
	if (ggiGAHasAttachedSeeAboves(reslist, res)) {
		ggiGA_resource_handle resnext = NULL;
		ggiGA_resource_handle tmp = NULL;


		LIB_ASSERT(ggiGAIsCompoundHead(res));

		while (resnext != res) {
			resnext = ggiGAFindLastSeeabove(reslist, res);

			/* All GA_RT_FRAMES are kept, since in order to
			 * change the mode you need to set() anyway.
			 */
			if (ggiGA_TYPE(resnext) == GA_RT_FRAME) goto install;

			tmp = ggiGAHandle(GA_RESLIST(vis),
					  reslist, resnext);

			//			rc = release(vis, tmp);
			LIB_ASSERT(rc == GALLOC_OK);

			tmp = ggiGAHandle(reqlist, reslist, resnext);
			rc = ggiGARemoveWithCap(&reqlist, &tmp);
			LIB_ASSERT(rc == GALLOC_OK);

			tmp = resnext;
			rc = ggiGARemoveWithCap(&reslist, &tmp);
			LIB_ASSERT(rc == GALLOC_OK);
		}	/* while */

	} else {
		ggiGA_resource_handle tmp;

		/* All GA_RT_FRAMES are kept, since in order to
		 * change the mode you need to set() anyway.
		 */
		if (ggiGA_TYPE(res) == GA_RT_FRAME) goto cleanup;

		tmp = ggiGAHandle(GA_RESLIST(vis), reslist, res);

		// rc = release(vis, tmp);
		LIB_ASSERT(rc == GALLOC_OK);

		rc = ggiGARemoveWithCap(&reslist, &res);
		LIB_ASSERT(rc == GALLOC_OK);

		rc = ggiGARemoveWithCap(&reqlist, &req);
		LIB_ASSERT(rc == GALLOC_OK);
	}	/* if */


 install:
	LIB_ASSERT(rc == GALLOC_OK);

	rc = ggiGAEmptyList(&(GA_RESLIST(vis)));
	LIB_ASSERT(rc == GALLOC_OK);

	rc = ggiGACopyList(reslist, &(GA_RESLIST(vis)));
	LIB_ASSERT(rc == GALLOC_OK);
	return rc;

 cleanup:
	LIB_ASSERT(rc == GALLOC_OK);
	ggiGAEmptyList(&reslist);
	return rc;

}	/* ggiGARelease_template */

