/*
******************************************************************************

   LibGalloc: extension internals

   Copyright (C) 2001 Christoph Egger   [Christoph_Egger@t-online.de]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#ifndef _GGI_GALLOC_INT_OPS_H
#define _GGI_GALLOC_INT_OPS_H


#include <ggi/internal/internal.h>




/* "black-ops" section.  People who use these functions drive cars with
 * tinted windows and duct-tape lethal weapons to their thigh.
 */


/* Some simple macros to reduce code size/improve legibility.  nuff said. */
#define ggiGAFlagActive(res) \
	res->res_state |= GA_STATE_NORESET | GA_STATE_NOCHANGE;
#define ggiGAFlagFailed(res) res->res_state |= GA_STATE_FAILED
#define ggiGAFlagSuccess(res) res->res_state &= ~GA_STATE_FAILED

/* Copy properties of a resource structure. Properties of the destination
 * resource structure is overwritten then. So you are warned.
 */
int ggiGACopyProps(ggiGA_resource_handle from_handle,
		   ggiGA_resource_handle to_handle);

/* Copy private data of a resource structure. The private data of the
 * destination resource structure is overwritten. So you are warned.
 */
int ggiGACopyPriv(ggiGA_resource_handle from_handle,
		  ggiGA_resource_handle to_handle);

/* Create a copy of a resource structure and any associated
 * private data.
 */
int ggiGAClone(ggiGA_resource_handle in, ggiGA_resource_handle *out);

/* Copy of a resource structure and any associated
 * private data. If out[0] == NULL, then it fails. This behaviour
 * allows you to make sure to not cause any kind of damage.
 */
int ggiGACopy(ggiGA_resource_handle in, ggiGA_resource_handle out);

/* Makes a copy of a resource structure but does not copy the props/priv
 * like ggiGACopy does, instead, sets them to NULL pointers;
 */
int ggiGACloneNoProps(ggiGA_resource_handle in, ggiGA_resource_handle *out);

/* Prepend a copy of the resource at "handle" to the list "*list". 
 * On success, "list" has a different value assigned to it.
 */
int ggiGAPrepend(ggiGA_resource_handle handle, ggiGA_resource_list *list);

/* Append a copy of the resource at "handle" to the list "*list". 
 */
int ggiGAAppend(ggiGA_resource_handle handle, ggiGA_resource_list *list);


/* Removes (and frees) the resource pointed to by "handle" out of the 
 * resource-list pointed to by "*list". if present, "list" is passed by
 * reference, as this may change the pointer to the
 * list (e.g. to NULL to indicate that the list is now empty.)
 */
int ggiGARemove(ggiGA_resource_list *list, ggiGA_resource_handle *handle);


/* Removes (and frees) the resource pointed to by "handle" out of the 
 * resource-list pointed to by "*list", INCLUDING the cap, if present.
 * "list" is passed by reference, as this may change the pointer to the
 * list (e.g. to NULL to indicate that the list is now empty.)
 */
int ggiGARemoveWithCap(ggiGA_resource_list *list,
			ggiGA_resource_handle *handle);

/* Frees a handle, setting the handle to NULL when done, but takes
 * care not to free any mutual props.  The handle should of course 
 * NOT be present in the list at the time.
 */
int ggiGAFreeDetachedHandle(ggiGA_resource_list list, 
			    ggiGA_resource_handle *handle);


/* Create a copy of a linked list of resource structures and any associated
 * private data.
 */
int ggiGACopyList(ggiGA_resource_list in, ggiGA_resource_list *out);

/* Link the list "this" to the end of the list "tothis".  No copying is done.
 * "tothis" is passed by reference, as this may change the pointer to that
 * list (e.g. from NULL to a real value if the list was empty.)
 */
int ggiGAAppendList(ggiGA_resource_list this, ggiGA_resource_list *tothis);

/* Take any extra resources from the end of "list" and put them on the end
 * of "onto", then empty the previous resources in "list".  This is used
 * by extensions in their lazy-API Create functions to keep the global
 * request list in sync without changing existing handles.
 */
int ggiGASpliceList(ggiGA_resource_list *onto, ggiGA_resource_list *list);

/* Link "tothis" to the end of "this" (syntax sugar).  Note swapped types. */
#define ggiGAPrependList(this, tothis)      \
                ggiGAAppendList(tothis, this);

/* Finds the first resource in the list which points to the 
 * same properties as "handle".  This is used by ggiGACopyList and
 * ggiGAFreeDetachedHandle to detect mutual resources.  You know a resource
 * is a mutual resource if the "handle" is in the list and ggiGAPropsHandle 
 * does not return the same value given in "handle".  (But you do NOT 
 * know the converse.)  This is a very internal function you must know
 * what you are doing to use correctly.
 */
ggiGA_resource_handle ggiGAPropsHandle(ggiGA_resource_list list, 
                                       ggiGA_resource_handle handle);

/* Find the last resource in the compound resource to which "handle"
 * belongs.  The "list" parameter is not needed now, but may be if we
 * change the list structure.  "handle" should not be a CAP, and CAPs
 * are not returned, rather the resource before them is. 
 */
ggiGA_resource_handle ggiGAFindLastSeeabove(ggiGA_resource_list list,
					    ggiGA_resource_handle handle);


/* Finds and returns the handle of the first (successful) GA_RT_FRAME
 * resource. If the returned "handle" is NULL, then there is _no_ 
 * GA_RT_FRAME in the list, which should never happen.
 */
ggiGA_resource_handle ggiGAFindFirstMode(ggiGA_resource_list reslist);


/* Find the last successful GA_RT_FRAME resource before "handle" in
 * "reslist", or if "handle" is NULL the last in the entire list.  By "last
 * successful" we mean that if a failure is found before "handle", the search
 * returns NULL, not the last one that would have succeeded.
 */
ggiGA_resource_handle ggiGAFindLastMode(ggiGA_resource_list reslist,
					ggiGA_resource_handle handle);

ggiGA_resource_list ggiGAReverseList(ggiGA_resource_list list);








/* Black ops - section: Utils for compound resource manipulation
 */


int ggiGAAddSeeabove(ggiGA_resource_list *reqlist, 
                     struct ggiGA_resource_props *props, 
                     enum ggiGA_resource_type rt, 
                     ggiGA_resource_handle *handle);

int ggiGAIsCompoundHead(ggiGA_resource_handle handle);

int ggiGAIsCap(ggiGA_resource_handle handle);

ggiGA_resource_handle ggiGAFindCompoundHead(ggiGA_resource_list list,
					    ggiGA_resource_handle handle);

int ggiGAIsSeeabove(ggiGA_resource_handle handle);

int ggiGAHasAttachedSeeAboves(ggiGA_resource_list list,
                              ggiGA_resource_handle handle);

/* Black ops - section: Utils for tagged groups manipulation
 */

int ggiGACompareTag(ggiGA_resource_handle handle1,
		    ggiGA_resource_handle handle2);

ggiGA_resource_handle ggiGAFindNextTag(ggiGA_resource_list reslist,
				       ggiGA_resource_handle handle);

ggiGA_resource_handle ggiGAFindFirstTag(ggiGA_resource_list reslist,
					ggiGA_resource_handle handle);

ggiGA_resource_handle ggiGAFindLastTag(ggiGA_resource_list reslist,
				       ggiGA_resource_handle handle);


/* Find last resource before "handle" in "reslist" with the same type and
 * tag as "handle" has.  Or, if "handle" is not in "reslist", finds the
 * last one in the entire list instead.  This function has asserts that
 * flag suspicious calls from API functions, so if you need it for other
 * purposes, dup it.
 */
ggiGA_resource_handle ggiGAFindPrevTag(ggiGA_resource_list reslist,
				       ggiGA_resource_handle handle);

/* Find the compound resource to which the first resource with the
 * same type and tag belongs.
 */
ggiGA_resource_handle ggiGAFindCompoundByTag(ggiGA_resource_list reslist,
					     ggiGA_resource_handle handle);


/* Basic sanity check functions for resources and their caps. */
int ggiGAModeCapSanity(ggiGA_resource_handle mode);
int ggiGAMotorCapSanity(ggiGA_resource_handle res);
int ggiGATankCapSanity(ggiGA_resource_handle res);
int ggiGACarbCapSanity(ggiGA_resource_handle res);

/* Some handy macros for dealing with caps/GGI_AUTO */

/* Check if x is GGI_AUTO or x <op> y is true. */
#define GA_AUTOCMP(x,op,y) ((x != GGI_AUTO) && (x op y))


/* Check if either x or y are set to GGI_AUTO, or x <op> y is true. */ 
#define GA_AUTO2CMP(x,op,y) \
((x == GGI_AUTO) || (y == GGI_AUTO) || (x op y))


/* Compare members in (struct *) "res" to those in (struct *) "cap".
 * NOTE: Variable names "res" and "cap" must be used and must be of 
 * correct type in order to use this.
 */
#define GA_CAPCMP(member, op) \
if (!GA_AUTO2CMP(cap->member, op, res->member)) return GALLOC_EFAILED;

/* As above, but throw in an assertion. */
#define GA_CAPCMP_ASSERT(member, op) \
LIB_ASSERT(GA_AUTO2CMP(cap->member, op, res->member));                   \
if (!GA_AUTO2CMP(cap->member, op, res->member)) return GALLOC_EFAILED;


#define GA_LIMIT(r, x, op, y)	\
if (r->x op y) {		\
	r->x = y;               \
	rc = GALLOC_EFAILED;	\
}

#define GA_AUTOLIMIT(r, x, op, y)	\
if (r->x == GGI_AUTO) {			\
	r->x = y;			\
} else GA_LIMIT(r, x, op, y)






#define GA_CHECK_GGIAUTO_1D(handle, maxsize, maxdepth)			\
									\
if (handle->props->size.linear == GGI_AUTO) {				\
	handle->props->size.linear = maxsize;				\
}	/* if */							\
if (handle->props->sub.tank.gt == GT_AUTO) {				\
	handle->props->sub.tank.gt = maxdepth;				\
}	/* if */


#define GA_CHECK_GGIAUTO_2D(handle, maxwidth, maxheight, maxdepth)	\
									\
if (handle->props->size.area.x == GGI_AUTO) {				\
	handle->props->size.area.x = maxwidth;				\
}	/* if */							\
if (handle->props->size.area.y == GGI_AUTO) {				\
	handle->props->size.area.y = maxheight;				\
}	/* if */							\
if (handle->props->sub.tank.gt == GT_AUTO) {				\
	handle->props->sub.tank.gt = maxdepth;				\
}	/* if */



#define GA_CHECK_CAPAUTO_1D(cap, maxsize, maxdepth)			\
									\
{									\
int cap_size = cap->props->size.linear;					\
int cap_depth = cap->props->sub.tank.gt;				\
									\
if ((cap_size == GGI_AUTO) || (cap_size > maxsize)) {			\
	cap_size = maxsize;						\
}	/* if */							\
if ((cap_depth == GGI_AUTO) || (cap_depth >  maxdepth)) {		\
	cap_depth = maxdepth;						\
}	/* if */							\
									\
maxsize = cap_size;							\
maxdepth = cap_depth;							\
}



#define GA_CHECK_CAPAUTO_2D(cap, maxwidth, maxheight, maxdepth)		\
									\
{									\
int cap_width = cap->props->size.area.x;				\
int cap_height = cap->props->size.area.y;				\
int cap_depth = cap->props->sub.tank.gt;				\
									\
if ((cap_width == GGI_AUTO) || (cap_width > maxwidth)) {		\
	cap_width = maxwidth;						\
}	/* if */							\
if ((cap_height == GGI_AUTO) || (cap_height > maxheight)) {		\
	cap_height = maxheight;						\
}	/* if */							\
if ((cap_depth == GGI_AUTO) || (cap_depth > maxdepth)) {		\
	cap_depth = maxdepth;						\
}	/* if */							\
									\
maxwidth = cap_width;							\
maxheight = cap_height;							\
maxdepth = cap_depth;							\
}



/* The "for internal use only" stubs.
 */
int ggiGA_Mode(ggi_visual_t vis, ggiGA_resource_handle *out);

#if 0
int ggiGACheckIfShareable(ggi_visual_t vis, 
		     ggiGA_resource_handle reshandle,
		     ggiGA_resource_handle tocompare);

/* Compares handle1 and handle2, if their props and priv-fields are
 * identical, even their own pointer is different...
 * Returns GALLOC_OK, if true.
 */
int ggiGACompare(ggiGA_resource_handle handle1, ggiGA_resource_handle handle2);


/* Compares the two lists. Note, that list1 may have more list-members
 * than list2. Returns GALLOC_OK, if successfull.
 */
int ggiGACompareList(ggiGA_resource_list list1, ggiGA_resource_list list2);

#endif

#endif	/* _GGI_GALLOC_INT_OPS_H */
