/*
******************************************************************************

   LibGAlloc implementation for KGI Radeon target -- 
   				Master resource list and associated callbacks
  
   Copyright (c) 2002 Brian S. Julin		[bri@tull.umassp.edu]
   
   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   The above copyright notice applies to all files in this package, unless 
   explicitly stated otherwise in the file itself or in a file named COPYING 
   in the same directory as the file.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include "kgi_Radeon_galloc.h"
#include <string.h>


/******************************** callbacks *********************************/

int kgi_Radeon_galloc_check_mode (ggi_visual_t vis,
				  ggiGA_resource_list reslist,
				  enum ggiGA_callback_command command,
				  ggiGA_resource_handle res,
				  ggiGA_resource_handle head,
				  ggiGA_taghead_t taghead,
				  ggiGA_resource_handle has,
				  struct ggiGA_template_state *state,
				  void *privstate) {
	ggi_mode mode;
	kgi_size_t	fb_vram;
	int rc;
	struct kgi_radeon_galloc_container_t *pstate;

	GALLOCDPRINT("kgi_radeon_galloc_check_mode reslist callback called\n");
	
	pstate = privstate;

	/* TODO: even if active process storage on POSTCHECK */
	if (res->res_state & GA_STATE_NORESET) return GALLOC_OK;

	if (command == GA_CB_PRECHECK) {
	  /* TODO: basic comparison of res vs. has including caps. */
	  return GALLOC_OK;
	}

	if (command == GA_CB_POSTCHECK ||
	    command == GA_CB_UNCHECK ||
	    command == GA_CB_RECHECK) goto dostorage;

	/* If we have a cap, try it first */
	if (!ggiGAIsCap(res->next)) goto nocap;

	memcpy(&mode, ggiGAGetGGIMode(res->next), sizeof(ggi_mode));
	rc = ggiCheckMode(vis, &mode);

        /* Now we try again, using the adjusted values that came 
	 * back from ggiCheckMode. 
	 */
        if (rc != GALLOC_OK) rc = ggiCheckMode(vis, &mode);

	if (rc == GALLOC_OK) {
		ggi_mode *minmode;
		/* The cap succeeded.  But did it get a mode above the 
		 * minimum requirements in res?
		 */
		minmode = ggiGAGetGGIMode(res);
		rc |= (minmode->visible.x > mode.visible.x);
		rc |= (minmode->visible.y > mode.visible.y);
		rc |= (minmode->virt.x > mode.virt.x);
		rc |= (minmode->virt.y > mode.virt.y);
		rc |= (GT_DEPTH(minmode->graphtype) > 
		       GT_DEPTH(mode.graphtype));
		rc |= (GT_SCHEME(minmode->graphtype) != 
		       GT_SCHEME(mode.graphtype));
		if (rc != GALLOC_OK) rc = GALLOC_EFAILED;
                return rc;
        }       /* if */

	if (rc == GALLOC_OK) goto done;

 nocap:
	/* cap failed, so let's try the original resource. */
	memcpy(&mode, ggiGAGetGGIMode(res), sizeof(ggi_mode));
        rc = ggiCheckMode(vis, &mode);

	if (rc == GALLOC_OK) {
		res->res_state |= GA_STATE_MODIFIED;
		memcpy(ggiGAGetGGIMode(res), &mode, sizeof(ggi_mode));
		if (!ggiGAIsCap(res->next)) return rc;
		res->next->res_state |= GA_STATE_MODIFIED;
		memcpy(ggiGAGetGGIMode(res->next), &mode, sizeof(ggi_mode));
		return rc;
	}
	
        /* Now we try again, using the adjusted values that came 
	 * back from ggiCheckMode. 
	 */
	rc = ggiCheckMode(vis, &mode);
	if (rc != GALLOC_OK) {
		mode.visible.x = GGI_AUTO;
		mode.visible.y = GGI_AUTO;
		mode.virt.x = GGI_AUTO;
		mode.virt.y = GGI_AUTO;
		mode.graphtype = GT_AUTO;
		rc = ggiCheckMode(vis, &mode);
	}

	if (rc == GALLOC_OK) {
		ggi_mode *minmode, *maxmode;

		/* We got something to succeed -- question is, what? */
		minmode = ggiGAGetGGIMode(res);
		rc |= (minmode->visible.x > mode.visible.x);
		rc |= (minmode->visible.y > mode.visible.y);
		rc |= (minmode->virt.x > mode.virt.x);
		rc |= (minmode->virt.y > mode.virt.y);
		rc |= (GT_DEPTH(minmode->graphtype) > 
		       GT_DEPTH(mode.graphtype));
		rc |= (GT_SCHEME(minmode->graphtype) != 
		       GT_SCHEME(mode.graphtype));

		if (!ggiGAIsCap(res->next)) goto done;
		maxmode = ggiGAGetGGIMode(res->next);
		rc |= (maxmode->visible.x < mode.visible.x);
		rc |= (maxmode->visible.y < mode.visible.y);
		rc |= (maxmode->virt.x < mode.virt.x);
		rc |= (maxmode->virt.y < mode.virt.y);
		rc |= (GT_DEPTH(maxmode->graphtype) <
		       GT_DEPTH(mode.graphtype));
	}

 done:	  
	res->res_state |= GA_STATE_MODIFIED;
	memcpy(ggiGAGetGGIMode(res), &mode, sizeof(ggi_mode));
	if (!ggiGAIsCap(res->next)) return rc;
	res->next->res_state |= GA_STATE_MODIFIED;
	memcpy(ggiGAGetGGIMode(res->next), &mode, sizeof(ggi_mode));
	return rc;

 dostorage:
	
	memcpy(&mode, ggiGAGetGGIMode(res), sizeof(ggi_mode));
#warning assure no hostile app multiply overflow possible here.
	fb_vram = mode.virt.x * mode.virt.y * 
	  (GT_SIZE(mode.graphtype) / 8) * mode.frames;
	if (command == GA_CB_POSTCHECK) goto postcheck;
	if (command == GA_CB_UNCHECK)	goto uncheck;
	if (command == GA_CB_RECHECK)	goto recheck;

 postcheck:
	GALLOCDPRINT("Marking %i bytes of VRAM for framebuffer.\n", fb_vram);
	if (pstate->vram_available < pstate->vram_used + fb_vram) return 1;
	pstate->vram_used += fb_vram;
	return GALLOC_OK;
	
 uncheck:
	GALLOCDPRINT("Unmarking %i bytes of VRAM.\n", fb_vram);
	pstate->vram_used -= fb_vram;
	return 2;

 recheck:
	GALLOCDPRINT("Re-marking %i bytes of VRAM.\n", fb_vram);
	pstate->vram_used += fb_vram;
	return GALLOC_OK;

}


int kgi_Radeon_galloc_set_mode (ggi_visual_t vis,
				ggiGA_resource_list reslist,
				enum ggiGA_callback_command command,
				ggiGA_resource_handle res,
				ggiGA_resource_handle head,
				ggiGA_taghead_t taghead,
				ggiGA_resource_handle has,
				struct ggiGA_template_state *state,
				void *privstate) {
	ggi_mode mode;
	kgi_size_t fb_vram;
	struct kgi_radeon_galloc_container_t *pstate;

	fprintf(stderr, "kgi_Radeon_galloc_set_mode called\n");
	fprintf(stderr, "res_state = %x\n", res->res_state);

	pstate = privstate;

	if (command == GA_CB_UNSET ||
	    command == GA_CB_RELEASE) {
	  fprintf(stderr, "Implement UNSET/RELEASE!!\n");
	  return -1;
	}

	memcpy(&mode, ggiGAGetGGIMode(res), sizeof(ggi_mode));
	/* We assume prior checks prevent multiply overflow here. */
	fb_vram = mode.virt.x * mode.virt.y * 
	  (GT_SIZE(mode.graphtype) / 8) * mode.frames;
	pstate->vram_used += fb_vram;

	if (!(res->res_state & GA_STATE_NORESET))
		return ggiSetMode(vis, ggiGAGetGGIMode(res));
	return GALLOC_OK;	/* Already set, nothing to do. */
}	/* kgi_Radeon_galloc_set_mode */


/* Check for buffers that clad the visual -- they match the 
 * main mode, which is always their compound head right now, until
 * we start supporting floaters.
 */
int kgi_Radeon_galloc_check_cladbuf (ggi_visual_t vis,
				ggiGA_resource_list reslist,
				enum ggiGA_callback_command command,
				ggiGA_resource_handle res,
				ggiGA_resource_handle head,
				ggiGA_taghead_t taghead,
				ggiGA_resource_handle has,
				struct ggiGA_template_state *state,
				void *privstate) {

        struct ggiGA_resource_props *res_p;
	ggi_mode *mode;

	/* We probably don't even need these asserts, remove later. */
	LIB_ASSERT(ggiGA_TYPE(head) == GA_RT_FRAME);
	LIB_ASSERT(head->priv != NULL);

	/* For now, we should not be getting any rechecks. */
        LIB_ASSERT (command != GA_CB_RECHECK);

	/* Storage type checks were handled by precheck */

	if (res == head) {
		fprintf(stderr, "No support for floaters yet.\n");
		goto fail;
	}

        res_p = res->props;

        mode = ggiGAGetGGIMode(head);

	if ((res_p->size.area.x != GGI_AUTO) &&
	    (res_p->size.area.x != mode->virt.x)) goto fail;
	if ((res_p->size.area.y != GGI_AUTO) &&
	    (res_p->size.area.y != mode->virt.y)) goto fail;

	res_p->size.area.x = mode->virt.x;
	res_p->size.area.y = mode->virt.y;

	if (command != GA_CB_POSTCHECK) goto nomark;

	/* Here we would use the storage manager */

 nomark:
	return GALLOC_OK;
 fail:
	return GALLOC_EFAILED;
}

/* Check hardware Z buffer resource.
 */
int kgi_Radeon_galloc_check_hwz (ggi_visual_t vis,
				 ggiGA_resource_list reslist,
				 enum ggiGA_callback_command command,
				 ggiGA_resource_handle res,
				 ggiGA_resource_handle head,
				 ggiGA_taghead_t taghead,
				 ggiGA_resource_handle has,
				 struct ggiGA_template_state *state,
				 void *privstate) {

	kgi_size_t	zbuf_vram;
        struct ggiGA_resource_props *res_p;
	ggi_mode *mode;
	struct kgi_radeon_galloc_container_t *pstate;

	GALLOCDPRINT("kgi_radeon_galloc_check_hwz reslist callback called\n");
	
	pstate = privstate;

	/* We probably don't even need these asserts, remove later. */
	LIB_ASSERT(ggiGA_TYPE(head) == GA_RT_FRAME);
	LIB_ASSERT(head->priv != NULL);

	/* For now, we should not be getting any rechecks. */
        LIB_ASSERT (command != GA_CB_RECHECK);

	/* Storage type checks were handled by precheck */

	if (res == head) {
		fprintf(stderr, "No support for floaters yet.\n");
		goto fail;
	}

        res_p = res->props;

        mode = ggiGAGetGGIMode(head);

	if ((res_p->size.area.x != GGI_AUTO) &&
	    (res_p->size.area.x != mode->virt.x)) goto fail;
	if ((res_p->size.area.y != GGI_AUTO) &&
	    (res_p->size.area.y != mode->virt.y)) goto fail;

	res_p->size.area.x = mode->virt.x;
	res_p->size.area.y = mode->virt.y;

	/* 16-bit Z support only for this whack at it. */
	if (GT_SIZE(res_p->sub.tank.gt) != 16) goto fail;

	if (command != GA_CB_POSTCHECK) goto nomark;

#warning assure no hostile app multiply overflow possible here.
	zbuf_vram = 
	  mode->virt.x * mode->virt.y * (GT_SIZE(res_p->sub.tank.gt)/8);
	/* Add enough padding to ensure we can align it. */
	zbuf_vram += 16;
	GALLOCDPRINT("Marking %i bytes of VRAM for Z buffer.\n", zbuf_vram);
	if (pstate->vram_available < pstate->vram_used + zbuf_vram) return 1;
	pstate->vram_used += zbuf_vram;

 nomark:
	return GALLOC_OK;
 fail:
	return GALLOC_EFAILED;
}

/* We need to calculate the address of the Z buffer. */
int kgi_Radeon_galloc_set_hwz(ggi_visual_t vis,
			      ggiGA_resource_list reslist,
			      enum ggiGA_callback_command command,
			      ggiGA_resource_handle res,
			      ggiGA_resource_handle head,
			      ggiGA_taghead_t taghead,
			      ggiGA_resource_handle has,
			      struct ggiGA_template_state *state,
			      void *privstate) {
	ggi_mode *mode;
	kgi_size_t zbuf_align, zbuf_vram;
	ggi_kgi_priv *kpriv;
	struct kgi_radeon_galloc_container_t *pstate;

	kpriv = LIBGGI_PRIVATE(vis);
	pstate = privstate;

	/* Here we assume that kpriv->fb is well aligned.  
	   Safe assumption though */
	zbuf_align = 16 - pstate->vram_used % 16;
	res->props->sub.tank.read = res->props->sub.tank.write = 
	  kpriv->fb + pstate->vram_used + zbuf_align;

	/* TODO: make sure this is the right mode structure to use. */
        mode = ggiGAGetGGIMode(head);

	/* Here we assume that prior checks prevent multiply overflow. */
	zbuf_vram = 
	  mode->virt.x * mode->virt.y * (GT_SIZE(res->props->sub.tank.gt)/8);

	pstate->vram_used += zbuf_vram + zbuf_align;

	return GALLOC_OK;
}

/********************************* haslist **********************************/

ggiGA_resource_list kgi_Radeon_create_master_haslist(ggi_visual_t vis) {
	ggiGA_resource_list ret, haslist;
	struct ggiGA_resource *res;
	struct ggiGA_resource_props *propslist, *props;
	struct ggiGA_mode *gamodelist, *gamode;
	struct ggiGA_tank_props *tank;
	struct ggiGA_carb_props *carb;
	struct ggiGA_motor_props *motor;
	int i;
	
	ret = haslist = calloc(9, sizeof(struct ggiGA_resource));
	propslist = calloc(7, sizeof(struct ggiGA_resource_props));
	gamodelist = calloc(2, sizeof(struct ggiGA_resource_props));
	if (!(haslist && propslist && gamodelist)) goto err;
	for (i = 0; i < 8; i++) haslist[i].next = haslist + i + 1;

	/* Tag group usage:

	   Group 1 and 2: used on the motor/carb level to separate
	   software-rendered from hardware-rendered resources, respectively.
	   LibBuf does not support mixing the two yet.

	   Group 3: used to designate that software-rendered Z and alpha 
	   tanks may share storage.

	   Group 4: used to designate that hardware-rendered Z and alpha 
	   tanks may share storage.

	*/
	
	/* Add drawops resource for visuals without hardware Z/alpha. */
	res = haslist; haslist++; 
	props = res->props = propslist; propslist++;
	motor = &(props->sub.motor);
	res->res_type = GA_RT_RENDERER_DRAWOPS | GA_RT_MOTOR;
	res->res_state = 1;	/* Tag group */
	props->qty = 1;
	motor->mul_min.x = motor->mul_max.x = 1;
	motor->mul_min.y = motor->mul_max.y = 1;
	motor->div_min.x = motor->div_max.x = 1;
	motor->div_min.y = motor->mul_max.y = 1;

	/* Add drawops resource for visuals with hardware Z/alpha. */
	res = haslist; haslist++; 
	props = res->props = propslist; propslist++;
	motor = &(props->sub.motor);
	res->res_type = GA_RT_RENDERER_DRAWOPS | GA_RT_MOTOR;
	res->res_state = 2;	/* Tag group */
	props->qty = 1;
	motor->mul_min.x = motor->mul_max.x = 1;
	motor->mul_min.y = motor->mul_max.y = 1;
	motor->div_min.x = motor->div_max.x = 1;
	motor->div_min.y = motor->mul_max.y = 1;

	/* Add a frame resource for visuals without hardware Z/Alpha */
	res = haslist; haslist++; 
	gamode = res->priv = gamodelist; gamodelist++;
	res->priv_size = sizeof(struct ggiGA_mode);
	res->res_type = GA_RT_FRAME;
        res->cbflags1 =
	  GA_CB_PRECHECK | GA_CB_CHECK | GA_CB_UNCHECK | GA_CB_RECHECK;
	res->cb1 = &kgi_Radeon_galloc_check_mode;
	res->cbflags2 = GA_CB_SET | GA_CB_UNSET | GA_CB_RELEASE;
	res->cb2 = &kgi_Radeon_galloc_set_mode;
	/* gamode is set to all GGI_AUTOS by virtue of calloc. */


	/* To the frame we attach a resource representing the 
	 * software-emulated alpha and Z side-buffer implemented by LibBuf's
	 * default renderers.
	 */
	res = haslist; haslist++; 
	props = res->props = propslist; propslist++;
	tank = &(props->sub.tank);
	res->res_type = GA_RT_BUFFER_ABUFFER;
	res->res_state = GA_STATE_SEEABOVE | 3;
	res->cbflags1 = GA_CB_CHECK | GA_CB_RECHECK | GA_CB_POSTCHECK;
	res->cb1 = &kgi_Radeon_galloc_check_cladbuf;
	props->storage_ok = GA_STORAGE_SWAP;
	props->storage_need = GA_STORAGE_REALRAM;
	/* Since aligning the Z buffer may help SWAR implementations, 
	 * we tell the user we expectthe Z origin to be aligned to a
	 * 128 byte boundary.
	 */
	tank->buf_snap.linear = 128;
	/* Currently LibBuf only supports 8-bit grayscale (that is, 
	 * one value affecting all channels) alpha buffers. 
	 */
	tank->gt = GT_CONSTRUCT(8, GT_GREYSCALE, 8);

	/* Now the software emulated Z buffer */
	res = haslist; haslist++; 
	props = res->props = propslist; propslist++;
	tank = &(props->sub.tank);
	res->res_type = GA_RT_BUFFER_ZBUFFER;
	res->res_state = GA_STATE_SEEABOVE | 3;
	res->cbflags1 = GA_CB_CHECK | GA_CB_RECHECK | GA_CB_POSTCHECK;
	res->cb1 = &kgi_Radeon_galloc_check_cladbuf;
	props->storage_ok = GA_STORAGE_SWAP;
	props->storage_need = GA_STORAGE_REALRAM;
	/* Since aligning the Z buffer may help SWAR implementations, 
	 * we tell the user we expectthe Z origin to be aligned to a
	 * 128 byte boundary.
	 */
	tank->buf_snap.linear = 128;
	/* Currently LibBuf only supports 8-bit grayscale (that is, 
	 * one value affecting all channels) alpha buffers. 
	 */
	tank->gt = GT_CONSTRUCT(8, GT_GREYSCALE, 8);

	/* We attach a carb representing the link between the available
	 * Z and alpha buffers to the LibBuf default drawops resource. 
	 */
	res = haslist; haslist++; 
	props = res->props = propslist; propslist++;
	carb = &(props->sub.carb);
	res->res_type =	GA_RT_RENDERER_DRAWOPS | GA_RT_CARB;
	res->res_state = GA_STATE_SEEABOVE | 1;
	carb->can.rop = LL_ROP_ALPHA_AUX | LL_ROP_Z;
	carb->can.math = 0xffffffff;  /* All math functions supported. */
	carb->must.rop = LL_ROP_AUX;  /* side-buffers only */

	/* Add a frame resource for visuals with hardware Z/Alpha */
	res = haslist; haslist++; 
	gamode = res->priv = gamodelist; gamodelist++;
	res->priv_size = sizeof(struct ggiGA_mode);
	res->res_type = GA_RT_FRAME;
        res->cbflags1 =
	  GA_CB_PRECHECK | GA_CB_CHECK | GA_CB_UNCHECK | GA_CB_RECHECK;
	res->cb1 = &kgi_Radeon_galloc_check_mode;
	res->cbflags2 = GA_CB_SET | GA_CB_UNSET | GA_CB_RELEASE;
	res->cb2 = &kgi_Radeon_galloc_set_mode;
	/* gamode is set to all GGI_AUTOS by virtue of calloc. */


	/* To the frame we attach a resource representing the 
	 * hardware alpha and Z side-buffer implemented by LibBuf's
	 * Radeon-specific renderers.  (Currently only Z.)
	 */
	res = haslist; haslist++; 
	props = res->props = propslist; propslist++;
	tank = &(props->sub.tank);
	res->res_type = GA_RT_BUFFER_ZBUFFER;
	res->res_state = GA_STATE_SEEABOVE | 4;
	res->cbflags1 = GA_CB_CHECK | GA_CB_RECHECK | GA_CB_POSTCHECK;
	res->cb1 = &kgi_Radeon_galloc_check_hwz;
	res->cbflags2 = GA_CB_SET;
	res->cb2 = &kgi_Radeon_galloc_set_hwz;
	props->storage_ok = GA_STORAGE_VRAM;
	props->storage_need = GA_STORAGE_VRAM;
	/* Radeon Z origin must be 16-byte aligned. */
	tank->buf_snap.linear = 16;
	tank->gt = GT_CONSTRUCT(16, GT_GREYSCALE, 16);

	/* We attach a carb representing the link between the available
	 * Z and alpha buffers to the LibBuf Radeon drawops resource. 
	 */
	res = haslist; haslist++; 
	props = res->props = propslist; propslist++;
	carb = &(props->sub.carb);
	res->res_type =	GA_RT_RENDERER_DRAWOPS | GA_RT_CARB;
	res->res_state = GA_STATE_SEEABOVE | 2;
	carb->can.rop = LL_ROP_Z;
	carb->can.math = LL_MATH_CMP_SIGNED |
	  LL_MATH_CMP_GT | LL_MATH_CMP_LT | LL_MATH_CMP_GTE | LL_MATH_CMP_LTE;

	return ret;

 err:
	if (haslist)	free(haslist);
	if (propslist)	free(propslist);
	if (gamodelist)	free(gamodelist);
	return NULL;
}

/* Since the above was allocated chunks we need to free the chunks. */
void kgi_Radeon_destroy_master_haslist(ggi_visual_t vis) {
	free(GA_PRIV(vis)->haslist[2].priv);
	free(GA_PRIV(vis)->haslist[0].props);
	free(GA_PRIV(vis)->haslist);
	GA_PRIV(vis)->haslist = NULL;
}
