/*
******************************************************************************

   LibGalloc: utility functions for list manipulation

   Copyright (C) 2001 Brian S. Julin	[bri@tull.umassp.edu]
   Copyright (C) 2001 Christoph Egger   [Christoph_Egger@t-online.de]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <ggi/internal/galloc.h>
#include <ggi/internal/galloc_debug.h>

#include <string.h>

/* Docs for functions listed here are in the ggi/internal/galloc.h or
 * ggi/internal/galloc_ops.h headers, or in the API docs.
 */


/* Functions on single resources.  (move to another file?) */

int ggiGACopyProps(ggiGA_resource_handle from_handle,
		   ggiGA_resource_handle to_handle)
{
	LIB_ASSERT(from_handle != NULL);
	LIB_ASSERT(to_handle != NULL);
	LIB_ASSERT(from_handle->props != NULL);

	if (from_handle->props == NULL) return GGI_EARGINVAL;

	if (to_handle->props == NULL) {
		to_handle->props = malloc(sizeof(struct ggiGA_resource_props));
		if (!to_handle->props) return GGI_ENOMEM;
	}	/* if */

	memcpy(to_handle->props, from_handle->props,
		sizeof(struct ggiGA_resource_props));
	return GALLOC_OK;
}	/* ggiGACopyProps */


int ggiGACopyPriv(ggiGA_resource_handle from_handle,
		  ggiGA_resource_handle to_handle)
{
	LIB_ASSERT(from_handle != NULL);
	LIB_ASSERT(to_handle != NULL);
	LIB_ASSERT(from_handle->priv != NULL);

	if (from_handle->priv == NULL) return GGI_EARGINVAL;

	if (to_handle->priv == NULL) {
		to_handle->priv = malloc(from_handle->priv_size);
		if (!to_handle->priv) return GGI_ENOMEM;	
		to_handle->priv_size = from_handle->priv_size;
	}	/* if */
	if (to_handle->priv_size != from_handle->priv_size) {
		to_handle->priv = realloc(from_handle->priv,
					  from_handle->priv_size);
		if (!to_handle->priv) return GGI_ENOMEM;	
		to_handle->priv_size = from_handle->priv_size;
	}	/* if */

	LIB_ASSERT(from_handle->priv_size == to_handle->priv_size);
	memcpy(to_handle->priv, from_handle->priv, from_handle->priv_size);

	return GALLOC_OK;
}	/* ggiGACopyPriv */


int ggiGAClone(ggiGA_resource_handle in, ggiGA_resource_handle *out)
{
	int rc = GALLOC_OK;
	ggiGA_resource_handle tmp = NULL;
	if (out == NULL) return GGI_EARGINVAL;

	if (in == out[0]) {
		/* don't copy yourself */
		return GALLOC_OK;
	}	/* if */

	out[0] = tmp = calloc(1, sizeof(struct ggiGA_resource));
	if (!tmp) {
		rc = GGI_ENOMEM;
		goto err0;
	}	/* if */

	memcpy(tmp, in, sizeof(struct ggiGA_resource));
	tmp->next = NULL;	
	tmp->priv = NULL;
	tmp->props = NULL;

	if (in->priv != NULL) {
		rc = ggiGACopyPriv(in, tmp);
		LIB_ASSERT(rc == GALLOC_OK);
		if (rc != GALLOC_OK) goto err1;
	}	/* if */

	if (in->props != NULL) {
		rc = ggiGACopyProps(in, tmp);
		LIB_ASSERT(rc == GALLOC_OK);
		if (rc != GALLOC_OK) goto err2;
	}	/* if */


	LIB_ASSERT(rc == GALLOC_OK);
	return rc;

err2:
	if (tmp->priv)  free(tmp->priv);
err1:
	free(tmp);
	out[0] = NULL;
err0:
	return rc;
}	/* ggiGAClone */


int ggiGACloneNoProps(ggiGA_resource_handle in, ggiGA_resource_handle *out)
{
	ggiGA_resource_handle tmp = NULL;

	if (out == NULL) return GGI_EARGINVAL;

	if (in == out[0]) {
		/* Copying a resource into itself is a noop. */
		return GALLOC_OK;
	}	/* if */

	out[0] = tmp = calloc(1, sizeof(struct ggiGA_resource));
	if (!tmp) return GGI_ENOMEM;

	memcpy(tmp, in, sizeof(struct ggiGA_resource));

	tmp->next  = NULL;
	tmp->props = NULL;
	tmp->priv  = NULL;

	return GALLOC_OK;
}	/* ggiGACloneNoProps */


int ggiGACopy(ggiGA_resource_handle in, ggiGA_resource_handle out)
{
	int rc = GALLOC_OK;
	ggiGA_resource_handle next = NULL;

	if (out == NULL) return GALLOC_EFAILED;

	if (in == out) {
		/* Copying a resource back into itself is a noop. */
		return GALLOC_OK;
	}	/* if */

	/* Save pointer to next handle */
	next = out->next;
	memcpy(out, in, sizeof(struct ggiGA_resource));

	/* Restore pointer to next handle */
	out->next = next;

	if (in->priv != NULL) {
		rc = ggiGACopyPriv(in, out);
		LIB_ASSERT(rc == GALLOC_OK);
		if (rc != GALLOC_OK) goto err0;
	}	/* if */

	if (in->props != NULL) {
		rc = ggiGACopyProps(in, out);
		LIB_ASSERT(rc == GALLOC_OK);
		if (rc != GALLOC_OK) goto err1;
	}	/* if */

	return GALLOC_OK;

err1:
	if (out->priv)  free(out->priv);
err0:
	return GGI_ENOMEM;
}	/* ggiGACopy */


/* Functions on resource + cap */

/* Check that the values in a GA_RT_FRAME resource are not ridiculous,
   including comparing to those in its resource cap if present. */
int ggiGAModeCapSanity(ggiGA_resource_handle handle)
{
	LIB_ASSERT(handle != NULL);

	LIB_ASSERT (handle->priv != NULL);
	if (handle->priv == NULL) {
		return GALLOC_ESPECIFY;
	}	/* if */

	/* Do we have a cap? */
	if (!ggiGAIsCap(handle->next)) {
		/* Nope. */
		return GALLOC_OK;
	}	/* if */

	LIB_ASSERT (handle->next->priv != NULL);
	if (handle->next->priv == NULL) {
		return GALLOC_ESPECIFY;
	}	/* if */

	{
		struct ggiGA_mode *res, *cap;
		res = (struct ggiGA_mode *)(handle->priv);
		cap = (struct ggiGA_mode *)(handle->next->priv);
		GA_CAPCMP_ASSERT(mode.visible.x, >=);
		GA_CAPCMP_ASSERT(mode.visible.y, >=);
		GA_CAPCMP_ASSERT(mode.virt.x, >=);
		GA_CAPCMP_ASSERT(mode.virt.y, >=);
		GA_CAPCMP_ASSERT(mode.size.x, >=);
		GA_CAPCMP_ASSERT(mode.size.y, >=);
		GA_CAPCMP_ASSERT(mode.frames, >=);
		/* TODO: db.* */
	}	/* if */

	return GALLOC_OK;
}	/* ggiGAModeCapSanity */


/* Check that the values in a motor resource are not ridiculous,
 * including comparing to those in its resource cap if present. */
int ggiGAMotorCapSanity(ggiGA_resource_handle res)
{
	ggiGA_resource_handle cap = NULL;

	/* If this was called on a tank, then we presume this is a
	 * motor-only resource, and the caller will want to return
	 * GALLOC_EUNAVAILABLE rather than GALLOC_EFAILED */
	LIB_ASSERT (ggiGAIsMotor(res));
	if (!ggiGAIsMotor(res)) return GALLOC_EUNAVAILABLE;

	LIB_ASSERT (res->props != NULL);
	if (res->props == NULL) return GALLOC_ESPECIFY;
	/* More to check here?  Feilds that cannot be GGI_AUTO? */

	/* Now, do we have a cap? */
	if (!ggiGAIsCap(res->next)) {
		/* Nope. */
		return GALLOC_OK;
	}	/* if */

	cap = res->next;
	LIB_ASSERT (ggiGAIsMotor(cap));
	if (!ggiGAIsMotor(cap)) return GALLOC_EUNAVAILABLE;
	LIB_ASSERT (cap->props != NULL);
	if (cap->props == NULL) return GALLOC_ESPECIFY;
	
	LIB_ASSERT((res->priv == NULL) || (cap->priv != NULL));
	if ((res->priv != NULL) && (cap->priv != NULL)) return GALLOC_EFAILED;
	
	GA_CAPCMP_ASSERT(props->qty, >=);
	GA_CAPCMP_ASSERT(props->sub.motor.pos_snap.x, <=);
	GA_CAPCMP_ASSERT(props->sub.motor.pos_snap.y, <=);
	GA_CAPCMP_ASSERT(props->sub.motor.size_snap.x, <=);
	GA_CAPCMP_ASSERT(props->sub.motor.size_snap.y, <=);
	GA_CAPCMP_ASSERT(props->sub.motor.grid_start.x, <=);
	GA_CAPCMP_ASSERT(props->sub.motor.grid_start.y, <=);
	GA_CAPCMP_ASSERT(props->sub.motor.grid_size.x, >=);
	GA_CAPCMP_ASSERT(props->sub.motor.grid_size.y, >=);
	GA_CAPCMP_ASSERT(props->sub.motor.mul_min.x, <=);
	GA_CAPCMP_ASSERT(props->sub.motor.mul_min.y, <=);
	GA_CAPCMP_ASSERT(props->sub.motor.mul_max.x, >=);
	GA_CAPCMP_ASSERT(props->sub.motor.mul_max.y, >=);
	GA_CAPCMP_ASSERT(props->sub.motor.div_min.x, <=);
	GA_CAPCMP_ASSERT(props->sub.motor.div_min.y, <=);
	GA_CAPCMP_ASSERT(props->sub.motor.div_max.x, >=);
	GA_CAPCMP_ASSERT(props->sub.motor.div_max.y, >=);
	return GALLOC_OK;
	
}	/* ggiGAMotorCapSanity */


/* Check that the values in a tank resource are not ridiculous,
 * including comparing to those in its resource cap if present.
 */
int ggiGATankCapSanity(ggiGA_resource_handle res)
{
	/* TODO */
	return GALLOC_OK;
}	/* ggiGATankCapSanity */


/* Check that the values in a carb resource are not ridiculous,
 * including comparing to those in its resource cap if present.
 */
int ggiGACarbCapSanity(ggiGA_resource_handle res)
{
	/* TODO */
	return GALLOC_OK;
}	/* ggiGACarbCapSanity */




/* Functions on lists. */

int ggiGAFind(ggiGA_resource_list list, ggiGA_resource_handle handle)
{
	ggiGA_resource_handle current = NULL;

	if (list == NULL) return GGI_EARGINVAL;
	if (handle == NULL) return GGI_EARGINVAL;

	current = list;
	while (current != NULL) {
		if (current == handle) return GALLOC_OK;
		current = current->next;
	}	/* while */

	/* There was no such resource */
	return GALLOC_EUNAVAILABLE;
}	/* ggiGAFind */


ggiGA_resource_handle ggiGAPropsHandle(ggiGA_resource_list list, 
				       ggiGA_resource_handle handle)
{
	ggiGA_resource_handle current = NULL;

	if (handle == NULL) return NULL;

	if (ggiGA_FULLTYPE(handle) == GA_RT_RESLIST_TAGHEAD) return handle;
	if ((handle->props == NULL) && (handle->priv == NULL)) return handle;

	current = list;
	while (current != NULL) {
		if ((handle->props != NULL) && 
		    (current->props == handle->props)) break;
		if ((handle->priv != NULL) &&
		    (current->priv  == handle->priv)) break;
		current = current->next;
	}	/* while */

	return current;
}	/* ggiGAPropsHandle */


int ggiGAFreeDetachedHandle(ggiGA_resource_list list, 
			     ggiGA_resource_handle *handle) 
{
	int rc = 0;

	if (ggiGAPropsHandle(list, handle[0]) == NULL) {
		if (handle[0]->props) {
			free(handle[0]->props);
			rc |= 1;
		}	/* if */

		if (handle[0]->priv) {
			free(handle[0]->priv);
			rc |= 2;
		}	/* if */
	}	/* if */

	free(handle[0]);
	handle[0] = NULL;

	return rc;
}	/* ggiGAFreeDetachedHandle */


int ggiGATruncateList(ggiGA_resource_list list, ggiGA_resource_handle *handle)
{
	ggiGA_resource_handle current = NULL;
	int rc;

	if (handle == NULL) return GGI_EARGINVAL;
	if (handle[0] == NULL) return GGI_EARGINVAL;

	/* ggiGAFind will check that list != NULL */
	rc = ggiGAFind(list, handle[0]);
	if (rc != GALLOC_OK) return rc;

	current = handle[0];
	while (current != NULL) {
		if (ggiGAPropsHandle(list, current) == current) {
			if (current->props) free(current->props);
			if (current->priv)  free(current->priv);
		}	/* if */

		current = current->next;
	}	/* while */

	current = handle[0];
	while (current != NULL) {
		ggiGA_resource_handle next;

		next = current->next;
		free(current);
		current = next;
	}	/* while */

	handle[0] = NULL;

	return GALLOC_OK;
}	/* ggiGATruncateList */


int ggiGAEmptyList(ggiGA_resource_list *list)
{
	if (list[0] == NULL) return 0; /* Already an empty list */
	return ggiGATruncateList(list[0], list);
}	/* ggiGAEmptyList */


int ggiGAEmptyFailures(ggiGA_resource_list *list)
{
	int rc = GALLOC_OK;
	ggiGA_resource_handle current = NULL;

	if (list == NULL) return GGI_EARGINVAL;
	if (list[0] == NULL) return GGI_EARGINVAL;

	if (ggiGAIsFailed(list[0])) {
		rc = ggiGAEmptyList(list);
		goto done;
	}	/* if */

	current = list[0];
	while (current->next != NULL) {

		if (ggiGAIsFailed(current->next)) {
			rc = ggiGATruncateList(list[0], &(current->next));
			goto done;
		}	/* if */

		current = current->next;
	}	/* while */
 done:
	return rc;
}	/* ggiGAEmptyFailures */




int ggiGAAddMutual(ggiGA_resource_list list, ggiGA_resource_handle *handle)
{
	int rc = GALLOC_OK;
	ggiGA_resource_handle result = NULL;

	if (ggiGAFind(list, handle[0]) != GALLOC_OK) {
		return GGI_EARGINVAL;
	}	/* if */

	rc = ggiGACloneNoProps(handle[0], &result);
	if (rc != GALLOC_OK) goto err1;

	result->props = handle[0]->props;
	result->priv = handle[0]->priv;

	result->next = NULL;

	/* Indicate that this resource has not been checked yet,
	 * even though it may be flagged active.
	 */
	ggiGAFlagFailed(result);

	/* We know that &list will not be changed, because we know
	 * It is not empty.  (As long as noone changes AppendList's behavior.)
	 */
	rc = ggiGAAppendList(result, &list);

	handle[0] = result;

	LIB_ASSERT(rc == GALLOC_OK);

	return GALLOC_OK;

 err1:
	if (handle != NULL) handle[0] = NULL;
	return GGI_ENOMEM;
}	/* ggiGAAddMutual */


int ggiGAAppend(ggiGA_resource_handle handle, ggiGA_resource_list *list)
{
	ggiGA_resource_handle current = NULL;
	ggiGA_resource_handle tmp = NULL;

	if (list == NULL) return GGI_EARGINVAL;

	LIB_ASSERT(handle != NULL);

	ggiGAClone(handle, &tmp);
	tmp->next = NULL;

	if (list[0] == NULL) {
		/* Oops! - The first element in this list! */
		list[0] = tmp;
		goto exit;
	}	/* if */

	current = list[0];
	while (current->next != NULL) {
		current = current->next;
	}	/* while */

	current->next = tmp;

exit:
	return GALLOC_OK;
}	/* ggiGAAppend */


/* Used by extensions for their Create* lazy API, to keep the global
 * reqlist in sync without altering the existing handles in it.
 */
int ggiGASpliceList(ggiGA_resource_list *onto, ggiGA_resource_list *list)
{
	ggiGA_resource_handle onto_h;

	if (list == NULL) return GGI_EARGINVAL;
	if (onto == NULL) return GGI_EARGINVAL;

	onto_h = onto[0];
	if (onto_h == NULL) {
		onto[0] = list[0];
		list[0] = NULL;
		return GALLOC_OK;
	}

	do {	
		if (list[0] == NULL) return GALLOC_EFAILED;
		/* Remove first resource and advance list[0] to next one. */
		ggiGARemove(list, list); 
		if (onto_h->next == NULL) break;
		onto_h = onto_h->next;
	} while (1);
	onto_h->next = list[0];
	return GALLOC_OK;
}

int ggiGARemove(ggiGA_resource_list *list, ggiGA_resource_handle *handle)
{
	int rc = GALLOC_OK;
	ggiGA_resource_handle curr = NULL;
	ggiGA_resource_handle res;

	if (list == NULL) return GGI_EARGINVAL;
	if (handle == NULL) return GGI_EARGINVAL;

	rc = ggiGAFind(list[0], handle[0]);
	if (rc != GALLOC_OK) {
		/* not present in list */
		return GALLOC_EUNAVAILABLE;
	}	/* if */

	curr = list[0];
	res = handle[0];

	/* Special case so below code will work. */
	if (curr == res) {
		/* They want to remove the head */
		list[0] = curr->next;
		curr->next = NULL;
		ggiGAFreeDetachedHandle(list[0], &curr);
		if (list != handle) handle[0] = NULL;
		goto exit;
	}	/* if */

	/* We know at this point that the head will stay. */
	while (curr->next != NULL) {
		if (res == curr->next) goto cont;
		curr = curr->next;
	}	/* while */

	GALLOCDPRINT("This should not happen\n");
	return GALLOC_EUNAVAILABLE;

 cont:
	/* Perform bypass operation */
	curr->next = curr->next->next;

	/* Make into one member list */
	res->next = NULL;

	/* Flushhh. */
	ggiGAFreeDetachedHandle(list[0], &res);

	if (handle != &(curr->next)) handle[0] = NULL;

 exit:
	return GALLOC_OK;

}	/* ggiGARemove */


int ggiGARemoveWithCap(ggiGA_resource_list *list,
			ggiGA_resource_handle *handle)
{
	int rc = GALLOC_OK;

	LIB_ASSERT(list != NULL);
	LIB_ASSERT(list[0] != NULL);
	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(handle[0] != NULL);

	if (list == NULL) return GGI_EARGINVAL;
	if (handle == NULL) return GGI_EARGINVAL;

	/* Obviously, you cannot remove anything from empty list */
	if (list[0] == NULL) return GALLOC_EUNAVAILABLE;

	rc = ggiGAFind(list[0], handle[0]);
	if (rc != GALLOC_OK) {
		/* not present in list */
		rc = GALLOC_EUNAVAILABLE;
		goto exit;
	}	/* if */

	if (ggiGAIsCap(handle[0]->next)) {
		rc = ggiGARemove(list, &(handle[0]->next));
		LIB_ASSERT(rc == GALLOC_OK);
	}	/* if */

	rc = ggiGARemove(list, handle);
	LIB_ASSERT(rc == GALLOC_OK);

exit:
	return rc;
}	/* ggiGARemoveWithCap */


int ggiGACopyList(ggiGA_resource_list in, ggiGA_resource_list *out)
{
	int rc = GALLOC_OK;

	ggiGA_resource_handle res = NULL;
	ggiGA_resource_handle tail = NULL;

	if (out == NULL) return GGI_EARGINVAL;

	if (out[0] != NULL) {
		rc = ggiGAEmptyList(out);
		LIB_ASSERT(rc == GALLOC_OK);
	}	/* if */
	LIB_ASSERT(out[0] == NULL);

	/* Return Empty list (NULL) if given empty list. */
	if (in == NULL) goto exit;

	/* Do the first one */
	rc = ggiGAClone(in, &tail);
	if (rc != GALLOC_OK) goto err1;
	LIB_ASSERT(tail->next == NULL);

	out[0] = tail;
	res = in->next;
	while (res != NULL) {
		ggiGA_resource_handle tmp1 = NULL;

		LIB_ASSERT(tail != NULL);
		LIB_ASSERT(tail->next == NULL);

		/* Detect mutual properties and duplicate them appropriately */
		tmp1 = ggiGAPropsHandle(in, res);
		LIB_ASSERT(tmp1 != NULL);
		if (tmp1 != res) {
			ggiGA_resource_handle tmp2;

			tmp2 = ggiGAHandle(out[0], in, tmp1);
			LIB_ASSERT(tmp2 != NULL);
			rc = ggiGACloneNoProps(res, &(tail->next));
			if (rc != GALLOC_OK) goto err2;
			tail->next->props = tmp2->props;
			tail->next->priv =  tmp2->priv;
		} else {
			rc = ggiGAClone(res, &(tail->next));
			if (rc != GALLOC_OK) goto err2;
		}	/* if */

		tail = tail->next;
		res = res->next;

	}	/* while */

 exit:
	return GALLOC_OK;

 err2:
	ggiGAEmptyList(out);

 err1:
	return GGI_ENOMEM;
}	/* ggiGACopyList */


int ggiGAAppendList(ggiGA_resource_list this, ggiGA_resource_list *tothis)
{
	ggiGA_resource_handle tmp;

	if (tothis == NULL) return GGI_EARGINVAL;

	if (tothis[0] == NULL) {
		tothis[0] = this;
		goto exit;
	}	/* if */

	tmp = tothis[0];
	while (tmp->next != NULL) tmp = tmp->next;
	tmp->next = this;
 exit:
	return GALLOC_OK;
}	/* ggiGAAppendList */


/* Finds and returns the handle of the first (successful) GA_RT_FRAME
 * resource. If the returned "handle" is NULL, then there is _no_
 * GA_RT_FRAME in the list, which should never happen.
 */
ggiGA_resource_handle ggiGAFindFirstMode(ggiGA_resource_list reslist)
{
	ggiGA_resource_handle current = NULL;

	LIB_ASSERT(reslist != NULL);

	current = reslist;
	while (current != NULL) {

		if (ggiGA_TYPE(current) != GA_RT_FRAME) goto onward;

		/* GA_RT_FRAMEs don't come in "seeabove" flavor */
		LIB_ASSERT(!(current->res_state & GA_STATE_SEEABOVE));

		/* GA_RT_FRAME might be a CAP, when the previous one failed */
		if (ggiGAIsCap(current)) goto onward;
		if (ggiGAIsFailed(current)) goto onward;

		break;
onward:
		current = current->next;
	}	/* while */

	/* Should never fail. */
	LIB_ASSERT(current != NULL);

	return current;
}	/* ggiGAFindFirstMode */



/* Find the last successful GA_RT_FRAME resource before "handle" in
 * "reslist", or if "handle" is NULL the last in the entire list.  By "last
 * successful" we mean that if a failure is found before "handle", the search
 * returns NULL, not the last one that would have succeeded.
 */
ggiGA_resource_handle ggiGAFindLastMode(ggiGA_resource_list reslist,
					ggiGA_resource_handle handle)
{
	ggiGA_resource_handle current = NULL;
	ggiGA_resource_handle result = NULL;

	LIB_ASSERT(reslist != NULL);

	current = reslist;
	while (current != NULL) {

		if (current == handle) goto finish;
		if (ggiGAIsFailed(current)) return NULL;
		if (ggiGA_TYPE(current) != GA_RT_FRAME) goto onward;

		/* GA_RT_FRAMEs don't come in "seeabove" flavor */
		LIB_ASSERT(!(current->res_state & GA_STATE_SEEABOVE));

		if (ggiGAIsCap(current)) goto onward;

		result = current;
	onward:
		current = current->next;
	}	/* while */

finish:
	return result;
}	/* ggiGAFindLastMode */

