/*
******************************************************************************

   Galloc implementation for X target - functions.

   Copyright (C) 2001	Christoph Egger	[Christoph_Egger@t-online.de]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include "xgalloc.h"



int GALLOC_X_check_callback(ggi_visual_t vis,
			    ggiGA_resource_handle handle,
			    ggiGA_resource_handle compound,
			    ggiGA_resource_handle lastmode,
			    void **checkstate)
{
	int rc = GALLOC_EUNAVAILABLE;
	int res_state;

	struct xgalloc_container_t *checkstate_t = NULL;

	GALLOCDPRINT("%s:%s:%i: called\n", DEBUG_INFO);

	/* This demonstrates how to hold state between callbacks. */
	if (handle == NULL) {
		LIB_ASSERT(compound == NULL);
		LIB_ASSERT(lastmode == NULL);
		if (checkstate[0] == NULL) {
			checkstate[0] = malloc(sizeof(struct xgalloc_container_t));
			if (checkstate[0] == NULL) {
				rc = GGI_ENOMEM;
				ggiGAFlagFailed(handle);
				return rc;
			}	/* if */

			checkstate_t = checkstate[0];
			checkstate_t->state = 1;
			checkstate_t->mode = NULL;
			checkstate_t->mode_handle = NULL;
		} else {
			free(checkstate[0]);
			checkstate[0] = NULL;
		}	/* if */
		return GALLOC_OK;
	}	/* if */

	LIB_ASSERT(checkstate != NULL);
	LIB_ASSERT(checkstate[0] != NULL);
	LIB_ASSERT(compound != NULL);
	LIB_ASSERT(handle != NULL);

	checkstate_t = checkstate[0];

	/* We should never be called directly on a cap. */
	LIB_ASSERT(!ggiGAIsCap(handle));

	res_state = (handle->res_state & GA_STATE_REQUEST_MASK);

	/* Resources which are already active have already been checked. */
	if ((res_state & GA_STATE_NORESET) == GA_STATE_NORESET) {
		rc = GALLOC_OK;
		return rc;
	}	/* if */


	if (lastmode == NULL) {
		/* We can assume here, that there was no GA_RT_FRAME
		 * before. So we can start searching for it from the
		 * current handle
		 */
		checkstate_t->mode_handle = ggiGAFindFirstMode(handle);

		/* If this fails, then this is a invalid list */
		LIB_ASSERT(checkstate_t->mode_handle != NULL);

		/* X always requires a mode as the _first_ resource
		 * to be able to open a window
		 */
		GALLOCDPRINT("%s:%s:%i: check frames\n", DEBUG_INFO);
		rc = check_frame(vis, checkstate_t->mode_handle,
				 compound, lastmode);

		checkstate_t->mode = ggiGAGetGGIMode(checkstate_t->mode_handle);
	} else {
		checkstate_t->mode_handle = lastmode;
		checkstate_t->mode = ggiGAGetGGIMode(checkstate_t->mode_handle);
	}	/* if */


	/* Check resources here
	 */
	switch (handle->res_type & GA_RT_TYPE_MASK) {
	case (GA_RT_FRAME & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->props == NULL);
		LIB_ASSERT(handle->priv != NULL);

		GALLOCDPRINT("%s:%s:%i: check frames\n", DEBUG_INFO);

		rc = check_frame(vis, handle, compound, lastmode);
		break;

	case (GA_RT_BUFFER & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->props != NULL);

		GALLOCDPRINT("%s:%s:%i: check buffer\n", DEBUG_INFO);

		/* check for subtypes */
		rc = check_buffer(vis, handle, compound, lastmode);
		break;

	case (GA_RT_BOB & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->props != NULL);

		GALLOCDPRINT("%s:%s:%i: check bob\n", DEBUG_INFO);

		rc = check_bob(vis, handle, compound, lastmode);
		break;

	case (GA_RT_RENDERER & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->props != NULL);

		GALLOCDPRINT("%s:%s:%i: check renderer\n", DEBUG_INFO);

		/* check for subtypes */
		rc = check_renderer(vis, handle, compound, lastmode);
		break;

	case (GA_RT_SPRITE & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->props != NULL);

		GALLOCDPRINT("%s:%s:%i: check sprite\n", DEBUG_INFO);

		/* check for subtypes */
		rc = check_sprite(vis, handle, compound, lastmode);
		break;

	case (GA_RT_VIDEO & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->props != NULL);

		/* not implemented */
		rc = GALLOC_EUNAVAILABLE;
		break;

	case (GA_RT_WINDOW & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->props != NULL);

		GALLOCDPRINT("%s:%s:%i: check window\n", DEBUG_INFO);

		/* check for subtypes */
		rc = check_window(vis, handle, compound, lastmode);
		break;

	case (GA_RT_MISC & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->props != NULL);

		GALLOCDPRINT("%s:%s:%i: check misc\n", DEBUG_INFO);

		/* check for subtypes */
		rc = check_misc(vis, handle, compound, lastmode);
		break;

	default:

		GALLOCDPRINT("%s:%s:%i: unknown resource-type: (%X/%X/%X)\n",
				DEBUG_INFO,
				handle->res_type,
				handle->res_type & GA_RT_TYPE_MASK,
				handle->res_type & GA_RT_SUBTYPE_MASK);
		break;
	}	/* switch */

	return rc;
}	/* GALLOC_X_check_callback */




int GALLOC_X_set_callback(ggi_visual_t vis,
			  ggiGA_resource_handle handle,
			  ggiGA_resource_handle compound,
			  ggiGA_resource_handle lastmode,
			  void **setstate)
{
	int rc = GALLOC_EFAILED;
	int res_state;

	LIB_ASSERT(handle != NULL);

	if (handle == lastmode) {
		if (*setstate == NULL) {
			/* This is the first callback. */
			*setstate = malloc(sizeof(int));
			*((int *)*setstate) = 1;
		} else {
			/* This is the last callback. */
			free(*setstate);
			*setstate = NULL;
			/* X target sets on the first callback. */
			return GALLOC_OK;
		}	/* if */
	}

	GALLOCDPRINT("%s:%s:%i: called\n", DEBUG_INFO);
	res_state = (handle->res_state & GA_STATE_REQUEST_MASK);

	if (res_state & GA_STATE_NORESET) {
		rc = GALLOC_OK;
		return rc;
	}	/* if */

	if (handle->res_count > 0) {
		/* already in use */
		handle->res_count++;
		rc = GALLOC_OK;
		return rc;
	}	/* if */


	switch (handle->res_type & GA_RT_TYPE_MASK) {
	case (GA_RT_FRAME & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->priv != NULL);

		GALLOCDPRINT("%s:%s:%i: set frame\n", DEBUG_INFO);

		rc = set_frame(vis, handle, compound, lastmode);
		break;

	case (GA_RT_BUFFER & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->props != NULL);
		LIB_ASSERT(handle->priv != NULL);

		GALLOCDPRINT("%s:%s:%i: set buffer\n", DEBUG_INFO);

		/* handle subtypes */
		rc = set_buffer(vis, handle, compound, lastmode);
		break;

	case (GA_RT_BOB & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->props != NULL);
		LIB_ASSERT(handle->priv != NULL);

		GALLOCDPRINT("%s:%s:%i: set bob\n", DEBUG_INFO);

		rc = set_bob(vis, handle, compound, lastmode);
		break;


	case (GA_RT_RENDERER & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->props != NULL);

		GALLOCDPRINT("%s:%s:%i: set renderer\n", DEBUG_INFO);

		/* check for subtypes */
		rc = set_renderer(vis, handle, compound, lastmode);
		break;


	case (GA_RT_SPRITE & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->props != NULL);

		GALLOCDPRINT("%s:%s:%i: set sprite\n", DEBUG_INFO);

		/* handle subtypes */
		rc = set_sprite(vis, handle, compound, lastmode);
		break;

	case (GA_RT_VIDEO & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->props != NULL);

		GALLOCDPRINT("%s:%s:%i: set video\n", DEBUG_INFO);

		/* not implemented */
		rc = GALLOC_EUNAVAILABLE;
		break;

	case (GA_RT_WINDOW & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->props != NULL);

		GALLOCDPRINT("%s:%s:%i: set window\n", DEBUG_INFO);

		/* handle subtypes */
		rc = set_window(vis, handle, compound, lastmode);
		break;

	case (GA_RT_MISC & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->props != NULL);

		GALLOCDPRINT("%s:%s:%i: set misc\n", DEBUG_INFO);

		/* handle subtypes */
		rc = set_misc(vis, handle, compound, lastmode);
		break;

	}	/* switch */

	if (rc == GALLOC_OK) {
		/* Increase usage-counter */
		handle->res_count++;
		handle->vis = vis;
		ggiGAFlagActive(handle);
	}	/* if */

	return rc;
}	/* GALLOC_X_set_callback */




int GALLOC_X_release_callback(ggi_visual_t vis,
			      ggiGA_resource_handle handle)
{
	int rc = GALLOC_EFAILED;


	GALLOCDPRINT("%s:%s:%i: called\n", DEBUG_INFO);

	LIB_ASSERT(handle != NULL);

	if (ggiGAFlagFailed(handle)) {
		/* allocation failed! */
		rc = GALLOC_OK;
		return rc;
	}	/* if */

	handle->res_count--;
	if (handle->res_count < 0) {
		/* Never allocated! */
		handle->res_count = 0;
		rc = GALLOC_OK;
		return rc;
	}	/* if */

	LIB_ASSERT(handle->res_count >= 0);
	GALLOCDPRINT("%s:%s:%i: handle->res_count: %i\n",
		DEBUG_INFO, handle->res_count);

	if (handle->res_count > 0) {
		/* already in use */
		rc = GALLOC_OK;
		return rc;
	}	/* if */


	switch (handle->res_type & GA_RT_TYPE_MASK) {
	case (GA_RT_FRAME & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->priv != NULL);

		GALLOCDPRINT("%s:%s:%i: release frame\n", DEBUG_INFO);

		rc = release_frame(vis, handle);
		break;

	case (GA_RT_BUFFER & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->priv != NULL);
		LIB_ASSERT(handle->props != NULL);

		GALLOCDPRINT("%s:%s:%i: release buffer\n", DEBUG_INFO);

		/* handle subtypes */
		rc = release_buffer(vis, handle);
		break;

	case (GA_RT_BOB & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->priv != NULL);
		LIB_ASSERT(handle->props != NULL);

		GALLOCDPRINT("%s:%s:%i: release bob\n", DEBUG_INFO);

		rc = release_bob(vis, handle);
		break;

	case (GA_RT_RENDERER & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->props != NULL);

		GALLOCDPRINT("%s:%s:%i: release renderer\n", DEBUG_INFO);

		/* check for subtypes */
		rc = release_renderer(vis, handle);
		break;

	case (GA_RT_SPRITE & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->priv != NULL);
		LIB_ASSERT(handle->props != NULL);

		GALLOCDPRINT("%s:%s:%i: release sprite\n", DEBUG_INFO);

		/* handle subtypes */
		rc = release_sprite(vis, handle);
		break;

	case (GA_RT_VIDEO & GA_RT_TYPE_MASK):
		GALLOCDPRINT("%s:%s:%i: release video\n", DEBUG_INFO);

		break;

	case (GA_RT_WINDOW & GA_RT_TYPE_MASK):
		LIB_ASSERT(handle->priv != NULL);
		LIB_ASSERT(handle->props != NULL);

		GALLOCDPRINT("%s:%s:%i: release window\n", DEBUG_INFO);

		/* handle subtypes */
		rc = release_window(vis, handle);
		break;

	case (GA_RT_MISC & GA_RT_TYPE_MASK):
		GALLOCDPRINT("%s:%s:%i: release misc\n", DEBUG_INFO);

		/* handle subtypes */
		rc = release_misc(vis, handle);
		break;

	}	/* switch */

	return rc;
}	/* GALLOC_X_release_callback */


int GALLOC_X_check(ggi_visual_t vis, ggiGA_resource_list request,
		   ggiGA_resource_list *result)
{
	return ggiGACheck_template(vis, request, result,
				   &GALLOC_X_check_callback);
}       /* ggiGACheck */


int GALLOC_X_set(ggi_visual_t vis, ggiGA_resource_list request,
		 ggiGA_resource_list *result)
{
	return ggiGASet_template(vis, request, result,
				 &GALLOC_X_set_callback);
}       /* ggiGASet */


int GALLOC_X_release(ggi_visual_t vis, ggiGA_resource_list *list,
		     ggiGA_resource_handle *handle)
{
	return ggiGARelease_template(vis, list, handle,
				     &GALLOC_X_release_callback);
}       /* ggiGARelease */


int GALLOC_X__mode(ggi_visual_t vis, ggiGA_resource_handle *out)
{
	ggiGA_resource_handle res;

	LIB_ASSERT(out != NULL);
	LIB_ASSERT(out[0] != NULL);
	LIB_ASSERT(out[0]->props == NULL);

	GALLOCDPRINT_MODE("GALLOC_X__mode is called\n");

	res = out[0];

	res->res_type = GA_RT_FRAME;

	/* etc. */
	return GALLOC_OK;
}	/* GALLOC_X__mode */



static struct ggiGA_shareable_types shareable[] = {

/* type1, that can be shared with type2
 * Note: This is _extremely_ target dependent!
 */

/*	type1			type 2 */

{GA_RT_BUFFER_DONTCARE,	GA_RT_BUFFER_RAW,     GA_TYPE_UNKNOWN, GA_TYPE_UNKNOWN},
{GA_RT_BUFFER_DONTCARE,	GA_RT_BUFFER_ZBUFFER, GA_TYPE_UNKNOWN, GA_TYPE_2D},
{GA_RT_BUFFER_DONTCARE,	GA_RT_BUFFER_ABUFFER, GA_TYPE_UNKNOWN, GA_TYPE_2D},
{GA_RT_BUFFER_DONTCARE,	GA_RT_BUFFER_SBUFFER, GA_TYPE_UNKNOWN, GA_TYPE_2D},
{GA_RT_BUFFER_DONTCARE,	GA_RT_BUFFER_TBUFFER, GA_TYPE_UNKNOWN, GA_TYPE_2D},
{GA_RT_BUFFER_RAW,	GA_RT_BUFFER_RAW,     GA_TYPE_2D, GA_TYPE_2D},
{GA_RT_BUFFER_ZBUFFER,	GA_RT_BUFFER_ZBUFFER, GA_TYPE_2D, GA_TYPE_2D},
{GA_RT_BUFFER_ABUFFER,	GA_RT_BUFFER_ABUFFER, GA_TYPE_2D, GA_TYPE_2D},
{GA_RT_BUFFER_SBUFFER,	GA_RT_BUFFER_SBUFFER, GA_TYPE_2D, GA_TYPE_2D},
{GA_RT_BUFFER_TBUFFER,	GA_RT_BUFFER_TBUFFER, GA_TYPE_2D, GA_TYPE_2D},
{GA_RT_BUFFER_SWATCH,	GA_RT_BUFFER_SWATCH,  GA_TYPE_1D, GA_TYPE_1D},
{GA_RT_BOB,		GA_RT_BOB,	      GA_TYPE_2D, GA_TYPE_2D}
};

#define NUM_SHAREABLE	(sizeof(shareable)/sizeof(struct ggiGA_shareable_types))


int GALLOC_X_checkifshareable(ggi_visual_t vis,
			 ggiGA_resource_handle reshandle,
			 ggiGA_resource_handle tocompare)
{
	int rc = GALLOC_EFAILED;
	int i, found;
	size_t size1 = 0, size2 = 0;

	/* compare the properties (graphic-type) */
	if (reshandle->props->sub.tank.gt != tocompare->props->sub.tank.gt) {
		goto exit;
	}	/* if */


	/* compare the (sub-)types */
	for (found = i = 0; i < NUM_SHAREABLE; i++) {

		if (reshandle->res_type != shareable[i].type1) {
			continue;
		}	/* if */

		if (tocompare->res_type != shareable[i].type2) {
			continue;
		}	/* if */

		found = 1;
		break;
	}	/* for */

	if (!found) {
		goto exit;
	}	/* if */

	switch (shareable[i].dimension1) {
	case GA_TYPE_1D:
		size1 = reshandle->props->size.linear;
		break;

	case GA_TYPE_2D:
		size1 = reshandle->props->size.area.x *
			reshandle->props->size.area.y;
		break;

	case GA_TYPE_UNKNOWN:
		size1 = 0;
		break;

	}	/* switch */


	switch (shareable[i].dimension2) {
	case GA_TYPE_1D:
		size2 = tocompare->props->size.linear;
		break;

	case GA_TYPE_2D:
		size2 = tocompare->props->size.area.x *
			tocompare->props->size.area.y;
		break;

	case GA_TYPE_UNKNOWN:
		size2 = 0;
		break;

	}	/* switch */


	/* compare the advanced buffer layout here */
#warning Compare the advanced buffer layout here


	/* compare the properties (resolution size) */
	if (size1 == 0 || size2 == 0) goto exit;

	if (size1 <= size2) {
		rc = GALLOC_OK;
		goto exit;
	}	/* if */

	if (size1 > size2) {
		/* return code > 0 means, that we must
		 * allocate (rc * colordepth) bytes additionally
		 * to be able to share. Note, that the allocation
		 * my fail (i.e. when there is not enough space).
		 */
		rc = (size2 - size1);
		goto exit;
	}	/* if */

exit:
#if 0
	/* fail as long as this doesn't work correctly */
	rc = GALLOC_EFAILED;
#endif
	return rc;
}	/* GALLOC_X_checkifshareable */
