/* $Id: control.c,v 1.1.1.1 2001/04/22 19:02:56 cegger Exp $
******************************************************************************

   LibGIC - Controls

   Copyright (C) 1999 Andreas Beck	[becka@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdlib.h>
#include <string.h>

#include <ggi/gic.h>

gic_control    *gicControlAllocate	(gic_handle_t hand,char *name,char *shortname)
{
	gic_control *ret;

	if ((ret=malloc(sizeof(gic_control)))) {
		strncpy(ret->name,name,65);
		ret->name[65-1]=0;
		strncpy(ret->shortname,shortname,5);
		ret->shortname[5-1]=0;
		ret->features=NULL;
	}
	return ret;
}

void		gicControlFree		(gic_handle_t hand,gic_control *control)
{
	free(control);
}

int gicControlAttachFeature 		(gic_handle_t hand,gic_control *control, gic_feature *feature)
{
	gic_featurelist *listitem, *cur;

	if (NULL==(cur=malloc(sizeof(gic_featurelist)))) {
		return -1;	/* NOMEM */
	}

	cur->next = NULL;
	cur->feature = feature;

        if (control->features == NULL) {
                control->features = cur;
                return 0;
        }

        for (listitem = control->features;
             listitem->next;
             listitem = listitem->next) {
        }

        listitem->next = cur;
	return 0;
}

int gicControlDetachFeature 		(gic_handle_t hand,gic_control *control,gic_feature *feature)
{
	gic_featurelist *listitem,**last;

	for(listitem=*(last=&control->features);
		listitem!=NULL;listitem=*(last=&listitem->next))
	{
		if (listitem->feature==feature) {
			*last=listitem->next;
			free(listitem);
			return 0;
		}
	}
	return -1;
}

gic_feature  *gicControlLookupFeature	(gic_handle_t hand,gic_control *control,const char *name)
{
	gic_featurelist *listitem;

	for(listitem=control->features;
		listitem!=NULL;listitem=listitem->next)
	{
		if (0==strcmp(listitem->feature->name,name)) {
			return listitem->feature;
		}
	}
	return NULL;
}

gic_feature *gicControlGetFeature	(gic_handle_t hand,gic_control *control, int number)
{
	gic_featurelist *listitem;

	for (listitem = control->features;
	     listitem != NULL;
	     listitem = listitem->next) {
		if (number--==0)
			return listitem->feature;
	}
	return NULL;
}

int gicControlNumFeatures		(gic_handle_t hand,gic_control *control)
{
	int count=0;
	gic_featurelist *listitem;

	for (listitem = control->features;
	     listitem != NULL;
	     listitem = listitem->next) {
		count++;
	}
	return count;
}

int     gicControlHandleEvent		(gic_handle_t hand,gic_control *control, gii_event *event)
{
	gic_featurelist *listitem;
	int rc=0;

	/* Check all features ... */
	for(listitem=control->features;listitem!=NULL;
		listitem=listitem->next) {
			rc+=gicFeatureHandleEvent(hand,listitem->feature,event);
	}
	return rc;
}

int	gicControlWrite			(gic_handle_t hand,gic_control *control,FILE *where)
{
	gic_featurelist *listitem;

	fprintf(where,"gic:    Control \"%s\" \"%s\"\n",
		control->name,control->shortname);
	for(listitem=control->features;listitem!=NULL;listitem=listitem->next)
	{
		gicFeatureWrite(hand,listitem->feature,where);
	}
	fprintf(where,"gic:    Control END\n");

	return 0;
}

gic_control	*gicControlRead		(gic_handle_t hand,FILE *where)
{
	gic_control	*control;
	gic_feature	*feature;
	char buffer[1024],*hlp,*hlp2;

	fgets(buffer, sizeof(buffer), where);
	control=NULL;

	if ( 0==strncmp("gic:    Control \"",buffer,17) && 
	     (hlp=strchr(buffer+17,'"')) ) {
		*hlp='\0';	/* Terminate string */
		if ( (hlp =strchr(hlp+1,'"')) &&
		     (hlp2=strchr(hlp+1,'"')) ) {
			*hlp2='\0';
			control=gicControlAllocate(hand,buffer+17,hlp+1);
		} 
		if (NULL==control) return NULL;

		/* Attention: The last gicContextRead will fail, as it reads
		 * the gic:Head END. This is intended behaviour. It detects
		 * the end of the contexts.
		 */
		while( (feature=gicFeatureRead(hand,where)) )
			gicControlAttachFeature(hand,control,feature);

		return control;
	}
	return NULL;
}

int	gicControlMapActions		(gic_handle_t hand,gic_control *control,gic_actionlist *actions)
{
	gic_featurelist *listitem;

	for(listitem=control->features;listitem!=NULL;listitem=listitem->next)
	{
		gicFeatureMapActions(hand,listitem->feature,actions);
	}

	return 0;
}

int		gicControlFindConflict	 (gic_handle_t hand,gic_control *control, gic_recognizer *rec,gic_recognizer **start_and_return,gic_feature **optional)
{
	gic_featurelist *listitem;
	int rc;

	for(listitem=control->features;listitem!=NULL;listitem=listitem->next)
	{
		if ((rc=gicFeatureFindConflict(hand,listitem->feature,rec,start_and_return))) {
			if (optional)
				*optional=listitem->feature;
			return rc;
		}
	}
	return 0;
}

int gicControlGetName 			(gic_handle_t hand,gic_control *control, char *string, int maxlen)
{
	strncpy(string, control->name, maxlen);
	string[maxlen-1] = 0;

	return 0;
}
