/* $Id: feature.c,v 1.1.1.1 2001/04/22 19:02:56 cegger Exp $
******************************************************************************

   LibGIC - Features

   Copyright (C) 1999 Andreas Beck	[becka@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <ggi/gic.h>
#include <ggi/internal/gic_debug.h>

gic_feature    *gicFeatureAllocate	 (gic_handle_t hand,char *name,char *shortname)
{
	gic_feature *ret;

	if ((ret=malloc(sizeof(gic_feature)))) {
		strncpy(ret->name,name,65);
		ret->name[65-1]=0;
		strncpy(ret->shortname,shortname,5);
		ret->shortname[5-1]=0;
		ret->recognizers=NULL;
		ret->actions=NULL;
	}
	return ret;
}

void		gicFeatureFree		(gic_handle_t hand,gic_feature *feature)
{
	free(feature);
}

int gicFeatureAttachRecognizer		(gic_handle_t hand,gic_feature *feature, gic_recognizer *reclist)
{
	gic_recognizer *listitem;

	reclist->next = NULL;

	if (feature->recognizers == NULL) {
		feature->recognizers = reclist;
		return 0;
	}

	for (listitem = feature->recognizers;
	     listitem->next;
	     listitem = listitem->next) {
	}

	listitem->next = reclist;
	return 0;
}

int gicFeatureAttachRecognizerDriver	(gic_handle_t hand,gic_feature *feature, gic_recognizerdriver *driver,void *privdata)
{
	gic_recognizer *listitem;

	if (NULL==(listitem=malloc(sizeof(gic_recognizer))))
		return -1;	/* NOMEM */
	listitem->driver    =driver;
	listitem->privdata  =privdata;
	return gicFeatureAttachRecognizer(hand,feature,listitem);
}

int gicFeatureDetachRecognizerdriver	(gic_handle_t hand,gic_feature *feature, gic_recognizerdriver *driver,void *privdata)
{
	gic_recognizer *listitem,**last;

	for(listitem=*(last=&feature->recognizers);
		listitem!=NULL;listitem=*(last=&listitem->next))
	{
		if (listitem->driver  ==driver &&
		    listitem->privdata==privdata     ) {
			*last=listitem->next;
			free(listitem);
			return 0;
		}
	}
	return -1;
}

int gicFeatureDetachRecognizer		(gic_handle_t hand,gic_feature *feature, gic_recognizer *rec)
{
	gic_recognizer *listitem, **last = &feature->recognizers;

	for (listitem = feature->recognizers;
	     listitem != NULL;
	     last = &listitem->next, listitem = listitem->next) {

		if (listitem == rec) {
			*last = listitem->next;
			free(listitem);
			return 0;
		}
	}
	return -1;
}

gic_recognizer *gicFeatureGetRecognizer	(gic_handle_t hand,gic_feature *feature, int number)
{
	gic_recognizer *listitem;

	for(listitem=feature->recognizers;
		listitem!=NULL;listitem=listitem->next)
	{
		if (number--==0)
			return listitem;
	}
	return NULL;
}

int gicFeatureNumRecognizers		(gic_handle_t hand,gic_feature *feature)
{
	int count=0;
	gic_recognizer *listitem;

	for (listitem = feature->recognizers;
	     listitem != NULL;
	     listitem = listitem->next) {
		count++;
	}
	return count;
}

int gicFeatureFindConflict		(gic_handle_t hand,gic_feature *feature, gic_recognizer *rec,gic_recognizer **start_and_return)
{
	gic_recognizer *listitem;
	int rc;
	int have_seen=0;

	if (*start_and_return==NULL) have_seen=1;	/* Start condition. */

	for(listitem=feature->recognizers;
		listitem!=NULL;listitem=listitem->next)
	{
		if (have_seen && (rc=gicRecognizerFindConflict(hand,rec,listitem))) {
			*start_and_return=listitem;
			return rc;
		}
		if (listitem==*start_and_return) have_seen=1;
	}
	return 0;
}

int gicFeatureAttachAction		(gic_handle_t hand, gic_feature *feature, void (*action)(gic_handle_t hand, gic_actionlist *action,gic_feature *feature,int newstate,gic_flag flag,int recnum), void *privdata, char *name)
{
	gic_actionlist *listitem;

	if (NULL==(listitem=malloc(sizeof(gic_actionlist))))
		return -1;	/* NOMEM */
	listitem->next=feature->actions;
	listitem->action=action;
	listitem->privdata=privdata;
	listitem->name=name;
	feature->actions=listitem;

	return 0;
}

int	gicFeatureDetachAction	  	(gic_handle_t hand, gic_feature *feature, void (*action)(gic_handle_t hand, gic_actionlist *action,gic_feature *feature,int newstate,gic_flag flag,int recnum), void *privdata, char *name)
{
	gic_actionlist *listitem,**last;

	for(listitem=*(last=&feature->actions);
		listitem!=NULL;listitem=*(last=&listitem->next))
	{
		if (listitem->action==action && listitem->privdata==privdata &&
			(listitem->name==name || name==NULL || 
				strcmp(listitem->name,name)==0 ) ) {
			*last=listitem->next;
			free(listitem);
			return 0;
		}
	}
	return -1;
}

int             gicFeatureActivate        (gic_handle_t hand,gic_feature *feature, int newstate,gic_flag flag,int recnum)
{
	gic_actionlist *listitem;

	/* Call all actions ... */
	for(listitem=feature->actions;listitem!=NULL;
		listitem=listitem->next) {
		listitem->action(hand, listitem,feature,newstate,flag,recnum);
	}
	return 0;
}

int             gicFeatureHandleEvent(gic_handle_t hand,gic_feature *feature, gii_event *event)
{
	gic_recognizer *listitem;
	int number;
	int rc=0;

	/* Call all recognizers. If they want, THEY call FeatureActivate ... */
	for(listitem=feature->recognizers,number=0;
		listitem!=NULL;
		listitem=listitem->next,number++) {
		if ( listitem->driver->check(hand, listitem,event,feature,number) )
			rc++;
	}
	return rc;
}

int	gicFeatureWrite	(gic_handle_t hand,gic_feature *feature,FILE *where)
{
	gic_recognizer *listitem;
	gic_actionlist *listitem2;

	fprintf(where,"gic:      Feature \"%s\" \"%s\"\n",
		feature->name,feature->shortname);

	for(listitem=feature->recognizers;listitem!=NULL;listitem=listitem->next)
	{
		gicRecognizerWrite(hand,listitem,where);
	}

	for(listitem2=feature->actions;listitem2!=NULL;listitem2=listitem2->next)
	{
		gicActionWrite(hand,listitem2,where);
	}

	fprintf(where,"gic:      Feature END\n");

	return 0;
}

static void default_action(gic_handle_t hand,gic_actionlist *action,gic_feature *feature,int newstate,gic_flag flag,int recnum)
{
	GICDPRINT_CORE("Warning: default_action called for action %s on feature %s.\n",
			action->name,feature->name);
}

gic_feature	*gicFeatureRead(gic_handle_t hand,FILE *where)
{
	gic_feature		*feature;
	char buffer[1024],*hlp,*hlp2;

	fgets(buffer, sizeof(buffer), where);
	feature=NULL;

	if ( 0==strncmp("gic:      Feature \"",buffer,19) && 
	     (hlp=strchr(buffer+19,'"')) ) {
		*hlp='\0';	/* Terminate string */
		if ( (hlp =strchr(hlp+1,'"')) &&
		     (hlp2=strchr(hlp+1,'"')) ) {
			*hlp2='\0';
			feature=gicFeatureAllocate(hand,buffer+19,hlp+1);
		} 
		if (NULL==feature) return NULL;

		/* Attention: The last gicContextRead will fail, as it reads
		 * the gic:Head END. This is intended behaviour. It detects
		 * the end of the contexts.
		 */
		fgets(buffer, sizeof(buffer), where);
		while(0==strncmp("gic:        Recognizer \"",buffer,24) &&
		     ( hlp=strchr(buffer+24,'"')) ) {
		     	gic_recognizer *reclist;
		     	gic_recognizerdriver	*rc;
		     	*hlp='\0';

		     	rc=gicRecognizerDriverLookup(hand,buffer+24);
		     	reclist=NULL;
		     	if (rc) {
				if ((reclist=malloc(sizeof(gic_recognizer)))) {
					reclist->driver  =rc;
					reclist->privdata=NULL;
				     	gicFeatureAttachRecognizer(hand,feature,reclist);
			     	}
			} else {
				GICDPRINT_CORE("Warning: unknown recognizer %s. Discarded.\n",buffer+24);
			}
			
			fgets(buffer, sizeof(buffer), where);
			if ( reclist &&
			     0==strncmp("gic:          \"",buffer,15) && 
	     		     (hlp=strchr(buffer+15,'"')) ) {
				*hlp='\0';	/* Terminate string */
				rc->read_pvtdata(hand, reclist, buffer+15);
			}

			fgets(buffer, sizeof(buffer), where);	/* Skip Recognizer END */
			fgets(buffer, sizeof(buffer), where);	/* Read next entry */
		}
		while(0==strncmp("gic:        Action \"",buffer,20) && 
		     ( hlp=strchr(buffer+20,'"')) ) {
			*hlp='\0';	/* Terminate string */
			gicFeatureAttachAction(hand,feature,default_action,NULL,strdup(buffer+20));

			fgets(buffer, sizeof(buffer), where);	/* Skip Action END */
			fgets(buffer, sizeof(buffer), where);	/* Read next entry */
		}

		return feature;
	}
	return NULL;
}

int	gicFeatureMapActions	(gic_handle_t hand,gic_feature *feature,gic_actionlist *actions)
{
	gic_actionlist *listitem;

	for(listitem=feature->actions;listitem!=NULL;listitem=listitem->next)
	{
		gicActionMapActions(hand,listitem,actions);
	}

	return 0;
}

int gicFeatureGetName (gic_handle_t hand,gic_feature *feature, char *string, int maxlen)
{
	strncpy(string, feature->name, maxlen);
	string[maxlen-1] = 0;

	return 0;
}
