/*
 ******************************************************************************

 LIBGPF: libggi io target initialisation

 Copyright (C) 2001-2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include "libggi_gpf.h"
#include <ggi/internal/gpf_internal.h>


static const gg_option optlist[] =
{
	{ "vis", "" },
	{ "opmode", "frame" },
	{ "newvis", "" },
	{ "mode", "" },
	{ "flush", "" },
};

#define OPT_VIS		0
#define OPT_OPMODE	1
#define OPT_NEWVIS	2
#define OPT_MODE	3
#define OPT_FLUSH	4

#define NUM_OPTS        (sizeof(optlist)/sizeof(gg_option))



static int GPFinit(struct gpf_handle_opio_t *opio,
		   const char *args, va_list ap)
{
	struct libggigpf_priv *priv;
	gg_option options[NUM_OPTS];
	ggi_visual_t vis = NULL;
	char str[GPF_MAXPARAM_LEN];
	enum libggi_opmode_t opmode;

	priv = LIBGGI_PRIV(opio);

	GPFDPRINT_TARGET("%s:%s:%i: entered\n", DEBUG_INFO);
	GPFDPRINT_TARGET("%s:%s:%i: args (%s)\n", DEBUG_INFO, args);


	if (!args) {
		/* there are no arguments given */
		return GGI_OK;
	}	/* if */

	memcpy(options, optlist, sizeof(options));
	args = ggParseOptions(args, options, NUM_OPTS);
	if (args == NULL) {
		fprintf(stderr, "io-libggi: error in arguments.\n");
		return GGI_EARGINVAL;
	}	/* if */

	GPFDPRINT_TARGET("%s:%s:%i: io-libggi has args: options[OPT_VIS].result (%s)\n",
		DEBUG_INFO, options[OPT_VIS].result);

	if (options[OPT_VIS].result) {
		GPFDPRINT_TARGET("io-libggi: vis=\"%s\"\n",
				options[OPT_VIS].result);

		_gpfParseArgs(options[OPT_VIS].result, &ap, &vis);

		priv->vis = vis;
	}	/* if */

	if (options[OPT_OPMODE].result) {
		GPFDPRINT_TARGET("io-libggi: opmode=\"%s\"\n",
				options[OPT_OPMODE].result);

#if 0
		_gpfParseArgs(options[OPT_OPMODE].result, &ap, &str);
#endif
		strncpy(str, options[OPT_OPMODE].result,
			strlen(options[OPT_OPMODE].result)+1);

		opmode = OPMODE_FRAME;
		if (strncmp(str, "frame", 6) == 0) {
			opmode = OPMODE_FRAME;
		}	/* if */
		if (strncmp(str, "hline", 6) == 0) {
			opmode = OPMODE_HLINE;
		}	/* if */
		if (strncmp(str, "vline", 6) == 0) {
			opmode = OPMODE_VLINE;
		}	/* if */

		priv->opmode = opmode;
	}	/* if */

	if (options[OPT_NEWVIS].result) {
		GPFDPRINT_TARGET("io-libggi: newvis=\"%s\"\n",
				options[OPT_NEWVIS].result);

#if 0
		_gpfParseArgs(options[OPT_NEWVIS].result, &ap, &str);
#endif
		strncpy(str, options[OPT_NEWVIS].result,
			strlen(options[OPT_NEWVIS].result)+1);

		if (vis == NULL) {
			priv->openvis = 1;

			if (strncmp(str, "NULL", 4) == 0) {
				priv->target[0] = '\0';
			} else {
				strncpy(priv->target,
					str, strlen(str) + 1);
			}	/* if */
		}	/* if */
	}	/* if */

	if (options[OPT_MODE].result) {
		GPFDPRINT_TARGET("io-libggi: newvis=\"%s\"\n",
				options[OPT_MODE].result);

#if 0
		_gpfParseArgs(options[OPT_NEWVIS].result, &ap, &str);
#endif

		if (priv->openvis) {
			ggiParseMode(options[OPT_MODE].result,
					&priv->mode);
		}	/* if */
	}	/* if */

	if (options[OPT_FLUSH].result) {
#if 0
		_gpfParseArgs(options[OPT_FLUSH].result, &ap, &str);
#endif
		priv->flush = 1;
	}	/* if */

	GPFDPRINT_TARGET("%s:%s:%i: visual (%p)\n", DEBUG_INFO, vis);
	GPFDPRINT_TARGET("%s:%s:%i: leaving\n", DEBUG_INFO);

	return GGI_OK;
}	/* GPFinit */



static int GPFopen(struct gpf_handle_opio_t *opio,
		   struct gpf_pipeline_item_t *pipe,
		   struct gpf_dlhandle *dlh, const char *args,
		   va_list ap, uint32 *dlret)
{
	int rc;
	struct libggigpf_priv *priv;

	GPFDPRINT_LIBS("GPFopen(%p, %p, %p, %s, %p, %p) called for libggi sublib\n",
			opio, pipe, dlh, args ? args : "(NULL)", ap, dlret);

	LIBGPF_IO_PRIVATE(opio) = malloc(sizeof(struct libggigpf_priv));
	if (LIBGPF_IO_PRIVATE(opio) == NULL) return GGI_ENOMEM;

	priv = LIBGGI_PRIV(opio);

	memset(priv, 0, sizeof(struct libggigpf_priv));
	priv->opmode = OPMODE_FRAME;
	priv->flush = 0;
	memset(&(priv->mode), GGI_AUTO, sizeof(ggi_mode));

	rc = GPFinit(opio, args, ap);

	if (rc == GGI_OK) {
		opio->open = GPF_LIBGGI_open;
		opio->close = GPF_LIBGGI_close;

		opio->setcurrframe = GPF_LIBGGI_setcurrframe;
		opio->getcurrframe = GPF_LIBGGI_getcurrframe;
		opio->getnumframes = GPF_LIBGGI_getnumframes;

		opio->read = GPF_LIBGGI_read;
		opio->write = GPF_LIBGGI_write;
		opio->doio = GPF_LIBGGI_doio;

		opio->pipe_getinfolist = GPF_LIBGGI_pipe_getinfolist;
		opio->pipe->getformat = GPF_LIBGGI_getformat;

		if (LIBGPF_IO_ISINPUT(opio)) {
			opio->pipe->setup = GPF_LIBGGI_setup_input;
			opio->pipe->transfernextstreamblock
				= GPF_LIBGGI_transfernextstreamblock_input;
		}	/* if */

		if (LIBGPF_IO_ISOUTPUT(opio)) {
			opio->pipe->setup = GPF_LIBGGI_setup_output;
			opio->pipe->transfernextstreamblock
				= GPF_LIBGGI_transfernextstreamblock_output;
		}	/* if */
	}	/* if */

	*dlret = (GPF_DL_OPIO);
	return rc;
}	/* GPFopen */


static int GPFclose(struct gpf_pipeline_item_t *pipe,
		    struct gpf_dlhandle *dlh)
{
	GPFDPRINT_LIBS("GPFclose(%p, %p) called for libggi sublib\n",
			pipe, dlh);

	if (pipe->buf_conv_size > 0) {
		free(pipe->buf_conv);
		pipe->buf_conv = NULL;
		pipe->buf_conv_size = 0;
	}	/* if */

	free(LIBGPF_PIPE_PRIVATE(pipe));
	LIBGPF_PIPE_PRIVATE(pipe) = NULL;

	return GGI_OK;
}	/* GPFclose */


int GPFdl_libggi(int func, void **funcptr)
{
	switch (func) {
	case GGIFUNC_open:
		*funcptr = GPFopen;
		return GGI_OK;
	case GGIFUNC_exit:
		*funcptr = NULL;
		return GGI_OK;
	case GGIFUNC_close:
		*funcptr = GPFclose;
		return GGI_OK;
	default:
		*funcptr = NULL;
	}	/* switch */

	return GGI_ENOTFOUND;
}	/* GPFdl_libggi */


/* This should be included by exactly _one_ file in each sublib. */
#include <ggi/internal/ggidlinit.h>
