/*
 ******************************************************************************

 LIBGPF: ppm io target functions

 Copyright (C) 2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include "ppm_gpf.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>





static ssize_t read_str(struct gpf_handle_opio_t *opio,
		void *buf, size_t size)
{
	ssize_t rc = 0;
	size_t maxsize;
	unsigned char *buf_str;
	struct ppmgpf_priv *priv;

	priv = PPM_PRIV(opio);

	GPFDPRINT_TARGET("%s:%s:%i: called (buf: %p, size: %i)\n",
			DEBUG_INFO, buf, size);
	maxsize = size;
	buf_str = (unsigned char *)buf;
	while (size--) {
		rc = opio->opproto->read(opio, buf_str, 1);
		if (rc <= 0) break;

		if (buf_str[0] == '\n') break;
		buf_str++;
	}	/* while */


	GPFDPRINT_TARGET("%s:%s:%i: read data (%i, %p) - rc (%i)\n",
			DEBUG_INFO, buf, size, rc);

	return (rc < 0) ? rc : (maxsize - size);
}	/* read_str */


static ssize_t write_str(struct gpf_handle_opio_t *opio,
			void *buf, size_t size)
{
	ssize_t rc = 0;
	size_t maxsize;
	unsigned char *str;
	struct ppmgpf_priv *priv;

	priv = PPM_PRIV(opio);

	GPFDPRINT_TARGET("%s:%s:%i: called (buf: %p, size: %i)\n",
			DEBUG_INFO, buf, size);

	maxsize = size;
	str = (unsigned char *)buf;
	for (; (size > 0) && (*str != '\0'); str++, size--) {
		rc = opio->opproto->write(opio, str, 1);
		if (rc <= 0) break;
	}	/* for */

	GPFDPRINT_TARGET("%s:%s:%i: read data (%i, %p) - rc (%i)\n",
			DEBUG_INFO, buf, size, rc);

	return (rc < 0) ? rc : (maxsize - size);
}	/* write_str */




static int open4output(struct gpf_handle_opio_t *opio,
			struct gpf_pipeline_item_t *pipe,
			struct ppmgpf_priv *priv)
{
	int rc = GGI_OK;
	char buf[MAX_LINELEN];

	LIB_ASSERT(pipe != NULL);
	LIB_ASSERT(priv != NULL);

	memset(buf, 0, MAX_LINELEN);


	LIB_ASSERT(pipe->prev != NULL);

	priv->width = opio->pipe->width = pipe->prev->width;
	priv->height = opio->pipe->height = pipe->prev->height;
	priv->bypp = ByPP(opio->pipe->size);

	switch (priv->bypp) {
	case 3:
		priv->depth = 255;
		GPFDPRINT_TARGET("%s:%s:%i: bypp: %i, depth: %i\n",
				DEBUG_INFO, priv->bypp, priv->depth);
		break;
	case 6:
		priv->depth = 65535;
		GPFDPRINT_TARGET("%s:%s:%i: bypp: %i, depth: %i\n",
				DEBUG_INFO, priv->bypp, priv->depth);
		break;
	default:
		fprintf(stderr, "unsupported pixel format (bypp = %i)\n",
			priv->bypp);
		rc = GGI_ENOMATCH;
		goto err0;
	}	/* switch */


	if (priv->doconv) {
		LIB_ASSERT(priv->width > 0);
		LIB_ASSERT(priv->height > 0);
		LIB_ASSERT(priv->bypp > 0);

		priv->bufsize = priv->width * priv->height * priv->bypp;
		priv->buf = calloc(1, priv->bufsize);
		if (!priv->buf) {
			rc = GGI_ENOMEM;
			priv->bufsize = 0;
			goto err0;
		}	/* if */
	}	/* if */


	if (priv->format == PPM_FORMAT_P6) {
		snprintf(buf, MAX_LINELEN, "P6\n%d %d\n%d\n",
			priv->width, priv->height, priv->depth);

#warning Get the information how much frames we have to store, if possible
		/* When the input target stores no information how much frames it has
		 *  - like the ppm target - then it is not possible to figure out
		 * how much frames we have to store.
		 *
		 * For the beginning we can assume, that at least one frame is
		 * available.
		 */
		priv->max_frames = 1;
	}	/* if */
	if (priv->format == PPM_FORMAT_P3) {
		snprintf(buf, MAX_LINELEN, "P3\n%d %d\n%d\n",
			priv->width, priv->height, priv->depth);

		/* This format can never store more than one frame.
		 * Therefore, we must either store each frame in a
		 * separate file or we must throw all other frames
		 * away!
		 */
		priv->max_frames = 1;
	}	/* if */

	rc = write_str(opio, buf, MAX_LINELEN);
	if (rc < 0) {
		goto err0;
	}	/* if */

	return GGI_OK;


err0:
	fprintf(stderr, "%s:%s:%i: err0\n", DEBUG_INFO);
	opio->opproto->close(opio);

	GPFDPRINT_TARGET("%s:%s:%i: leave with rc: %i\n",
			DEBUG_INFO, rc);
	return rc;
}	/* open4output */





int GPF_PPM_open(struct gpf_handle_opio_t *opio)
{
	int rc = GGI_OK;
	int flags = 0;
	struct ppmgpf_priv *priv;
	char buf[MAX_LINELEN];


	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);

	priv = PPM_PRIV(opio);
	memset(buf, 0, MAX_LINELEN);


	if (LIBGPF_IO_ISINPUT(opio)) {
		flags = O_SYNC | O_RDONLY;
		goto read;
	}	/* if */

	if (LIBGPF_IO_ISOUTPUT(opio)) {
		flags = O_SYNC | O_CREAT | O_TRUNC | O_WRONLY /* O_RDWR */;
		goto write;
	}	/* if */

	rc = GGI_ENODEVICE;
	goto exit;

read:
	rc = opio->opproto->open(opio, flags);
	if (rc != GGI_OK) {
		goto err0;
	}	/* if */

	/* ppm's P3 format has never more than 1 frame stored.
	 * The P6 format can store multiple frames, but it stores
	 * no information how much frames it actually has.
	 * That means, we can figure that out during
	 * the reading transfer process. So, for the beginning
	 * we can only assume, that there is only one frame.
	 */
	priv->max_frames = 1;

#define READSTR	\
	do {								\
		rc = read_str(opio, buf, MAX_LINELEN);			\
		if (rc < 0) {						\
			/* read error occured */			\
			GPFDPRINT_TARGET("%s:%s:%i: read error: %s\n",	\
					DEBUG_INFO, strerror(rc));	\
			goto err1;					\
		}	/* if */					\
	} while ( (buf[0] == COMMENT_CHAR) || (isspace(buf[0])));


	READSTR;

	GPFDPRINT_TARGET("%s:%s:%i: buf: %s\n", DEBUG_INFO, buf);

	if (strncmp(buf, "P6", 2) == 0) {
		GPFDPRINT_TARGET("%s:%s:%i: format: P6\n", DEBUG_INFO);
		priv->format = PPM_FORMAT_P6;
	} else if (strncmp(buf, "P3", 2) == 0) {
		GPFDPRINT_TARGET("%s:%s:%i: format: P3\n", DEBUG_INFO);
		priv->format = PPM_FORMAT_P3;
	} else {
		fprintf(stderr, "io-ppm (output): Unknown file format!\n");
		rc = GGI_ENOMATCH;
		goto err1;
	}	/* if */

	READSTR;

	sscanf(buf, "%d %d", &priv->width, &priv->height);

	READSTR;

	sscanf(buf, "%d", &priv->depth);

	priv->bypp = (priv->depth > 255) ? 2*3 : 1*3;
	if (priv->depth > 65535) {
		rc = GGI_ENOMATCH;
		goto err1;
	}	/* if */

	GPFDPRINT_TARGET("%s:%s:%i: width: %i, height: %i, depth: %i, bypp: %i\n",
		DEBUG_INFO, priv->width, priv->height, priv->depth, priv->bypp);

	opio->pipe->width = priv->width;
	opio->pipe->height = priv->height;
	LIB_ASSERT(priv->bypp > 0);
	opio->pipe->depth = opio->pipe->size = (priv->bypp * 8);

	rc = GGI_OK;
	goto exit;

#undef READSTR


write:
	if ((opio->pipe->width == 0) || (opio->pipe->height == 0)) {
		/* Ooops! This target is probably opened _before_
		 * the input one. Fail!
		 */
		rc = GGI_ENOMATCH;
		goto err0;
	}	/* if */

	rc = opio->opproto->open(opio, flags);
	if (rc != GGI_OK) {
		goto err0;
	}	/* if */

	priv->width = opio->pipe->width;
	priv->height = opio->pipe->height;
	priv->bypp = ByPP(opio->pipe->size);

exit:
	LIB_ASSERT(rc == GGI_OK);
	return GGI_OK;

err1:
	opio->opproto->close(opio);
err0:
	GPFDPRINT_TARGET("%s:%s:%i: leave with rc: %i\n",
			DEBUG_INFO, rc);
	return rc;
}	/* GPF_PPM_open */


int GPF_PPM_close(struct gpf_handle_opio_t *opio)
{
	struct ppmgpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = PPM_PRIV(opio);

	if (priv->doconv) {
		free(priv->buf);
		priv->bufsize = 0;
		priv->buf = NULL;
	}	/* if */

	return opio->opproto->close(opio);
}	/* GPF_PPM_close */




int GPF_PPM_setcurrframe(struct gpf_handle_opio_t *opio, int num_frame)
{
	struct ppmgpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = PPM_PRIV(opio);


	if (num_frame < 0) return GGI_ENODEVICE;

	if (LIBGPF_IO_ISINPUT(opio)) {
		if (num_frame > priv->max_frames) {
			return GGI_ENODEVICE;
		}	/* if */
		opio->cur_frame = num_frame;
		return GGI_OK;
	}	/* if */

	if (LIBGPF_IO_ISOUTPUT(opio)) {
		if (num_frame > priv->max_frames) {
			return GGI_ENODEVICE;
		}	/* if */
		opio->cur_frame = num_frame;
		return GGI_OK;
	}	/* if */

	return GGI_ENODEVICE;
}	/* GPF_PPM_setcurrframe */


int GPF_PPM_getcurrframe(struct gpf_handle_opio_t *opio, int *num_frame)
{
	struct ppmgpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = PPM_PRIV(opio);


	if (LIBGPF_IO_ISINPUT(opio)) {
		*num_frame = 1;
		LIB_ASSERT(*num_frame == opio->cur_frame);
		return (*num_frame > 0) ? GGI_OK : GGI_ENODEVICE;
	}	/* if */

	if (LIBGPF_IO_ISOUTPUT(opio)) {
		*num_frame = 1;
		LIB_ASSERT(*num_frame == opio->cur_frame);
		return (*num_frame > 0) ? GGI_OK : GGI_ENODEVICE;
	}	/* if */

	return GGI_ENODEVICE;
}	/* GPF_PPM_getcurrframe */


int GPF_PPM_getnumframes(struct gpf_handle_opio_t *opio, int *num_frames)
{
	struct ppmgpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = PPM_PRIV(opio);

	if (priv->format == PPM_FORMAT_P3) {
		LIB_ASSERT(priv->max_frames == 1);
		return 1;
	}	/* if */

	if (priv->format == PPM_FORMAT_P6) {
		/* The P6 format may have more than one
		 * images in one file...
		 */
		return priv->max_frames;
	}	/* if */

	return 1;
}	/* GPF_PPM_getnumframes */



ssize_t GPF_PPM_read(struct gpf_handle_opio_t *opio,
			void *buf, size_t size)
{
	ssize_t rc;
	struct ppmgpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: opio: %p, buf: %p, size: %i\n",
			DEBUG_INFO, opio, buf, size);
	priv = PPM_PRIV(opio);

	if (priv->doconv) {
		LIB_ASSERT(priv->buf != NULL);
		LIB_ASSERT(priv->bufsize > 0);

		rc = opio->opproto->read(opio, priv->buf,
					priv->bufsize);
		GPFDPRINT_TARGET("%s:%s:%i: read data: %i\n",
				DEBUG_INFO, rc);

		if (rc < 0) {
			/* error occured */
			goto exit;
		}	/* if */

		rc = priv->func_conv(priv, priv->buf, buf,
					priv->bufsize, size);
	} else {
		rc = opio->opproto->read(opio, buf, size);

		GPFDPRINT_TARGET("%s:%s:%i: read data: %i\n",
				DEBUG_INFO, rc);
		if (rc < 0) {
			/* error occured */
			goto exit;
		}	/* if */
	}	/* if */


exit:
	return rc;
}	/* GPF_PPM_read */


ssize_t GPF_PPM_write(struct gpf_handle_opio_t *opio,
			void *buf, size_t size)
{
	ssize_t rc;
	struct ppmgpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = PPM_PRIV(opio);

	if (priv->doconv) {
		LIB_ASSERT(priv->buf != NULL);
		LIB_ASSERT(priv->bufsize > 0);

		rc = priv->func_conv(priv, buf, priv->buf,
					size, priv->bufsize);
		if (rc < 0) {
			/* an error occured */
			return rc;
		}	/* if */

		rc = opio->opproto->write(opio, priv->buf, priv->bufsize);
	} else {
		rc = opio->opproto->write(opio, buf, size);
	}	/* if */

	return rc;
}	/* GPF_PPM_write */




static int GPF_PPM_perform_conversion(struct ppmgpf_priv *priv)
{
	LIB_ASSERT(priv != NULL);


	switch (priv->nr) {
	case 0:
		/* This is r8g8b8 */

		switch (priv->bypp) {
		case 1:
			/* no conversion */
			return 0;
		case 2:
			/* perform conversion */
			return 1;
		}	/* switch */

	case 1:
		/* This is r16g16b16 */

		switch (priv->bypp) {
		case 1:
			/* perform conversion */
			return 1;
		case 2:
			/* no conversion */
			return 0;
		}	/* switch */
	}	/* switch */

	LIB_ASSERT(priv->nr < 2);
	return 0;
}	/* GPF_PPM_perform_conversion */




int GPF_PPM_setup_input(gpf_handle_t handle,
			struct gpf_pipeline_item_t *pipe)
{
	int rc;
	size_t data_size;
	struct ppmgpf_priv *priv = NULL;


	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(pipe != NULL);

	priv = PPM_PRIV(handle->opio_input);

	LIB_ASSERT(PIPE_READBUF(pipe) == NULL);
	LIB_ASSERT(PIPE_WRITEBUF(pipe) == NULL);

	GPFDPRINT("io-ppm (input): setting it up to act as input target\n");

	LIB_ASSERT(pipe->width > 0);
	LIB_ASSERT(pipe->height > 0);


	LIB_ASSERT(PIPELINEITEM_MARK(pipe, INFOLIST_INPUT) != NULL);
	LIB_ASSERT(PIPELINEITEM_MARK(pipe, INFOLIST_OUTPUT) == NULL);

	memcpy(&priv->pf, &PIPELINEITEM_MARK(pipe, INFOLIST_INPUT)->pf,
		sizeof(struct gpf_pixelformat_t));
	priv->nr = PIPELINEITEM_MARK(pipe, INFOLIST_INPUT)->nr;

	/* set up some internal function pointers */
	priv->is_frame_finished = _ppm_input_is_frame_finished;

	data_size = pipe->width * pipe->height * ByPP(priv->pf.size);
	LIB_ASSERT(data_size > 0);

	/* the amount of memory, that we have to transfer */
	rc = _gpf_alloc_readbuf(pipe, data_size);
	if (rc != GGI_OK) {
		GPFDPRINT("%s:%s:%i: io-ppm (input): memory allocation failed\n",
			DEBUG_INFO);
		goto err0;
	}	/* if */

	rc = _gpf_alloc_writebuf(pipe, data_size);
	if (rc != GGI_OK) {
		GPFDPRINT("%s:%s:%i: io-ppm (input): memory allocation failed\n",
			DEBUG_INFO);
		goto err1;
	}	/* if */

	LIB_ASSERT(pipe->buf_conv == NULL);
	LIB_ASSERT(pipe->buf_conv_size == 0);

	if (!GPF_PPM_perform_conversion(priv)) {
		GPFDPRINT("io-ppm (input): performs no conversion\n");
		priv->doconv = 0;
		priv->func_conv = NULL;
		priv->buf = NULL;
		priv->bufsize = 0;
		return GGI_OK;
	}	/* if */

	GPFDPRINT("io-ppm (input): conversion enabled\n");
	priv->doconv = 1;

	LIB_ASSERT(priv->width > 0);
	LIB_ASSERT(priv->height > 0);
	LIB_ASSERT(ByPP(priv->pf.size) > 0);

	priv->bufsize = priv->width * priv->height * ByPP(priv->pf.size);
	priv->buf = calloc(1, priv->bufsize);
	if (!priv->buf) {
		priv->bufsize = 0;
		goto err2;
	}	/* if */

	switch (priv->bypp) {
	case 1:
		priv->func_conv = GPF_PPM_conv_8to16;
		break;
	case 2:
		priv->func_conv = GPF_PPM_conv_16to8;
		break;
	}	/* switch */

	return GGI_OK;

err2:
	_gpf_free_writebuf(pipe);
err1:
	_gpf_free_readbuf(pipe);
err0:
	return GGI_ENOMEM;
}	/* GPF_PPM_setup_input */




int GPF_PPM_setup_output(gpf_handle_t handle,
			struct gpf_pipeline_item_t *pipe)
{
	int rc;
	struct ppmgpf_priv *priv = NULL;

	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(pipe != NULL);

	priv = PPM_PRIV(handle->opio_output);

	GPFDPRINT("io-ppm (output): setting it up to act as output target\n");

	LIB_ASSERT(PIPELINEITEM_MARK(pipe, INFOLIST_INPUT) == NULL);
	LIB_ASSERT(PIPELINEITEM_MARK(pipe, INFOLIST_OUTPUT) != NULL);

	memcpy(&priv->pf, &PIPELINEITEM_MARK(pipe, INFOLIST_OUTPUT)->pf,
		sizeof(struct gpf_pixelformat_t));
	priv->nr = PIPELINEITEM_MARK(pipe, INFOLIST_OUTPUT)->nr;

	/* set up some internal function pointers */
	priv->is_frame_finished = _ppm_output_is_frame_finished;

	if (!GPF_PPM_perform_conversion(priv)) {
		GPFDPRINT("io-ppm (output): performs no conversion\n");
		priv->doconv = 0;
	} else {
		GPFDPRINT("io-ppm (output): conversion enabled\n");
		priv->doconv = 1;
	}	/* if */


	rc = open4output(handle->opio_output, pipe, priv);


	return rc;
}	/* GPF_PPM_setup_output */


int GPF_PPM_getformat(struct gpf_pipeline_item_t *pipe,
			struct gpf_pipeline_item_t *from_pipe,
			int chain)
{
	GPFDPRINT_PIPEMGR("io-ppm: no rely on \"foreign\" format descriptions\n");

	/* the ppm target has no GPF_PIPE_DF_ALLVARIANTS flag.
	 * So we only implement this as a dummy function to avoid
	 * a crash in the pipeline management.
	 */

	return GGI_OK;
}	/* GPF_PPM_getformat */





int _ppm_input_is_frame_finished(struct ppmgpf_priv *priv,
				struct gpf_pipeline_item_t *pipe)
{
	return 0;
}	/* _ppm_input_is_frame_finished */

int _ppm_output_is_frame_finished(struct ppmgpf_priv *priv,
				struct gpf_pipeline_item_t *pipe)
{
	return 0;
}	/* _ppm_output_is_frame_finished */
