/* $Id: blockout.c,v 1.2 2001/05/05 21:27:11 cegger Exp $
******************************************************************************

   Test program for the "BSE" i.e. BobSpriteEmusprite extension.

   Copyright (C) 1999 Andreas Beck - becka@ggi-project.org
  
   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <math.h>

#include <ggi/gg.h>
#include <ggi/ggi.h>
#include <ggi/ggi_ext.h>

#include <ggi/bse.h>

void make_alphahardcircle(unsigned char *alpha,int width,int height)
{
	int x,y;

	/* Circle */
	for(x = 0; x < height; x++) {
		for(y = 0; y < width; y++) {
			*alpha++ = ((y - height/2) * (y - height/2)
				+ (x - width/2) * (x - width/2))
				< width * height/4 ? 0x00 : 0xff;
		}	/* for */
	}	/* for */
}	/* make_alphahardcircle */


char *fgetsnoc(char *buffer,size_t size,FILE *inf)
{
	char *rc;
	while(1) {
		rc = fgets(buffer,size,inf);
		if (rc && *rc != '#') return rc;
	}	/* while */
}	/* fgetsnoc */


int loadppm(ggi_visual_t vis,int x,int y,char *name)
{
	struct rgb{ unsigned char r,g,b; } rgb;
	ggi_color col;
	FILE *inf;
	int w,h,xx,yy;
	char buffer[256];
	
	if (NULL == (inf = fopen(name,"r"))) return 1;

	fgetsnoc(buffer, sizeof(buffer),inf);
	if (strcmp(buffer, "P6\n")) return 1;

	fgetsnoc(buffer,sizeof(buffer),inf);
	w = h = 0;
	sscanf(buffer, "%d %d", &w,&h);
	fgetsnoc(buffer, sizeof(buffer), inf);
	for(yy = 0; yy < h; yy++) {
		for(xx = 0; xx < w; xx++) {
			fread(&rgb, sizeof(rgb), 1, inf);
			col.r = rgb.r * 257;
			col.g = rgb.g * 257;
			col.b = rgb.b * 257;
			ggiPutPixel(vis, x+xx, y+yy, ggiMapColor(vis, &col));
		}	/* for */
	}	/* for */

	fclose(inf);
	return 0;
}	/* loadppm */


struct visual {
	ggi_visual_t vis;
	int width, height;
};

struct path {
	double x,y;
	double vx,vy;
	double ax,ay;
};

enum type {
	T_NONE,	/* No block ... */
	T_SIMPLE,	/* just disappear. */
	T_DOUBLE,	/* Need to hit twice */
	T_TRIPLE,	/* Need to hit three times */
	T_QUAD,		/* Need to hit four times */
	T_FIVE,		/* Need to hit five times */
	T_SIX,		/* Need to hit six times */
	T_SEVEN,	/* Need to hit seven times */
	T_EIGHT,	/* Need to hit eight times */
	T_INDESTRUCT,	/* This block can't be destroyed */
	T_LAST		/* Marker */
};

char typedes[T_LAST]={
	' ',
	'1',
	'2',
	'3',
	'4',
	'5',
	'6',
	'7',
	'8',
	'I'
};

enum bonus {
	B_NONE,	/* No block ... */
	B_NEWBALL,	/* give a new ball */
	B_FASTBALL,	/* make ball faster */
	B_FASTBALLS,	/* make balls faster */
	B_SLOWBALL,	/* make ball slower */
	B_SLOWBALLS,	/* make balls slower */
	B_LAST		/* Marker */
};

char bonusdes[T_LAST]={
	' ',
	'N',
	'f',
	'F',
	's',
	'S'
};

struct block {
	int x,y;
	int type;
	int bonus;
};

struct mysprite {
	ggiBse_t sprite;
	int w,h;
};

#define MAXBALLS 10

struct gamestate {

	struct visual vis;

	struct sprites {
		int numballs;
		struct mysprite ball[MAXBALLS], racket, brick[T_LAST];
	} sprites;

	struct physics {
		struct path ball[MAXBALLS];
		struct path racket;
		int minx,miny,maxx,maxy;
	} physics;

	struct blocks {
		int total;
		int numx,numy;
		int gridx,gridy;
		int offx,offy;
		ggi_visual_t bgvis;
		struct block *block;
	} blocks;

} gamestate;

void show_sprites(void)
{
	int n;
	for(n = 0; n < gamestate.sprites.numballs; n++) {
		ggiBseShow(gamestate.sprites.ball[n].sprite);
	}	/* for */
	ggiBseShow(gamestate.sprites.racket.sprite);
}	/* show_sprites */


void hide_sprites(void){
	int n;
	ggiBseHide(gamestate.sprites.racket.sprite);
	for(n = gamestate.sprites.numballs - 1; n >= 0; n--) {
		ggiBseHide(gamestate.sprites.ball[n].sprite);
	}	/* for */
}	/* hide_sprites */


void do_physics(struct path *path)
{
	path->vx += path->ax;
	path->vy += path->ay;
	path->x  += path->vx;
	path->y  += path->vy;
}	/* do_physics */


#define abs(x) (x<0 ? -x : x)

void check_coll_wall(struct path *path,int objw,int objh)
{
	if (path->x+objw >=gamestate.physics.maxx) path->vx = -abs(path->vx);
	if (path->y+objh >=gamestate.physics.maxy) path->vy = -abs(path->vy);
	if (path->x      <=gamestate.physics.minx) path->vx =  abs(path->vx);
	if (path->y      <=gamestate.physics.miny) path->vy =  abs(path->vy);
}	/* check_coll_wall */


/* Check ball/ball colisions.
 *
 * Assumptions: 
 * 1 Balls are of equal weight. 
 * 2 Impulse and energy is conserved.
 * 3 Impulse in only transferred perpendicular to the surface (no friction)
 * 
 * This leads to the equations 
 * v1+v2=const=v1'+v2' (1+2a) => v1'=v1-v3, v2'=v2+v3 with v3||(x1-x2) (3). 
 * Note, that the length of v3 is not yet defined.
 *
 * v1^2+v2^2=const=(v1-v3)^2+(v2+v3)^2 (with 2b).
 * => v1^2+v2^2=v1^2-2*v1v3+v3^2+v2^2+2*v2v3+v3^2
 * => 0 = 2*v3v3-2*v1v3+2*v2v3
 * => 0 = (v3+(v2-v1))v3
 * This has two solutions. v3=0 (trivial) or (v3+(v2-v1)) perpendicular to v3
 * To yield that, we have to scale v3 to compensate for any component of
 * vd=(v2-v1) in direction v3. This component is:
 * v3'=cos(alpha)*|vd|*v3/|v3|
 * If we now initially choose v3 to be a unit vector v3u=v3/|v3|, we can write
 * v3'=cos(alpha)*|vd|*|v3u|*v3u	[ |v3u| is 1, so we insert that ]
 * v3'=(vd v3u)*v3u
 * A v3 which fulfills the perpendicularity condition is then v3=-v3'.
 * => v1'=v1+v3', v2'=v2-v3'
 * Wow ... much math for so little code ...
 */
int check_coll_ball_ball(struct path *p1,struct path *p2,double r)
{
	double v3x,v3y;
	double len;
	
	v3x = (p2->x-p1->x); 	/* Calculate direction for impulse transfer */
	v3y = (p2->y-p1->y);
	len = sqrt(v3x*v3x+v3y*v3y);	/* Check distance */
	if (len>r) return 0;		/* No hit. */

#if 0	/* This prints out the constants of motion they should be the same before */
	printf("EHG v.: %f %f - %f\n", p1->vx + p2->vx, p1->vy + p2->vy,
	p1->vx * p1->vx + p1->vy * p1->vy + p2->vx * p2->vx + p2->vy * p2->vy);
#endif

	v3x /= len; v3y /= len;	/* v3 is now a unit vector (v3u)*/
	len = (p2->vx - p1->vx) * v3x + (p2->vy - p1->vy) * v3y; /* (vd v3u) */
	p1->vx += len * v3x; p1->vy += len * v3y;	/* v1'=v1+v3' */
	p2->vx -= len * v3x; p2->vy -= len * v3y;	/* v2'=v2-v3' */
	
#if 0	/* and after the collision. */
	printf("EHG n.: %f %f - %f\n", p1->vx + p2->vx, p1->vy + p2->vy,
	p1->vx * p1->vx + p1->vy * p1->vy + p2->vx * p2->vx + p2->vy * p2->vy);
#endif
	return 1;
}	/* check_coll_ball_ball */


/* We now check for a collision with a "point". Same calculation as above,
 * but now, v1'=v1+v3, v1'^2=v1^2. Yields the same equation with vd=2*v1.
 * Note, that impulse it not retained. The point is considered to have
 * a mass of infinity.
 */
int check_coll_ball_point(struct path *p1,double r,double x,double y)
{
	double v3x,v3y;
	double len;
	
	v3x = (x - p1->x);	/* Calculate direction for impulse transfer */
	v3y = (y - p1->y);
	len = sqrt(v3x * v3x + v3y * v3y);	/* Check distance */
	if (len > r) return 0;		/* No hit. */

#if 0	/* This prints out the constants of motion they should be the same before */
	printf("EHG v.: %f\n", p1->vx * p1->vx + p1->vy * p1->vy);
#endif

	v3x /= len; v3y /= len;	/* v3 is now a unit vector (v3u)*/
	len = (2 * p1->vx) * v3x + (2 * p1->vy) * v3y;	/* (vd v3u) */
	p1->vx -= len * v3x; p1->vy -= len * v3y;	/* v1'=v1+v3' */
	
#if 0	/* and after the collision. */
	printf("EHG n.: %f\n", p1->vx * p1->vx + p1->vy * p1->vy);
#endif
	return 1;
}	/* check_coll_ball_point */


int check_coll_ball_rect(struct path *path,int objw,int objh,
			 double rx,double ry,int rw,int rh)
{
	if (path->x + objw <= rx ||
	    path->y + objh <= ry ||
	    path->x   >= rx + rw ||
	    path->y   >= ry + rh ) return 0;	/* Bounds check failed. */

	rx -= objw / 2.0; ry -= objh / 2.0;	/* Transform coordinates relative to ball center */

	if (path->x >= rx && path->x <= rx+rw) {/* flat area, below or above. */
		if (path->y < ry) {
			path->vy = -abs(path->vy);
		} else {
			path->vy =  abs(path->vy);
		}	/* if */
		return 1;
	}	/* if */

	if (path->y >= ry && path->y <= ry+rh) {/* flat area, right/left. */
		if (path->x<rx) {
			path->vx = -abs(path->vx);
		} else {
			path->vx =  abs(path->vx);
		}	/* if */
		return 1;
	}	/* if */

	if (check_coll_ball_point(path, objw/2.0, rx   ,ry   ) ||
	    check_coll_ball_point(path, objw/2.0, rx+rw,ry   ) ||
	    check_coll_ball_point(path, objw/2.0, rx   ,ry+rh) ||
	    check_coll_ball_point(path, objw/2.0, rx+rw,ry   )) return 1;

	return 0;
}	/* check_coll_ball_rect */


void do_bonus(struct block *block,int ballnumber)
{
	switch (block->bonus) {
	default: break;
	case B_NEWBALL:
		if (gamestate.sprites.numballs < MAXBALLS) {
			int n = gamestate.sprites.numballs;
			double phi;

			gamestate.physics.ball[n].x =
				block->x + gamestate.blocks.gridx/2;
			gamestate.physics.ball[n].y =
				block->y + gamestate.blocks.gridy/2;
			phi = (double)rand() / RAND_MAX*M_PI;
			gamestate.physics.ball[n].vx = sin(phi);
			gamestate.physics.ball[n].vy = cos(phi);
			gamestate.physics.ball[n].ax = 0;
			gamestate.physics.ball[n].ay = 0;
			ggiBseShow(gamestate.sprites.ball[n].sprite);
			gamestate.sprites.numballs++;
		}	/* if */
		break;

	case B_FASTBALL:
		gamestate.physics.ball[ballnumber].vx *= 2.0;
		gamestate.physics.ball[ballnumber].vy *= 2.0;
		break;

	case B_FASTBALLS:
		{
		int n;

		for(n = 0; n < gamestate.sprites.numballs; n++) {
			gamestate.physics.ball[n].vx *= 2.0;
			gamestate.physics.ball[n].vy *= 2.0;
		}	/* for */
		}
		break;

	case B_SLOWBALL:
		gamestate.physics.ball[ballnumber].vx *= 0.5;
		gamestate.physics.ball[ballnumber].vy *= 0.5;
		break;

	case B_SLOWBALLS:
		{	int n;
			for(n = 0; n < gamestate.sprites.numballs; n++) {
				gamestate.physics.ball[n].vx *= 0.5;
				gamestate.physics.ball[n].vy *= 0.5;
			}	/* for */
		}
		break;
	}	/* switch */
}	/* do_bonus */


int block_ishit(struct block *block,struct visual *vis,int ballnumber)
{
	switch(block->type) {
	case T_NONE:
	case T_INDESTRUCT:
		break;

	case T_SIMPLE:
	case T_DOUBLE:
	case T_TRIPLE:
	case T_QUAD:
	case T_FIVE:
	case T_SIX:
	case T_SEVEN:
	case T_EIGHT:
		block->type--;
		hide_sprites();
		ggiBseMove(gamestate.sprites.brick[block->type].sprite,
			   block->x, block->y);
		show_sprites();

		/* do beep */
		printf("\a");fflush(stdout);

		if (block->type == 0) {
			if (block->bonus) do_bonus(block, ballnumber);
			if (gamestate.blocks.bgvis) {
				hide_sprites();
				ggiCrossBlit(gamestate.blocks.bgvis,
					     block->x-gamestate.physics.minx,
					     block->y-gamestate.physics.miny,
					     gamestate.blocks.gridx,
					     gamestate.blocks.gridy,
					     gamestate.vis.vis,
					     block->x,block->y);
				show_sprites();
			}	/* if */
			return 1;
		}	/* if */
		break;
	}	/* switch */
	return 0;
}	/* block_ishit */


int chartotype(char chr){
	int type;

	for(type = 0; type < T_LAST; type++) {
		if (typedes[type] == chr) return type;
	}	/* for */

	return T_NONE;
}	/* chartotype */


int chartobonus(char chr){
	int type;

	for(type = 0; type < B_LAST; type++) {
		if (bonusdes[type] == chr) return type;
	}	/* for */

	return B_NONE;
}	/* chartobonus */


int load_level(struct blocks *blocks,char *filename)
{
	FILE *inf;
	char buffer[256];
	int x,y,num;

	gamestate.physics.minx = 0;
	gamestate.physics.miny = 0;
	gamestate.physics.maxx = gamestate.vis.width;
	gamestate.physics.maxy = gamestate.vis.height;
	gamestate.blocks.gridx = 44;
	gamestate.blocks.gridy = 24;
	gamestate.blocks.offx  = 0;
	gamestate.blocks.offy  = 0;

	if (NULL == (inf = fopen(filename,"r"))) return 1;

	while (fgets(buffer, sizeof(buffer), inf)) {
		while (strchr(buffer,'\n')) *strchr(buffer,'\n')='\0';

		if (strncmp(buffer,"panel",5) == 0) {
			loadppm(gamestate.vis.vis, 0,0, buffer + 6);
		} else 	if (strncmp(buffer,"borders", 7) == 0) {
			sscanf(buffer+8,"%d %d %d %d",
				&gamestate.physics.minx,
				&gamestate.physics.miny,
				&gamestate.physics.maxx,
				&gamestate.physics.maxy);
		} else 	if (strncmp(buffer,"grid", 4) == 0) {
			sscanf(buffer+5,"%d %d %d %d",
				&gamestate.blocks.gridx,
				&gamestate.blocks.gridy,
				&gamestate.blocks.offx,
				&gamestate.blocks.offy);
		} else 	if (strncmp(buffer,"size", 4) == 0) {
			sscanf(buffer+5,"%d %d",
				&gamestate.blocks.numx,
				&gamestate.blocks.numy);
		} else 	if (strncmp(buffer,"bgpic", 5) == 0) {
			if (gamestate.blocks.bgvis) {
				ggiClose(gamestate.blocks.bgvis);
			}	/* if */

			gamestate.blocks.bgvis = ggiOpen("display-memory",NULL);
			if ((gamestate.blocks.bgvis)) {
				ggi_mode mode;

				ggiGetMode(gamestate.vis.vis, &mode);
				mode.visible.x = mode.virt.x =
					gamestate.physics.maxx - gamestate.physics.minx;
				mode.visible.y = mode.virt.y =
					gamestate.physics.maxy - gamestate.physics.miny;

				ggiCheckMode(gamestate.blocks.bgvis, &mode);

				if (ggiSetMode(gamestate.blocks.bgvis, &mode)) {
					ggiClose(gamestate.blocks.bgvis);
					gamestate.blocks.bgvis = NULL;
				} else {
					if (GT_SCHEME(mode.graphtype) == GT_PALETTE) {
						ggiSetColorfulPalette(gamestate.blocks.bgvis);
					}
					loadppm(gamestate.blocks.bgvis, 0,0, buffer+6);
				}	/* if */
			}	/* if */
		} else if (strncmp(buffer,"layout",6)==0) break;
	}	/* while */

	for(x = 0; x < MAXBALLS; x++) {
		gamestate.physics.ball[x].x  = (gamestate.physics.minx+gamestate.physics.maxx)/2-5;
		gamestate.physics.ball[x].y  = gamestate.physics.maxy-1-30;
		gamestate.physics.ball[x].vx =
		gamestate.physics.ball[x].vy = -2;
		gamestate.physics.ball[x].ax = 0;
		gamestate.physics.ball[x].ay = 0;
		gamestate.physics.ball[x].vx = ((random()%101)-50)/10.0;
	}
	gamestate.sprites.numballs  = 1;
	gamestate.physics.racket.x  = (gamestate.physics.minx+gamestate.physics.maxx)/2-25;
	gamestate.physics.racket.y  = gamestate.physics.maxy-1-20;
	gamestate.physics.racket.vx =
	gamestate.physics.racket.vy = 0;
	gamestate.physics.racket.ax =
	gamestate.physics.racket.ay = 0;

	if (gamestate.blocks.block) free(gamestate.blocks.block);
	if (gamestate.blocks.numx > (gamestate.physics.maxx -
		gamestate.physics.minx - gamestate.blocks.offx) /
		gamestate.blocks.gridx ||
	    gamestate.blocks.numy > (gamestate.physics.maxy -
	    	gamestate.physics.miny - gamestate.blocks.offy) /
		gamestate.blocks.gridy)
	{
		fprintf(stderr, "size argument is not good. We will probably soon segafult ...\n");
	}
	gamestate.blocks.block = malloc(sizeof(struct block) *
		gamestate.blocks.numx * gamestate.blocks.numy);

	{
	int x,y;

	for(y = 0; y < gamestate.blocks.numy; y++)
		for(x = 0; x < gamestate.blocks.numx; x++) {
			gamestate.blocks.block[x + gamestate.blocks.numx *y].x=
				gamestate.physics.minx
				+ x * gamestate.blocks.gridx
				+ gamestate.blocks.offx;
			gamestate.blocks.block[x + gamestate.blocks.numx *y].y=
				gamestate.physics.miny
				+ y * gamestate.blocks.gridy
				+ gamestate.blocks.offy;
			gamestate.blocks.block[x + gamestate.blocks.numx *y].type=0;
		}	/* for */
	}	/* for */

	if (gamestate.blocks.bgvis) {
		ggiCrossBlit(gamestate.blocks.bgvis,
		0,
		0,
		gamestate.physics.maxx-gamestate.physics.minx,
		gamestate.blocks.offy,
		gamestate.vis.vis,
		gamestate.physics.minx,
		gamestate.physics.miny);
		ggiCrossBlit(gamestate.blocks.bgvis,
		0,
		0,
		gamestate.blocks.offx,
		gamestate.physics.maxy-gamestate.physics.miny,
		gamestate.vis.vis,
		gamestate.physics.minx,
		gamestate.physics.miny);
		ggiCrossBlit(gamestate.blocks.bgvis,
		0,
		gamestate.blocks.offy+gamestate.blocks.gridy*gamestate.blocks.numy,
		gamestate.physics.maxx-gamestate.physics.minx,
		gamestate.physics.maxy-gamestate.physics.miny-
			gamestate.blocks.offy-gamestate.blocks.gridy*gamestate.blocks.numy,
		gamestate.vis.vis,
		gamestate.physics.minx,
		gamestate.physics.miny+
			gamestate.blocks.offy+gamestate.blocks.gridy*gamestate.blocks.numy);
		ggiCrossBlit(gamestate.blocks.bgvis,
		gamestate.blocks.offx+gamestate.blocks.gridx*gamestate.blocks.numx,
		0,
		gamestate.physics.maxx-gamestate.physics.minx-
			gamestate.blocks.offx-gamestate.blocks.gridx*gamestate.blocks.numx,
		gamestate.physics.maxy-gamestate.physics.miny,
		gamestate.vis.vis,
		gamestate.physics.minx+
			gamestate.blocks.offx+gamestate.blocks.gridx*gamestate.blocks.numx,
		gamestate.physics.miny);
	}	/* if */

	y = 0; blocks->total = 0;
	while(fgets(buffer, sizeof(buffer), inf)) {

		if (y >= blocks->numy) break;

		for(x = 0; x < blocks->numx; x++) {
			num = x + gamestate.blocks.numx * y;
			blocks->block[num].type = chartotype(buffer[x+x]);
			blocks->block[num].bonus= chartobonus(buffer[x+x+1]);
			ggiBseMove(gamestate.sprites.brick[blocks->block[num].type].sprite,
				blocks->block[num].x,
				blocks->block[num].y);

			switch(blocks->block[num].type) {
			case T_NONE:
				if (gamestate.blocks.bgvis) {
					ggiCrossBlit(gamestate.blocks.bgvis,
						blocks->block[num].x-gamestate.physics.minx,
						blocks->block[num].y-gamestate.physics.miny,
						gamestate.blocks.gridx,
						gamestate.blocks.gridy,
						gamestate.vis.vis,
						blocks->block[num].x,
						blocks->block[num].y);
				}	/* if */
			case T_INDESTRUCT:
				break;

			default:
				blocks->total++;
			}	/* switch */
		}	/* for */
		y++;
	}	/* while */

	fclose(inf);
	return 0;
}	/* load_level */


void setup_visual(struct visual *vis)
{
	ggi_mode mode;

	/* Initialize LibGGI */
	if (ggiInit() != 0) {
		fprintf(stderr, "Unable to initialize LibGGI\n");
		exit(1);
	}	/* if */
	
	/* Initialize BSE extension */
	if (ggiBseInit() != 0) {
		ggiPanic("Unable to initialize LibGGIBSE extension\n");
	}	/* if */

	/* Open the default visual */
	if ((vis->vis = ggiOpen(NULL)) == NULL) {
		ggiPanic("Unable to open default visual\n");
	}	/* if */
	
	/* Turn on asynchronous mode (which should always be used) */
	ggiSetFlags(vis->vis, GGIFLAG_ASYNC);
	
	/* Set the default mode */
	if (ggiSetSimpleMode(vis->vis, GGI_AUTO, GGI_AUTO, GGI_AUTO, GT_AUTO) < 0) {
		ggiPanic("Unable to set default mode\n");
	}	/* if */
	
	/* Attach the WMH extension to the visual */
	if (ggiBseAttach(vis->vis) < 0) {
		ggiPanic("Unable to attach BSE extension to visual\n");
	}	/* if */

	/* Check what mode we actually got */
	ggiGetMode(vis->vis, &mode);
	vis->width  = mode.visible.x;
	vis->height = mode.visible.y;
}	/* setup_visual */


void cleanup_visual(struct visual *vis) {

	/* Detach extension from gamestate.vis.visual */
	ggiBseDetach(vis->vis);

	/* Close visual */
	ggiClose(vis->vis);

	/* Deinitialize WMH extension */
	ggiBseExit();

	/* Deinitialize LibGGI */
	ggiExit();
}	/* cleanup_visual */


void allocate_sprites(struct sprites *sprites,ggi_visual_t vis)
{
	int x;
	
	for(x = 0; x < MAXBALLS; x++) {
		sprites->ball[x].sprite	= ggiBseCreateSprite(vis,10,10,GT_AUTO);
		if (!sprites->ball[x].sprite) {
			fprintf(stderr, "%s (%i) ggiBseCreateSprite() failed!\n",
				__PRETTY_FUNCTION__, __LINE__);
		}	/* if */
	}	/* for */

	sprites->racket.sprite = ggiBseCreateSprite(vis,50,10, GT_AUTO);
	if (!sprites->racket.sprite) {
		fprintf(stderr, "%s (%i) ggiBseCreateSprite() failed!\n",
			__PRETTY_FUNCTION__, __LINE__);
	}	/* if */

	for(x = 0; x < T_LAST; x++) {
		sprites->brick[x].sprite = ggiBseCreateBob(vis,40,20,GT_AUTO);
		if (!sprites->brick[x].sprite) {
			fprintf(stderr, "%s (%i) ggiBseAllocate failed!\n",
				__PRETTY_FUNCTION__, __LINE__);
		}	/* if */
	}	/* for */
}	/* allocate_sprites */


int loadsprite(struct mysprite *sprite,const char *pathname)
{
	if (sprite->sprite) {
		ggiBseDestroy(sprite->sprite);
	}	/* if */

	return 1;
}	/* loadsprite */


void load_sprites(struct sprites *sprites,ggi_visual_t vis)
{
	ggi_color col;
	unsigned char alpha[50*50];
	int x;

	col.r = 0xffff;	/* Make a red ball ... */
	col.g = 0x0000;
	col.b = 0x0000;
	ggiSetGCForeground(vis, ggiMapColor(vis, &col));

	ggiDrawBox(vis, 0, 0, 10, 10);
	loadppm(vis,0,0,"ball.ppm");	/* If that works, it's better ... */
	ggiFlush(vis);
	
	make_alphahardcircle(alpha,10,10);
	for(x = 0; x < MAXBALLS; x++) {
		/* Fixme - different colors */
		ggiBseSetImage(sprites->ball[x].sprite, vis, 0,0, alpha);
		ggiBseMove(sprites->ball[x].sprite,60,60);
	}	/* for */

	col.r = 0x0000;	/* and a blue racket ... */
	col.g = 0x0000;
	col.b = 0xffff;
	ggiSetGCForeground(vis, ggiMapColor(vis, &col));
	ggiDrawBox(vis, 0, 0, 50, 10);
	loadppm(vis,0,0,"racket.ppm");	/* If that works, it's better ... */
	ggiFlush(vis);

	ggiBseSetImage(sprites->racket.sprite, vis, 0,0, NULL);
	ggiBseMove(sprites->racket.sprite,60,60);

	for(x = 0; x < T_LAST; x++) {
		char buffer[80];
		col.r = 0x0000;	/* and green blocks ... */
		col.g = 0xffff;
		col.b = 0x0000;
		ggiSetGCForeground(vis, ggiMapColor(vis, &col));
		ggiDrawBox(vis, 0, 0, 40, 20);
		sprintf(buffer,"brick%d.ppm",x);
		loadppm(vis,0,0,buffer);	/* If that works, it's better ... */
		ggiFlush(vis);

		ggiBseSetImage(sprites->brick[x].sprite, vis, 0,0, NULL);
	}	/* for */
}	/* load_sprites */


void cleanup_sprites(struct sprites *sprites)
{
	int x;
	
	for(x = 0; x < MAXBALLS; x++) {
		ggiBseHide(sprites->ball[x].sprite);
		ggiBseDestroy(sprites->ball[x].sprite);
	}	/* for */

	ggiBseHide(sprites->racket.sprite);
	ggiBseDestroy(sprites->racket.sprite);

	for(x = 0; x < T_LAST; x++) {
		/* actually nonsense ... */
		ggiBseHide(sprites->brick[x].sprite);

		ggiBseDestroy(sprites->brick[x].sprite);
	}	/* for */

}	/* cleanup_sprites */


int gameloop(void)
{
	ggi_event ev;
	int n;

	while(1) {
		struct timeval timeout={0,1000};
		
		ggiEventPoll(gamestate.vis.vis, emAll, &timeout);
		usleep(timeout.tv_usec);
		
		n = ggiEventsQueued(gamestate.vis.vis, emAll);

		gamestate.physics.racket.vx=0;
		while (n) {
			ggiEventRead(gamestate.vis.vis, &ev, emAll);
			
			switch(ev.any.type) {
			case evKeyPress:
			case evKeyRepeat:
				switch(ev.key.sym) {
				case 'q':
				case 'Q':
				case 27: return -1;
				case 'l': /* level cheat */
				case 'L': return 0;
				case GIIK_Left:
					gamestate.physics.racket.vx=-1;
					break;
				case GIIK_Right:
					gamestate.physics.racket.vx=+1;
					break;
				}	/* switch */
				break;

			case evKeyRelease:
				switch(ev.key.sym) {
					case GIIK_Left:
					case GIIK_Right:
						gamestate.physics.racket.vx=0;
						break;
				}
				break;

			case evPtrRelative:
				gamestate.physics.racket.vx=ev.pmove.x;
				break;

			case evPtrAbsolute:
				gamestate.physics.racket.x=ev.pmove.x-25;
				break;

			}	/* switch */

			n--;
		}	/* while */

		for(n = 0; n < gamestate.sprites.numballs; n++) {
			do_physics(&gamestate.physics.ball[n]);
		}	/* for */

		do_physics(&gamestate.physics.racket);
		for(n = 0; n < gamestate.sprites.numballs; n++) {
			check_coll_wall(&gamestate.physics.ball[n], 10,10);
			if (check_coll_ball_rect(&gamestate.physics.ball[n],10,10,
				gamestate.physics.racket.x,gamestate.physics.racket.y,50,10)) { 
				gamestate.physics.ball[n].vx+=gamestate.physics.racket.vx/5.0;
				gamestate.physics.ball[n].vx*=0.9;
			}	/* if */
		}	/* for */

		check_coll_wall(&gamestate.physics.racket,50,10);

		{
		int x,y;

		for(y = 0; y < gamestate.blocks.numy; y++) {
			for(x = 0; x < gamestate.blocks.numx; x++) {
				for(n = 0;n < gamestate.sprites.numballs;n++) {

					if (gamestate.blocks.block[x+gamestate.blocks.numx*y].type &&
						check_coll_ball_rect(&gamestate.physics.ball[n],10,10,
						gamestate.blocks.block[x+gamestate.blocks.numx*y].x,
						gamestate.blocks.block[x+gamestate.blocks.numx*y].y,40,20))
					{
						if (block_ishit(
							&gamestate.blocks.block[x+gamestate.blocks.numx*y],
							&gamestate.vis,n))
						{
							gamestate.blocks.total--;
						}	/* if */
					}	/* if */
				}	/* for */
			}	/* for */
		}	/* for */
		}

		{
		int n1,n2;

		for(n1 = 0; n1 < gamestate.sprites.numballs; n1++) {
			for(n2 = n1 + 1; n2 < gamestate.sprites.numballs;n2++) {
				check_coll_ball_ball(&gamestate.physics.ball[n1],&gamestate.physics.ball[n2],10);
			}	/* for */
		}	/* for */
		}

		ggiBseMove(gamestate.sprites.racket.sprite,
				gamestate.physics.racket.x,
				gamestate.physics.racket.y);

		for(n = 0; n <gamestate.sprites.numballs; n++) {
			ggiBseMove(gamestate.sprites.ball[n].sprite,
				gamestate.physics.ball[n].x,
				gamestate.physics.ball[n].y);

			if (gamestate.physics.ball[n].y > gamestate.physics.maxy-1-20+6) {
				gamestate.physics.ball[n] =
					gamestate.physics.ball[gamestate.sprites.numballs-1];
				gamestate.sprites.numballs--;
				ggiBseHide(gamestate.sprites.ball[gamestate.sprites.numballs].sprite);
			}	/* if */
		}	/* for */

		ggiFlush(gamestate.vis.vis);
		if (gamestate.blocks.total <= 0) return 0;
		if (gamestate.sprites.numballs <= 0) return 1;
	}	/* while */

	return 1;
}	/* gameloop */


int main(void)
{
	int level;

	setup_visual(&gamestate.vis);
	allocate_sprites(&gamestate.sprites,gamestate.vis.vis);
	load_sprites(&gamestate.sprites,gamestate.vis.vis);

	gamestate.blocks.block=NULL;
	gamestate.blocks.bgvis=NULL;
	show_sprites();

	for (level = 1; level < 999; level++) {

		char buffer[256];

		hide_sprites();

		/*  */
		ggiSetGCForeground(gamestate.vis.vis, 0);
		ggiFillscreen(gamestate.vis.vis);
		sprintf(buffer,"level%d.dat", level);

		if (load_level(&gamestate.blocks,buffer)) break;
		if (gamestate.blocks.total == 0) break;

		show_sprites();
		switch(gameloop()) {
		case 0: /* Level completed */
			break;

		case 1: /* Lost ball */
			level--;
			break;

		default:	/* explicit quit */
			level = 1000;
			break;

		}	/* switch */
	}	/* for */

  cleanup:
	cleanup_sprites(&gamestate.sprites);
	cleanup_visual(&gamestate.vis);
	return 0;
}	/* main */
