/*
 ******************************************************************************

 LIBGPF: plain functions

 Copyright (C) 2001-2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include <ggi/internal/gpf.h>
#include <ggi/internal/gpf_debug.h>
#include <ggi/internal/gpf_helpers.h>

#include <string.h>


int gpfIsStopped(gpf_handle_t handle)
{
	GPF_APPASSERT(handle != NULL, "Invalid handle! Please gpfOpen() it properly.\n");

	return (!handle->started) ? 1 : 0;
}	/* gpfIsStopped */



int gpfStartStream(gpf_handle_t handle)
{
	GPF_APPASSERT(handle != NULL, "Invalid handle! Please gpfOpen() it properly.\n");

	if (gpfIsEOS(handle)) {
		GPFDPRINT("gpfStartStream: End Of Stream reached - stop transfer!\n");
		gpfStopStream(handle);
	} else {
		handle->started = 1;
		ggCurTime(&LIBGPF_TS_MARK(handle));

		if ((handle->pipe_index != NULL)
		  && (handle->props.flags & GPF_TIMESTAMP_CONTINUE))
		{
			/* we continue here and do not restart */
			GPFDPRINT("gpfStreamStart: stream transfer continues\n");
			return GGI_OK;
		}	/* if */

		handle->pipe_lasttransfer = NULL;
		handle->pipe_index = handle->pipeline->item_head;
		GPFDPRINT("gpfStreamStart: stream transfer (re)started\n");
	}	/* if */

	return GGI_OK;
}	/* gpfStartStream */


int gpfStopStream(gpf_handle_t handle)
{
	GPF_APPASSERT(handle != NULL, "Invalid handle! Please gpfOpen() it properly.\n");

	handle->started = 0;
	GPFDPRINT_CORE("gpfStopStream: stream transfer stopped\n");
	return GGI_OK;
}	/* gpfStopStream */


int gpfIsEOS(gpf_handle_t handle)
{
	int is_eos;

	GPF_APPASSERT(handle != NULL, "Invalid handle! Please gpfOpen() it properly.\n");

	is_eos = handle->opio_input->opproto->eos(handle->opio_input);
	GPFDPRINT_CORE("gpfIsEOS: input stream %s\n",
			(is_eos) ? "reached End Of Stream" : "can continue");
	if (is_eos) return is_eos;

	is_eos = handle->opio_output->opproto->eos(handle->opio_output);
	GPFDPRINT_CORE("gpfIsEOS: output stream %s\n",
			(is_eos) ? "reached End Of Stream" : "can continue");
	if (is_eos) return is_eos;

	return GGI_OK;
}	/* gpfIsEOS */



enum gpf_flags gpfSetFlags(struct gpf_properties_t *props,
			enum gpf_flags flags)
{
	enum gpf_flags tmp;

	LIB_ASSERT(props != NULL);

	tmp = (props->flags & GPF_RESPONSE_MASK);
	flags &= GPF_REQUEST_MASK;
	flags |= (props->flags & ~GPF_REQUEST_MASK);

	props->flags = flags;

	return tmp;
}	/* gpfSetFlags */


enum gpf_flags gpfGetFlags(const struct gpf_properties_t *props)
{
	LIB_ASSERT(props != NULL);

	return (props->flags & GPF_RESPONSE_MASK);
}	/* gpfGetFlags */



int gpfClearProperties(struct gpf_properties_t *props)
{
	LIB_ASSERT(props != NULL);
	memset(props, GPF_AUTO, sizeof(struct gpf_properties_t));

	props->flags = GPF_TIMESTAMP_CONTINUE;
	props->flags |= GPF_PIPE_OPT4SPEED;

	return GGI_OK;
}	/* gpfClearProperties */


int gpfSetProperties(gpf_handle_t handle, const struct gpf_properties_t *props)
{
	int flags;
	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(props != NULL);

	if (!gpfIsStopped(handle)) {
		return GGI_EBUSY;
	}	/* if */

	flags = props->flags;
	if ((flags & GPF_TIMESTAMP_MASK) == 0) {
		flags |= (LIBGPF_FLAGS(handle) & GPF_TIMESTAMP_MASK);
	}	/* if */
	if ((flags & GPF_PIPE_MASK) == 0) {
		flags |= (LIBGPF_FLAGS(handle) & GPF_PIPE_MASK);
	}	/* if */
	memcpy(&(LIBGPF_PROPS(handle)), props, sizeof(struct gpf_properties_t));
	LIBGPF_FLAGS(handle) = flags;

	return GGI_OK;
}	/* gpfSetProperties */


int gpfGetProperties(gpf_handle_t handle, struct gpf_properties_t *props)
{
	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(props != NULL);

	memcpy(props, &(LIBGPF_PROPS(handle)), sizeof(struct gpf_properties_t));

	return GGI_OK;
}	/* gpfGetProperties */




/* The whence param requires one GPF_SEEK_OPMASK value and
 * one GPF_SEEK_POSMASK.
 */
int gpfTimeStamp(gpf_handle_t handle, struct timeval *tv, int whence)
{
	/* check for validity */
	switch (whence & GPF_SEEK_OPMASK) {
	case 0:
	case GPF_SEEK_OPMASK:
		return GGI_ENOMATCH;
	}	/* switch */

	switch (whence & GPF_SEEK_POSMASK) {
	case 0:
	case GPF_SEEK_POSMASK:
		return GGI_ENOMATCH;
	}	/* switch */


	/* from here, we can assume, that all values
	 * are valid.
	 */
	switch (whence & GPF_SEEK_OPMASK) {
	case GPF_SEEK_GET:
		if (whence & GPF_SEEK_REL) {
			if (whence & GPF_SEEK_LAST) {
				/* get current position relative to last
				 * available timestamp
				 */
				handle->opio_input->timestamp(
					handle->opio_input,
					tv, GPF_SEEK_GET | GPF_SEEK_LAST);
				_gpf_ts_sub(tv, tv, &LIBGPF_TS_CURRENT(handle));
				return GGI_OK;
			} else if (whence & GPF_SEEK_END) {
				/* get current position relative to the
				 * end
				 */
				handle->opio_input->timestamp(
					handle->opio_input,
					tv, GPF_SEEK_GET | GPF_SEEK_END);
				_gpf_ts_sub(tv, tv, &LIBGPF_TS_CURRENT(handle));
				return GGI_OK;
			}	/* if */

			/* else get the relative timestamp position
			 * to the start, which is the absolute value
			 */
			memcpy(tv, &LIBGPF_TS_CURRENT(handle),
				sizeof(struct timeval));
			return GGI_OK;

		} else if (whence & GPF_SEEK_ABS) {
			if (whence & GPF_SEEK_LAST) {
				goto get_seek_last_abs;
			} else if (whence & GPF_SEEK_END) {
				goto get_seek_end_abs;
			}	/* if */

			memcpy(tv, &LIBGPF_TS_CURRENT(handle),
				sizeof(struct timeval));
			return GGI_OK;
		}	/* if */

		switch (whence & GPF_SEEK_POSMASK) {
		case GPF_SEEK_LAST:
get_seek_last_abs:
			/* this information can only come from
			 * the input target
			 */
			return handle->opio_input->timestamp(handle->opio_input,
				tv, GPF_SEEK_GET | GPF_SEEK_LAST);

		case GPF_SEEK_END:
get_seek_end_abs:
			return handle->opio_input->timestamp(handle->opio_input,
				tv, GPF_SEEK_GET | GPF_SEEK_END);
		}	/* switch */

		return GGI_ENOMATCH;

	case GPF_SEEK_SET:
		{
		int rc;
		struct timeval old_tv;

		/* First inform the targets about the change
		 */

		handle->opio_input->timestamp(handle->opio_input,
			&old_tv, GPF_SEEK_GET | GPF_SEEK_ABS);

		rc = handle->opio_input->timestamp(handle->opio_input,
						tv, whence);
		if (rc != GGI_OK) return rc;


		rc = handle->opio_output->timestamp(handle->opio_output,
						tv, whence);
		if (rc != GGI_OK) {
			rc = handle->opio_input->timestamp(handle->opio_input,
					&old_tv, GPF_SEEK_SET | GPF_SEEK_ABS);
			return rc;
		}	/* if */

		/* Then do it
		 */
		switch (whence & GPF_SEEK_POSMASK) {
		case GPF_SEEK_REL:
			_gpf_ts_add(&LIBGPF_TS_CURRENT(handle),
					&LIBGPF_TS_CURRENT(handle),
					tv);
			break;
		case GPF_SEEK_ABS:
			LIBGPF_TS_CURRENT(handle) = *tv;
			break;

		case GPF_SEEK_LAST:
			return handle->opio_input->timestamp(handle->opio_input,
				&LIBGPF_TS_CURRENT(handle),
				GPF_SEEK_GET | GPF_SEEK_LAST);
		case GPF_SEEK_END:
			LIBGPF_TS_CURRENT(handle) = LIBGPF_TS_TOTAL(handle);
			break;
		}	/* switch */

		return GGI_OK;
		}

	}	/* switch */

	return GGI_ENOMATCH;
}	/* gpfTimeStamp */




int gpfTransferNextStreamBlock(gpf_handle_t handle, gpfIndicatorCallback cbI)
{
	int rc = GGI_OK;
	struct timeval newtv, diff;
	struct gpf_pipeline_item_t *pipe_tmp;
	struct timeval transfer_timestamps;
	struct timeval tmp, sum;


	GPFDPRINT_TRANSFER("gpfTransferNextStreamBlock(%p, %p) called\n",
			handle, cbI);

	if (gpfIsStopped(handle)) {
		GPFDPRINT("stream transfer is stopped - abort\n");

		/* stream is stopped/hasn't been started */
		return 0;
	}	/* if */


	LIB_ASSERT(LIBGPF_TS_ISSET(&LIBGPF_TS_TRANSFER(handle)));
	transfer_timestamps = LIBGPF_TS_TRANSFER(handle);


	/* get the time the application took */
	ggCurTime(&newtv);
	_gpf_ts_sub(&diff, &newtv, &LIBGPF_TS_MARK(handle));

	GPFDPRINT("The application spent %.03f seconds\n",
		diff.tv_sec + (diff.tv_usec / 1000000.0F));


	/* The application took some time here, which very 
	 * likely causes some transfer fluctuation.
	 * So we speed up or slow down here a bit
	 * to keep to compensate it as good as possible.
	 */
	_gpf_ts_add(&transfer_timestamps, &transfer_timestamps, &diff);

	memcpy(&LIBGPF_TS_MARK(handle), &newtv, sizeof(struct timeval));

#if 0	/* enable this to artificially slow down the transfer.
	 * This makes it much easier to find/reproduce some bugs
	 * occuring occasionally (and in particular during slow transfer)
	 */
	transfer_timestamps.tv_sec = 0;
	transfer_timestamps.tv_usec = 1000;
#endif
	GPFDPRINT("try to transfer %.03f timestamps\n",
		transfer_timestamps.tv_sec
		+ (transfer_timestamps.tv_usec / 1000000.0F));

	LIBGPF_TS_CLEAR(&sum);
	tmp = transfer_timestamps;
	while (LIBGPF_TS_ISSET(&transfer_timestamps)) {

		pipe_tmp = handle->pipe_index;
		rc = handle->pipe_index->transfernextstreamblock(
					handle, handle->pipe_index,
					&tmp);
		handle->pipe_lasttransfer = pipe_tmp;

		if (handle->pipe_index == NULL) {
			gpfStopStream(handle);
			fprintf(stderr, "%s:%s:%i: transfer stopped!\n",
				DEBUG_INFO);
			goto exit;
		}	/* if */

		if (rc < 0) {
			/* error occured */
			fprintf(stderr, "unexpected conversion error: %i\n",
				rc);
			fprintf(stderr, "transfer stopped\n");
			gpfStopStream(handle);

			handle->pipe_index = handle->pipeline->item_head;
			goto exit;
		}	/* if */

		_gpf_ts_add(&sum, &sum, &tmp);
		_gpf_ts_sub(&transfer_timestamps,
				&transfer_timestamps, &tmp);

		/* status indicator */
		if (cbI != NULL) {
			cbI(&LIBGPF_TS_CURRENT(handle),
				&LIBGPF_TS_TOTAL(handle));
		}	/* if */

		GPFDPRINT_TRANSFER("%.03f timestamps totally transferd here\n",
			LIBGPF_TS_CURRENT(handle).tv_sec
			+ (LIBGPF_TS_CURRENT(handle).tv_usec / 1000000.0F));
	}	/* while */

	ggCurTime(&LIBGPF_TS_MARK(handle));
	GPFDPRINT_TRANSFER("gpfDoNextStreamDataBlock(): leave this function and let the app do something\n");

exit:
	_gpf_ts_add(&LIBGPF_TS_CURRENT(handle),
		&LIBGPF_TS_CURRENT(handle), &tmp);

	return GGI_OK;
}	/* gpfDoNextStreamDataBlock */
