/*
 ******************************************************************************

 LIBGPF: helper functions

 Copyright (C) 2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include <ggi/internal/gpf.h>
#include <ggi/internal/gpf_debug.h>
#include <ggi/internal/gpf_helpers.h>

#include <math.h>


/*
******************************************************************************
 Memory allocation
******************************************************************************
*/

void _gpf_mem_error(void) {
        ggiPanic("LibGPF is out of memory!\n");
}       /* _gpf_mem_error */


void *_gpf_malloc(size_t siz)
{
        void *mem = calloc(1, siz);

        if (mem == NULL) {
                _gpf_mem_error();
        }       /* if */

        return mem;
}       /* _gpf_malloc */

void *_gpf_calloc(size_t siz)
{
        void *mem = calloc(1, siz);

        if (mem == NULL) {
                _gpf_mem_error();
        }       /* if */

        return mem;
}       /* _gpf_calloc */


void *_gpf_realloc(void *ptr, size_t siz)
{
        void *mem = realloc(ptr, siz);

        if (mem == NULL) {
                _gpf_mem_error();
        }       /* if */

        return mem;
}       /* _gpf_realloc */


/*
******************************************************************************
 Readbuffer + Cache allocation
******************************************************************************
*/


int _gpf_alloc_readbuf(struct gpf_pipeline_item_t *pipe, size_t siz)
{
	LIB_ASSERT(PIPE_READBUF(pipe) == NULL);
	LIB_ASSERT(PIPE_READBUFSIZE(pipe) == 0);

	PIPE_READBUF(pipe) = calloc(1, siz);
	if (PIPE_READBUF(pipe) == NULL) goto err0;

	PIPE_READBUFSIZE(pipe) = siz;
	return GGI_OK;

err0:
	return GGI_ENOMEM;
}	/* _gpf_alloc_readbuf */


int _gpf_realloc_readbuf(struct gpf_pipeline_item_t *pipe, size_t siz)
{
	void *tmp;
	size_t oldsize;
	if (siz == 0) return _gpf_free_readbuf(pipe);

	oldsize = PIPE_READBUFSIZE(pipe);
	tmp = realloc(PIPE_READBUF(pipe), siz);
	if (tmp == NULL) {
		goto err0;
	}	/* if */
	PIPE_READBUF(pipe) = tmp;

	PIPE_READBUFSIZE(pipe) = siz;
	return GGI_OK;

err0:
	return GGI_ENOMEM;
}	/* _gpf_realloc_readbuf */


int _gpf_free_readbuf(struct gpf_pipeline_item_t *pipe)
{
	LIB_ASSERT(PIPE_READBUF(pipe) != NULL);
	LIB_ASSERT(PIPE_READBUFSIZE(pipe) > 0);

	free(PIPE_READBUF(pipe));

	PIPE_READBUF(pipe) = NULL;
	PIPE_READBUFSIZE(pipe) = 0;

	return GGI_OK;
}	/* _gpf_free_readbuf */



/*
******************************************************************************
 Writebuffer + Cache allocation
******************************************************************************
*/


int _gpf_alloc_writebuf(struct gpf_pipeline_item_t *pipe, size_t siz)
{
	LIB_ASSERT(PIPE_WRITEBUF(pipe) == NULL);
	LIB_ASSERT(PIPE_WRITEBUFSIZE(pipe) == 0);

	PIPE_WRITEBUF(pipe) = calloc(1, siz);
	if (PIPE_WRITEBUF(pipe) == NULL) goto err0;


	PIPE_WRITEBUFSIZE(pipe) = siz;
	return GGI_OK;

err0:
	return GGI_ENOMEM;
}	/* _gpf_alloc_writebuf */


int _gpf_realloc_writebuf(struct gpf_pipeline_item_t *pipe, size_t siz)
{
	void *tmp;
	size_t oldsize;
	if (siz == 0) return _gpf_free_writebuf(pipe);

	oldsize = PIPE_WRITEBUFSIZE(pipe);
	tmp = realloc(PIPE_WRITEBUF(pipe), siz);
	if (tmp == NULL) {
		goto err0;
	}	/* if */
	PIPE_WRITEBUF(pipe) = tmp;

	PIPE_WRITEBUFSIZE(pipe) = siz;
	return GGI_OK;

err0:
	return GGI_ENOMEM;
}	/* _gpf_realloc_writebuf */


int _gpf_free_writebuf(struct gpf_pipeline_item_t *pipe)
{
	LIB_ASSERT(PIPE_WRITEBUF(pipe) != NULL);
	LIB_ASSERT(PIPE_WRITEBUFSIZE(pipe) > 0);

	free(PIPE_WRITEBUF(pipe));

	PIPE_WRITEBUF(pipe) = NULL;
	PIPE_WRITEBUFSIZE(pipe) = 0;

	return GGI_OK;
}	/* _gpf_free_writebuf */


/*
******************************************************************************
 Timestamp helpers
******************************************************************************
*/


size_t _gpf_gettransfersize(size_t frame_size, struct timeval *interframe_delay)
{
	size_t transfersize;
	double idelay;
	double tmp;

	/* The math formular is this:
	 * 	z(x,y) = exp(-x) + y*exp(100/y);
	 * where z = transfersize, x = interframe_delay and y = frame_size
	 */

#if 0
	fprintf(stderr, "exp(-idelay) = %.03f, frame_size * exp(100.0 / frame_size) = %f\n",
		exp(-idelay), exp(1000.0 / (double)frame_size));
#endif

	idelay = interframe_delay->tv_sec + (interframe_delay->tv_usec / 1000000);
	tmp = exp(idelay);
#if 0
	transfersize = (size_t)( 1000.0 * exp(-idelay) + 1000.0 * exp(1000.0 / (double)frame_size));
#else
	transfersize = (size_t)( (double)frame_size * exp(tmp / (double)frame_size));
#endif


#if 0
	fprintf(stderr, "%s:%s:%i: idelay = %.03f, frame_size = %i,"
		" transfersize = %i\n",
		DEBUG_INFO, idelay, frame_size, transfersize);
#endif


	return transfersize;
}	/* _gpf_gettransfersize */



/* makes sure, that ->tv_sec and ->tv_usec are always valid */
int _gpf_ts_inrange(struct timeval *ts)
{
	while (ts->tv_usec > 1000000) {
		ts->tv_usec -= 1000000;
		ts->tv_sec++;
	}	/* while */

	while (ts->tv_usec < -1000000) {
		ts->tv_usec += 1000000;
		ts->tv_sec--;
	}	/* while */

	return GGI_OK;
}	/* _gpf_ts_inrange */



/* adds ts1 and ts2 and returns the result */
int _gpf_ts_add(struct timeval *result, const struct timeval *ts1,
                const struct timeval *ts2)
{
	result->tv_sec = ts1->tv_sec + ts2->tv_sec;
	result->tv_usec = ts1->tv_usec + ts2->tv_usec;
	while (result->tv_usec > 1000000) {
		result->tv_usec -= 1000000;
		result->tv_sec++;
	}	/* while */

	return GGI_OK;
}	/* _gpf_ts_add */



/* subtracts t2 from ts1 and returns the result */
int _gpf_ts_sub(struct timeval *result, const struct timeval *ts1,
                const struct timeval *ts2)

{
	result->tv_sec = ts1->tv_sec - ts2->tv_sec;
	result->tv_usec = ts1->tv_usec - ts2->tv_usec;
	while (result->tv_usec < -1000000) {
		result->tv_usec += 1000000;
		result->tv_sec--;
	}	/* while */

	return GGI_OK;
}	/* _gpf_ts_sub */
