/*
 ******************************************************************************

 LIBGPF: pipe functions

 Copyright (C) 2001-2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include <ggi/internal/gpf.h>
#include <ggi/internal/gpf_pipe.h>
#include <ggi/internal/gpf_internal.h>
#include <ggi/internal/gpf_debug.h>

#include <stdlib.h>
#include <string.h>
#include <math.h>


/*******************************
 * Pipline Item Helpers
 */

static int _gpfCompareRGBA(const struct gpf_pixelformat_t *pf_src,
			const struct gpf_pixelformat_t *pf_sink)
{
	LIB_ASSERT(pf_src != NULL);
	LIB_ASSERT(pf_sink != NULL);


	GPFDPRINT_PIPEMGR("compare two RGBA pixelformats now:\n");

	if (pf_src->depth != pf_sink->depth) {
		GPFDPRINT_PIPEMGR(" bitdepth doesn't fit: %i != %i\n",
				pf_src->depth, pf_sink->depth);
		return 0;
	}	/* if */

	if (pf_src->size != pf_sink->size) {
		GPFDPRINT_PIPEMGR(" size per pixel doesn't fit: %i != %i\n",
				pf_src->size, pf_sink->size);
		return 0;
	}	/* if */

	if ((LIBGPF_PF_RGBA(pf_src).red_mask !=
		LIBGPF_PF_RGBA(pf_sink).red_mask)
	|| (LIBGPF_PF_RGBA(pf_src).green_mask !=
		LIBGPF_PF_RGBA(pf_sink).green_mask)
	|| (LIBGPF_PF_RGBA(pf_src).blue_mask !=
		LIBGPF_PF_RGBA(pf_sink).blue_mask))
	{
		GPFDPRINT_PIPEMGR(" rgb bitmask doesn't fit\n");
		return 0;
	}	/* if */

	if ((LIBGPF_PF_RGBA(pf_src).alpha_mask != 0)
	   && (LIBGPF_PF_RGBA(pf_sink).alpha_mask != 0))
	{
		if (LIBGPF_PF_RGBA(pf_src).alpha_mask
		   != LIBGPF_PF_RGBA(pf_sink).alpha_mask)
		{
			GPFDPRINT_PIPEMGR(" alpha bitmask doesn't fit\n");
			return 0;
		}	/* if */
	}	/* if */


	GPFDPRINT_PIPEMGR(" pf_src and pf_sink fits together\n");
	return 1;
}	/* _gpfCompareRGBA */



static int _gpfCompareYUV(const struct gpf_pixelformat_t *pf_src,
			const struct gpf_pixelformat_t *pf_sink)
{
	LIB_ASSERT(pf_src != NULL);
	LIB_ASSERT(pf_sink != NULL);

	GPFDPRINT_PIPEMGR("compare two YUV pixelformats now:\n");

	fprintf(stderr, "fixme: implement _gpfCompareYUV()\n");

	return 0;
}	/* _gpfCompareYUV */



static int _gpfCompareYIQ(const struct gpf_pixelformat_t *pf_src,
			const struct gpf_pixelformat_t *pf_sink)
{
	LIB_ASSERT(pf_src != NULL);
	LIB_ASSERT(pf_sink != NULL);

	GPFDPRINT_PIPEMGR("compare two YIQ pixelformats now:\n");

	fprintf(stderr, "fixme: implement _gpfCompareYIQ()\n");

	return 0;
}	/* _gpfCompareYIQ */



static int _gpfCompareYCBCR(const struct gpf_pixelformat_t *pf_src,
			const struct gpf_pixelformat_t *pf_sink)
{
	LIB_ASSERT(pf_src != NULL);
	LIB_ASSERT(pf_sink != NULL);

	GPFDPRINT_PIPEMGR("compare two YCBCR pixelformats now:\n");

	fprintf(stderr, "fixme: implement _gpfCompareYCBCR()\n");

	return 0;
}	/* _gpfCompareYCBCR */



static int _gpfCompareEntries(struct gpf_infolist_t *input,
			struct gpf_infolist_t *output)
{
	LIB_ASSERT(input != NULL);
	LIB_ASSERT(output != NULL);

	if (input->pf.cf != output->pf.cf) return 0;

	if ((input->pf.flags & GPF_PIPE_DF_ALLVARIANTS)
	   == GPF_PIPE_DF_ALLVARIANTS)
	{
		GPFDPRINT_PIPEMGR("input (%p) accepts all variants\n",
				input);
		return 1;
	}	/* if */

	if ((output->pf.flags & GPF_PIPE_DF_ALLVARIANTS)
	   == GPF_PIPE_DF_ALLVARIANTS)
	{
		GPFDPRINT_PIPEMGR("output (%p) accepts all variants\n",
				output);
		return 1;
	}	/* if */


	/* Now we can assume, that the colorspaces
	 * are identical.
	 */
	switch (input->pf.cf) {
	case GPF_CF_RGBA:
		return _gpfCompareRGBA(&input->pf, &output->pf);

	case GPF_CF_YUV:
		return _gpfCompareYUV(&input->pf, &output->pf);

	case GPF_CF_YIQ:
		return _gpfCompareYIQ(&input->pf, &output->pf);

	case GPF_CF_YCbCr:
		return _gpfCompareYCBCR(&input->pf, &output->pf);

	default:
		fprintf(stderr, "%s:%s:%i: fixme: unknown colorspace: %i\n",
			DEBUG_INFO, input->pf.cf);
		return 0;

	}	/* switch */

	return 1;
}	/* _gpfCompareEntries */






/********************************
 * InfoList - Linklist
 */

int  _gpfInfoListConnectAppend(struct gpf_infolist_t *inflist,
				struct gpf_infolist_link_t *linkto)
{
	struct gpf_infolist_link_t *index = NULL;

	LIB_ASSERT(inflist != NULL);
	LIB_ASSERT(linkto != NULL);

	/* go to last element */
	if (inflist->linkto_head == NULL) {
		inflist->linkto_head = linkto;
		return GGI_OK;
	}	/* if */
	for (index = inflist->linkto_head;
		index->next != NULL;
		index = index->next);

	index->next = linkto;

	return GGI_OK;
}	/* _gpfInfoListConnectAppend */


int _gpfInfoListConnect(struct gpf_infolist_t *input_item,
			struct gpf_infolist_t *output_item)
{
	struct gpf_infolist_link_t *index = NULL;

	if (input_item->linkto_head == NULL) {
		input_item->linkto_head = calloc(1,
				sizeof(struct gpf_infolist_link_t));
		if (!input_item) return GGI_ENOMEM;

		/* connect first item */
		input_item->linkto_head->linkto = output_item;
		return GGI_OK;
	}	/* if */

	/* goto last element */
	for (index = input_item->linkto_head;
				index->next != NULL;
				index = index->next);

	index->next = calloc(1, sizeof(struct gpf_infolist_link_t));
	if (!index->next) {
		fprintf(stderr, "out of memory\n");
		return GGI_ENOMEM;
	}	/* if */

	/* connect here */
	index->linkto = output_item;

	return GGI_OK;
}	/* _gpfInfoListConnect */


int _gpfInfoListUnconnect(struct gpf_infolist_t *item)
{
	struct gpf_infolist_link_t *index = NULL;
	struct gpf_infolist_link_t *tmp = NULL;

	index = item->linkto_head;
	while (index != NULL) {
		tmp = index->next;
		free(index);
		index = tmp;
	}	/* while */

	item->linkto_head = NULL;

	return GGI_OK;
}	/* _gpfInfoListUnconnect */




/********************************
 * InfoList
 */

struct gpf_infolist_t *_gpfInfoListCreate(void)
{
	struct gpf_infolist_t *item = NULL;

	item = calloc(1, sizeof(struct gpf_infolist_t));

	return item;
}	/* _gpfInfoListCreate */


int _gpfInfoListDestroy(struct gpf_infolist_t **head)
{
	struct gpf_infolist_t *index = NULL;
	struct gpf_infolist_t *tmp = NULL;

	LIB_ASSERT(head != NULL);

	index = head[0];
	while (index != NULL) {
		tmp = index->next;

		if (index->linkto_head != NULL) {
			_gpfInfoListUnconnect(index);
		}	/* if */

		free(index);
		index = tmp;
	}	/* while */

	head[0] = NULL;

	return GGI_OK;
}	/* _gpfInfoListDestroy */


int _gpfInfoListAppend(struct gpf_infolist_t *head,
			struct gpf_infolist_t *item)
{
	struct gpf_infolist_t *index = NULL;

	LIB_ASSERT(item != NULL);
	LIB_ASSERT(head != NULL);

	_gpfInfoListGotoLastOne(head, index);
	LIB_ASSERT(index != NULL);
	LIB_ASSERT(index->next == NULL);

	index->next = item;

	return GGI_OK;
}	/* _gpfInfoListAppend */


static int _gpfInfoListSetMarkPos(struct gpf_pipeline_item_t *pipe,
			struct gpf_infolist_t *infolist,
			int sortof, int nr)
{
	LIB_ASSERT(pipe != NULL);
	LIB_ASSERT(infolist != NULL);
	LIB_ASSERT(nr == 0 || nr == 1);
	LIB_ASSERT(sortof == 0 || sortof == 1);

	pipe->mark[sortof][nr] = infolist;

	if (_gpfPipelineItemIsTarget(pipe)) {
		LIB_ASSERT(infolist->runtime > 0);
		LIB_ASSERT(infolist->memory_usage > 0);

		pipe->runtime = infolist->runtime;
		pipe->memory_usage = infolist->memory_usage;
		goto exit;
	}	/* if */

	if (sortof == PIPELINEITEM_FITS && nr == INFOLIST_OUTPUT) {
		LIB_ASSERT(infolist->runtime > 0);
		LIB_ASSERT(infolist->memory_usage > 0);

		pipe->runtime = infolist->runtime;
		pipe->memory_usage = infolist->memory_usage;
	}	/* if */

exit:
	return GGI_OK;
}	/* _gpfInfoListSetMarkPos */


static struct gpf_infolist_t *_gpfInfoListGetMarkPos(
				struct gpf_pipeline_item_t *pipe,
				int sortof, int nr)
{
	struct gpf_infolist_t *infolist;

	LIB_ASSERT(pipe != NULL);
	LIB_ASSERT(nr == 0 || nr == 1);
	LIB_ASSERT(sortof == 0 || sortof == 1);

	infolist = pipe->mark[sortof][nr];

	return infolist;
}	/* _gpfInfoListGetMarkPos */


static int _gpfInfoListClearMarkPos(struct gpf_pipeline_item_t *pipe,
				int sortof, int nr)
{
	LIB_ASSERT(pipe != NULL);
	LIB_ASSERT(nr == 0 || nr == 1);
	LIB_ASSERT(sortof == 0 || sortof == 1);

	pipe->mark[sortof][nr] = NULL;

	if (sortof == PIPELINEITEM_FITS && nr == INFOLIST_OUTPUT) {
		pipe->runtime = 0.0f;
		pipe->memory_usage = 0.0f;
	}	/* if */

	return GGI_OK;
}	/* _gpfInfoListClearMarkPos */








/********************************
 * Pipeline Item Helpers
 */

ssize_t _gpfPipelineItemRead(struct gpf_pipeline_item_t *pipe,
				size_t size, size_t maxsize)
{
	char *bufc;

	LIB_ASSERT(pipe != NULL);
	LIB_ASSERT(pipe->buf_conv != NULL);
	LIB_ASSERT(pipe->buf_conv_size > 0);
	LIB_ASSERT(PIPE_READBUF(pipe) != NULL);
	LIB_ASSERT(size <= pipe->buf_conv_size);

	bufc = (char *)PIPE_READBUF(pipe);
	bufc += PIPE_READBUFPOS(pipe);

	if (maxsize > PIPE_READBUFSIZE(pipe)) {
		maxsize = PIPE_READBUFSIZE(pipe);
	}	/* if */

	if ((PIPE_READBUFPOS(pipe) + size) > maxsize) {
		if (PIPE_READBUFPOS(pipe) >= maxsize) {
			return 0;
		}	/* if */

		size = maxsize - PIPE_READBUFPOS(pipe);
		LIB_ASSERT(size >= 0);
		if (size == 0) {
			/* EOS case */
			return 0;
		}	/* if */

		PIPE_READBUFPOS(pipe) += size;
		memcpy(pipe->buf_conv, bufc, size);
		return size;
	}	/* if */

	PIPE_READBUFPOS(pipe) += size;
	memcpy(pipe->buf_conv, bufc, size);

	return size;
}	/* _gpfPipelineItemRead */


ssize_t _gpfPipelineItemWrite(struct gpf_pipeline_item_t *pipe,
				size_t size, size_t maxsize)
{
	char *bufc;

	LIB_ASSERT(pipe != NULL);
	LIB_ASSERT(pipe->buf_conv != NULL);
	LIB_ASSERT(pipe->buf_conv_size > 0);
	LIB_ASSERT(PIPE_WRITEBUF(pipe) != NULL);
	LIB_ASSERT(size <= pipe->buf_conv_size);

	bufc = (char *)PIPE_WRITEBUF(pipe);
	bufc += PIPE_WRITEBUFPOS(pipe);

	if (maxsize > PIPE_WRITEBUFSIZE(pipe)) {
		maxsize = PIPE_WRITEBUFSIZE(pipe);
	}	/* if */

	if ((PIPE_WRITEBUFPOS(pipe) + size) > maxsize) {
		if (PIPE_WRITEBUFPOS(pipe) >= maxsize) {
			return 0;
		}	/* if */

		size = maxsize - PIPE_WRITEBUFPOS(pipe);
		LIB_ASSERT(size >= 0);
		if (size == 0) return 0;

		PIPE_WRITEBUFPOS(pipe) += size;
		memcpy(bufc, pipe->buf_conv, size);
		return size;
	}	/* if */

	PIPE_WRITEBUFPOS(pipe) += size;
	memcpy(bufc, pipe->buf_conv, size);

	return size;
}	/* _gpfPipelineItemWrite */






/********************************
 * Pipeline Items
 */

struct gpf_pipeline_item_t *_gpfPipelineItemCreate(void)
{
	struct gpf_pipeline_item_t *pipe = NULL;

	pipe = calloc(1, sizeof(struct gpf_pipeline_item_t));

	return pipe;
}	/* _gpfPipelineItemCreate */


int _gpfPipelineItemDestroy(struct gpf_pipeline_item_t **pipeitem)
{
	int rc = GGI_OK;

	LIB_ASSERT(pipeitem != NULL);
	LIB_ASSERT(pipeitem[0] != NULL);

	if (_gpfPipelineItemIsTarget(pipeitem[0])) {

		free(pipeitem[0]);
		pipeitem[0] = NULL;
		return GGI_OK;
	}	/* if */


	/* this item belongs to no target
	 */
	rc = _gpfInfoListDestroy(&(pipeitem[0]->input_head));
	if (rc != GGI_OK) {
		/* an error occured */
		goto exit;
	}	/* if */
	rc = _gpfInfoListDestroy(&(pipeitem[0]->output_head));
	if (rc != GGI_OK) {
		/* an error occured */
		goto exit;
	}	/* if */

	rc = _gpfClosePipe(pipeitem);
	LIB_ASSERT(rc == GGI_OK);
	LIB_ASSERT(pipeitem[0] == NULL);

exit:
	return rc;
}	/* _gpfPipelineItemDestroy */


int _gpfPipelineItemIsTarget(struct gpf_pipeline_item_t *pipe)
{
	LIB_ASSERT(pipe != NULL);

	return pipe->istarget;
}	/* _gpfPipelineItemIsTarget */



int _gpfPipelineItemAppend(struct gpf_pipeline_t *pipeline,
			struct gpf_pipeline_item_t *pipeitem)
{
	struct gpf_pipeline_item_t *index = NULL;

	LIB_ASSERT(pipeline != NULL);
	LIB_ASSERT(pipeitem != NULL);
	LIB_ASSERT(pipeitem->next == NULL);

	if (pipeline->item_head == NULL) {

		/* this is the first item in this list
		 */
		pipeline->item_head = pipeitem;
		LIB_ASSERT(pipeitem->prev == NULL);

		return GGI_OK;
	}	/* if */

	_gpfPipelineItemGotoLastOne(pipeline, index);
	LIB_ASSERT(index != NULL);
	LIB_ASSERT(index->next == NULL);

	index->next = pipeitem;
	pipeitem->prev = index;

	return GGI_OK;
}	/* _gpfPipelineItemAppend */


int _gpfPipelineItemInsert(struct gpf_pipeline_item_t *item,
			struct gpf_pipeline_item_t *pipe)
{
	LIB_ASSERT(item != NULL);
	LIB_ASSERT(pipe != NULL);
	LIB_ASSERT(pipe->next == NULL);
	LIB_ASSERT(pipe->prev == NULL);

	pipe->next = item->next;
	item->next = pipe;

	pipe->prev = item;
	if (pipe->next != NULL) {
		pipe->next->prev = pipe;
	}	/* if */

	return GGI_OK;
}	/* _gpfPipelineItemInsert */


int _gpfPipelineItemRemove(struct gpf_pipeline_item_t *item,
			struct gpf_pipeline_item_t *pipe)
{
	LIB_ASSERT(item != NULL);
	LIB_ASSERT(item->next == pipe);
	LIB_ASSERT(pipe != NULL);

	item->next = pipe->next;
	if (item->next != NULL) {
		item->next->prev = item;
	}	/* if */

	pipe->next = NULL;
	pipe->prev = NULL;

	return GGI_OK;
}	/* _gpfPipelineItemRemove */



int _gpfPipelineItemIsEqual(struct gpf_pipeline_item_t *pipe1,
			struct gpf_pipeline_item_t *pipe2)
{
	int equal;
	struct gpf_infolist_t *pipe1_idx = NULL;
	struct gpf_infolist_t *pipe2_idx = NULL;

	LIB_ASSERT(pipe1 != NULL);
	LIB_ASSERT(pipe2 != NULL);

	if (pipe1 == pipe2) {
		fprintf(stderr, "%s:%s:%i: pointer equals\n",
			DEBUG_INFO);
		return 1;
	}	/* if */

	/* Compare input */
	if ((pipe1->input_head != NULL) && (pipe2->input_head != NULL)) {
		for (pipe1_idx = pipe1->input_head,
			pipe2_idx = pipe2->input_head;
			pipe1_idx != NULL && pipe2_idx != NULL;
			pipe1_idx = pipe1_idx->next,
			pipe2_idx = pipe2_idx->next)
		{
			equal = _gpfCompareEntries(pipe1_idx, pipe2_idx);
			if (!equal) {
				GPFDPRINT_PIPEMGR("%s:%s:%i: input don't equals\n",
						DEBUG_INFO);
				return 0;
			}	/* if */

		}	/* for */
	}	/* if */

	/* Compare output */
	if ((pipe1->output_head != NULL) && (pipe2->output_head != NULL)) {
		for (pipe1_idx = pipe1->output_head,
			pipe2_idx = pipe2->output_head;
			pipe1_idx != NULL && pipe2_idx != NULL;
			pipe1_idx = pipe1_idx->next,
			pipe2_idx = pipe2_idx->next)
		{
			equal = _gpfCompareEntries(pipe1_idx, pipe2_idx);
			if (!equal) {
				GPFDPRINT_PIPEMGR("%s:%s:%i: output don't equals\n",
						DEBUG_INFO);
				return 0;
			}	/* if */

		}	/* for */
	}	/* if */

	GPFDPRINT_PIPEMGR("%s:%s:%i: items are equal\n",
			DEBUG_INFO);
	return 1;
}	/* _gpfPipelineItemIsEqual */



double _gpfPipelineItemGetRuntime(struct gpf_pipeline_item_t *pipe)
{
	LIB_ASSERT(pipe != NULL);
	LIB_ASSERT(pipe->runtime > 0);

	return pipe->runtime;
}	/* _gpfPipelineItemGetRuntime */


size_t _gpfPipelineItemGetMemoryUsage(struct gpf_pipeline_item_t *pipe)
{
	LIB_ASSERT(pipe != NULL);
	LIB_ASSERT(pipe->memory_usage > 0);

	return pipe->memory_usage;
}	/* _gpfPipelineItemGetMemoryUsage */





int _gpfPipelineItemHasAllVariants(struct gpf_pipeline_item_t *pipe, int chain)
{
	int flags = 0;

	LIB_ASSERT(pipe != NULL);

	switch (chain) {
	case INFOLIST_INPUT:
		if (PIPELINEITEM_MARK(pipe, INFOLIST_OUTPUT) == NULL) {
			return 0;
		}	/* if */

		flags = PIPELINEITEM_MARK(pipe, INFOLIST_OUTPUT)->pf.flags;
		break;

	case INFOLIST_OUTPUT:
		if (PIPELINEITEM_MARK(pipe, INFOLIST_INPUT) == NULL) {
			return 0;
		}	/* if */
		flags = PIPELINEITEM_MARK(pipe, INFOLIST_INPUT)->pf.flags;
		break;

	default:
		gpfPanic("%s:%s:%i: Unknown chain: %i\n",
			DEBUG_INFO, chain);
	}	/* switch */

	if (flags & GPF_PIPE_DF_ALLVARIANTS) {
		GPFDPRINT_PIPEMGR("pipe: %p on chain %i has allvariants\n",
				pipe, chain);
		return 1;
	}	/* if */

	return 0;
}	/* _gpfPipelineItemHasAllVariants */



int _gpfPipelineItemMakesSense(struct gpf_pipeline_item_t *source,
				struct gpf_pipeline_item_t *sink)
{
	int tmp1, tmp2;

	tmp1 = _gpfPipelineItemHasAllVariants(source, INFOLIST_OUTPUT);
	tmp2 = _gpfPipelineItemHasAllVariants(sink, INFOLIST_INPUT);

	if (!(tmp1 && tmp2)) return 1;

	if (source->prev != NULL) {
		tmp1 = _gpfPipelineItemHasAllVariants(source->prev,
							INFOLIST_OUTPUT);
		if (!tmp1) return 1;

		/* That equals to a raw transfer through
		 * additional pipe sublib
		 */
		GPFDPRINT_PIPEMGR("%s:%s:%i: this pipe item makes no sense!\n",
				DEBUG_INFO);
		return 0;
	}	/* if */
	if (sink->next != NULL) {
		tmp1 = _gpfPipelineItemHasAllVariants(sink->next,
			INFOLIST_INPUT);
		if (!tmp1) return 1;

		/* That equals to a raw transfer through
		 * additional pipe sublib
		 */
		GPFDPRINT_PIPEMGR("%s:%s:%i: this pipe item makes no sense!\n",
				DEBUG_INFO);
		return 0;
	}	/* if */

	return 1;
}	/* _gpfPipelineItemMakesSense */




int  _gpfPipelineItemFits(struct gpf_pipeline_item_t *source,
			struct gpf_pipeline_item_t *sink)
{
	int found = 0;

	struct gpf_infolist_t *index_src = NULL;
	struct gpf_infolist_t *index_sink = NULL;

	LIB_ASSERT(source != NULL);
	LIB_ASSERT(sink != NULL);

	index_src = _gpfInfoListGetMarkPos(source,
				PIPELINEITEM_FITS, INFOLIST_INPUT);
	index_sink = _gpfInfoListGetMarkPos(sink,
				PIPELINEITEM_FITS, INFOLIST_OUTPUT);

	if (index_sink == NULL) {
		index_sink = sink->input_head;
	} else {
		/* continue with next item */
		index_sink = index_sink->next;
	}	/* if */

	if (index_src == NULL) {
		index_src = source->output_head;
	} else {
		/* continue with next item */
		index_src = index_src->next;
		if (index_sink == NULL) index_sink = sink->input_head;
	}	/* if */


	while (index_src != NULL) {
		while (index_sink != NULL) {

			found = _gpfCompareEntries(index_src, index_sink);
			if (found) {
				_gpfInfoListSetMarkPos(source, index_src,
						PIPELINEITEM_FITS,
						INFOLIST_INPUT);
				_gpfInfoListSetMarkPos(sink, index_sink,
						PIPELINEITEM_FITS,
						INFOLIST_OUTPUT);
				GPFDPRINT_PIPEMGR("items fits together\n");

				if (_gpfPipelineItemMakesSense(source, sink)) {
					/* items fits together */
					return 1;
				}	/* if */

				return 0;
			}	/* if */

			index_sink = index_sink->next;
		}	/* while */

		index_sink = sink->input_head;
		index_src = index_src->next;
	}	/* while */

	/* items don't fit */

	LIB_ASSERT(found == 0);

	_gpfInfoListClearMarkPos(source,
			PIPELINEITEM_FITS, INFOLIST_INPUT);
	_gpfInfoListClearMarkPos(sink,
			PIPELINEITEM_FITS, INFOLIST_OUTPUT);

	return 0;
}	/* _gpfPipelineItemFits */



enum gpf_pipe_sublib_t {

	GPF_SUBLIB_RGBA		= 0,
	GPF_SUBLIB_YUV		= 1,

	GPF_SUBLIB_MAXNUM	= 2,
	GPF_SUBLIB_UNDEFINED	= GPF_SUBLIB_MAXNUM+1,
};

static char *sublibname[GPF_SUBLIB_MAXNUM] = {

	"rgba",
	"yuv",
};


int _gpfPipelineItemFindString(char *pipe_string,
			struct gpf_pipeline_item_t *prev,
			struct gpf_pipeline_item_t *next)
{
	struct gpf_infolist_t *prev_idx = NULL;
	struct gpf_infolist_t *next_idx = NULL;

	struct gpf_pixelformat_t *pf_input = NULL;
	struct gpf_pixelformat_t *pf_output = NULL;
	enum gpf_pipe_sublib_t input_sublib, output_sublib;

	LIB_ASSERT(pipe_string != NULL);
	LIB_ASSERT(prev != NULL);
	LIB_ASSERT(next != NULL);
	LIB_ASSERT(prev->output_head != NULL);
	LIB_ASSERT(next->input_head != NULL);


	prev_idx = _gpfInfoListGetMarkPos(prev,
				PIPELINEITEM_FINDSTRING, INFOLIST_INPUT);
	next_idx = _gpfInfoListGetMarkPos(next,
				PIPELINEITEM_FINDSTRING, INFOLIST_OUTPUT);

	if (next_idx == NULL) {
		next_idx = next->input_head;
	} else {
		/* continue with next item */
		next_idx = next_idx->next;
	}	/* if */

	if (prev_idx == NULL) {
		prev_idx = prev->output_head;
	} else {
		/* continue with next item */
		prev_idx = prev_idx->next;
		if (next_idx == NULL) next_idx = next->input_head;
	}	/* if */


	while (prev_idx != NULL) {

		pf_output = &prev_idx->pf;
		while (next_idx != NULL) {

			pf_input = &next_idx->pf;

			input_sublib = GPF_SUBLIB_UNDEFINED;
			output_sublib = GPF_SUBLIB_UNDEFINED;

			switch (pf_input[0].cf) {
			case GPF_CF_RGBA:
				input_sublib = GPF_SUBLIB_RGBA;
				break;
			case GPF_CF_YUV:
				input_sublib = GPF_SUBLIB_YUV;
				break;
			default:
				input_sublib = GPF_SUBLIB_UNDEFINED;
				break;
			}	/* switch */


			switch (pf_output[0].cf) {
			case GPF_CF_RGBA:
				output_sublib = GPF_SUBLIB_RGBA;
				break;
			case GPF_CF_YUV:
				output_sublib = GPF_SUBLIB_YUV;
				break;
			default:
				output_sublib = GPF_SUBLIB_UNDEFINED;
				break;
			}	/* switch */

			if ((input_sublib == GPF_SUBLIB_UNDEFINED)
			   || (output_sublib == GPF_SUBLIB_UNDEFINED))
			{
				next_idx = next_idx->next;
				continue;
			}	/* if */


			if (input_sublib == output_sublib) {
				if (_gpfCompareEntries(next_idx, prev_idx)) {
					GPFDPRINT_PIPEMGR("using found pipe "
						"sublib would result in raw "
						"transfer ==> through away\n");

					/* raw transfer through an _additional_ pipe sublib
					 * never makes sense, because:
					 * - it never speeds a transfer up
					 * - it never reduces the needed amount of memory
					 * => search for another alternate
					 */
					next_idx = next_idx->next;
					continue;
				}	/* if */
			}	/* if */


			/* add "pipe-" prefix to string */
#ifdef HAVE_SNPRINTF
			snprintf(pipe_string, GGI_MAX_APILEN, "pipe-%s2%s",
				sublibname[input_sublib],
				sublibname[output_sublib]);
#else
			sprintf(pipe_string, "pipe-%s2%s",
				sublibname[input_sublib],
				sublibname[output_sublib]);
#endif

			_gpfInfoListSetMarkPos(prev, prev_idx,
					PIPELINEITEM_FINDSTRING,
					INFOLIST_INPUT);
			_gpfInfoListSetMarkPos(next, next_idx,
					PIPELINEITEM_FINDSTRING,
					INFOLIST_OUTPUT);


			GPFDPRINT_PIPEMGR("pipe string = %s\n",
					pipe_string);
			return 1;
		}	/* while */
		if (next_idx == NULL) next_idx = next->input_head;
		prev_idx = prev_idx->next;

	}	/* while */

	GPFDPRINT_PIPEMGR("no (more) suitable string found\n");

	_gpfInfoListClearMarkPos(prev,
			PIPELINEITEM_FINDSTRING, INFOLIST_INPUT);
	_gpfInfoListClearMarkPos(next,
			PIPELINEITEM_FINDSTRING, INFOLIST_OUTPUT);

	return 0;
}	/* _gpfPipelineItemFindString */



struct gpf_pipeline_item_t *_gpfPipelineItemClone(
			struct gpf_pipeline_item_t *pipe)
{
	int rc;
	struct gpf_pipeline_item_t *pipe_new = NULL;

	LIB_ASSERT(pipe != NULL);

	if (_gpfPipelineItemIsTarget(pipe)) {
		pipe_new = _gpfPipelineItemCreate();
		if (!pipe_new) {
			/* no mem */
			goto err0;
		}	/* if */

		memcpy(pipe_new, pipe,
			sizeof(struct gpf_pipeline_item_t));
		pipe_new->next = pipe_new->prev = NULL;
	} else {
		pipe_new = _gpfOpenPipe(pipe->pipe_string);
		if (!pipe_new) {
			/* no mem */
			goto err0;
		}	/* if */

		LIB_ASSERT(pipe->prev != NULL);
		LIB_ASSERT(pipe->next != NULL);
		pipe_new->getformat(pipe_new, pipe->prev, INFOLIST_OUTPUT);
		pipe_new->getformat(pipe_new, pipe->next, INFOLIST_INPUT);

		rc = pipe_new->getinfolist(pipe_new);
		if (rc != GGI_OK) {
			goto err1;
		}	/* if */
	}	/* if */

	return pipe_new;

err1:
	_gpfPipelineItemDestroy(&pipe_new);
err0:
	LIB_ASSERT(pipe_new == NULL);
	return NULL;
}	/* _gpfPipelineItemClone */








/********************************
 * Pipeline
 */

struct gpf_pipeline_t *_gpfPipelineCreate(void)
{
	struct gpf_pipeline_t *pipeline = NULL;

	pipeline = calloc(1, sizeof(struct gpf_pipeline_t));

	return pipeline;
}	/* _gpfPipelineCreate */


int _gpfPipelineDestroy(struct gpf_pipeline_t **pipeline_head,
			struct gpf_pipeline_t *except_pipeline)
{
	int rc = GGI_OK;
	struct gpf_pipeline_t *pipeline_idx = NULL;
	struct gpf_pipeline_t *pipeline_tmp = NULL;

	struct gpf_pipeline_item_t *item_idx = NULL;
	struct gpf_pipeline_item_t *item_tmp = NULL;

	LIB_ASSERT(pipeline_head != NULL);

	if (pipeline_head[0] == NULL) {
		goto exit;
	}	/* if */

	pipeline_idx = pipeline_head[0];
	while (pipeline_idx != NULL) {

		/* skip the pipeline we don't destroy */
		if (pipeline_idx == except_pipeline) {
			pipeline_idx = pipeline_idx->next;
			continue;
		}	/* if */

		pipeline_tmp = pipeline_idx->next;

		item_idx = pipeline_idx->item_head;
		while (item_idx != NULL) {
			item_tmp = item_idx->next;
			rc = _gpfPipelineItemDestroy(&item_idx);
			LIB_ASSERT(rc == GGI_OK);

			item_idx = item_tmp;
		}	/* while */

		free(pipeline_idx);
		pipeline_idx = pipeline_tmp;
	}	/* for */

	pipeline_head[0] = except_pipeline;
	if (except_pipeline != NULL) {
		except_pipeline->next = NULL;
	}	/* if */

exit:
	LIB_ASSERT(rc == GGI_OK);
	return rc;
}	/* _gpfPipelineDestroy */


int _gpfPipelineAppend(struct gpf_pipeline_t *pipeline_head,
			struct gpf_pipeline_t *pipeline_append)
{
	struct gpf_pipeline_t *index = NULL;

	LIB_ASSERT(pipeline_head != NULL);
	LIB_ASSERT(pipeline_append != NULL);

	/* goto last item/element */
	_gpfPipelineGotoLastOne(pipeline_head, index);
	LIB_ASSERT(index != NULL);
	LIB_ASSERT(index->next == NULL);

	index->next = pipeline_append;
	LIB_ASSERT(pipeline_append->next == NULL);

	return GGI_OK;
}	/* _gpfPipelineAppend */


int  _gpfPipelineRemoveItem(struct gpf_pipeline_t *pipeline,
				struct gpf_pipeline_item_t *pipe)
{
	int rc = GGI_OK;
	struct gpf_pipeline_item_t *prev_pipe = NULL;

	LIB_ASSERT(pipeline != NULL);
	LIB_ASSERT(pipe != NULL);


	if (pipeline->item_head == pipe) {
		pipeline->item_head = pipe->next;
		pipe->next = NULL;
		return rc;
	}	/* if */

	for_each_pipeline_item_c(pipeline,
				prev_pipe->next != pipe,
				prev_pipe);
	LIB_ASSERT(prev_pipe != NULL);

	rc = _gpfPipelineItemRemove(prev_pipe, pipe);
	LIB_ASSERT(rc == GGI_OK);

	return rc;
}	/* _gpfPipelineRemoveItem */


int _gpfPipelineIsEqual(struct gpf_pipeline_t *pipeline1,
			struct gpf_pipeline_t *pipeline2)
{
	struct gpf_pipeline_item_t *index1 = NULL;
	struct gpf_pipeline_item_t *index2 = NULL;

	LIB_ASSERT(pipeline1 != NULL);
	LIB_ASSERT(pipeline2 != NULL);

	if (pipeline1 == pipeline2) {
		GPFDPRINT_PIPEMGR("%s:%s:%i: pointer equals\n",
			DEBUG_INFO);
		return 1;
	}	/* if */

	index1 = pipeline1->item_head;
	index2 = pipeline2->item_head;

	while ((index1 != NULL) && (index2 != NULL)) {

		if (!_gpfPipelineItemIsEqual(index1, index2)) {
			GPFDPRINT_PIPEMGR("%s:%s:%i: don't equals\n",
					DEBUG_INFO);
			return 0;
		}	/* if */

		index1 = index1->next;
		index2 = index2->next;
	}	/* while */

	if ((index1 == NULL) && (index2 == NULL)) {
		GPFDPRINT_PIPEMGR("%s:%s:%i: equals\n",
				DEBUG_INFO);
		return 1;
	}	/* if */

	if (index1 == NULL) {
		GPFDPRINT_PIPEMGR("%s:%s:%i: don't equals\n",
				DEBUG_INFO);
		return 0;
	}	/* if */
	if (index2 == NULL) {
		GPFDPRINT_PIPEMGR("%s:%s:%i: don't equals\n",
				DEBUG_INFO);
		return 0;
	}	/* if */

	GPFDPRINT_PIPEMGR("%s:%s:%i: equals\n", DEBUG_INFO);
	return 1;
}	/* _gpfPipelineIsEqual */


int _gpfPipelineIsUnique(struct gpf_pipeline_t *pipeline_head,
                        struct gpf_pipeline_t *pipeline)
{
	struct gpf_pipeline_t *index = NULL;

	LIB_ASSERT(pipeline != NULL);

	if (pipeline_head == NULL) {
		/* It's assumed here, that pipeline
		 * is the first pipeline being created.
		 * So it's always unique.
		 */
		GPFDPRINT_PIPEMGR("first pipeline (%p) is always unique :)\n");
		return 1;
	}	/* if */

	LIB_ASSERT(pipeline_head != NULL);

	for_each_pipeline(pipeline_head, index) {

		if (index == pipeline) continue;

		/* compare pipelines */
		if (_gpfPipelineIsEqual(pipeline, index)) {
			GPFDPRINT_PIPEMGR("pipeline (%p) is not unique\n",
					pipeline);
			return 0;
		}	/* if */
	}	/* for */

	GPFDPRINT_PIPEMGR("pipeline (%p) is unique\n", pipeline);
	return 1;
}	/* _gpfPipelineIsUnique */



double _gpfPipelineComputeRuntime(struct gpf_pipeline_t *pipeline)
{
	double runtime = 0.0;
	struct gpf_pipeline_item_t *index = NULL;

	for_each_pipeline_item(pipeline, index) {
		runtime += _gpfPipelineItemGetRuntime(index);
	}	/* for */

	GPFDPRINT_PIPEMGR("pipeline (%p) has a runtime of %lf\n",
			pipeline, runtime);

	return runtime;
}	/* _gpfPipelineComputeRuntime */


size_t _gpfPipelineGetMemoryUsage(struct gpf_pipeline_t *pipeline)
{
	size_t memory_usage = 0;
	struct gpf_pipeline_item_t *index = NULL;

	for_each_pipeline_item(pipeline, index) {
		memory_usage += _gpfPipelineItemGetMemoryUsage(index);
	}	/* for */

	GPFDPRINT_PIPEMGR("pipeline (%p) requires %lf bytes memory\n",
			pipeline, memory_usage);

	return memory_usage;
}	/* _gpfPipelineGetMemoryUsage */



struct gpf_pipeline_t *_gpfPipelineSelect(struct gpf_pipeline_t *pipeline_head,
					struct gpf_properties_t *props)
{
	struct gpf_pipeline_t *index = NULL;
	struct gpf_pipeline_t *best_pipeline = NULL;


	double diff_ratio, best_diff_ratio;
	double runtime, best_runtime;
	size_t memory_usage, best_memory_usage;


	LIB_ASSERT(pipeline_head != NULL);
	LIB_ASSERT(props != NULL);

	diff_ratio = best_diff_ratio = 0.0;

	/* maximum positive value */
	memory_usage = best_memory_usage = (~0) >> 1;

	runtime = best_runtime = (double)memory_usage;


	switch (props->flags & GPF_PIPE_MASK) {
	case (GPF_PIPE_OPT4MEMORY | GPF_PIPE_OPT4SPEED):

		GPFDPRINT_PIPEMGR("searching for proper speed<->memory ratio\n");
		if (props->speed_memory_ratio < 1E-5) {
			fprintf(stderr, "%s:%s:%i: no proper speed<->memory"
				"optimization ratio given\n",
				DEBUG_INFO);
			return NULL;
		}	/* if */

		for_each_pipeline(pipeline_head, index) {

			runtime = _gpfPipelineComputeRuntime(index);
			memory_usage = _gpfPipelineGetMemoryUsage(index);
			diff_ratio = runtime / memory_usage;

			diff_ratio = fabs(props->speed_memory_ratio
					- diff_ratio);

			if (diff_ratio < best_diff_ratio) {
				best_diff_ratio = diff_ratio;
				best_pipeline = index;
			}	/* if */
		}	/* for */
		break;


	case (GPF_PIPE_OPT4MEMORY):

		GPFDPRINT_PIPEMGR("searching for pipeline requiring the less memory\n");
		for_each_pipeline(pipeline_head, index) {

			runtime = _gpfPipelineComputeRuntime(index);
			memory_usage = _gpfPipelineGetMemoryUsage(index);

			if (memory_usage < best_memory_usage) {
				best_memory_usage = memory_usage;
				best_pipeline = index;
			}	/* if */
		}	/* for */
		break;


	case (GPF_PIPE_OPT4SPEED):

		GPFDPRINT_PIPEMGR("searching for the fastest pipeline\n");
		for_each_pipeline(pipeline_head, index) {

			runtime = _gpfPipelineComputeRuntime(index);
			memory_usage = _gpfPipelineGetMemoryUsage(index);

			if (runtime < best_runtime) {
				best_runtime = runtime;
				best_pipeline = index;
			}	/* if */
		}	/* for */
		break;

	default:
		/* Unknown flags */
		fprintf(stderr, "%s:%s:%i: Unknown flags: %X (%X)\n",
			DEBUG_INFO, props->flags & GPF_PIPE_MASK,
			props->flags);
		return NULL;
	}	/* switch */


	return best_pipeline;
}	/* _gpfPipelineSelect */



struct gpf_pipeline_item_t *_gpfPipelineFindSuitableItem(
					struct gpf_pipeline_t *pipeline,
					struct gpf_pipeline_item_t *prev,
					struct gpf_pipeline_item_t *next)
{
	struct gpf_pipeline_item_t *pipe = NULL;
	char pipe_string[GGI_MAX_APILEN];
	int found;


	LIB_ASSERT(pipeline != NULL);
	LIB_ASSERT(prev != NULL);
	LIB_ASSERT(next != NULL);

	found = _gpfPipelineItemFindString(pipe_string, prev, next);
	if (!found) {
		/* no matching pipe available */
		goto err0;
	}	/* if */

	GPFDPRINT_PIPEMGR("opening pipe = %s\n", pipe_string);
	pipe = _gpfOpenPipe(pipe_string);
	if (!pipe) {
		goto err0;
	}	/* if */

	return pipe;

err0:
	return NULL;
}	/* _gpfPipelineFindSuitableItem */


struct gpf_pipeline_t *_gpfPipelineClone(struct gpf_pipeline_t *pipeline)
{
	int rc;
	struct gpf_pipeline_t *pipeline_new = NULL;
	struct gpf_pipeline_item_t *index = NULL;
	struct gpf_pipeline_item_t *pipe_new = NULL;

	LIB_ASSERT(pipeline != NULL);

	pipeline_new = _gpfPipelineCreate();
	if (!pipeline_new) {
		/* no mem */
		goto err0;
	}	/* if */

	for_each_pipeline_item(pipeline, index) {
		pipe_new = _gpfPipelineItemClone(index);
		if (!pipe_new) {
			/* no mem */
			goto err1;
		}	/* if */

		rc = _gpfPipelineItemAppend(pipeline_new, pipe_new);
		LIB_ASSERT(rc == GGI_OK);
	}	/* for */

	return pipeline_new;

err1:
	_gpfPipelineDestroy(&pipeline_new, NULL);
err0:
	LIB_ASSERT(pipeline_new == NULL);
	return NULL;
}	/* _gpfPipelineClone */


struct gpf_pipeline_t *_gpfPipelineFind(gpf_handle_t handle,
					struct gpf_pipeline_t *head)
{
	int rc;
	int found;
	struct gpf_pipeline_t *pipeline = NULL;

	struct gpf_pipeline_item_t *source = NULL;
	struct gpf_pipeline_item_t *sink = NULL;

	struct gpf_pipeline_item_t *item_idx = NULL;
	struct gpf_pipeline_item_t *pipe = NULL;


	LIB_ASSERT(handle != NULL);

	if (head == NULL) {
		GPFDPRINT_PIPEMGR("Try to build the first pipeline\n");
		GPFDPRINT_PIPEMGR("First step is to clone the targets\n");

		/* ah - we create the first pipeline at all
		 */
		pipeline = _gpfPipelineCreate();
		if (!pipeline) {
			rc = GGI_ENOMEM;
			goto err0;
		}	/* if */

		/* The input target sends data through the pipeline.
		 * So the input target shows the pipeline, what
		 * it can offer as its "output".
		 */

		rc = handle->opio_input->pipe_getinfolist(handle->opio_input);
		LIB_ASSERT(rc == GGI_OK);
		if (rc != GGI_OK) {
			goto err1;
		}	/* if */

		/* The output target revieves data from a pipe.
		 * So the output target shows the pipeline, what
		 * it can handle as its "input".
		 */

		rc = handle->opio_output->pipe_getinfolist(handle->opio_output);
		LIB_ASSERT(rc == GGI_OK);
		if (rc != GGI_OK) {
			goto err1;
		}	/* if */

		source = _gpfPipelineItemClone(handle->opio_input->pipe);
		if (!source) {
			rc = GGI_ENOMEM;
			goto err2;
		}	/* if */

		sink = _gpfPipelineItemClone(handle->opio_output->pipe);
		if (!sink) {
			rc = GGI_ENOMEM;
			goto err3;
		}	/* if */

		LIB_ASSERT(_gpfPipelineItemIsTarget(source) != 0);
		LIB_ASSERT(_gpfPipelineItemIsTarget(sink) != 0);

		pipeline->item_head = source;

		rc = _gpfPipelineItemAppend(pipeline, sink);
		LIB_ASSERT(rc == GGI_OK);

	} else {
		struct gpf_pipeline_t *index = NULL;

		GPFDPRINT_PIPEMGR("Clone previous found pipeline\n");

		/* We clone the last created pipeline, so that
		 * we can continue searching for another pipeline
		 * from the last point.
		 */

		_gpfPipelineGotoLastOne(head, index);
		pipeline = _gpfPipelineClone(index);
		if (!pipeline) {
			rc = GGI_ENOMEM;
			goto err0;
		}	/* if */

		source = pipeline->item_head;
		_gpfPipelineItemGotoLastOne(pipeline, item_idx);
		sink = item_idx;

	}	/* if */


	found = 0;
	item_idx = pipeline->item_head;
	while (item_idx->next != NULL) {

		GPFDPRINT_PIPEMGR("compare items %p with %p\n",
				item_idx, item_idx->next);
		found = _gpfPipelineItemFits(item_idx, item_idx->next);
		if (found) {
			item_idx = item_idx->next;
			continue;
		}	/* if */

		GPFDPRINT_PIPEMGR("need to find a suitable pipeitem\n");
		pipe = _gpfPipelineFindSuitableItem(pipeline,
						item_idx, item_idx->next);
		if (!pipe) {
			/* no suitable pipe item found */
			GPFDPRINT_PIPEMGR("no suitable pipeitem found\n");
			break;
		}	/* if */

		rc = _gpfPipelineItemInsert(item_idx, pipe);
		LIB_ASSERT(rc == GGI_OK);

		LIB_ASSERT(pipe->prev != NULL);
		LIB_ASSERT(pipe->next != NULL);

		pipe->getformat(pipe, pipe->prev, INFOLIST_OUTPUT);
		pipe->getformat(pipe, pipe->next, INFOLIST_INPUT);

		/* Ok, here we can assume, that there are no nonsense
		 * combinations.
		 */
		pipe->prev->getformat(pipe->prev, pipe,
					INFOLIST_INPUT);

		pipe->getinfolist(pipe);

		if (!_gpfPipelineIsUnique(head, pipeline)) {
			_gpfPipelineItemRemove(item_idx, pipe);
			GPFDPRINT_PIPEMGR("pipeline (%p) is not unique - retry\n",
					pipeline);

			_gpfPipelineItemDestroy(&pipe);

			continue;
		}	/* if */


		LIB_ASSERT(item_idx->next != NULL);
		LIB_ASSERT(item_idx->next->next != NULL);

		_gpfInfoListClearMarkPos(item_idx,
				PIPELINEITEM_FITS, INFOLIST_INPUT);
		_gpfInfoListClearMarkPos(item_idx->next->next,
				PIPELINEITEM_FITS, INFOLIST_OUTPUT);


		/* don't go one step ahead to force a recheck,
		 * that the found sublib actually fits
		 */
	}	/* while */

	if (!found) goto err1;

	GPFDPRINT_PIPEMGR("  found pipeline = %p\n", pipeline);
	LIB_ASSERT(found != 0);
	return pipeline;

err3:
	_gpfPipelineItemDestroy(&sink);
err2:
	_gpfPipelineItemDestroy(&source);
err1:
	_gpfPipelineDestroy(&pipeline, NULL);
err0:
	GPFDPRINT_PIPEMGR("no (more) pipeline found\n");
	LIB_ASSERT(pipeline == NULL);
	return NULL;
}	/* _gpfPipelineFind */



struct gpf_pipeline_t *_gpfPipelineFindOptimal(gpf_handle_t handle)
{
	int rc;
	struct gpf_pipeline_t *head = NULL;
	struct gpf_pipeline_t *pipeline = NULL;


	LIB_ASSERT(handle != NULL);

	head = _gpfPipelineFind(handle, NULL);
	if (head == NULL) {
		GPFDPRINT_PIPEMGR("no pipeline found! Transfer impossible!\n");
		/* very bad: absolutely no pipeline found
		 * => no transfer possible
		 */
		return NULL;
	}	/* if */

	do {
		pipeline = _gpfPipelineFind(handle, head);
		if (pipeline != NULL) {
			LIB_ASSERT(pipeline->next == NULL);
			_gpfPipelineAppend(head, pipeline);
		}	/* if */
	} while (pipeline != NULL);


	GPFDPRINT_PIPEMGR("select proper pipeline\n");
	/* Select the right pipeline in dependence of some flags
	 * being set.
	 */
	LIB_ASSERT(head != NULL);
	pipeline = _gpfPipelineSelect(head, &(LIBGPF_PROPS(handle)));
	LIB_ASSERT(pipeline != NULL);

	GPFDPRINT_PIPEMGR("optimal pipeline = %p - this one will be used\n",
			pipeline);
	GPFDPRINT_PIPEMGR("destroy all other pipelines\n");

	/* Destroy all pipelines, except 'pipeline' */
	rc = _gpfPipelineDestroy(&head, pipeline);
	LIB_ASSERT(rc == GGI_OK);

	/* Hint: If a crash occurs here, then the pipeline was
	 * accidently freed and not set to NULL.
	 * pipeline shouldn't be freed here at all!
	 */
	LIB_ASSERT(pipeline != NULL);
	return pipeline;
}	/* _gpfPipelineFindOptimal */



int gpfPipelineSetup(gpf_handle_t handle)
{
	int rc = GGI_OK;
	struct gpf_pipeline_t *pipeline = NULL;
	struct gpf_pipeline_item_t *index = NULL;

	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(handle->pipeline == NULL);

	if ((handle->opio_output == NULL)
	   || (handle->opio_input == NULL))
	{
		rc = GGI_ENOMATCH;
		goto err0;
	}	/* if */

	pipeline = _gpfPipelineFindOptimal(handle);
	if (!pipeline) {
		rc = GGI_ENOMATCH;
		goto err0;
	}	/* if */

	LIB_ASSERT(pipeline != NULL);
	LIB_ASSERT(handle->pipeline == NULL);

	pipeline->read = _gpfPipelineItemRead;
	pipeline->write = _gpfPipelineItemWrite;

	handle->pipeline = pipeline;
	handle->pipe_index = handle->pipeline->item_head;

	GPFDPRINT_PIPEMGR("initalize pipeline\n");
	for_each_pipeline_item(pipeline, index) {
		rc = index->setup(handle, index);
		if (rc < 0) {
			goto err1;
		}	/* if */

		_gpfInfoListDestroy(&index->input_head);
		_gpfInfoListDestroy(&index->output_head);

		_gpfInfoListClearMarkPos(index, PIPELINEITEM_FITS, INFOLIST_INPUT);
		_gpfInfoListClearMarkPos(index, PIPELINEITEM_FITS, INFOLIST_OUTPUT);
		_gpfInfoListClearMarkPos(index, PIPELINEITEM_FINDSTRING, INFOLIST_INPUT);
		_gpfInfoListClearMarkPos(index, PIPELINEITEM_FINDSTRING, INFOLIST_OUTPUT);
	}	/* for */

	LIB_ASSERT(LIBGPF_TS_ISSET(&LIBGPF_TS_TOTAL(handle)));

	GPFDPRINT_PIPEMGR("total playtime (in seconds) to transfer: "
			"%.02f\n",
			(float)(LIBGPF_TS_TOTAL(handle).tv_sec +
			(LIBGPF_TS_TOTAL(handle).tv_usec / 1000000)));

	return GGI_OK;

err1:
	_gpfPipelineDestroy(&pipeline, NULL);
err0:
	return rc;
}	/* gpfPipelineSetup */


int _gpfPipelineShutdown(gpf_handle_t handle)
{
	int rc;
	LIB_ASSERT(handle != NULL);

	rc = _gpfPipelineDestroy(&(handle->pipeline), NULL);
	LIB_ASSERT(rc == GGI_OK);
	LIB_ASSERT(handle->pipeline == NULL);

	GPFDPRINT_PIPEMGR("%s:%s:%i: pipeline management shutdown finished\n",
			DEBUG_INFO);
	return rc;
}	/* _gpfPipelineShutdown */
