/*
 ******************************************************************************

 LIBGPF: Debugging macros

 Copyright (C) 2000-2001 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */


#ifndef _GPF_INTERNAL_DEBUG_H
#define _GPF_INTERNAL_DEBUG_H

#include <stdio.h>
#include <stdarg.h>
#include <ggi/types.h>
#include <ggi/gg.h>
#include <ggi/gpf.h>


__BEGIN_DECLS

/* Exported variables */
#ifdef BUILDING_GPF
extern uint32     _gpfDebugState;
extern int        _gpfDebugSync;
#else
IMPORTVAR uint32  _gpfDebugState;
IMPORTVAR int     _gpfDebugSync;
#endif
__END_DECLS


/* Debugging types
 * bit 0 is reserved! */

#define GPFDEBUG_CORE		(1<<1)	/*   2 */
#define GPFDEBUG_PIPEMGR	(1<<2)	/*   4 */
#define GPFDEBUG_TRANSFER	(1<<3)	/*   8 */
#define GPFDEBUG_TARGET		(1<<4)	/*  16 */
#define GPFDEBUG_PROTO		(1<<5)	/*  32 */
#define GPFDEBUG_LIBS		(1<<6)	/*  64 */
#define GPFDEBUG_CONV		(1<<7)	/* 128 */

#define GPFDEBUG_ALL	0xffffffff

#define DEBUG_INFO      __FILE__,__PRETTY_FUNCTION__,__LINE__

#define FIXME(args...)	ggDPrintf(_gpfDebugSync, "fixme: LIBGPF", args);

#ifdef DEBUG
#define LIB_ASSERT(x)   \
        if (!(x))       \
                fprintf(stderr,"LIBGPF:%s:%s:%d: assertion \"%s\" failed.\n", DEBUG_INFO, #x);

#else
#define LIB_ASSERT(x)	do{}while(0)
#endif


#ifdef __GNUC__

#ifdef DEBUG
#define GPFDPRINT(args...)	  if (_gpfDebugState) { ggDPrintf(_gpfDebugSync, "LibGPF",args); }
#define GPFDPRINT_CORE(args...)	  if (_gpfDebugState & GPFDEBUG_CORE) { ggDPrintf(_gpfDebugSync,"LibGPF",args); }
#define GPFDPRINT_PIPEMGR(args...)  if (_gpfDebugState & GPFDEBUG_PIPEMGR) { ggDPrintf(_gpfDebugSync,"LibGPF",args); }
#define GPFDPRINT_TRANSFER(args...)  if (_gpfDebugState & GPFDEBUG_TRANSFER) { ggDPrintf(_gpfDebugSync,"LibGPF",args); }
#define GPFDPRINT_TARGET(args...) if (_gpfDebugState & GPFDEBUG_TARGET) { ggDPrintf(_gpfDebugSync,"LibGPF",args); }
#define GPFDPRINT_PROTO(args...)  if (_gpfDebugState & GPFDEBUG_PROTO) { ggDPrintf(_gpfDebugSync,"LibGPF",args); }
#define GPFDPRINT_LIBS(args...)	  if (_gpfDebugState & GPFDEBUG_LIBS) { ggDPrintf(_gpfDebugSync,"LibGPF",args); }
#define GPFDPRINT_CONV(args...)	  if (_gpfDebugState & GPFDEBUG_CONV) { ggDPrintf(_gpfDebugSync,"LibGPF",args); }

#else /* DEBUG */
#define GPFDPRINT(args...)		do{}while(0)
#define GPFDPRINT_CORE(args...)		do{}while(0)
#define GPFDPRINT_PIPEMGR(args...)	do{}while(0)
#define GPFDPRINT_TRANSFER(args...)	do{}while(0)
#define GPFDPRINT_TARGET(args...)	do{}while(0)
#define GPFDPRINT_PROTO(args...)	do{}while(0)
#define GPFDPRINT_LIBS(args...)		do{}while(0)
#define GPFDPRINT_CONV(args...)		do{}while(0)

#endif /* DEBUG */

#else /* __GNUC__ */

__BEGIN_DECLS

static inline void GPFDPRINT(const char *form,...)
{
#ifdef DEBUG
	if (_ggiDebugState) {
		va_list args;

		fprintf(stderr, "GPF: ");
		va_start(args, form);
		vfprintf(stderr, form, args);
		va_end(args);
		if (_gpfDebugSync) fflush(stderr);
	}
#endif
}

static inline void GPFDPRINT_CORE(const char *form,...)
{
#ifdef DEBUG
	if (_ggiDebugState & GGIDEBUG_CORE) {
		va_list args;

		fprintf(stderr, "GPF: ");
		va_start(args, form);
		vfprintf(stderr, form, args);
		va_end(args);
		if (_gpfDebugSync) fflush(stderr);
	}
#endif
}

static inline void GPFDPRINT_PIPEMGR(const char *form,...)
{
#ifdef DEBUG
	if (_ggiDebugState & GGIDEBUG_PIPEMGR) {
		va_list args;

		fprintf(stderr, "GPF: ");
		va_start(args, form);
		vfprintf(stderr, form, args);
		va_end(args);
		if (_gpfDebugSync) fflush(stderr);
	}
#endif
}

static inline void GPFDPRINT_TRANSFER(const char *form,...)
{
#ifdef DEBUG
	if (_ggiDebugState & GGIDEBUG_TRANSFER) {
		va_list args;

		fprintf(stderr, "GPF: ");
		va_start(args, form);
		vfprintf(stderr, form, args);
		va_end(args);
		if (_gpfDebugSync) fflush(stderr);
	}
#endif
}

static inline void GPFDPRINT_TARGET(const char *form,...)
{
#ifdef DEBUG
	if (_ggiDebugState & GGIDEBUG_TARGET) {
		va_list args;

		fprintf(stderr, "GPF: ");
		va_start(args, form);
		vfprintf(stderr, form, args);
		va_end(args);
		if (_gpfDebugSync) fflush(stderr);
	}
#endif
}

static inline void GPFDPRINT_PROTO(const char *form,...)
{
#ifdef DEBUG
	if (_ggiDebugState & GGIDEBUG_PROTO) {
		va_list args;

		fprintf(stderr, "GPF: ");
		va_start(args, form);
		vfprintf(stderr, form, args);
		va_end(args);
		if (_gpfDebugSync) fflush(stderr);
	}
#endif
}


static inline void GPFDPRINT_LIBS(const char *form,...)
{
#ifdef DEBUG
	if (_ggiDebugState & GGIDEBUG_LIBS) {
		va_list args;

		fprintf(stderr, "GPF: ");
		va_start(args, form);
		vfprintf(stderr, form, args);
		va_end(args);
		if (_gpfDebugSync) fflush(stderr);
	}
#endif
}

static inline void GPFDPRINT_CONV(const char *form,...)
{
#ifdef DEBUG
	if (_ggiDebugState & GGIDEBUG_CONV) {
		va_list args;

		fprintf(stderr, "GPF: ");
		va_start(args, form);
		vfprintf(stderr, form, args);
		va_end(args);
		if (_gpfDebugSync) fflush(stderr);
	}
#endif
}

__END_DECLS

#endif /* __GNUC__ */

#ifdef DEBUG
#define GPF_ASSERT(x,str) \
{ if (!(x)) { \
	fprintf(stderr,"GPF:%s:%d: INTERNAL ERROR: %s\n",__FILE__,__LINE__,str); \
	exit(1); \
} }
#define GPF_APPASSERT(x,str) \
{ if (!(x)) { \
	fprintf(stderr,"GPF:%s:%d: APPLICATION ERROR: %s\n",__FILE__,__LINE__,str); \
	exit(1); \
} }
#else /* DEBUG */
#define GPF_ASSERT(x,str)	do{}while(0)
#define GPF_APPASSERT(x,str)	do{}while(0)
#endif /* DEBUG */

#ifdef DEBUG
# define GPFD0(x)	x
#else
# define GPFD0(x)	/* empty */
#endif

#ifdef GPFDLEV
# if GPFDLEV == 1
#  define GPFD1(x)	x
#  define GPFD2(x)	/* empty */
#  define GPFD3(x)	/* empty */
# elif GPFDLEV == 2
#  define GPFD1(x)	x
#  define GPFD2(x)	x
#  define GPFD3(x)	/* empty */
# elif GPFDLEV > 2
#  define GPFD1(x)	x
#  define GPFD2(x)	x
#  define GPFD3(x)	x
# endif
#else
# define GPFD1(x)	/* empty */
# define GPFD2(x)	/* empty */
# define GPFD3(x)	/* empty */
#endif

#endif /* _GPF_INTERNAL_DEBUG_H */
