/*
 ******************************************************************************

 LIBGPF: library flags and macros

 Copyright (C) 2001-2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#ifndef _GPF_INTERNAL_TYPES_H
#define _GPF_INTERNAL_TYPES_H


#include <ggi/internal/gpf.h>


/* flags
 */

/* flags for communication between
 * pipe and targets sublibs
 */
enum gpf_pipe_dataformats_t {

	GPF_PIPE_DF_COLOR_MASK		= 0x0000FFFF,

	/* rgba types */
	GPF_PIPE_DF_RGBA		= 0x00000001,
	GPF_PIPE_DF_YUV			= 0x00000002,
	GPF_PIPE_DF_YIQ			= 0x00000003,
	GPF_PIPE_DF_YCbCr		= 0x00000004,

	/* special things */
	GPF_PIPE_DF_SPECIAL_MASK	= 0x0000F000,
	GPF_PIPE_DF_GREYSCALE		= 0x00001000,
	GPF_PIPE_DF_RAW			= 0x00002000,
	GPF_PIPE_DF_ALLVARIANTS		= 0x00004000,


	/* sound data */

	/* means data contains non-graphic stuff like sound,
	 * which is the case on videos, for example
	 */
	GPF_PIPE_DF_SOUND_MASK		= 0x00FF0000,

	/* audio only formats */
	GPF_PIPE_DF_SOUND_WAV		= 0x00110000,
	GPF_PIPE_DF_SOUND_MP3		= 0x00120000,

	/* video only formats */
	GPF_PIPE_DF_SOUND_FLI		= 0x00210000,
	GPF_PIPE_DF_SOUND_FLC		= 0x00220000,

	/* video with sound */
	GPF_PIPE_DF_SOUND_AVI		= 0x00410000,
	GPF_PIPE_DF_SOUND_QUICKTIME	= 0x00420000,
	GPF_PIPE_DF_SOUND_MPEG1		= 0x00430000,
	GPF_PIPE_DF_SOUND_MPEG2		= 0x00440000,
	GPF_PIPE_DF_SOUND_MPEG4		= 0x00450000,


	/* pipe communication bits */
	GPF_PIPE_DF_COMM_MASK		= 0xFF000000,
	GPF_PIPE_DF_NOTHING		= 0x01000000,
	GPF_PIPE_DF_GIVEMEINFORMATION	= 0x02000000,
	GPF_PIPE_DF_NOMOREALTERNATIVES	= 0x04000000,
	GPF_PIPE_DF_UNKNOWN		= 0x80000000,
};



/* convenient macros
 */

/* timestamps */
#define LIBGPF_TS_ISSET(ts)		((ts)->tv_sec || (ts)->tv_usec)
#define LIBGPF_TS_CMP(ts1, ts2, cmp)	(((ts1)->tv_sec cmp (ts2)->tv_sec) ||	\
					((ts1)->tv_sec == (ts2)->tv_sec &&	\
					(ts1)->tv_usec cmp (ts2)->tv_usec))

#define LIBGPF_TS_GTZ(ts)		(((ts)->tv_sec >= 0) && ((ts)->tv_usec >= 0))
#define LIBGPF_TS_LTZ(ts)		(((ts)->tv_sec <= 0) && ((ts)->tv_usec <= 0))


#define LIBGPF_TS_CLEAR(ts)		(ts)->tv_sec = (ts)->tv_usec = 0;


#define LIBGPF_TS_TRANSFER(handle)	(handle)->pipeline->transfer_timestamps
#define LIBGPF_TS_CURRENT(handle)	(handle)->pipeline->current_timestamp
#define LIBGPF_TS_TOTAL(handle)		(handle)->pipeline->total_timestamps
#define LIBGPF_TS_MARK(handle)		(handle)->pipeline->tv_mark



#define GPF_MAXPARAM_LEN       512

#define LIBGPF_PF_TYPE(pf)		((pf)->type)
#define LIBGPF_PF_RGBA(pf)		LIBGPF_PF_TYPE(pf).rgba
#define LIBGPF_PF_YUV(pf)		LIBGPF_PF_TYPE(pf).yuv
#define LIBGPF_PF_YIQ(pf)		LIBGPF_PF_TYPE(pf).yiq
#define LIBGPF_PF_YCBCR(pf)		LIBGPF_PF_TYPE(pf).ycbcr

#define ByPP(size)			((size) / 8)
#define LIBGPF_PF_ByPP(pf)		(ByPP(pf->size))


/* pipeline macros */
#define PIPE_READBUF(pipe)		(pipe)->buf
#define PIPE_READBUFSIZE(pipe)		(pipe)->bufsize
#define PIPE_READBUFPOS(pipe)		(pipe)->read_buf_pos
#define PIPE_WRITEBUF(pipe)		(pipe)->next->buf
#define PIPE_WRITEBUFSIZE(pipe)		(pipe)->next->bufsize
#define PIPE_WRITEBUFPOS(pipe)		(pipe)->write_buf_pos

#define PIPE_READCACHESIZE(pipe)	(pipe)->cache_size
#define PIPE_WRITECACHESIZE(pipe)	(pipe)->next->cache_size


#define LIMIT_TRANSFERSIZE					\
	if (transfer_size > PIPE_READBUFSIZE(pipe)) {		\
		transfer_size = PIPE_READBUFSIZE(pipe);		\
	}	/* if */


#define GET_TARGET_TRANSFERSIZE					\
	if (PIPE_ISLASTTRANSFER(handle, pipe)) {		\
		transfer_size = PIPE_READCACHESIZE(pipe);	\
	} else {						\
		if (PIPE_WRITEBUFPOS(pipe->prev) >= transfer_size) {	\
			transfer_size += PIPE_READCACHESIZE(pipe);	\
		} else {					\
			transfer_size = PIPE_WRITEBUFPOS(pipe->prev)	\
				+ PIPE_READCACHESIZE(pipe);	\
		}	/* if */				\
	}	/* if */


#define GET_PIPE_TRANSFERSIZE					\
	if (PIPE_ISLASTTRANSFER(handle, pipe->prev)) {		\
		transfer_size = pipe->prev->writebytes		\
				+ PIPE_READCACHESIZE(pipe);	\
	} else {						\
		if (PIPE_READCACHESIZE(pipe) > 0) {		\
			transfer_size = PIPE_READCACHESIZE(pipe);	\
		} else {					\
			handle->pipe_index = handle->pipe_index->prev;	\
			return 0;				\
		}	/* if */				\
	}	/* if */


#define REALLOC_BUFCONVSIZE(size)				\
	if (pipe->buf_conv_size != size) {			\
		void *buf_tmp;					\
								\
		buf_tmp = realloc(pipe->buf_conv, size);	\
		if (!buf_tmp) {					\
			free(pipe->buf_conv);			\
			pipe->buf_conv = NULL;			\
			pipe->buf_conv_size = 0;		\
			return GGI_ENOMEM;			\
		}	/* if */				\
		pipe->buf_conv = buf_tmp;			\
		pipe->buf_conv_size = size;			\
	}	/* if */



#define PIPE_READCACHEDATA(pipe, size)				\
	memmove(PIPE_READBUF(pipe), PIPE_READBUF(pipe) +	\
		(PIPE_READBUFPOS(pipe) - 			\
		(pipe->readbytes - pipe->writebytes)), size);	\
	PIPE_READCACHESIZE(pipe) = size;

#define PIPE_WRITECACHEDATA(pipe, size)				\
	memmove(PIPE_WRITEBUF(pipe), PIPE_WRITEBUF(pipe) +	\
		(PIPE_WRITEBUFPOS(pipe) -			\
		(pipe->readbytes - pipe->writebytes)), size);	\
	PIPE_WRITECACHESIZE(pipe) = size;

#define PIPE_ISLASTTRANSFER(handle, pipe)			\
	((handle)->pipe_lasttransfer == (pipe))

#define PIPELINEITEM_MARK(pipe, chain)	pipe->mark[PIPELINEITEM_FITS][chain]
#define PIPELINEITEM_STRMARK(pipe, chain)	pipe->mark[PIPELINEITEM_FINDSTRING][chain]



/* handle macros */

/* properties */
#define LIBGPF_PROPS(hnd)		((hnd)->props)
#define LIBGPF_RATIO(hnd)		(LIBGPF_PROPS(hnd).speed_memory_ratio)
#define LIBGPF_RATE(hnd)		(LIBGPF_PROPS(hnd).transfer_rate)

/* dltype flags */
#define LIBGPF_FLAGS(hnd)		(LIBGPF_PROPS(hnd).flags)

/* pipe macros */
#define LIBGPF_PIPE_PRIVATE(pipe)	((pipe)->pipepriv)
#define LIBGPF_PIPE_DLHANDLE(pipe)	((pipe)->dlhandle)

/* proto macros */
#define LIBGPF_PROTO_PRIVATE(io)	(LIBGPF_PIPE_PRIVATE((io)->opproto->pipe))
#define LIBGPF_PROTO_DLHANDLE(io)	(LIBGPF_PIPE_DLHANDLE((io)->opproto->pipe))

/* io macros */
#define LIBGPF_IO_PRIVATE(io)		(LIBGPF_PIPE_PRIVATE((io)->pipe))
#define LIBGPF_IO_DLHANDLE(io)		(LIBGPF_PIPE_DLHANDLE((io)->pipe))

#define LIBGPF_IO_MARK_EOS(io)		(io)->opproto->is_eos = 1
#define LIBGPF_IO_UNMARK_EOS(io)	(io)->opproto->is_eos = 0
#define LIBGPF_IO_IS_EOS(io)		(io)->opproto->is_eos

#define LIBGPF_IO_ISINPUT(io)		((io)->type & GPF_DLTYPE_INPUT)
#define LIBGPF_IO_ISOUTPUT(io)		((io)->type & GPF_DLTYPE_OUTPUT)

#endif	/* _GPF_INTERNAL_TYPES_H */
