/*
 ******************************************************************************

 LIBGPF: libggi io target functions

 Copyright (C) 2001-2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include "libggi_gpf.h"


int GPF_LIBGGI_open(struct gpf_handle_opio_t *opio)
{
	int rc;
	struct libggigpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = LIBGGI_PRIV(opio);


	rc = opio->opproto->open(opio, 0);

	/* Probably you wanna add an option,
	 * which allows you to choose between the
	 * visible and the virtual visual size...
	 */
	opio->pipe->width = priv->mode.virt.x;
	opio->pipe->height = priv->mode.virt.y;
	opio->pipe->depth = GT_DEPTH(priv->mode.graphtype);
	opio->pipe->size = GT_SIZE(priv->mode.graphtype);

	return rc;
}	/* GPF_LIBGGI_open */


int GPF_LIBGGI_close(struct gpf_handle_opio_t *opio)
{
	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);

	return opio->opproto->close(opio);
}	/* GPF_LIBGGI_close */



int GPF_LIBGGI_timestamp(struct gpf_handle_opio_t *opio,
			struct timeval *tv, int whence)
{
	struct libggigpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = LIBGGI_PRIV(opio);

	switch (whence & GPF_SEEK_OPMASK) {
	case GPF_SEEK_GET:
		switch (whence & GPF_SEEK_POSMASK) {

		/* case GPF_SEEK_REL and GPF_SEEK_ABS are
		 * already handled in libgpf/gpf/gpf.c
		 */
		case GPF_SEEK_LAST:
			if (LIBGPF_IO_ISINPUT(opio)) {
				memcpy(tv, &opio->last_ts,
					sizeof(struct timeval));
				return GGI_OK;
			}	/* if */
			if (LIBGPF_IO_ISOUTPUT(opio)) {
				return GGI_ENOMATCH;
			}	/* if */
			return GGI_ENOMATCH;

		case GPF_SEEK_END:

			/* Return the result of GPF_SEEK_END - tv seconds,
			 * where GPF_SEEK_END is the timestamp
			 * of the last frame. tv is simply the
			 * given parameter. If the result is negative
			 * then fail by returning GGI_ENOMATCH
			 *
			 * Hint: Use _gpf_ts_sub() to do the subtraction.
			 */

			FIXME("%s:%s:%i: GPF_SEEK_GET | GPF_SEEK_END "
				"not implemented\n",
				DEBUG_INFO);
			return GGI_ENOMATCH;

		}	/* switch */
		break;

	case GPF_SEEK_SET:

		switch (whence & GPF_SEEK_POSMASK) {
		case GPF_SEEK_REL:
			FIXME("%s:%s:%i: GPF_SEEK_SET | GPF_SEEK_REL "
				"not implemented\n",
				DEBUG_INFO);
			break;
		case GPF_SEEK_ABS:
			FIXME("%s:%s:%i: GPF_SEEK_SET | GPF_SEEK_ABS "
				"not implemented\n",
				DEBUG_INFO);
			break;

		case GPF_SEEK_LAST:
			FIXME("%s:%s:%i: GPF_SEEK_SET | GPF_SEEK_LAST "
				"not implemented\n",
				DEBUG_INFO);
			break;
		case GPF_SEEK_END:
			FIXME("%s:%s:%i: GPF_SEEK_SET | GPF_SEEK_END "
				"not implemented\n",
				DEBUG_INFO);
			break;
		}	/* switch */
		break;
	}	/* switch */

	return GGI_ENOMATCH;
}	/* GPF_LIBGGI_timestamp */



#if 0
int GPF_LIBGGI_setcurrframe(struct gpf_handle_opio_t *opio,
				int num_frame)
{
	struct libggigpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = LIBGGI_PRIV(opio);


	if (LIBGPF_IO_ISINPUT(opio)) {
		int rc;

		rc = ggiSetReadFrame(priv->vis, num_frame);
		ggiSetDisplayFrame(priv->vis, num_frame);
		if (rc == GGI_OK) opio->cur_frame = num_frame;
		return rc;
	}	/* if */

	if (LIBGPF_IO_ISOUTPUT(opio)) {
		int rc;

		rc = ggiSetWriteFrame(priv->vis, num_frame);
		ggiSetDisplayFrame(priv->vis, num_frame);
		if (rc == GGI_OK) opio->cur_frame = num_frame;
		return rc;
	}	/* if */

	return GGI_ENODEVICE;
}	/* GPF_LIBGGI_setcurrframe */


int GPF_LIBGGI_getcurrframe(struct gpf_handle_opio_t *opio,
				int *num_frame)
{
	struct libggigpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = LIBGGI_PRIV(opio);


	if (LIBGPF_IO_ISINPUT(opio)) {
		*num_frame = ggiGetReadFrame(priv->vis);
		LIB_ASSERT(*num_frame == opio->cur_frame);
		return (*num_frame > 0) ? GGI_OK : GGI_ENODEVICE;
	}	/* if */

	if (LIBGPF_IO_ISOUTPUT(opio)) {
		*num_frame = ggiGetWriteFrame(priv->vis);
		LIB_ASSERT(*num_frame == opio->cur_frame);
		return (*num_frame > 0) ? GGI_OK : GGI_ENODEVICE;
	}	/* if */

	return GGI_ENODEVICE;
}	/* GPF_LIBGGI_getcurrframe */


int GPF_LIBGGI_getnumframes(struct gpf_handle_opio_t *opio,
				int *num_frames)
{
	struct libggigpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = LIBGGI_PRIV(opio);

	return priv->mode.frames;
}	/* GPF_LIBGGI_getnumframes */
#endif



ssize_t GPF_LIBGGI_read(struct gpf_handle_opio_t *opio,
			void *buf, size_t size)
{
	struct libggigpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = LIBGGI_PRIV(opio);

	return opio->opproto->read(opio, buf, size);
}	/* GPF_LIBGGI_read */


ssize_t GPF_LIBGGI_write(struct gpf_handle_opio_t *opio,
			void *buf, size_t size)
{
	struct libggigpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = LIBGGI_PRIV(opio);

	return opio->opproto->write(opio, buf, size);
}	/* GPF_LIBGGI_write */






static int GPF_LIBGGI_perform_conversion(struct libggigpf_priv *priv)
{
	LIB_ASSERT(priv != NULL);


	switch (priv->nr) {
	case 0:
		if ((GT_SCHEME(priv->mode.graphtype) & GT_PALETTE)
		   || (GT_SCHEME(priv->mode.graphtype) & GT_STATIC_PALETTE))
		{
			/* perform conversion */
			return 1;
		}	/* if */

		/* no conversion */
		return 0;
	case 1:
		/* perform conversion */
		return 1;
	}	/* switch */

	LIB_ASSERT(priv->nr < 2);
	return 0;
}	/* GPF_LIBGGI_perform_conversion */







int GPF_LIBGGI_setup_input(gpf_handle_t handle,
			struct gpf_pipeline_item_t *pipe)
{
	int rc;
	size_t data_size;
	struct libggigpf_priv *priv = NULL;
	struct gpf_pipeline_t *pipeline = NULL;


	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(pipe != NULL);

	pipeline = handle->pipeline;
	priv = LIBGGI_PRIV(handle->opio_input);

	GPFDPRINT("io-libggi (input): setting it up to act as input target\n");


	LIB_ASSERT(PIPE_READBUF(pipe) == NULL);
	LIB_ASSERT(PIPE_WRITEBUF(pipe) == NULL);

	LIB_ASSERT(pipe->width > 0);
	LIB_ASSERT(pipe->height > 0);

	LIB_ASSERT(PIPELINEITEM_MARK(pipe, INFOLIST_INPUT) != NULL);
	LIB_ASSERT(PIPELINEITEM_MARK(pipe, INFOLIST_OUTPUT) == NULL);

	memcpy(&priv->pf, &PIPELINEITEM_MARK(pipe, INFOLIST_INPUT)->pf,
		sizeof(struct gpf_pixelformat_t));
	priv->nr = PIPELINEITEM_MARK(pipe, INFOLIST_INPUT)->nr;

	/* Set up some internal function pointers, if necessary
	 */

	data_size = pipe->width * pipe->height * ByPP(priv->pf.size);
	LIB_ASSERT(data_size > 0);

	/* set total amount of transfering timestamps */
	LIBGPF_TS_CLEAR(&LIBGPF_TS_TOTAL(handle));
	LIBGPF_TS_TOTAL(handle).tv_sec = -1;	/* -1 means unknown */
	FIXME("%s:%s:%i: use number of frames here instead of -1\n",
		DEBUG_INFO);

	/* set transfer rate (1 second means 1 frame) */
	LIBGPF_TS_CLEAR(&LIBGPF_TS_TRANSFER(handle));
	LIBGPF_TS_TRANSFER(handle).tv_sec = 1;

	/* the amount of memory, that we have to transfer */
	rc = _gpf_alloc_readbuf(pipe, data_size);
	if (rc != GGI_OK) {
		GPFDPRINT_PIPEMGR("%s:%s:%i: io-libggi (input): memory allocation failed\n",
			DEBUG_INFO);
		goto err0;
	}	/* if */

	rc = _gpf_alloc_writebuf(pipe, data_size);
	if (rc != GGI_OK) {
		GPFDPRINT_PIPEMGR("%s:%s:%i: io-libggi (input): memory allocation failed\n",
			DEBUG_INFO);
		goto err1;
	}	/* if */

	LIB_ASSERT(pipe->buf_conv == NULL);
	LIB_ASSERT(pipe->buf_conv_size == 0);

	if (!GPF_LIBGGI_perform_conversion(priv)) {
		GPFDPRINT("io-libggi (input): performs no conversion\n");
		priv->doconv = 0;
		return GGI_OK;
	}	/* if */

	GPFDPRINT("io-libggi (input): conversion enabled\n");
	priv->doconv = 1;
	return GGI_OK;

err1:
	_gpf_free_readbuf(pipe);
err0:
	return GGI_ENOMEM;
}	/* GPF_LIBGGI_setup_input */





int GPF_LIBGGI_setup_output(gpf_handle_t handle,
			struct gpf_pipeline_item_t *pipe)
{
	struct libggigpf_priv *priv = NULL;
	struct gpf_pipeline_t *pipeline = NULL;

	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(pipe != NULL);

	pipeline = handle->pipeline;
	priv = LIBGGI_PRIV(handle->opio_output);

	GPFDPRINT("io-libggi (output): setting it up to act as output target\n");


	LIB_ASSERT(PIPELINEITEM_MARK(pipe, INFOLIST_INPUT) == NULL);
	LIB_ASSERT(PIPELINEITEM_MARK(pipe, INFOLIST_OUTPUT) != NULL);

	memcpy(&priv->pf, &PIPELINEITEM_MARK(pipe, INFOLIST_OUTPUT)->pf,
		sizeof(struct gpf_pixelformat_t));
	priv->nr = PIPELINEITEM_MARK(pipe, INFOLIST_OUTPUT)->nr;

	/* Set up some internal function pointers, if necessary
	 */

	if (!GPF_LIBGGI_perform_conversion(priv)) {
		GPFDPRINT("io-libggi (output): performs no conversion\n");
		priv->doconv = 0;
		return GGI_OK;
	}	/* if */

	GPFDPRINT("io-libggi (output): conversion enabled\n");
	priv->doconv = 1;

	return GGI_OK;
}	/* GPF_LIBGGI_setup_output */



int GPF_LIBGGI_getformat(struct gpf_pipeline_item_t *pipe,
			struct gpf_pipeline_item_t *from_pipe,
			int chain)
{
	GPFDPRINT_PIPEMGR("io-libggi: no rely on \"foreign\" format descriptions\n");

	/* the libggi target is no GPF_PIPE_DF_ALLVARIANTS flag.
	 * So we only implement this as a dummy function to avoid
	 * a crash in the pipeline management.
	 */

	return GGI_OK;
}	/* GPF_LIBGGI_getformat */
