/*
 ******************************************************************************

 LIBGPF: libggi io target functions (read/write)

 Copyright (C) 2001-2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include "libggi_gpf.h"


static int GPF_proto_open(struct gpf_handle_opio_t *opio,
			int flags)
{
	int rc = GGI_OK;
	struct libggigpf_priv *priv;

	GPFDPRINT_PROTO("%s:%s:%i: called\n", DEBUG_INFO);
	priv = LIBGGI_PRIV(opio);

	LIB_ASSERT(priv->vis != NULL);

	if (priv->openvis) {
		priv->vis = ggiOpen( (priv->target[0] == '\0') ?
				NULL : priv->target );
		if (!priv->vis) {
			rc = GGI_ENOMEM;
			goto err0;
		}	/* if */

		ggiCheckMode(priv->vis, &(priv->mode));
		rc = ggiSetMode(priv->vis, &(priv->mode));
		if (rc != GGI_OK) {
			goto err1;
		}	/* if */

		return GGI_OK;
	}	/* if */

	priv->read_xoffset = priv->read_yoffset = 0;
	priv->write_xoffset = priv->write_yoffset = 0;
	ggiGetMode(priv->vis, &(priv->mode));

	if ((priv->mode.graphtype == 0)
	  || (priv->mode.visible.x == 0)
	  || (priv->mode.visible.y == 0))
	{
		fprintf(stderr, "io-libggi: No proper mode set!\n");
		rc = GGI_ENOMATCH;
		goto err0;
	}	/* if */


	/* nothing more do here, because we can assume here
	 * that we have a valid ggi visual given by
	 * params.
	 */
	return GGI_OK;

err1:
	ggiClose(priv->vis);
err0:
	return rc;
}	/* GPF_proto_open */


static int GPF_proto_close(struct gpf_handle_opio_t *opio)
{
	struct libggigpf_priv *priv;

	GPFDPRINT_PROTO("%s:%s:%i: called\n", DEBUG_INFO);
	priv = LIBGGI_PRIV(opio);

	LIB_ASSERT(priv->vis != NULL);

	if (priv->openvis) {
		ggiClose(priv->vis);
	}	/* if */

	/* The visual given by a parameter must be
	 * closed external.
	 */
	return GGI_OK;
}	/* GPF_proto_close */



#define LIMIT_X(x,y, width)		\
	if (x >= width) {		\
		x = 0;			\
		y++;			\
	}	/* if */
#define LIMIT_Y(x,y, height, io)	\
	if (y >= height) {		\
		x = y = 0;		\
		LIBGPF_IO_MARK_EOS(io);	\
	}	/* if */


static ssize_t GPF_proto_read(struct gpf_handle_opio_t *opio,
			void *buf, size_t size)
{
	int rc = GGI_OK;
	struct libggigpf_priv *priv;
	char *base_buf;	
	char *read_buf;
	size_t width, height;
	size_t size_bypp_total;
	size_t size_bypp;	/* size in bytes per pixel */
	size_t stride;		/* bytes per line */

	GPFDPRINT_PROTO("%s:%s:%i: called\n", DEBUG_INFO);
	priv = LIBGGI_PRIV(opio);

	if (LIBGPF_IO_IS_EOS(opio)) return 0;

	if (priv->doconv) {
		read_buf = (char *)malloc(size);
		if (!read_buf) return GGI_ENOMEM;

		size /= ByPP(priv->pf.size);

		base_buf = (char *)read_buf;
	} else {
		base_buf = read_buf = (char *)buf;
	}	/* if */

	size_bypp_total = size / GT_ByPP(priv->mode.graphtype);
	size_bypp = size_bypp_total;


	switch (priv->opmode) {
	case OPMODE_FRAME:
		width = priv->mode.virt.x;
		height = priv->mode.virt.y;
		stride = GT_ByPP(priv->mode.graphtype) * width;

		if ((priv->read_xoffset > 0)
		   || ((priv->read_xoffset >= 0)
		   && (size_bypp_total < opio->pipe->width)))
		{
			do {
				/* complete line */
				width = priv->mode.virt.x - priv->read_xoffset;
				if (size_bypp < width) width = size_bypp;

				fprintf(stderr, "%s:%s:%i: A offset: (%i,%i), width: %i, size_bypp: %i\n",
					DEBUG_INFO, priv->read_xoffset, priv->read_yoffset, width, size_bypp);

				ggiGetBox(priv->vis, priv->read_xoffset,
					priv->read_yoffset,
					width, 1, read_buf);

				/* be aware: that's != stride */
				read_buf += width * GT_ByPP(priv->mode.graphtype);

				size_bypp -= width;
				priv->read_xoffset += width;

				fprintf(stderr, "%s:%s:%i: A offset: (%i,%i), width: %i, size_bypp: %i\n",
					DEBUG_INFO, priv->read_xoffset, priv->read_yoffset, width, size_bypp);

				LIMIT_X(priv->read_xoffset, priv->read_yoffset,
					priv->mode.virt.x);
				LIMIT_Y(priv->read_xoffset, priv->read_yoffset,
					priv->mode.virt.y, opio);
				LIB_ASSERT(priv->read_xoffset < priv->mode.virt.x);
				LIB_ASSERT(priv->read_yoffset < priv->mode.virt.y);

				fprintf(stderr, "%s:%s:%i: A offset: (%i,%i), width: %i, size_bypp: %i\n",
					DEBUG_INFO, priv->read_xoffset, priv->read_yoffset, width, size_bypp);

				if (size_bypp == 0) break;
			} while ((priv->write_xoffset + size_bypp) <= opio->pipe->width);
		}	/* if */


		fprintf(stderr, "%s:%s:%i: B offset: (%i,%i), width: %i, height: %i, size_bypp: %i\n",
			DEBUG_INFO, priv->read_xoffset, priv->read_yoffset, width, height, size_bypp);

		width = priv->mode.virt.x;
		height = size_bypp / width;
		if (height > priv->mode.virt.y) height = priv->mode.virt.y;
		if ((height + priv->read_yoffset) > priv->mode.virt.y) {
			LIB_ASSERT(priv->read_yoffset < priv->mode.virt.y);
			height = priv->mode.virt.y - priv->read_yoffset;
		}	/* if */

		fprintf(stderr, "%s:%s:%i: B offset: (%i,%i), width: %i, height: %i, size_bypp: %i\n",
			DEBUG_INFO, priv->read_xoffset, priv->read_yoffset, width, height, size_bypp);

		ggiGetBox(priv->vis, priv->read_xoffset, priv->read_yoffset,
				width, height, read_buf);
		read_buf += stride * height;
		size_bypp -= (width * height);
		priv->read_xoffset += width;
		priv->read_yoffset += (height-1);

		fprintf(stderr, "%s:%s:%i: B offset: (%i,%i), width: %i, height: %i, size_bypp: %i\n",
			DEBUG_INFO, priv->read_xoffset, priv->read_yoffset, width, height, size_bypp);

		LIMIT_X(priv->read_xoffset, priv->read_yoffset,
			priv->mode.virt.x);
		LIMIT_Y(priv->read_xoffset, priv->read_yoffset,
			priv->mode.virt.y, opio);
		LIB_ASSERT(priv->read_xoffset < priv->mode.virt.x);
		LIB_ASSERT(priv->read_yoffset < priv->mode.virt.y);

		fprintf(stderr, "%s:%s:%i: B offset: (%i,%i), width: %i, height: %i, size_bypp: %i\n",
			DEBUG_INFO, priv->read_xoffset, priv->read_yoffset, width, height, size_bypp);

		if (size_bypp > 0) {
			fprintf(stderr, "%s:%s:%i: C offset: (%i,%i), width: %i, size_bypp: %i\n",
				DEBUG_INFO, priv->read_xoffset, priv->read_yoffset, width, size_bypp);

			LIB_ASSERT(size_bypp < priv->mode.virt.x);
			width = priv->mode.virt.x - priv->read_xoffset;
			if (size_bypp < width) width = size_bypp;

			fprintf(stderr, "%s:%s:%i: C offset: (%i,%i), width: %i, size_bypp: %i\n",
				DEBUG_INFO, priv->read_xoffset, priv->read_yoffset, width, size_bypp);

			ggiGetHLine(priv->vis,
				priv->read_xoffset, priv->read_yoffset,
				width, read_buf);
			size_bypp -= width;
			LIB_ASSERT(size_bypp == 0);
			GPFDPRINT_PROTO("%s:%s:%i: size_bypp: %i, width: %i\n",
					DEBUG_INFO, size_bypp, width);

			priv->read_xoffset += width;

			fprintf(stderr, "%s:%s:%i: C offset: (%i,%i), width: %i, size_bypp: %i\n",
				DEBUG_INFO, priv->read_xoffset, priv->read_yoffset, width, size_bypp);

			LIMIT_X(priv->read_xoffset, priv->read_yoffset,
				priv->mode.virt.x);
			LIMIT_Y(priv->read_xoffset, priv->read_yoffset,
				priv->mode.virt.y, opio);
			LIB_ASSERT(priv->read_xoffset < priv->mode.virt.x);
			LIB_ASSERT(priv->read_yoffset < priv->mode.virt.y);

			fprintf(stderr, "%s:%s:%i: C offset: (%i,%i), width: %i, size_bypp: %i\n",
				DEBUG_INFO, priv->read_xoffset, priv->read_yoffset, width, size_bypp);
		}	/* if */

		LIB_ASSERT(size_bypp == 0);
		break;

	case OPMODE_HLINE:
		width = priv->mode.virt.x;
		height = priv->mode.virt.y;
		stride = GT_ByPP(priv->mode.graphtype) * width;

		if ((priv->read_xoffset > 0)
		   || ((priv->read_xoffset >= 0)
		   && (size_bypp_total < opio->pipe->width)))
		{
			do {
				/* complete line */
				width = priv->mode.virt.x - priv->read_xoffset;

				if (size_bypp < width) width = size_bypp;

				ggiGetHLine(priv->vis, priv->read_xoffset, priv->read_yoffset,
						width, read_buf);

				/* be aware: that's != stride */
				read_buf += width * GT_ByPP(priv->mode.graphtype);
				size_bypp -= width;
				priv->read_xoffset += width;

				LIMIT_X(priv->read_xoffset, priv->read_yoffset,
					priv->mode.virt.x);
				LIMIT_Y(priv->read_xoffset, priv->read_yoffset,
					priv->mode.virt.y, opio);
				LIB_ASSERT(priv->read_xoffset < priv->mode.virt.x);
				LIB_ASSERT(priv->read_yoffset < priv->mode.virt.y);

				if (size_bypp == 0) break;
			} while ((priv->write_xoffset + size_bypp) <= opio->pipe->width);
		}	/* if */


		width = priv->mode.virt.x;
		height = size_bypp / width;
		if (height > priv->mode.virt.y) height = priv->mode.virt.y;
		if ((height + priv->read_yoffset) > priv->mode.virt.y) {
			LIB_ASSERT(priv->read_yoffset < priv->mode.virt.y);
			height = priv->mode.virt.y - priv->read_yoffset;
		}	/* if */

		while (height--) {
			ggiGetHLine(priv->vis, priv->read_xoffset,
					priv->read_yoffset, width, read_buf);
			read_buf += stride;
			size_bypp -= width;
			priv->read_xoffset += width;

			LIMIT_X(priv->read_xoffset, priv->read_yoffset,
				priv->mode.virt.x);
			LIMIT_Y(priv->read_xoffset, priv->read_yoffset,
				priv->mode.virt.y, opio);
			LIB_ASSERT(priv->read_xoffset < priv->mode.virt.x);
			LIB_ASSERT(priv->read_yoffset < priv->mode.virt.y);
		}	/* while */


		if (size_bypp > 0) {
			LIB_ASSERT(size_bypp < priv->mode.virt.x);
			width = priv->mode.virt.x - priv->read_xoffset;

			if (size_bypp < width) width = size_bypp;

			ggiGetHLine(priv->vis, priv->read_xoffset,
				priv->read_yoffset, width, read_buf);
			size_bypp -= width;
			LIB_ASSERT(size_bypp == 0);

			priv->read_xoffset += width;

			LIMIT_X(priv->read_xoffset, priv->read_yoffset,
				priv->mode.virt.x);
			LIMIT_Y(priv->read_xoffset, priv->read_yoffset,
				priv->mode.virt.y, opio);
			LIB_ASSERT(priv->read_xoffset < priv->mode.virt.x);
			LIB_ASSERT(priv->read_yoffset < priv->mode.virt.y);
		}	/* if */

		LIB_ASSERT(size_bypp == 0);
		break;

	case OPMODE_VLINE:
		width = priv->mode.virt.x;
		height = priv->mode.virt.y;
		stride = GT_ByPP(priv->mode.graphtype) * height;

		/* don't use the LIMIT_X() and LIMIT_Y() macros here.
		 * It's wrong in this section
		 */

		if ((priv->read_yoffset > 0)
		   || ((priv->read_yoffset >= 0)
		   && (size_bypp_total < opio->pipe->height)))
		{
			do {
				/* complete line */
				height = priv->mode.virt.y - priv->read_yoffset;

				if (size_bypp < height) height = size_bypp;

				ggiGetVLine(priv->vis,
					priv->read_xoffset, priv->read_yoffset,
					height, read_buf);

				/* be aware: that's != stride */
				read_buf += height * GT_ByPP(priv->mode.graphtype);
				size_bypp -= height;
				priv->read_yoffset += height;

				if (priv->read_yoffset >= priv->mode.virt.y) {
					priv->read_xoffset++;
					priv->read_yoffset = 0;
				}	/* if */
				if (priv->read_xoffset >= priv->mode.virt.x) {
					priv->read_xoffset = 0;
					priv->read_yoffset = 0;
					LIBGPF_IO_MARK_EOS(opio);
				}	/* if */
				LIB_ASSERT(priv->read_xoffset < priv->mode.virt.x);
				LIB_ASSERT(priv->read_yoffset < priv->mode.virt.y);

				if (size_bypp == 0) break;			
			} while ((priv->write_yoffset + size_bypp) < opio->pipe->height);
		}	/* if */


		height = priv->mode.virt.y;
		width = size_bypp / height;
		if (width > priv->mode.virt.x) height = priv->mode.virt.x;
		if ((width + priv->read_xoffset) > priv->mode.virt.x) {
			LIB_ASSERT(priv->read_xoffset < priv->mode.virt.x);
			width = priv->mode.virt.x - priv->read_xoffset;
		}	/* if */

		while (width--) {
			ggiGetVLine(priv->vis,
				priv->read_xoffset, priv->read_yoffset,
				height, read_buf);
			read_buf += stride;
			size_bypp -= height;
			priv->read_yoffset += height;

			if (priv->read_yoffset >= priv->mode.virt.y) {
				priv->read_xoffset++;
				priv->read_yoffset = 0;
			}	/* if */
			if (priv->read_xoffset >= priv->mode.virt.x) {
				priv->read_xoffset = 0;
				priv->read_yoffset = 0;
				LIBGPF_IO_MARK_EOS(opio);
			}	/* if */
			LIB_ASSERT(priv->read_xoffset < priv->mode.virt.x);
			LIB_ASSERT(priv->read_yoffset < priv->mode.virt.y);
		}	/* while */


		if (size_bypp > 0) {
			LIB_ASSERT(size_bypp < priv->mode.virt.y);
			height = priv->mode.virt.y - priv->read_yoffset;

			if (size_bypp < height) height = size_bypp;

			ggiGetVLine(priv->vis,
				priv->read_xoffset, priv->read_yoffset,
				height, read_buf);
			size_bypp -= height;
			LIB_ASSERT(size_bypp == 0);

			priv->read_yoffset += height;

			if (priv->read_yoffset >= priv->mode.virt.y) {
				priv->read_xoffset++;
				priv->read_yoffset = 0;
			}	/* if */
			if (priv->read_xoffset >= priv->mode.virt.x) {
				priv->read_xoffset = 0;
				priv->read_yoffset = 0;
				LIBGPF_IO_MARK_EOS(opio);
			}	/* if */
			LIB_ASSERT(priv->read_xoffset < priv->mode.virt.x);
			LIB_ASSERT(priv->read_yoffset < priv->mode.virt.y);
		}	/* if */

		LIB_ASSERT(size_bypp == 0);
		break;

	}	/* switch */

	rc = (size_bypp_total - size_bypp) * ByPP(priv->pf.size);
	GPFDPRINT_PROTO("%s:%s:%i: rc: %i\n", DEBUG_INFO, rc);

	if (priv->doconv) {
		size_t size_bypp, conv_size;

		read_buf = base_buf;
		size_bypp = ByPP(priv->pf.size);
		conv_size = size_bypp_total;

		GPFDPRINT_PROTO("%s:%s:%i: priv->pf.size: %i, size_bypp: %i\n",
			DEBUG_INFO, priv->pf.size, size_bypp);

		while (conv_size > size_bypp || conv_size == size_bypp) {
			ggiUnmapPixel(priv->vis,
					((ggi_pixel *)read_buf)[0],
					((ggi_color *)buf));
			conv_size -= size_bypp;

			read_buf += GT_ByPP(priv->mode.graphtype);
			((ggi_color *)buf)++;
		}	/* while */

		free(base_buf);
	}	/* if */

	GPFDPRINT_PROTO("%s:%s:%i: rc: %i\n", DEBUG_INFO, rc);
	return rc;
}	/* GPF_proto_read */



static ssize_t GPF_proto_write(struct gpf_handle_opio_t *opio,
				void *buf, size_t size)
{
	int rc = GGI_OK;
	struct libggigpf_priv *priv;
	char *base_buf;
	char *write_buf;
	size_t width, height;
	size_t maxwidth, maxheight;
	size_t size_bypp_total;
	size_t size_bypp;	/* size in bytes per pixel */
	size_t stride;		/* bytes per line */

	GPFDPRINT_PROTO("%s:%s:%i: called\n", DEBUG_INFO);
	priv = LIBGGI_PRIV(opio);

	size_bypp_total = size / GT_ByPP(priv->mode.graphtype);
	size_bypp = size_bypp_total;


	if (priv->doconv) {
		size_t conv_size;

		write_buf = (char *)malloc(size);
		if (!write_buf) return GGI_ENOMEM;

		base_buf = (char *)write_buf;

		size_bypp = ByPP(priv->pf.size);
		conv_size = size_bypp_total;

		GPFDPRINT_PROTO("%s:%s:%i: priv->pf.size: %i, size_bypp: %i\n",
				DEBUG_INFO, priv->pf.size, size_bypp);

		while (conv_size > size_bypp || conv_size == size_bypp) {
			((ggi_pixel *)write_buf)[0] = ggiMapColor(priv->vis,
							(ggi_color *)buf);
			conv_size -= size_bypp;

			write_buf += GT_ByPP(priv->mode.graphtype);
			((ggi_color *)buf)++;
		}	/* while */

		size /= ByPP(priv->pf.size);

		size_bypp_total = size / GT_ByPP(priv->mode.graphtype);
		size_bypp = size_bypp_total;
		write_buf = base_buf;
	} else {
		base_buf = write_buf = (char *)buf;
	}	/* if */

	switch (priv->opmode) {
	case OPMODE_FRAME:
		maxwidth = width = opio->pipe->width;
		maxheight = height = opio->pipe->height;
		if (opio->pipe->width > priv->mode.virt.x) {
			maxwidth = priv->mode.virt.x;
		}	/* if */
		if (opio->pipe->height > priv->mode.virt.y) {
			maxheight = priv->mode.virt.y;
		}	/* if */

		stride = GT_ByPP(priv->mode.graphtype) * opio->pipe->width;

		if ((priv->write_xoffset > 0)
		   || ((priv->write_xoffset >= 0)
		   && (size_bypp_total < opio->pipe->width)))
		{
			do {
				/* complete line */
				width = opio->pipe->width - priv->write_xoffset;
				if (size_bypp < width) width = size_bypp;

				ggiPutBox(priv->vis,
					priv->write_xoffset, priv->write_yoffset,
					width, 1, write_buf);

				/* be aware: that's != stride */
				write_buf += width * GT_ByPP(priv->mode.graphtype);

				size_bypp -= width;
				priv->write_xoffset += width;

				LIMIT_X(priv->write_xoffset, priv->write_yoffset,
					opio->pipe->width);
				LIMIT_Y(priv->write_xoffset, priv->write_yoffset,
					opio->pipe->height, opio);
				LIB_ASSERT(priv->write_xoffset < opio->pipe->width);
				LIB_ASSERT(priv->write_yoffset < opio->pipe->height);

				if (size_bypp == 0) break;
			} while ((priv->write_xoffset + size_bypp) <= opio->pipe->width);
		}	/* if */


		width = opio->pipe->width;
		height = size_bypp / width;

		if (width > priv->mode.virt.x) width = priv->mode.virt.x;
		if (height > priv->mode.virt.y) height = priv->mode.virt.y;
		if (height >= 1) {
			ggiPutBox(priv->vis,
				priv->write_xoffset, priv->write_yoffset,
				width, height, write_buf);
			write_buf += stride * height;
			size_bypp -= (width * height);
			priv->write_xoffset += width;
			priv->write_yoffset += (height-1);

			LIMIT_X(priv->write_xoffset, priv->write_yoffset,
				maxwidth);
			LIMIT_Y(priv->write_xoffset, priv->write_yoffset,
				maxheight, opio);
			LIB_ASSERT(priv->write_xoffset < maxwidth);
			LIB_ASSERT(priv->write_yoffset < maxheight);
		}	/* if */

		if (size_bypp > 0) {
			if (maxheight < opio->pipe->height) {
				/* happens, when image is larger than virtual y size
				 */
				size_bypp = 0;
				break;
			}	/* if */

			LIB_ASSERT(size_bypp < opio->pipe->width);
			width = opio->pipe->width - priv->write_xoffset;
			if (size_bypp < width) width = size_bypp;

			ggiPutHLine(priv->vis,
				priv->write_xoffset, priv->write_yoffset,
				width, write_buf);
			size_bypp -= width;
			LIB_ASSERT(size_bypp == 0);
			GPFDPRINT_PROTO("%s:%s:%i: size_bypp: %i, width: %i\n",
					DEBUG_INFO, size_bypp, width);

			priv->write_xoffset += width;

			LIMIT_X(priv->write_xoffset, priv->write_yoffset,
				opio->pipe->width);
			LIMIT_Y(priv->write_xoffset, priv->write_yoffset,
				opio->pipe->height, opio);
			LIB_ASSERT(priv->write_xoffset < opio->pipe->width);
			LIB_ASSERT(priv->write_yoffset < opio->pipe->height);
		}	/* if */

		LIB_ASSERT(size_bypp == 0);
		break;

	case OPMODE_HLINE:
		maxwidth = width = opio->pipe->width;
		maxheight = height = opio->pipe->height;
		if (opio->pipe->width > priv->mode.virt.x) {
			maxwidth = priv->mode.virt.x;
		}	/* if */
		if (opio->pipe->height > priv->mode.virt.y) {
			maxheight = priv->mode.virt.y;
		}	/* if */

		stride = GT_ByPP(priv->mode.graphtype) * opio->pipe->width;

		if ((priv->write_xoffset > 0)
		   || ((priv->write_xoffset >= 0)
		   && (size_bypp_total < opio->pipe->width)))
		{
			do {
				/* complete line */
				width = opio->pipe->width - priv->write_xoffset;
				if (size_bypp < width) width = size_bypp;

				ggiPutHLine(priv->vis,
						priv->write_xoffset,
						priv->write_yoffset,
						width, write_buf);

				/* be aware: that's != stride */
				write_buf += width *
						GT_ByPP(priv->mode.graphtype);
				size_bypp -= width;
				priv->write_xoffset += width;

				LIMIT_X(priv->write_xoffset,
					priv->write_yoffset,
					opio->pipe->width);
				LIMIT_Y(priv->write_xoffset,
					priv->write_yoffset,
					opio->pipe->height, opio);
				LIB_ASSERT(priv->write_xoffset < opio->pipe->width);
				LIB_ASSERT(priv->write_yoffset < opio->pipe->height);

				if (size_bypp == 0) break;
			} while ((priv->write_xoffset + size_bypp) < opio->pipe->width);
		}	/* if */

		width = opio->pipe->width;
		height = size_bypp / width;

		if (width > priv->mode.virt.x) width = priv->mode.virt.x;
		if (height > priv->mode.virt.y) height = priv->mode.virt.y;
		while (height--) {
			ggiPutHLine(priv->vis,
				priv->write_xoffset, priv->write_yoffset,
				width, write_buf);

			write_buf += stride;
			size_bypp -= opio->pipe->width;
			priv->write_xoffset += opio->pipe->width;

			LIMIT_X(priv->write_xoffset, priv->write_yoffset,
				opio->pipe->width);

			LIB_ASSERT(priv->write_xoffset < opio->pipe->width);
			LIB_ASSERT(priv->write_yoffset < opio->pipe->height || height == 0);
		}	/* while */

		if (size_bypp > 0) {
			if ((maxheight < opio->pipe->height)
			  && (priv->write_yoffset >= maxheight))
			{
				/* happens, when image is larger
				 * than virtual y size
				 */
				size_bypp = 0;
				break;
			}	/* if */

			LIB_ASSERT(size_bypp < opio->pipe->width);
			width = opio->pipe->width - priv->write_xoffset;
			if (size_bypp < width) width = size_bypp;

			ggiPutHLine(priv->vis,
				priv->write_xoffset, priv->write_yoffset,
				width, write_buf);
			size_bypp -= width;
			LIB_ASSERT(size_bypp == 0);

			priv->write_xoffset += width;

			LIMIT_X(priv->write_xoffset,
				priv->write_yoffset,
				opio->pipe->width);
			LIMIT_Y(priv->write_xoffset,
				priv->write_yoffset,
				opio->pipe->height, opio);
			LIB_ASSERT(priv->write_xoffset < opio->pipe->width);
			LIB_ASSERT(priv->write_yoffset < opio->pipe->height);
		}	/* if */

		LIB_ASSERT(size_bypp == 0);
		break;

	case OPMODE_VLINE:
		maxwidth = width = opio->pipe->width;
		maxheight = height = opio->pipe->height;
		if (opio->pipe->width > priv->mode.virt.x) {
			maxwidth = priv->mode.virt.x;
		}	/* if */
		if (opio->pipe->height > priv->mode.virt.y) {
			maxheight = priv->mode.virt.y;
		}	/* if */

		stride = GT_ByPP(priv->mode.graphtype) * opio->pipe->height;

		/* don't use the LIMIT_X() and LIMIT_Y() macros here.
		 * It's wrong in this section
		 */

		if ((priv->write_yoffset > 0)
		   || ((priv->write_yoffset >= 0)
		   && (size_bypp_total < opio->pipe->height)))
		{
			do {
				/* complete line */
				height = opio->pipe->height - priv->write_yoffset;
				if (size_bypp < height) height = size_bypp;

				ggiPutVLine(priv->vis,
						priv->write_xoffset,
						priv->write_yoffset,
						height, write_buf);

				/* be aware: that's != stride */
				write_buf += height *
						GT_ByPP(priv->mode.graphtype);
				size_bypp -= height;
				priv->write_xoffset += height;

				if (priv->write_yoffset >= opio->pipe->height) {
					priv->write_xoffset++;
					priv->write_yoffset = 0;
				}	/* if */
				if (priv->write_xoffset >= opio->pipe->width) {
					priv->write_xoffset = 0;
					priv->write_yoffset = 0;
					LIBGPF_IO_MARK_EOS(opio);
				}	/* if */
				LIB_ASSERT(priv->write_xoffset < opio->pipe->width);
				LIB_ASSERT(priv->write_yoffset < opio->pipe->height);

				if (size_bypp == 0) break;			
			} while ((priv->write_yoffset + size_bypp) < opio->pipe->height);
		}	/* if */


		height = opio->pipe->height;
		width = size_bypp / height;

		if (width > priv->mode.virt.x) width = priv->mode.virt.x;
		if (height > priv->mode.virt.y) height = priv->mode.virt.y;
		while (width--) {
			ggiPutVLine(priv->vis,
				priv->write_xoffset, priv->write_yoffset,
				height, write_buf);

			write_buf += stride;
			size_bypp -= opio->pipe->height;
			priv->write_yoffset += opio->pipe->height;

			if (priv->write_yoffset >= maxheight) {
				priv->write_xoffset++;
				priv->write_yoffset = 0;
			}	/* if */

			if (priv->write_xoffset >= priv->mode.virt.x) {
				/* happens, when image is larger
				 * than virtual y size
				 */
				size_bypp = 0;
				break;
			}	/* if */

			if (priv->write_xoffset >= opio->pipe->width) {
				priv->write_xoffset = 0;
				priv->write_yoffset = 0;
				LIBGPF_IO_MARK_EOS(opio);
			}	/* if */
			LIB_ASSERT(priv->write_xoffset < opio->pipe->width);
			LIB_ASSERT(priv->write_yoffset < opio->pipe->height || width == 0);
		}	/* while */


		if (size_bypp > 0) {
			if (maxwidth < opio->pipe->width) {
				/* happens, when image is larger than virtual x size
				 */
				size_bypp = 0;
				break;
			}	/* if */

			LIB_ASSERT(size_bypp < opio->pipe->height);
			height = opio->pipe->height - priv->write_yoffset;

			if (size_bypp < height) height = size_bypp;

			ggiPutVLine(priv->vis,
				priv->write_xoffset, priv->write_yoffset,
				height, write_buf);
			size_bypp -= height;
			LIB_ASSERT(size_bypp == 0);

			priv->write_yoffset += height;

			if (priv->write_yoffset >= opio->pipe->height) {
				priv->write_xoffset++;
				priv->write_yoffset = 0;
			}	/* if */
			if (priv->write_xoffset >= opio->pipe->width) {
				priv->write_xoffset = 0;
				priv->write_yoffset = 0;
				LIBGPF_IO_MARK_EOS(opio);
			}	/* if */
			LIB_ASSERT(priv->write_xoffset < opio->pipe->width);
			LIB_ASSERT(priv->write_yoffset < opio->pipe->height);
		}	/* if */

		LIB_ASSERT(size_bypp == 0);
		break;

	}	/* switch */

	rc = (size_bypp_total - size_bypp) * ByPP(priv->pf.size);

	if (priv->doconv) {
		free(base_buf);
	}	/* if */

	if (priv->flush) {
#warning Use ggiFlushRegion() instead
		ggiFlush(priv->vis);
	}	/* if */

	return rc;
}	/* GPF_proto_write */



static off_t GPF_proto_lseek(struct gpf_handle_opio_t *opio,
			off_t offset, int whence)
{
	off_t rc;
	int x,y;
	struct libggigpf_priv *priv;

	GPFDPRINT_PROTO("%s:%s:%i: called\n", DEBUG_INFO);
	priv = LIBGGI_PRIV(opio);

	rc = offset;
	x = offset % priv->mode.virt.x;
	offset -= x;
	y = offset / priv->mode.virt.x;
	LIB_ASSERT((y * priv->mode.virt.x + x) == rc);
	LIBGPF_IO_UNMARK_EOS(opio);

	switch (opio->type) {
	case GPF_DLTYPE_INPUT:
		switch (whence) {
		case SEEK_SET:
			priv->read_xoffset = x;
			priv->read_yoffset = y;

			LIMIT_X(priv->read_xoffset, priv->read_yoffset,
				priv->mode.virt.x);
			LIMIT_Y(priv->read_xoffset, priv->read_yoffset,
				priv->mode.virt.y, opio);
			LIB_ASSERT(priv->read_xoffset < priv->mode.virt.x);
			LIB_ASSERT(priv->read_yoffset < priv->mode.virt.y);

			rc = (priv->read_yoffset * priv->mode.virt.x)
				+ priv->read_xoffset;
			break;

		case SEEK_CUR:
			priv->read_xoffset += x;
			priv->read_yoffset += y;

			LIMIT_X(priv->read_xoffset, priv->read_yoffset,
				priv->mode.virt.x);
			LIMIT_Y(priv->read_xoffset, priv->read_yoffset,
				priv->mode.virt.y, opio);
			LIB_ASSERT(priv->read_xoffset < priv->mode.virt.x);
			LIB_ASSERT(priv->read_yoffset < priv->mode.virt.y);

			rc = (priv->read_yoffset * priv->mode.virt.x)
				+ priv->read_xoffset;
			break;

		case SEEK_END:
			rc = -1;
			break;
		}	/* switch */
		break;

	case GPF_DLTYPE_OUTPUT:
		switch (whence) {
		case SEEK_SET:
			priv->write_xoffset = x;
			priv->write_yoffset = y;

			LIMIT_X(priv->read_xoffset, priv->read_yoffset,
				priv->mode.virt.x);
			LIMIT_Y(priv->read_xoffset, priv->read_yoffset,
				priv->mode.virt.y, opio);
			LIB_ASSERT(priv->write_xoffset < priv->mode.virt.x);
			LIB_ASSERT(priv->write_yoffset < priv->mode.virt.y);

			rc = (priv->write_yoffset * priv->mode.virt.x)
				+ priv->write_xoffset;
			break;

		case SEEK_CUR:
			priv->write_xoffset += x;
			priv->write_yoffset += y;

			LIMIT_X(priv->read_xoffset, priv->read_yoffset,
				priv->mode.virt.x);
			LIMIT_Y(priv->read_xoffset, priv->read_yoffset,
				priv->mode.virt.y, opio);
			LIB_ASSERT(priv->write_xoffset < priv->mode.virt.x);
			LIB_ASSERT(priv->write_yoffset < priv->mode.virt.y);

			rc = (priv->write_yoffset * priv->mode.virt.x)
				+ priv->write_xoffset;
			break;

		case SEEK_END:
			rc = -1;
			break;
		}	/* switch */
		break;

	default:
		rc = -1;
		break;
	}	/* switch */

	return rc;
}	/* GPF_proto_lseek */


static int GPF_proto_flush(struct gpf_handle_opio_t *opio)
{
	int rc;
	struct libggigpf_priv *priv;

	GPFDPRINT_PROTO("%s:%s:%i: called\n", DEBUG_INFO);
	priv = LIBGGI_PRIV(opio);

	rc = ggiFlush(priv->vis);

	return rc;
}	/* GPF_proto_flush */



static int GPF_proto_eos(struct gpf_handle_opio_t *opio)
{
	GPFDPRINT_TARGET("%s:%s:%i: is_eos (%i)\n",
			DEBUG_INFO, LIBGPF_IO_IS_EOS(opio));

	return LIBGPF_IO_IS_EOS(opio);
}	/* GPF_proto_eos */




int GPF_LIBGGI_doio(struct gpf_handle_opio_t *opio,
		const char *args, va_list ap)
{
	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);

	opio->opproto->open = GPF_proto_open;
	opio->opproto->close = GPF_proto_close;

	opio->opproto->read = GPF_proto_read;
	opio->opproto->write = GPF_proto_write;

	opio->opproto->lseek = GPF_proto_lseek;
	opio->opproto->flush = GPF_proto_flush;
	opio->opproto->eos = GPF_proto_eos;

	return GGI_OK;
}	/* GPF_LIBGGI_doio */
