/* $Id: gic.h,v 1.1.1.1 2001/04/22 19:02:58 cegger Exp $
******************************************************************************

   Generic Input Configuration library for GII.

   Copyright (C) 1999 Andreas Beck	[becka@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#ifndef _GGI_GIC_H
#define _GGI_GIC_H

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#ifndef _BUILDING_LIBGIC
typedef void *gic_handle_t;
#endif

#include <stdio.h>
#include <stdlib.h>
#include <ggi/errors.h>
#include <ggi/gii.h>
#include <ggi/gic_structs.h>

/* General Library control
 */
int gicInit(void);
int gicExit(void);

/* Open a new GIC instance.
 */
gic_handle_t	gicOpen(const char *recognizers,...);
int		gicClose(gic_handle_t hand);

/*
 * Miscellaneous stuff that can be associated to a GIC handle:
 */

/* Register a GII gii_input_t with the GIC instance. This is needed by some
 * recognizers to be able to store their configuration in a portable way.
 */
int		gicInputRegister(gic_handle_t hand,gii_input_t inp);

/*
 * Recognizerdrivers. The work-monkeys. The recognizers given in gicOpen are automatically
 * open. But you might want to adjust that list.
 */

int			gicRecognizerDriverRegister  (gic_handle_t hand,gic_recognizerdriver *driver);
int			gicRecognizerDriverUnregister(gic_handle_t hand,gic_recognizerdriver *driver);
gic_recognizerdriver   *gicRecognizerDriverLookup    (gic_handle_t hand,const char *name);

/*
 * Stage 5: Recognizers
 * the helper libraries that decide whether some event is matched.
 */

int	gicRecognizerWrite	 (gic_handle_t hand,gic_recognizer *recognizer, FILE *file);
int    	gicRecognizerTrain	 (gic_handle_t hand,gic_recognizer **recognizerlist,gii_event *event);

#define	gicRecognizerTrainStart(hand,x) gicRecognizerTrain(hand,x,NULL)
#define	gicRecognizerTrainStop(hand,x)  gicRecognizerTrain(hand,x,NULL)

int	gicRecognizerTrainAdd	 (gic_handle_t hand,gic_recognizer **recognizerlist,gic_recognizer *newone);
int	gicRecognizerTrainMove	 (gic_handle_t hand,gic_recognizer **recognizerlist,gic_recognizer *newone);
int	gicRecognizerFindConflict(gic_handle_t hand,gic_recognizer *recognizer, gic_recognizer *compare_to);
int	gicRecognizerGetName	 (gic_handle_t hand,gic_recognizer *recognizer, char *string, int maxlen);
int	gicRecognizerGetOpposite (gic_handle_t hand,gic_recognizer *recognizer,gic_recognizer **opposite);

/*
 * Stage 5B: Actions
 * the functions called, when features get active
 */
int     gicActionWrite		(gic_handle_t hand,gic_actionlist *aclist,FILE *where);
int	gicActionMapActions	(gic_handle_t hand,gic_actionlist *aclist, gic_actionlist *actions);

/*
 * Stage 4: Features.
 * Features are individual "moves" that control something. They can have multiple
 * actions and recognizers attached.
 */

gic_feature    *gicFeatureAllocate	  (gic_handle_t hand,char *name,char *shortname);
void		gicFeatureFree		  (gic_handle_t hand,gic_feature *feature);

int		gicFeatureAttachRecognizerDriver(gic_handle_t hand, gic_feature *feature, gic_recognizerdriver *driver,void *privdata);
int		gicFeatureAttachRecognizer	(gic_handle_t hand, gic_feature *feature, gic_recognizer *recognizer);
int		gicFeatureDetachRecognizer	(gic_handle_t hand, gic_feature *feature, gic_recognizer *recognizer);
gic_recognizer *gicFeatureGetRecognizer		(gic_handle_t hand, gic_feature *feature, int number);
int		gicFeatureNumRecognizers	(gic_handle_t hand, gic_feature *feature);
int		gicFeatureGetName		(gic_handle_t hand, gic_feature *feature, char *string, int maxlen);

int		gicFeatureAttachAction	  	(gic_handle_t hand, gic_feature *feature, void (*action)(gic_handle_t hand, gic_actionlist *action,gic_feature *feature,int newstate,gic_flag flag,int recnum), void *privdata, char *name);
int		gicFeatureDetachAction	  	(gic_handle_t hand, gic_feature *feature, void (*action)(gic_handle_t hand, gic_actionlist *action,gic_feature *feature,int newstate,gic_flag flag,int recnum), void *privdata, char *name);

int		gicFeatureWrite		  	(gic_handle_t hand, gic_feature *feature, FILE *file);
gic_feature    *gicFeatureRead		  	(gic_handle_t hand, FILE *file);

int		gicFeatureMapActions	  	(gic_handle_t hand, gic_feature *feature, gic_actionlist *actions);
int		gicFeatureFindConflict	  	(gic_handle_t hand, gic_feature *feature, gic_recognizer *rec,gic_recognizer **start_and_return);

/* "Fake" a feature activation. Pretty much like a SendEvent.
 */
int		gicFeatureActivate		(gic_handle_t hand, gic_feature *feature, int newstate,gic_flag flag,int recnum);

/* Check for a feature activation. Tries all recognizers. RC is number of 
 * activations, i.e. zero for an unmatched event.
 */
int		gicFeatureHandleEvent		(gic_handle_t hand, gic_feature *feature, gii_event *event);

/*
 * Stage 3: Controls.
 * Contexts contain all "Features" that make up a given set like all things
 * you might need to do for navigating a menu.
 */

gic_control    *gicControlAllocate	(gic_handle_t hand, char *name,char *shortname);
void		gicControlFree		(gic_handle_t hand, gic_control *control);
int		gicControlAttachFeature	(gic_handle_t hand, gic_control *control,gic_feature *feature);
int		gicControlDetachFeature	(gic_handle_t hand, gic_control *control,gic_feature *feature);
gic_feature    *gicControlLookupFeature	(gic_handle_t hand, gic_control *context,const char *name);
gic_feature    *gicControlGetFeature	(gic_handle_t hand, gic_control *context,int number);
int		gicControlNumFeatures	(gic_handle_t hand, gic_control *context);
int		gicControlGetName	(gic_handle_t hand, gic_control *control, char *string, int maxlen);
int		gicControlWrite		(gic_handle_t hand, gic_control *control,FILE *file);
gic_control    *gicControlRead		(gic_handle_t hand, FILE *file);
int		gicControlMapActions	(gic_handle_t hand, gic_control *control,gic_actionlist *actions);
int		gicControlFindConflict	(gic_handle_t hand, gic_control *control,gic_recognizer *feature,gic_recognizer **start_and_return,gic_feature **optional);

/* Check for a control activation. Loops through all features.
 * RC is number of total activations, i.e. zero for an unmatched event.
 */
int		gicControlHandleEvent	 (gic_handle_t hand,gic_control *control, gii_event *event);

/*
 * Stage 2: Contexts.
 * Contexts describe _all_ "Controls" that are applicable in a given state of
 * an application.
 */

gic_context    *gicContextAllocate	(gic_handle_t hand,char *name);
void		gicContextFree		(gic_handle_t hand,gic_context *context);
int		gicContextAttachControl	(gic_handle_t hand,gic_context *context,gic_control *control);
int		gicContextDetachControl	(gic_handle_t hand,gic_context *context,gic_control *control);
gic_control    *gicContextLookupControl	(gic_handle_t hand,gic_context *context,const char *name);
gic_control    *gicContextGetControl	(gic_handle_t hand,gic_context *context,int number);
int		gicContextNumControls	(gic_handle_t hand,gic_context *context);
int		gicContextGetName	(gic_handle_t hand,gic_context *context, char *string, int maxlen);
int		gicContextWrite		(gic_handle_t hand,gic_context *context,FILE *file);
gic_context    *gicContextRead		(gic_handle_t hand,FILE *file);
int		gicContextMapActions	(gic_handle_t hand,gic_context *context,gic_actionlist *actions);
int		gicContextFindConflict	(gic_handle_t hand,gic_context *context,gic_recognizer *rec,gic_recognizer **start_and_return,gic_feature **optional);

/* Check for a context activation. Loops through all controls and features.
 * RC is number of total activations, i.e. zero for an unmatched event.
 */
int		gicContextHandleEvent	(gic_handle_t hand,gic_context *context, gii_event *event);

/*
 * Stage 1: Heads.
 * This is a convenience structure that can be used to group contexts.
 * There are some convenience functions that will load/restore complete heads
 * allowing to save configuration with a single call.
 */

gic_head       *gicHeadAllocate		(gic_handle_t hand,char *name);
void		gicHeadFree		(gic_handle_t hand,gic_head *head);
int		gicHeadAttachContext	(gic_handle_t hand,gic_head *head,gic_context *context);
int		gicHeadDetachContext	(gic_handle_t hand,gic_head *head,gic_context *context);
gic_context    *gicHeadLookupContext	(gic_handle_t hand,gic_head *head,const char *name);
gic_context    *gicHeadGetContext	(gic_handle_t hand,gic_head *head,int number);
int		gicHeadNumContexts	(gic_handle_t hand,gic_head *head);
int		gicHeadWrite		(gic_handle_t hand,gic_head *head,FILE *file);
gic_head       *gicHeadRead		(gic_handle_t hand,FILE *file);
int		gicHeadMapActions	(gic_handle_t hand,gic_head *head,gic_actionlist *actions);

/* 
 * Miscellaneous:
 */

/* This function is NOT part of LibGIC. It is autogenerated by gic2c and used
 * for embedding a default configuration in the program executable itself.
 * You _have_ to generate the source for it with gic2c and link your program 
 * with it.
 */
size_t gicWriteDefaultConfig(FILE *outfile);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* _GGI_GIC_H */
