/*
 ******************************************************************************

 LIBGPF: plain functions

 Copyright (C) 2001-2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include <ggi/internal/gpf.h>
#include <ggi/internal/gpf_debug.h>
#include <ggi/internal/gpf_helpers.h>
#include <ggi/internal/gpf_internal.h>

#include <string.h>


int gpfIsStopped(gpf_handle_t handle)
{
	GPF_APPASSERT(handle != NULL, "Invalid handle! Please gpfOpen() it properly.\n");

	return (!handle->started) ? 1 : 0;
}	/* gpfIsStopped */



int gpfStartStream(gpf_handle_t handle)
{
	int idx;

	GPF_APPASSERT(handle != NULL,
			"Invalid handle! Please gpfOpen() it properly.\n");

	if (gpfIsEOS(handle)) {
		GPFDPRINT("gpfStartStream: End Of Stream reached - stop transfer!\n");
		gpfStopStream(handle);
	} else {
		handle->started = 1;

		ggCurTime(&LIBGPF_TS_MARK(handle));

		for_each_handle_pipeline(handle, idx) {

			LIBGPF_SKIP_CLOSED_PIPELINE(handle, idx);

			if ((LIBGPF_PIPELINE(handle, idx)->pipe_index != NULL)
			  && (handle->props.flags & GPF_TIMESTAMP_CONTINUE))
			{
				/* we continue here and do not restart */
				GPFDPRINT("gpfStreamStart: stream transfer continues on pipeline %i\n",
					idx);
				continue;
			}	/* if */

			LIBGPF_PIPELINE(handle, idx)->pipe_lasttransfer = NULL;
			LIBGPF_PIPELINE(handle, idx)->pipe_index
			   = LIBGPF_PIPELINE(handle, idx)->pipeline->item_head;

			GPFDPRINT("gpfStreamStart: stream transfer (re)started on pipeline %i\n",
				idx);
		}	/* for */

	}	/* if */

	return GGI_OK;
}	/* gpfStartStream */


int gpfStopStream(gpf_handle_t handle)
{
	GPF_APPASSERT(handle != NULL, "Invalid handle! Please gpfOpen() it properly.\n");

	handle->started = 0;
	GPFDPRINT_CORE("gpfStopStream: stream transfer stopped\n");
	return GGI_OK;
}	/* gpfStopStream */


int gpfIsEOS(gpf_handle_t handle)
{
	int is_eos;
	int idx;

	GPF_APPASSERT(handle != NULL,
		"Invalid handle! Please gpfOpen() it properly.\n");

	for_each_handle_input(handle, idx) {

		LIBGPF_SKIP_CLOSED_INPUT(handle, idx);

		is_eos = LIBGPF_INPUT(handle, idx)->opproto->eos(LIBGPF_INPUT(handle, idx));
		GPFDPRINT_CORE("gpfIsEOS: input stream (%i) %s\n",
			idx, (is_eos) ? "reached End Of Stream" : "can continue");

		if (is_eos) return is_eos;
	}	/* for */

	for_each_handle_output(handle, idx) {

		LIBGPF_SKIP_CLOSED_OUTPUT(handle, idx);

		is_eos = LIBGPF_OUTPUT(handle, idx)->opproto->eos(LIBGPF_OUTPUT(handle, idx));
		GPFDPRINT_CORE("gpfIsEOS: output stream (%i) %s\n",
			idx, (is_eos) ? "reached End Of Stream" : "can continue");

		if (is_eos) return is_eos;
	}	/* for */

	return GGI_OK;
}	/* gpfIsEOS */



enum gpf_flags gpfSetFlags(struct gpf_properties_t *props,
			enum gpf_flags flags)
{
	enum gpf_flags tmp;

	LIB_ASSERT(props != NULL);

	tmp = (props->flags & GPF_RESPONSE_MASK);
	flags &= GPF_REQUEST_MASK;
	flags |= (props->flags & ~GPF_REQUEST_MASK);

	props->flags = flags;

	return tmp;
}	/* gpfSetFlags */


enum gpf_flags gpfGetFlags(const struct gpf_properties_t *props)
{
	LIB_ASSERT(props != NULL);

	return (props->flags & GPF_RESPONSE_MASK);
}	/* gpfGetFlags */



int gpfClearProperties(struct gpf_properties_t *props)
{
	LIB_ASSERT(props != NULL);
	memset(props, GPF_AUTO, sizeof(struct gpf_properties_t));

	props->flags = GPF_TIMESTAMP_CONTINUE;
	props->flags |= GPF_PIPE_OPT4SPEED;

	return GGI_OK;
}	/* gpfClearProperties */


int gpfSetProperties(gpf_handle_t handle, const struct gpf_properties_t *props)
{
	int flags;
	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(props != NULL);

	if (!gpfIsStopped(handle)) {
		return GGI_EBUSY;
	}	/* if */

	flags = props->flags;
	if ((flags & GPF_TIMESTAMP_MASK) == 0) {
		flags |= (LIBGPF_FLAGS(handle) & GPF_TIMESTAMP_MASK);
	}	/* if */
	if ((flags & GPF_PIPE_MASK) == 0) {
		flags |= (LIBGPF_FLAGS(handle) & GPF_PIPE_MASK);
	}	/* if */
	memcpy(&(LIBGPF_PROPS(handle)), props, sizeof(struct gpf_properties_t));
	LIBGPF_FLAGS(handle) = flags;

	return GGI_OK;
}	/* gpfSetProperties */


int gpfGetProperties(gpf_handle_t handle, struct gpf_properties_t *props)
{
	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(props != NULL);

	memcpy(props, &(LIBGPF_PROPS(handle)), sizeof(struct gpf_properties_t));

	return GGI_OK;
}	/* gpfGetProperties */




/* The whence param requires one GPF_SEEK_OPMASK value and
 * one GPF_SEEK_POSMASK.
 */
int gpfTimeStamp(gpf_handle_t handle, struct timeval *tv, int whence)
{
	int idx;

	/* check for validity */
	switch (whence & GPF_SEEK_OPMASK) {
	case 0:
	case GPF_SEEK_OPMASK:
		return GGI_ENOMATCH;
	}	/* switch */

	switch (whence & GPF_SEEK_POSMASK) {
	case 0:
	case GPF_SEEK_POSMASK:
		return GGI_ENOMATCH;
	}	/* switch */


	/* from here, we can assume, that all values
	 * are valid.
	 */
	switch (whence & GPF_SEEK_OPMASK) {
	case GPF_SEEK_GET:
		if (whence & GPF_SEEK_REL) {
			if (whence & GPF_SEEK_LAST) {
				/* get current position relative to last
				 * available timestamp
				 */
				LIBGPF_INPUT(handle, 0)->timestamp(
					LIBGPF_INPUT(handle, 0),
					tv, GPF_SEEK_GET | GPF_SEEK_LAST);
				_gpf_ts_sub(tv, tv,
					&LIBGPF_TS_CURRENT(handle));
				return GGI_OK;

			} else if (whence & GPF_SEEK_END) {
				/* get current position relative to the
				 * end
				 */
				LIBGPF_INPUT(handle, 0)->timestamp(
					LIBGPF_INPUT(handle, 0),
					tv, GPF_SEEK_GET | GPF_SEEK_END);
				_gpf_ts_sub(tv, tv, &LIBGPF_TS_CURRENT(handle));
				return GGI_OK;

			}	/* if */

			/* else get the relative timestamp position
			 * to the start, which is the absolute value
			 */
			memcpy(tv, &LIBGPF_TS_CURRENT(handle),
				sizeof(struct timeval));
			return GGI_OK;

		} else if (whence & GPF_SEEK_ABS) {
			if (whence & GPF_SEEK_LAST) {
				goto get_seek_last_abs;
			} else if (whence & GPF_SEEK_END) {
				goto get_seek_end_abs;
			}	/* if */

			memcpy(tv, &LIBGPF_TS_CURRENT(handle),
				sizeof(struct timeval));
			return GGI_OK;
		}	/* if */

		switch (whence & GPF_SEEK_POSMASK) {
		case GPF_SEEK_LAST:
get_seek_last_abs:
			/* this information can only come from
			 * the (first) input target
			 */
			return LIBGPF_INPUT(handle, 0)->timestamp(
					LIBGPF_INPUT(handle, 0),
					tv, GPF_SEEK_GET | GPF_SEEK_LAST);

		case GPF_SEEK_END:
get_seek_end_abs:
			return LIBGPF_INPUT(handle, 0)->timestamp(
					LIBGPF_INPUT(handle, 0),
					tv, GPF_SEEK_GET | GPF_SEEK_END);
		}	/* switch */

		return GGI_ENOMATCH;

	case GPF_SEEK_SET:
		{
		int rc;
		struct timeval old_tv;

		/* First inform the targets about the change
		 */

		for_each_handle_input(handle, idx) {

			LIBGPF_SKIP_CLOSED_INPUT(handle, idx);

			LIBGPF_INPUT(handle, idx)->timestamp(
				LIBGPF_INPUT(handle, idx),
				&old_tv, GPF_SEEK_GET | GPF_SEEK_ABS);

			rc = LIBGPF_INPUT(handle, idx)->timestamp(
				LIBGPF_INPUT(handle, idx), tv, whence);
			if (rc < GGI_OK) return rc;
		}	/* for */


		for_each_handle_output(handle, idx) {

			LIBGPF_SKIP_CLOSED_OUTPUT(handle, idx);

			rc = LIBGPF_OUTPUT(handle, idx)->timestamp(
				LIBGPF_OUTPUT(handle, idx), tv, whence);
			if (rc < GGI_OK) {
				rc = LIBGPF_INPUT(handle, idx)->timestamp(
					LIBGPF_INPUT(handle, idx),
					&old_tv, GPF_SEEK_SET | GPF_SEEK_ABS);
				return rc;
			}	/* if */
		}	/* for */

		/* Then do it
		 */
		switch (whence & GPF_SEEK_POSMASK) {
		case GPF_SEEK_REL:
			_gpf_ts_add(&LIBGPF_TS_CURRENT(handle),
					&LIBGPF_TS_CURRENT(handle),
					tv);
			break;

		case GPF_SEEK_ABS:
			LIBGPF_TS_CURRENT(handle) = *tv;
			break;

		case GPF_SEEK_LAST:
			rc = LIBGPF_INPUT(handle, 0)->timestamp(
				LIBGPF_INPUT(handle, 0),
				&LIBGPF_TS_CURRENT(handle),
				GPF_SEEK_GET | GPF_SEEK_LAST);
			if (rc < GGI_OK) {
				return rc;
			}	/* if */

		case GPF_SEEK_END:
			LIBGPF_TS_CURRENT(handle) = LIBGPF_TS_TOTAL(handle);
			break;

		}	/* switch */

		return GGI_OK;
		}

	}	/* switch */

	return GGI_ENOMATCH;
}	/* gpfTimeStamp */




int gpfTransferNextStreamBlock(gpf_handle_t handle, gpfIndicatorCallback cbI)
{
	int rc = GGI_OK;
	struct timeval newtv, diff;
	struct gpf_pipeline_item_t *pipe_tmp;
	struct timeval transfer_timestamps;
	struct timeval tmp, sum;
	int idx;


	GPFDPRINT_TRANSFER("gpfTransferNextStreamBlock(%p, %p) called\n",
			handle, cbI);

	if (gpfIsStopped(handle)) {
		GPFDPRINT("stream transfer is stopped - abort\n");

		/* stream is stopped/hasn't been started */
		return 0;
	}	/* if */


	LIB_ASSERT(LIBGPF_TS_ISSET(&LIBGPF_TS_TRANSFER(handle)));
	transfer_timestamps = LIBGPF_TS_TRANSFER(handle);


	/* get the time the application took */
	ggCurTime(&newtv);
	_gpf_ts_sub(&diff, &newtv, &LIBGPF_TS_MARK(handle));

	GPFDPRINT("The application spent %.03f seconds\n",
		diff.tv_sec + (diff.tv_usec / 1000000.0F));


	/* The application took some time here, which very 
	 * likely causes some transfer fluctuation.
	 * So we speed up or slow down here a bit
	 * to keep to compensate it as good as possible.
	 */
	_gpf_ts_add(&transfer_timestamps, &transfer_timestamps, &diff);

	memcpy(&LIBGPF_TS_MARK(handle), &newtv, sizeof(struct timeval));

#if 0	/* enable this to artificially slow down the transfer.
	 * This makes it much easier to find/reproduce some bugs
	 * occuring occasionally (and in particular during slow transfer)
	 */
	transfer_timestamps.tv_sec = 0;
	transfer_timestamps.tv_usec = 1000;
#endif
	GPFDPRINT("try to transfer %.03f timestamps\n",
		transfer_timestamps.tv_sec
		+ (transfer_timestamps.tv_usec / 1000000.0F));

	LIBGPF_TS_CLEAR(&sum);
	tmp = transfer_timestamps;
	while (LIBGPF_TS_ISSET(&transfer_timestamps)) {

		for_each_handle_pipeline(handle, idx) {

			LIBGPF_SKIP_CLOSED_PIPELINE(handle, idx);

			pipe_tmp = LIBGPF_PIPELINE(handle, idx)->pipe_index;
			rc = LIBGPF_PIPELINE(handle, idx)->pipe_index->transfernextstreamblock(
				handle, LIBGPF_PIPELINE(handle, idx),
				LIBGPF_PIPELINE(handle, idx)->pipe_index,
				&tmp);
			LIBGPF_PIPELINE(handle, idx)->pipe_lasttransfer
				 = pipe_tmp;

			if (LIBGPF_PIPELINE(handle, idx)->pipe_index == NULL) {
				gpfStopStream(handle);
				fprintf(stderr, "%s:%s:%i: transfer stopped!\n",
					DEBUG_INFO);
				goto exit;
			}	/* if */

			if (rc < 0) {
				/* error occured */
				fprintf(stderr, "unexpected conversion error: %i\n",
					rc);
				fprintf(stderr, "transfer stopped\n");
				gpfStopStream(handle);

				LIBGPF_PIPELINE(handle, idx)->pipe_index
					= LIBGPF_PIPELINE(handle, idx)->pipeline->item_head;
				goto exit;
			}	/* if */
		}	/* for */


		FIXME("%s:%s:%i: use correct transfer_timestamps value here\n",
			DEBUG_INFO);

		_gpf_ts_add(&sum, &sum, &tmp);
		_gpf_ts_sub(&transfer_timestamps,
				&transfer_timestamps, &tmp);

		/* status indicator */
		if (cbI != NULL) {
			cbI(&LIBGPF_TS_CURRENT(handle),
				&LIBGPF_TS_TOTAL(handle));
		}	/* if */

		GPFDPRINT_TRANSFER("%.03f timestamps totally transferd here\n",
			LIBGPF_TS_CURRENT(handle).tv_sec
			+ (LIBGPF_TS_CURRENT(handle).tv_usec / 1000000.0F));
	}	/* while */

	ggCurTime(&LIBGPF_TS_MARK(handle));
	GPFDPRINT_TRANSFER("gpfDoNextStreamDataBlock(): leave this function and let the app do something\n");

exit:
	_gpf_ts_add(&LIBGPF_TS_CURRENT(handle),
		&LIBGPF_TS_CURRENT(handle), &tmp);

	return GGI_OK;
}	/* gpfDoNextStreamDataBlock */
