/*
 ******************************************************************************

 LIBGPF: library internals

 Copyright (C) 2000-2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#ifndef _GPF_INT_H
#define _GPF_INT_H


/* Forward declaration */
struct gpf_handle_opio_t;
struct gpf_handle_pipeline_t;


#define HAVE_STRUCT_GPF

#include <ggi/internal/internal.h>
#include <ggi/gpf.h>
#include <ggi/internal/gpf_pipe.h>
#include <ggi/internal/gpf_types.h>
#include <ggi/internal/gpf_dl.h>


enum gpf_protocol_type {

	/* invalid or unknown protocol */
	GPF_PROTO_NONE		= 0x00000000,

	/* protocol to reference a target directly */
	GPF_PROTO_TARGET	= 0x00000001,

	/* use one or more of these protocols for data-transfer
	 */
	GPF_PROTO_FILE		= 0x00000002,
	GPF_PROTO_HTTP		= 0x00000003,
	GPF_PROTO_FTP		= 0x00000004,
	GPF_PROTO_TCP		= 0x00000005,
	GPF_PROTO_UDP		= 0x00000006,
};


/* Data types for the API functions
 */

typedef int (GPFproto_open)(struct gpf_handle_opio_t *opio,
			int flags);
typedef int (GPFproto_close)(struct gpf_handle_opio_t *opio);

typedef ssize_t (GPFproto_read)(struct gpf_handle_opio_t *opio,
			void *buf, size_t size);
typedef ssize_t (GPFproto_write)(struct gpf_handle_opio_t *opio,
			void *buf, size_t size);


typedef off_t (GPFproto_lseek)(struct gpf_handle_opio_t *opio,
			off_t offset, int whence);
typedef int (GPFproto_lseek_available)(struct gpf_handle_opio_t *opio);
typedef int (GPFproto_flush)(struct gpf_handle_opio_t *opio);
typedef int (GPFproto_eos)(struct gpf_handle_opio_t *opio);


#define GPF_VERSION_HANDLE_OPPROTO	GPF_VERSION_MK(1)
struct gpf_handle_opproto_t {
	struct gpf_op_head head;

	/* only here to hold the dl stuff */
	struct gpf_pipeline_item_t *pipe;

	int is_eos;
	enum gpf_protocol_type ptype;

	GPFproto_open			*open;
	GPFproto_close			*close;
	GPFproto_read			*read;
	GPFproto_write			*write;

	GPFproto_lseek			*lseek;
	GPFproto_lseek_available	*lseek_available;
	GPFproto_flush			*flush;
	GPFproto_eos			*eos;

	void *protopriv;
};



/* communication with the conversion module */
typedef int (GPFpipe_TransferNextStreamBlock)(gpf_handle_t handle,
				struct gpf_handle_pipeline_t *hnd_pipeline,
				struct gpf_pipeline_item_t *pipe,
				struct timeval *transfered_timestamps);


typedef int (GPFpipe_getinfolist)(struct gpf_pipeline_item_t *pipe);
typedef int (GPFpipe_getformat)(struct gpf_pipeline_item_t *pipe,
				struct gpf_pipeline_item_t *from_pipe,
				int chain);
typedef int (GPFpipe_setup)(gpf_handle_t handle,
				struct gpf_handle_pipeline_t *hnd_pipeline,
				struct gpf_pipeline_item_t *pipe,
				int pipeline_channel);


#define PIPELINEITEM_FITS		0
#define PIPELINEITEM_FINDSTRING		1
#define PIPELINEITEM_MAXELEMENTS	2

#define INFOLIST_INPUT			0
#define INFOLIST_OUTPUT			1
#define INFOLIST_MAXCHAINS		2


#define GPF_VERSION_HANDLE_PIPE		GPF_VERSION_MK(1)
struct gpf_pipeline_item_t {
	struct gpf_op_head head;

	gpf_dlhandle_l  *dlhandle;


	struct gpf_pipeline_item_t *prev;
	struct gpf_pipeline_item_t *next;

	struct gpf_infolist_t *input_head;
	struct gpf_infolist_t *output_head;

	struct gpf_infolist_t *mark[PIPELINEITEM_MAXELEMENTS][INFOLIST_MAXCHAINS];

	GPFpipe_setup		*setup;
	GPFpipe_getformat	*getformat;
	GPFpipe_getinfolist     *getinfolist;
	GPFpipe_TransferNextStreamBlock   *transfernextstreamblock;

	/* used for cloning this pipe */
	char pipe_string[GGI_MAX_APILEN];

	/* true, when this belongs to a target */
	int istarget;

	/* buffer holding the data coming from the input.
	 * For the output use simply the buffer of the next
	 * pipe element.
	 */
	void *buf;
	size_t bufsize;
	size_t cache_size;
	size_t read_buf_pos, write_buf_pos;
	size_t readbytes, writebytes;

	/* one buffer for conversion purpose only
	 * - only used by the sublibs themself
	 */
	void *buf_conv;
	size_t buf_conv_size;

	/* resolution in pixels coming from the input */
	int width;
	int height;

	/* pixelsize coming from the input */
	int depth, size;


	/* runtime */
	double runtime;

	/* memory usage */
	size_t memory_usage;

	/* sublib specific data */
	void *pipepriv;
};



typedef ssize_t (GPFpipe_read)(struct gpf_pipeline_item_t *pipe,
				size_t size, size_t maxsize);

typedef ssize_t (GPFpipe_write)(struct gpf_pipeline_item_t *pipe,
				size_t size, size_t maxsize);


#define GPF_VERSION_HANDLE_PIPELINE	GPF_VERSION_MK(1)
struct gpf_pipeline_t {
	struct gpf_op_head head;

	struct gpf_pipeline_t *next;

	/* head of the list */
	struct gpf_pipeline_item_t *item_head;


	GPFpipe_read		*read;
	GPFpipe_write		*write;
};




typedef int (GPFio_open)(struct gpf_handle_opio_t *opio);
typedef int (GPFio_close)(struct gpf_handle_opio_t *opio);

typedef int (GPFio_timestamp)(struct gpf_handle_opio_t *opio,
				struct timeval *tv, int whence);

typedef ssize_t (GPFio_read)(struct gpf_handle_opio_t *opio,
			void *buf, size_t size);
typedef ssize_t (GPFio_write)(struct gpf_handle_opio_t *opio,
			void *buf, size_t size);


/* called by the target protocol */
typedef int (GPFio_doio)(struct gpf_handle_opio_t *opio,
			const char *args, va_list ap);






typedef int (GPFio_pipe_getinfolist)(struct gpf_handle_opio_t *opio);


#define GPF_VERSION_HANDLE_OPIO		GPF_VERSION_MK(1)
struct gpf_handle_opio_t {
	struct gpf_op_head head;

	struct gpf_handle_opproto_t	*opproto;
	struct gpf_pipeline_item_t	*pipe;

	/* This type field allows the sublib to identify themself,
	 * if they interact as input or output target
	 */
	int type;

	/* current frame-number */
	int cur_frame;		/* needed for (interpolated) frames
				 * with timestamps
				 */

	/* timestamps */
	struct timeval cur_ts;	/* current timestamp */
	struct timeval last_ts;	/* last available timestamp - input only */

	/* open/close are called _after_ parsing is done
	 */
	GPFio_open			*open;
	GPFio_close			*close;

	GPFio_timestamp			*timestamp;

	GPFio_read			*read;
	GPFio_write			*write;


	/* called by the protocol */
	GPFio_doio		*doio;

	/* used to determine pipe sublib that most fits */
	GPFio_pipe_getinfolist	*pipe_getinfolist;
};


struct gpf_handle_pipeline_t {

	struct gpf_handle_opio_t *opio_input;
	struct gpf_handle_opio_t *opio_output;

	struct gpf_pipeline_t *pipeline;
	struct gpf_pipeline_item_t *pipe_index;
	struct gpf_pipeline_item_t *pipe_lasttransfer;
};



#define GPF_VERSION_HANDLE		GPF_VERSION_MK(1)
struct gpf_handle {
	unsigned int	version;
	int		mutex_valid;	/* != 0, when mutex is successfully created */
	void		*mutex;		/* Lock when changing.. */
	gpf_handle_t	next;		/* Next one in the list... */


	struct gpf_properties_t props;

	/* true, when started, false, when stopped */
	int started;

	int num_inputs;
	int num_outputs;

	struct gpf_handle_opio_t **opio_input;
	struct gpf_handle_opio_t **opio_output;

	int num_pipelines;
	struct gpf_handle_pipeline_t *pipeline;


	/* how much to timestamps to transfer */
	struct timeval transfer_timestamps;

	/* tranfer state */
	struct timeval current_timestamp;

	/* only to be filled out by the input target */
	struct timeval total_timestamps;

	/* helper variable to determine how long the application
	 * break the transfer. This is used to eventually skip
	 * some timestamps then.
	 */
	struct timeval tv_mark;
};



#endif	/* _GPF_INT_H */
