/*
 ******************************************************************************

 LIBGPF: libggi io pipeline communication functions

 Copyright (C) 2001-2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include "libggi_gpf.h"


static int _LIBGGI_getinfonr(struct libggigpf_priv *priv,
				int chain)
{
	switch (chain) {
	case INFOLIST_INPUT:
		if ((GT_SCHEME(priv->mode.graphtype) & GT_PALETTE)
		   || (GT_SCHEME(priv->mode.graphtype) & GT_STATIC_PALETTE))
		{
			return 1;
		}	/* if */
		return 2;

	case INFOLIST_OUTPUT:
		if ((GT_SCHEME(priv->mode.graphtype) & GT_PALETTE)
		   || (GT_SCHEME(priv->mode.graphtype) & GT_STATIC_PALETTE))
		{
			return 1;
		}	/* if */
		return 2;

	}	/* switch */

	return 0;
}	/* _LIBGGI_getinfonr */




/* fills in the information of pixelformats it can handle
 * from the input
 */
static void _LIBGGI_fillinfo_input(struct gpf_handle_opio_t *opio,
				struct gpf_infolist_t *item,
				int nr)
{
	struct libggigpf_priv *priv;
	const ggi_pixelformat *ggi_pf = NULL;

	priv = LIBGGI_PRIV(opio);


	ggi_pf = ggiGetPixelFormat(priv->vis);

	memset(&item->pf, 0,
		sizeof(struct gpf_pixelformat_t));
	item->pf.cf = GPF_CF_RGBA;
	item->pf.flags = GPF_PIPE_DF_RGBA;

	item->nr = nr;

	if ((GT_SCHEME(priv->mode.graphtype) & GT_PALETTE)
	   || (GT_SCHEME(priv->mode.graphtype) & GT_STATIC_PALETTE))
	{
		LIB_ASSERT(nr == 0);
		goto palette_only;
	}	/* if */


	switch (nr) {
	case 0:
		/* native RGB format
		 */

		GPFDPRINT_TARGET("%s:%s:%i: native format\n", DEBUG_INFO);

		item->pf.depth = GT_DEPTH(priv->mode.graphtype);
		item->pf.size = GT_SIZE(priv->mode.graphtype);

		if (GT_SCHEME(priv->mode.graphtype) == GT_GREYSCALE) {
			item->pf.flags |= GPF_PIPE_DF_GREYSCALE;
		}	/* if */

		item->pf.type.rgba.red_mask = ggi_pf[0].red_mask;
		item->pf.type.rgba.red_shift = ggi_pf[0].red_shift;
		item->pf.type.rgba.green_mask = ggi_pf[0].green_mask;
		item->pf.type.rgba.green_shift = ggi_pf[0].green_shift;
		item->pf.type.rgba.blue_mask = ggi_pf[0].blue_mask;
		item->pf.type.rgba.blue_shift = ggi_pf[0].blue_shift;
		item->pf.type.rgba.alpha_mask = ggi_pf[0].alpha_mask;
		item->pf.type.rgba.alpha_shift = ggi_pf[0].alpha_shift;

		item->memory_usage = PF_ByPP(item->pf.size)
					* priv->mode.virt.x
					* priv->mode.virt.y;

		/* algorithm with a linear runtime */
		item->runtime = item->memory_usage * 1.0;
		break;

	case 1:
		/* RGB: r16g16b16a16
		 */
palette_only:
		GPFDPRINT_TARGET("%s:%s:%i: unmapped format\n", DEBUG_INFO);

		item->pf.depth = sizeof(ggi_color) * 8;
		item->pf.size = sizeof(ggi_color) * 8;

		item->memory_usage = PF_ByPP(item->pf.size)
				* priv->mode.virt.x
				* priv->mode.virt.y;

		/* unmapping the data costs an extra loop
		 * and doubles the runtime
		 */
		item->runtime = item->memory_usage * 2.0;


		item->pf.type.rgba.red_mask = 0xFFFF;
		item->pf.type.rgba.green_mask = 0xFFFF;
		item->pf.type.rgba.blue_mask = 0xFFFF;
		item->pf.type.rgba.alpha_mask = 0xFFFF;

		break;


	default:
		fprintf(stderr, "Unknown nr: %i\n", nr);
		break;
	}	/* switch */

	return;
}	/* _LIBGGI_fillinfo_input */



/* fills in the information of pixelformats it can offer
 */
#define _LIBGGI_fillinfo_output	_LIBGGI_fillinfo_input




static void _LIBGGI_fillinfo(struct gpf_handle_opio_t *opio,
			struct gpf_infolist_t *item,
			int nr, int chain)
{
	LIB_ASSERT(opio != NULL);
	LIB_ASSERT(item != NULL);

	switch (chain) {
	case INFOLIST_INPUT:
		_LIBGGI_fillinfo_input(opio, item, nr);
		break;

	case INFOLIST_OUTPUT:
		_LIBGGI_fillinfo_output(opio, item, nr);
		break;

	}	/* switch */

	return;
}	/* _LIBGGI_fillinfo */



static struct gpf_infolist_t *_LIBGGI_getInfolist(
					struct gpf_handle_opio_t *opio,
					int chain)
{
	int nr, i;
	struct gpf_infolist_t *head = NULL;
	struct gpf_infolist_t *item = NULL;


	LIB_ASSERT(opio != NULL);

	nr = _LIBGGI_getinfonr(LIBGGI_PRIV(opio), chain);
	item = _gpfInfoListCreate();
	if (!item) {
		goto err0;
	}	/* if */

	_LIBGGI_fillinfo(opio, item, 0, chain);
	head = item;

	for (i = 1; i < nr; i++) {
		item = _gpfInfoListCreate();
		if (!item) {
			goto err1;
		}	/* if */

		_LIBGGI_fillinfo(opio, item, i, chain);

		_gpfInfoListAppend(head, item);
	}	/* for */


	return head;


err1:
	_gpfInfoListDestroy(&item);
	_gpfInfoListDestroy(&head);
err0:
	return NULL;
}	/* _LIBGGI_getInfolist */



int GPF_LIBGGI_pipe_getinfolist(struct gpf_handle_opio_t *opio)
{
	struct gpf_pipeline_item_t *pipe;
	int rc = GGI_OK;


	pipe = opio->pipe;

	LIB_ASSERT(pipe != NULL);

	switch (opio->type) {
	case GPF_DLTYPE_INPUT:
		LIB_ASSERT(pipe->output_head == NULL);

		GPFDPRINT_TARGET("%s:%s:%i: get infolist of the input target\n",
				DEBUG_INFO);

		/* This target interacts as an input, so we need the
		 * information, what it offers...
		 */
		pipe->output_head = _LIBGGI_getInfolist(opio, INFOLIST_OUTPUT);
		if (!pipe->output_head) {
			rc = GGI_ENOMEM;
			goto err0;
		}	/* if */
		break;


	case GPF_DLTYPE_OUTPUT:
		LIB_ASSERT(pipe->input_head == NULL);

		GPFDPRINT_TARGET("%s:%s:%i: get infolist of the output target\n",
				DEBUG_INFO);

		/* This target interacts as an output, so we need the
		 * information, what it can handle from the input.
		 */
		pipe->input_head = _LIBGGI_getInfolist(opio, INFOLIST_INPUT);
		if (!pipe->input_head) {
			rc = GGI_ENOMEM;
			goto err0;
		}	/* if */
		break;

	}	/* switch */

err0:
	return rc;
}	/* GPF_LIBGGI_pipe_getinfolist */
