/*
 ******************************************************************************

 LIBGPF: null io pipeline communication functions

 Copyright (C) 2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include "null_gpf.h"


static int _null_getinfonr(int chain)
{
	switch (chain) {
	case INFOLIST_INPUT:

		/* ... when this target acts as input */
		return 1;

	case INFOLIST_OUTPUT:

		/* ... when this target acts as output */
		return 1;

	}	/* switch */

	return 0;
}	/* _null_getinfonr */


/* fills in the information of pixelformats it can handle
 * from the input
 */
static void _null_fillinfo_input(struct gpf_handle_opio_t *opio,
			struct gpf_infolist_t *item,
			int nr)
{
	struct nullgpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = NULL_PRIV(opio);

	/* Initizalize everything with zero */
	memset(&item->pf, 0, sizeof(struct gpf_pixelformat_t));

	switch (nr) {
	case 0:
		item->pf.cf = GPF_CF_RAW;
		item->pf.flags = GPF_PIPE_DF_RAW | GPF_PIPE_DF_ALLVARIANTS;

		LIB_ASSERT(priv->input_data_size > 0);
		item->memory_usage = priv->input_data_size;
		item->runtime = item->memory_usage * 1.0;
		break;

	default:
		fprintf(stderr, "Unknown nr: %i\n", nr);
		break;
	}	/* switch */

	return;
}	/* _null_fillinfo_input */



/* fills in the information of pixelformats it can convert to
 */
static void _null_fillinfo_output(struct gpf_handle_opio_t *opio,
			struct gpf_infolist_t *item,
			int nr)
{
	struct nullgpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = NULL_PRIV(opio);


	/* Initizalize everything with zero */
	memset(&item->pf, 0, sizeof(struct gpf_pixelformat_t));

	switch (nr) {
	case 0:
		item->pf.cf = GPF_CF_RAW;
		item->pf.flags = GPF_PIPE_DF_RAW | GPF_PIPE_DF_ALLVARIANTS;

		if (priv->output_data_size >= priv->input_data_size) {
			item->memory_usage = priv->output_data_size;
		} else {
			item->memory_usage = priv->input_data_size;
		}	/* if */

		item->runtime = item->memory_usage * 1.0;
		break;

	default:
		fprintf(stderr, "Unknown nr: %i\n", nr);
		break;
	}	/* switch */

	return;
}	/* _null_fillinfo_output */




static void _null_fillinfo(struct gpf_handle_opio_t *opio,
			struct gpf_infolist_t *item,
			int nr, int chain)
{
	switch (chain) {
	case INFOLIST_INPUT:
		_null_fillinfo_input(opio, item, nr);
		break;

	case INFOLIST_OUTPUT:
		_null_fillinfo_output(opio, item, nr);
		break;

	}	/* switch */

	return;
}	/* _null_fillinfo */



static struct gpf_infolist_t *_null_getInfolist(
					struct gpf_handle_opio_t *opio,
					int chain)
{
	int nr, i;
	struct gpf_infolist_t *head = NULL;
	struct gpf_infolist_t *item = NULL;


	nr = _null_getinfonr(chain);
	item = _gpfInfoListCreate();
	if (!item) {
		goto err0;
	}	/* if */
	_null_fillinfo(opio, item, 0, chain);
	head = item;

	for (i = 1; i < nr; i++) {
		item = _gpfInfoListCreate();
		if (!item) {
			goto err1;
		}	/* if */

		_null_fillinfo(opio, item, i, chain);

		_gpfInfoListAppend(head, item);
	}	/* for */


	return head;


err1:
	_gpfInfoListDestroy(&item);
	_gpfInfoListDestroy(&head);
err0:
	return NULL;
}	/* _null_getInfolist */


int GPF_NULL_pipe_getinfolist(struct gpf_handle_opio_t *opio)
{
	int rc = GGI_OK;
	struct gpf_pipeline_item_t *pipe;

	pipe = opio->pipe;

	LIB_ASSERT(pipe != NULL);

	LIB_ASSERT(pipe->input_head == NULL);
	LIB_ASSERT(pipe->output_head == NULL);


	switch (opio->type) {
	case GPF_DLTYPE_INPUT:
		/* This target interacts as an input, so we need the
		 * information, what it offers to the next pipe...
		 */
		pipe->output_head = _null_getInfolist(opio, INFOLIST_OUTPUT);
		if (!pipe->output_head) {
			rc = GGI_ENOMEM;
			goto err0;
		}	/* if */
		break;


	case GPF_DLTYPE_OUTPUT:
		/* This target interacts as an output, so we need the
		 * information, what it can handle from the previous pipe...
		 */
		pipe->input_head = _null_getInfolist(opio, INFOLIST_INPUT);
		if (!pipe->input_head) {
			rc = GGI_ENOMEM;
			goto err0;
		}	/* if */
		break;

	}	/* switch */

err0:
	return rc;
}	/* GPF_NULL_pipe_getinfolist */
