/*
 ******************************************************************************

 LIBGPF: ppm io target initialisation

 Copyright (C) 2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include "ppm_gpf.h"
#include <ggi/internal/gpf_internal.h>


static const gg_option optlist[] =
{
	{ "opmode", "frame" },
	{ "format", "P6" },
};

#define OPT_OPMODE	0
#define OPT_FORMAT	1

#define NUM_OPTS        (sizeof(optlist)/sizeof(gg_option))



static int GPFinit(struct gpf_handle_opio_t *opio,
		   const char *args, va_list ap)
{
	gg_option options[NUM_OPTS];
	char str[GPF_MAXPARAM_LEN];
	enum ppm_opmode_t opmode;
	enum ppm_format_t format;

	GPFDPRINT_TARGET("%s:%s:%i: entered - args (%s)\n",
			DEBUG_INFO, args);

	if (!args) {
		/* there are no arguments given */
		return GGI_OK;
	}	/* if */

	memcpy(options, optlist, sizeof(options));
	args = ggParseOptions(args, options, NUM_OPTS);
	if (args == NULL) {
		fprintf(stderr, "io-ppm: error in arguments.\n");
		return GGI_EARGINVAL;
	}	/* if */

	GPFDPRINT_TARGET("%s:%s:%i: io-ppm has args.\n",
			DEBUG_INFO);

	if (options[OPT_OPMODE].result) {
		GPFDPRINT_TARGET("io-ppm: opmode=\"%s\"\n",
				options[OPT_OPMODE].result);

		_gpfParseArgs(options[OPT_OPMODE].result, &ap, &str);

		opmode = OPMODE_FRAME;
		if (strncmp(str, "frame", 6) == 0) {
			opmode = OPMODE_FRAME;
		}	/* if */
		if (strncmp(str, "hline", 6) == 0) {
			opmode = OPMODE_HLINE;
		}	/* if */
		if (strncmp(str, "vline", 6) == 0) {
			opmode = OPMODE_VLINE;
		}	/* if */

		PPM_PRIV(opio)->opmode = opmode;
	}	/* if */


	if (options[OPT_FORMAT].result) {
		GPFDPRINT_TARGET("io-ppm: format=\"%s\"\n",
				options[OPT_FORMAT].result);

		_gpfParseArgs(options[OPT_FORMAT].result, &ap, &str);

		format = PPM_FORMAT_P6;
		if (strncmp(str, "P6", 3) == 0) {
			format = PPM_FORMAT_P6;
		}	/* if */

		if (strncmp(str, "P3", 3) == 0) {
			format = PPM_FORMAT_P3;
		}	/* if */

		PPM_PRIV(opio)->format = format;
	}	/* if */

	GPFDPRINT_TARGET("%s:%s:%i: leaving\n", DEBUG_INFO);

	return GGI_OK;
}	/* GPFinit */



static int GPFopen(struct gpf_handle_opio_t *opio,
		   struct gpf_pipeline_item_t *pipe,
		   struct gpf_dlhandle *dlh, const char *args,
		   va_list ap, uint32 *dlret)
{
	int rc;
	struct ppmgpf_priv *priv;

	GPFDPRINT_LIBS("GPFopen(%p, %p, %p, %s, %p, %p) called for ppm sublib\n",
		opio, pipe, dlh, args ? args : "(NULL)", ap, dlret);

	LIBGPF_IO_PRIVATE(opio) = malloc(sizeof(struct ppmgpf_priv));
	if (LIBGPF_IO_PRIVATE(opio) == NULL) return GGI_ENOMEM;


	priv = PPM_PRIV(opio);

	memset(priv, 0, sizeof(struct ppmgpf_priv));
	priv->opmode = OPMODE_FRAME;

	rc = GPFinit(opio, args, ap);

	if (rc == GGI_OK) {
		opio->open = GPF_PPM_open;
		opio->close = GPF_PPM_close;

		opio->timestamp = GPF_PPM_timestamp;

		opio->read = GPF_PPM_read;
		opio->write = GPF_PPM_write;

		opio->pipe->getformat = GPF_PPM_getformat;
		opio->pipe_getinfolist = GPF_PPM_pipe_getinfolist;

		if (LIBGPF_IO_ISINPUT(opio)) {
			opio->pipe->setup = GPF_PPM_setup_input;
			opio->pipe->transfernextstreamblock
				= GPF_PPM_transfernextstreamblock_input;
		}	/* if */

		if (LIBGPF_IO_ISOUTPUT(opio)) {
			opio->pipe->setup = GPF_PPM_setup_output;
			opio->pipe->transfernextstreamblock
				= GPF_PPM_transfernextstreamblock_output;
		}	/* if */
	}	/* if */

	*dlret = (GPF_DL_OPIO);
	return rc;
}	/* GPFopen */


static int GPFclose(struct gpf_pipeline_item_t *pipe,
		    struct gpf_dlhandle *dlh)
{
	GPFDPRINT_LIBS("GPFclose(%p, %p) called for ppm sublib\n",
			pipe, dlh);

	if (pipe->buf_conv_size > 0) {
		free(pipe->buf_conv);
		pipe->buf_conv = NULL;
		pipe->buf_conv_size = 0;
	}	/* if */

	free(LIBGPF_PIPE_PRIVATE(pipe));
	LIBGPF_PIPE_PRIVATE(pipe) = NULL;

	return GGI_OK;
}	/* GPFclose */


int GPFdl_ppm(int func, void **funcptr)
{
	switch (func) {
	case GGIFUNC_open:
		*funcptr = GPFopen;
		return GGI_OK;
	case GGIFUNC_exit:
		*funcptr = NULL;
		return GGI_OK;
	case GGIFUNC_close:
		*funcptr = GPFclose;
		return GGI_OK;
	default:
		*funcptr = NULL;
	}	/* switch */

	return GGI_ENOTFOUND;
}	/* GPFdl_ppm */


/* This should be included by exactly _one_ file in each sublib. */
#include <ggi/internal/ggidlinit.h>
