/*
 ******************************************************************************

 LIBGPF: pipe functions

 Copyright (C) 2001-2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include "rgba_gpf.h"

static int _RGBA_getinfonr(int chain)
{
	switch (chain) {
	case INFOLIST_INPUT:
		return 1;

	case INFOLIST_OUTPUT:
		return 1;

	}	/* switch */

	return 0;
}	/* _RGBA_getinfonr */


/* fills in the information of pixelformats it can handle
 * from the input
 */
static void _RGBA_fillinfo_input(struct gpf_pipeline_item_t *pipe,
			struct gpf_infolist_t *item,
			int nr)
{
	LIB_ASSERT(pipe != NULL);
	LIB_ASSERT(item != NULL);

	memset(&item->pf, 0, sizeof(struct gpf_pixelformat_t));

	switch (nr) {
	case 0:
		item->pf.cf = GPF_CF_RGBA;
		item->pf.flags = GPF_PIPE_DF_RGBA | GPF_PIPE_DF_ALLVARIANTS;

		LIB_ASSERT(RGBA_PRIV(pipe)->input_data_size > 0);
		item->memory_usage = RGBA_PRIV(pipe)->input_data_size;
		item->runtime = item->memory_usage * 1.0;
		break;
	default:
		fprintf(stderr, "Unknown nr: %i\n", nr);
		break;
	}	/* switch */

	return;
}	/* _RGBA_fillinfo_input */


/* fills in the information of pixelformats it can convert to
 */
static void _RGBA_fillinfo_output(struct gpf_pipeline_item_t *pipe,
			struct gpf_infolist_t *item,
			int nr)
{
	LIB_ASSERT(pipe != NULL);
	LIB_ASSERT(item != NULL);

	memset(&item->pf, 0, sizeof(struct gpf_pixelformat_t));

	switch (nr) {
	case 0:
		item->pf.cf = GPF_CF_RGBA;
		item->pf.flags = GPF_PIPE_DF_RGBA | GPF_PIPE_DF_ALLVARIANTS;


		LIB_ASSERT(RGBA_PRIV(pipe)->output_data_size > 0);
		LIB_ASSERT(RGBA_PRIV(pipe)->input_data_size > 0);

		if (RGBA_PRIV(pipe)->output_data_size
		  >= RGBA_PRIV(pipe)->input_data_size)
		{
			item->memory_usage = RGBA_PRIV(pipe)->output_data_size;
		} else {
			item->memory_usage = RGBA_PRIV(pipe)->input_data_size;
		}	/* if */

		item->runtime = item->memory_usage * 2.0;
		break;
	default:
		fprintf(stderr, "Unknown nr: %i\n", nr);
		break;
	}	/* switch */

	return;
}	/* _RGBA_fillinfo_output */




static void _RGBA_fillinfo(struct gpf_pipeline_item_t *pipe,
			struct gpf_infolist_t *item,
			int nr, int chain)
{
	LIB_ASSERT(pipe != NULL);
	LIB_ASSERT(item != NULL);

	switch (chain) {
	case INFOLIST_INPUT:
		_RGBA_fillinfo_input(pipe, item, nr);
		break;

	case INFOLIST_OUTPUT:
		_RGBA_fillinfo_output(pipe, item, nr);
		break;

	}	/* switch */

	return;
}	/* _RGBA_fillinfo */



static struct gpf_infolist_t *_RGBA_getInfolist(
					struct gpf_pipeline_item_t *pipe,
					int chain)
{
	int nr, i;
	struct gpf_infolist_t *head = NULL;
	struct gpf_infolist_t *item = NULL;

	LIB_ASSERT(pipe != NULL);

	nr = _RGBA_getinfonr(chain);
	item = _gpfInfoListCreate();
	if (!item) {
		goto err0;
	}	/* if */
	_RGBA_fillinfo(pipe, item, 0, chain);
	head = item;

	for (i = 1; i < nr; i++) {
		item = _gpfInfoListCreate();
		if (!item) {
			goto err1;
		}	/* if */

		_RGBA_fillinfo(pipe, item, i, chain);

		_gpfInfoListAppend(head, item);
	}	/* for */


	return head;


err1:
	_gpfInfoListDestroy(&item);
	_gpfInfoListDestroy(&head);
err0:
	return NULL;
}	/* _RGBA_getInfolist */



/* connect the items from the input and output information list
 * together, that can be converted from/to.
 */
static int _RGBA_connectinfolist(struct gpf_infolist_t *input_head,
				struct gpf_infolist_t *output_head)
{
	struct gpf_infolist_t *input_idx = NULL;
	struct gpf_infolist_t *output_idx = NULL;

	for_each_infolist(input_head, input_idx) {
		for_each_infolist(output_head, output_idx) {

			/* Connect all items together */
			_gpfInfoListConnect(input_idx, output_idx);

		}	/* for */
	}	/* for */

	return GGI_OK;
}	/* _RGBA_connectinfolist */



int GPF_RGBA_getinfolist(struct gpf_pipeline_item_t *pipe)
{
	int rc = GGI_OK;

	LIB_ASSERT(pipe != NULL);

	LIB_ASSERT(pipe->input_head == NULL);
	LIB_ASSERT(pipe->output_head == NULL);


	pipe->input_head = _RGBA_getInfolist(pipe, INFOLIST_INPUT);
	if (!pipe->input_head) {
		goto err0;
	}	/* if */

	pipe->output_head = _RGBA_getInfolist(pipe, INFOLIST_OUTPUT);
	if (!pipe->output_head) {
		goto err1;
	}	/* if */

	rc = _RGBA_connectinfolist(pipe->input_head, pipe->output_head);
	LIB_ASSERT(rc == GGI_OK);

	return rc;

err1:
	_gpfInfoListDestroy(&pipe->input_head);
err0:
	return rc;
}	/* GPF_RGBA_getinfolist */

