/*
 ******************************************************************************

 LIBGPF: file protocol functions

 Copyright (C) 2001-2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include "file_gpf.h"
#include <string.h>


int GPF_FILE_open(struct gpf_handle_opio_t *opio, int flags)
{
	int rc = GGI_OK;
	struct filegpf_priv *priv;

	priv = FILE_PRIV(opio);

	priv->fd = open(priv->location, flags,
			S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
	if (priv->fd == -1) {
		rc = errno;
		perror("file-protocol: Unable to open file");
	}	/* if */

	GPFDPRINT_PROTO("%s:%s:%i: opened file (%s, %i) - rc (%i)\n",
			DEBUG_INFO, priv->location, priv->fd, rc);
	opio->opproto->is_eos = 0;

	priv->read_buf_idx = priv->read_buf_len = 0;
	priv->write_buf_len = 0;

	return rc;
}	/* GPF_FILE_open */


int GPF_FILE_close(struct gpf_handle_opio_t *opio)
{
	int rc;
	struct filegpf_priv *priv;

	priv = FILE_PRIV(opio);

	rc = opio->opproto->flush(opio);
	if (rc < 0) {
		/* An error occured */
		return rc;
	}	/* if */

	rc = close(priv->fd);
	if (rc == -1) rc = errno;

	GPFDPRINT_PROTO("%s:%s:%i: closed file (%i) - rc (%i)\n",
			DEBUG_INFO, priv->fd, rc);

	priv->fd = -1;

	return rc;
}	/* GPF_FILE_close */


off_t GPF_FILE_lseek(struct gpf_handle_opio_t *opio,
			off_t offset, int whence)
{
	off_t rc;
	struct filegpf_priv *priv;

	priv = FILE_PRIV(opio);

	rc = lseek(priv->fd, offset, whence);

	return rc;
}	/* GPF_FILE_lseek */


int GPF_FILE_flush(struct gpf_handle_opio_t *opio)
{
	int rc;
	struct filegpf_priv *priv;

	priv = FILE_PRIV(opio);

	if (priv->write_buf_len <= 0) {
		return 0;
	}	/* if */

	rc = write(priv->fd, priv->write_buffer, priv->write_buf_len);
	if (rc < 0) {
		perror("file-protocol: write error");
	}	/* if */

	priv->write_buf_len = 0;

	return rc;
}	/* GPF_FILE_flush */



static int read_byte(struct filegpf_priv *priv, uint8 *val)
{
	int rc = 1;

	if (priv->read_buf_idx >= priv->read_buf_len) {
		rc = read(priv->fd, priv->read_buffer,
			  FILE_BUFFER_SIZE);
		if (rc < 0) {
			rc = (errno >= 0) ? -errno : errno;
			perror("file-protocol: read error");
		}	/* if */

		priv->read_buf_len = rc;
		priv->read_buf_idx = 0;
		if (rc == 0) return 0;
	}	/* if */

	val[0] = priv->read_buffer[priv->read_buf_idx];
	priv->read_buf_idx++;

	return (rc > 0) ? 1 : rc;
}	/* read_byte */



ssize_t GPF_FILE_read(struct gpf_handle_opio_t *opio,
		void *buf, size_t size)
{
	ssize_t rc = 0;
	size_t maxsize;
	struct filegpf_priv *priv;

	priv = FILE_PRIV(opio);

	GPFDPRINT_PROTO("%s:%s:%i: called (fd: %i, buf: %p, size: %i)\n",
			DEBUG_INFO, priv->fd, buf, size);

	maxsize = size;
	while (size--) {
		rc = read_byte(priv, (uint8 *)buf++);
		if (rc == 0) {
			GPFDPRINT_PROTO("%s:%s:%i: end of stream\n",
					DEBUG_INFO);
			opio->opproto->is_eos = 1;
			break;
		}	/* if */

		if (rc < 0) {
			GPFDPRINT_PROTO("%s:%s:%i: error occured: %i\n",
					DEBUG_INFO, rc);
			/* error occured */
			size--;
			break;
		}	/* if */
	}	/* while */

	size++;		/* add 1 to not underflow */

	GPFDPRINT_PROTO("%s:%s:%i: read data (%i, %p, %i) - rc (%i)\n",
			DEBUG_INFO, priv->fd, buf, size, rc);

	return (rc < 0) ? rc : (maxsize - size);
}	/* GPF_FILE_read */



static int write_byte(struct gpf_handle_opio_t *opio,
			uint8 val)
{
	int rc = 1;
	struct filegpf_priv *priv;

	priv = FILE_PRIV(opio);

	GPFDPRINT_PROTO("%s:%s:%i: called (fd: %i, val: %i)\n",
			DEBUG_INFO, priv->fd, val);

	if (priv->write_buf_len >= FILE_BUFFER_SIZE) {
		rc = opio->opproto->flush(opio);
	}	/* if */

	priv->write_buffer[priv->write_buf_len] = val;
	priv->write_buf_len++;

	return (rc > 0) ? 1 : rc;
}	/* write_byte */



ssize_t GPF_FILE_write(struct gpf_handle_opio_t *opio,
		void *buf, size_t size)
{
	ssize_t rc = 0;
	size_t maxsize;
	struct filegpf_priv *priv;

	priv = FILE_PRIV(opio);

	GPFDPRINT_PROTO("%s:%s:%i: called (fd: %i, buf: %p, size: %i)\n",
			DEBUG_INFO, priv->fd, buf, size);

	maxsize = size;
	while (size--) {
		rc = write_byte(opio, ((uint8 *)buf)[0]);
		((uint8 *)buf)++;
		if (rc < 0) {
			/* error occured */
			size--;
			break;
		}	/* if */
	}	/* while */

	size++;		/* add 1 to not underflow */

	GPFDPRINT_PROTO("%s:%s:%i: wrote data (%i, %p, %i) - rc (%i)\n",
			DEBUG_INFO, priv->fd, buf, size, rc);

	return (rc < 0) ? rc : (maxsize - size);
}	/* GPF_FILE_write */


int GPF_FILE_eos(struct gpf_handle_opio_t *opio)
{
	return opio->opproto->is_eos;
}	/* GPF_FILE_eos */
