/* $Id: bse.c,v 1.6 2001/07/25 12:57:22 cegger Exp $
******************************************************************************

   Bse extension stubs.

   Copyright (C) 1999 Andreas Beck	[becka@ggi-project.org]
		 2001 Christoph Egger	[Christoph_Egger@t-online.de]
  
   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdio.h>

#include <ggi/internal/bse.h>


/* Allocating a Sprite.
 */
ggiBse_t ggiBseCreateSprite(ggi_visual_t vis, int width, int height,
			    ggi_graphtype gt)
{
	ggiBse_t newsprite;
	ggiBse_t bse_index;

	struct resourcelist_t *cur_res = NULL;
	struct ggiGA_resource_props props;
	enum bsetype type;

	if (vis == NULL) return NULL;	/* invalid argument */

	printf("%s:%s:%i: search resource\n", DEBUG_INFO);

	FIND_RESOURCE(vis, cur_res);
	if (cur_res == NULL) {
		fprintf(stderr, "Sorry! This visual was never attached before!\n");
		return NULL;
	}	/* if */


	printf("%s:%s:%i: allocate newsprite\n", DEBUG_INFO);

	newsprite = malloc(sizeof(*newsprite));
	if (!newsprite) {
		/* GGI_ENOMEM error */
		return NULL;
	}	/* if */

	ggiGAClearProperties(&props);
	props.size.area.x = width;
	props.size.area.y = height;
	props.sub.tank.gt = gt;


	printf("%s:%s:%i: call ggiOvlCreateSprite\n", DEBUG_INFO);
	type = BSE_SPRITE;
	newsprite->subtype.sprite = ggiOvlCreateSprite(vis, &props,
					GA_RT_SPRITE_SPRITE /*, gt */ );

	printf("%s:%s:%i: after ggiOvlCreateSprite()\n", DEBUG_INFO);

	if (!newsprite->subtype.sprite) {
		/* fall back to emusprite */
		printf("sprite allocation failed! Fall back to emusprite!\n");

		type = BSE_EMUSPRITE;
		newsprite->subtype.emusprite.bob = ggiBltCreateBob(vis, &props);

		if (!newsprite->subtype.emusprite.bob) {
			printf("%s:%s:%i: emusprite allocation failed!\n",
				DEBUG_INFO);
			return NULL;
		}	/* if */
		newsprite->subtype.emusprite.backgr = NULL;
	}	/* if */

	newsprite->vis = vis;
	newsprite->x   = 0;
	newsprite->y   = 0;
	newsprite->w   = width;
	newsprite->h   = height;
	newsprite->cw  = newsprite->w;
	newsprite->ch  = newsprite->h;

	/* come up hidden ! */
	newsprite->hidecount = 1;
	newsprite->type = type;

	ADD_NEW_BSE(bse_index, cur_res->first, newsprite);

	printf("%s:%s:%i: %s allocation successful!\n",
		DEBUG_INFO, (type == BSE_SPRITE) ? "sprite" : "emusprite");

	return newsprite;
}	/* ggiBseCreateSprite */


/* Allocating a Bob.
 */
ggiBse_t ggiBseCreateBob(ggi_visual_t vis, int width, int height,
			 ggi_graphtype gt)
{
	ggiBse_t newbob;
	ggiBse_t bse_index;

	struct resourcelist_t *cur_res = NULL;
	struct ggiGA_resource_props props;

	if (vis == NULL) return NULL;	/* invalid argument */


	FIND_RESOURCE(vis, cur_res);
	if (cur_res == NULL) {
		fprintf(stderr, "Sorry! This visual was never attached before!\n");
		return NULL;
	}	/* if */


	newbob = malloc(sizeof(*newbob));
	if (!newbob) {
		/* GGI_ENOMEM error */
		return NULL;
	}	/* if */

	newbob->vis = vis;
	newbob->x   = 0;
	newbob->y   = 0;
	newbob->w   = width;
	newbob->h   = height;

	/* BOBs have no notion of being hidden */
	newbob->hidecount = 0; 
	newbob->type = BSE_BOB;

	ggiGAClearProperties(&props);
	props.size.area.x = width;
	props.size.area.y = height;
	props.sub.tank.gt = gt;

	newbob->subtype.bob = ggiBltCreateBob(vis, &props /* , gt */);
	if (!newbob->subtype.bob) {
		printf("bob allocation failed!\n");
		return NULL;
	}	/* if */

	ADD_NEW_BSE(bse_index, cur_res->first, newbob);

	return newbob;
}	/* ggiBseCreateBob */




int ggiBseDestroy(ggiBse_t bse)
{
	int rc = 0;
	struct resourcelist_t *cur_res = NULL;


	if (bse == NULL) return GGI_EARGINVAL;


	FIND_RESOURCE(bse->vis, cur_res);
	if (cur_res == NULL) {
		fprintf(stderr, "Sorry! This visual was never attached before!\n");
		return GGI_EARGINVAL;
	}	/* if */


	switch (bse->type) {
	case BSE_BOB:
		rc = ggiBltDestroyBob(&(bse->subtype.bob));
		break;

	case BSE_SPRITE:
		rc = ggiOvlDestroySprite(&(bse->subtype.sprite));
		break;

	case BSE_EMUSPRITE:
		if (bse->subtype.emusprite.backgr) {
			free(bse->subtype.emusprite.backgr);
		}	/* if */
		rc = ggiBltDestroyBob(&(bse->subtype.emusprite.bob));
		break;

	}	/* switch */


	DESTROY_BSE(bse, cur_res->first);

	return rc;
}	/* ggiBseDestroy */



/* BOB converting functions
 * Allows to use the libBlt API for BOBs.
 */
ggiBlt_t ggiBob2Blt(ggiBse_t bse)
{
	if (bse == NULL) return NULL;
	if (bse->type != BSE_BOB) return NULL;

	return bse->subtype.bob;
}	/* ggiBob2Blt */



int ggiBseIsSpriteEmulated(ggiBse_t bse)
{
	if (bse == NULL) return GGI_EARGINVAL;
	if (bse->type == BSE_BOB) return GGI_EARGINVAL;

	return (bse->type == BSE_EMUSPRITE) ? 1 : 0;
}	/* ggiBseIsSpriteEmulated */



/* Hiding and showing
 */
int ggiBseHideAboveCollide(ggiBse_t bse,int x,int y) 
{
	ggiBse_t curr;

	if (bse == NULL) return GGI_EARGINVAL;

	curr = bse;
	while ( (curr = curr->prev) ) {
		if (curr->type != BSE_EMUSPRITE) continue;

		if (curr->x + curr->w <= x ||
		    curr->y + curr->h <= y ||
			  x + bse->w <=curr->x ||
			  y + bse->h <=curr->y )
			continue;
		ggiBseHideAboveCollide(curr,curr->x,curr->y);
		ggiBseHide(curr);
	}	/* while */

	return GALLOC_OK;
}	/* ggiBseHideAboveCollide */


int ggiBseShowAboveCollide(ggiBse_t bse,int x,int y) 
{
	ggiBse_t curr;

	if (bse == NULL) return GGI_EARGINVAL;

	curr = bse;
	while ( (curr = curr->prev) ) {
		if (curr->type != BSE_EMUSPRITE) continue;

		if (curr->x + curr->w <= x ||
		    curr->y + curr->h <= y ||
			  x + bse->w <= curr->x ||
			  y + bse->h <= curr->y )
			continue; 
		ggiBseShow(curr);
		ggiBseShowAboveCollide(curr,curr->x,curr->y);
	}	/* while */

	return GALLOC_OK;
}	/* ggiBseShowAboveCollide */



/* Populate the sprite
 */
int ggiBseSetImage(ggiBse_t bse, ggi_visual_t vis, int x,int y,
		   unsigned char *alpha)
{
	int rc = 0;

	if (bse == NULL) return GGI_EARGINVAL;
	if (vis == NULL) return GGI_EARGINVAL;

	switch(bse->type) {
	case BSE_BOB:
		rc = ggiBltSetImage(bse->subtype.bob, vis, x, y, alpha);
		break;

	case BSE_SPRITE:
		rc = ggiOvlSetImage(bse->subtype.sprite, vis, x, y, alpha);
		break;

	case BSE_EMUSPRITE:
		if (bse->subtype.emusprite.backgr == NULL) {
			bse->subtype.emusprite.backgr =
				malloc(bse->w * bse->h * sizeof(ggi_pixel));
			if (!bse->subtype.emusprite.backgr) {
				return GGI_ENOMEM;
			}	/* if */
		}	/* if */

		rc = ggiBltSetImage(bse->subtype.emusprite.bob,
				    vis, x,y, alpha);
		break;
	}	/* switch */

	return rc;
}	/* ggiBseSetImage */



/* Move the sprite around.
 */
int ggiBseMove(ggiBse_t bse, int x,int y)
{
	int rc = GALLOC_OK;

	if (bse == NULL) return GGI_EARGINVAL;
	
	if (x == bse->x && y == bse->y) {
		return GALLOC_OK;	/* It's already there ! */
	}	/* if */


	switch(bse->type) {
	case BSE_BOB:
		rc = ggiBltBlit2Vis(bse->subtype.bob, x,y, 0,0, ROP_NOP);
		break;

	case BSE_SPRITE:
		rc = ggiOvlMove(bse->subtype.sprite, x, y);
		break;

	case BSE_EMUSPRITE:
		{
		ggi_mode mode;
		int oldx;
		int oldy;

		if (bse->hidecount) {
			rc = GALLOC_OK;		/* It's hidden. */
			break;
		}	/* if */

		oldx = bse->x;
		oldy = bse->y;

		ggiBseHideAboveCollide(bse, oldx, oldy);
		ggiBseHideAboveCollide(bse, x, y);

		ggiPutBox(bse->vis, bse->x,bse->y, bse->cw,bse->ch,
			  bse->subtype.emusprite.backgr);

		ggiGetMode(bse->vis, &mode);
		if ((x + bse->w) >= mode.virt.x) {
			bse->cw = mode.virt.x - x;
		} else {
			bse->cw = bse->w;
		}	/* if */

		if ((y + bse->h) >= mode.virt.y) {
			bse->ch = mode.virt.y - y;
		} else {
			bse->ch = bse->h;
		}	/* if */

		ggiGetBox(bse->vis, x,y, bse->cw,bse->ch,
			  bse->subtype.emusprite.backgr);

		rc = ggiBltBlit2Vis(bse->subtype.emusprite.bob, x,y,
				    0,0, ROP_NOP);

		ggiBseShowAboveCollide(bse, x, y);
		ggiBseShowAboveCollide(bse, oldx, oldy);
		break;
		}

	}	/* switch */

	bse->x = x;
	bse->y = y;

	return rc;
}	/* ggiBseMove */


int ggiBseHide(ggiBse_t bse)
{
	int rc = GALLOC_OK;

	if (bse == NULL) return GGI_EARGINVAL;

	if (bse->type == BSE_BOB) {	
		return GGI_EARGINVAL;
	}	/* if */

	switch(bse->type) {
	case BSE_BOB:
		/* A BOB cannot be hidden. */
		return GGI_EARGINVAL;

	case BSE_SPRITE:
		/* Hide it. */
		rc = ggiOvlHide(bse->subtype.sprite);
		break;

	case BSE_EMUSPRITE:
		if (bse->hidecount) {	/* It's already hidden. */
			rc = GALLOC_OK;
			goto exit;
		}	/* if */

		ggiBseHideAboveCollide(bse, bse->x,bse->y);
		rc = ggiPutBox(bse->vis, bse->x,bse->y, bse->cw,bse->ch,
				bse->subtype.emusprite.backgr);
		ggiBseShowAboveCollide(bse, bse->x,bse->y);
		break;
	}	/* switch */

exit:
	bse->hidecount++;

	return rc;
}	/* ggiBseHide */


int ggiBseShow(ggiBse_t bse)
{
	int rc = GALLOC_OK;

	if (bse == NULL) return GGI_EARGINVAL;
	
	if (bse->type == BSE_BOB) {
		/* A BOB cannot be hidden. */
		return GGI_EARGINVAL;
	}	/* if */

	if (bse->hidecount <= 0) {
		return GALLOC_EFAILED;	/* Too often! */
	}	/* if */

	bse->hidecount--;

	switch(bse->type) {
	case BSE_BOB:
		/* A BOB cannot be hidden. */
		return GGI_EARGINVAL;

	case BSE_SPRITE:
		/* Hide it. */
		rc = ggiOvlShow(bse->subtype.sprite);
		return rc;

	case BSE_EMUSPRITE:
		{
		ggi_mode mode;

		if (bse->hidecount != 0) {
			return GALLOC_OK;	/* We are still hidden. */
		}	/* if */

		ggiBseHideAboveCollide(bse, bse->x,bse->y);

		/* Perform clipping */
		ggiGetMode(bse->vis, &mode);
		if ((bse->x + bse->w) >= mode.virt.x) {
			bse->cw = mode.virt.x - bse->x;
		} else {
			bse->cw = bse->w;
		}	/* if */

		if ((bse->y + bse->h) >= mode.virt.y) {
			bse->ch = mode.virt.y - bse->y;
		} else {
			bse->ch = bse->h;
		}	/* if */

		/* Save box */
		ggiGetBox(bse->vis, bse->x,bse->y, bse->cw,bse->ch,
			  bse->subtype.emusprite.backgr);

		rc = ggiBltBlit2Vis(bse->subtype.emusprite.bob,
				    bse->x,bse->y, 0,0, ROP_NOP);

		ggiBseShowAboveCollide(bse, bse->x,bse->y);
		break;
		}

	}	/* switch */

	return rc;
}	/* ggiBseShow */


/* Hiding and showing of _emulated_ Sprites.
 * Has no effect to Bob's and real Sprites.
 */
int ggiBseHideIfEmulated(ggiBse_t bse)
{
	if (bse == NULL) return GGI_EARGINVAL;

	switch(bse->type) {
	case BSE_BOB:
	default:
		return GGI_EARGINVAL;

	case BSE_SPRITE:
		return 0;	/* NOP for real sprites */

	case BSE_EMUSPRITE:
		return ggiBseHide(bse);
	}	/* switch */

}	/* ggiBseHideIfEmulated */


int ggiBseShowIfEmulated(ggiBse_t bse)
{
	if (bse == NULL) return GGI_EARGINVAL;

	switch(bse->type) {
	case BSE_BOB:
	default:
		return GGI_EARGINVAL;

	case BSE_SPRITE:
		return 0;	/* NOP for real sprites */

	case BSE_EMUSPRITE:
		return ggiBseShow(bse);
	}	/* switch */

}	/* ggiBseShowIfEmulated */




/* Helper functions for Alpha channels
 */
 
unsigned char *ggiBseMakeAlphaFromPixelkey(ggiBse_t bse,
			ggi_visual_t vis,int x,int y, ggi_pixel pix)
{
	unsigned char *rc,*alph;
	ggi_pixel gp;
	int xx,yy;

	if (bse == NULL) return NULL;

	if ((rc = malloc(bse->w * bse->h * sizeof(unsigned char)))) {
		alph = rc;
		for (yy = 0; yy < bse->h; yy++) {
			for (xx = 0;xx < bse->w; xx++) {
				ggiGetPixel(vis, x + xx, y + yy, &gp);
				*alph++ = (gp == pix) ? 0xff : 0x00;
			}	/* for */
		}	/* for */
	}	/* if */

	return rc;
}	/* ggiBseMakeAlphaFromPixelkey */


unsigned char *ggiBseMakeAlphaFromColorkey(ggiBse_t bse,
			ggi_visual_t vis, int x,int y, ggi_color *col)
{
	return ggiBseMakeAlphaFromPixelkey(bse, vis, x,y,
				ggiMapColor(vis,col));
}	/* ggiBseMakeAlphaFromColorkey */



static unsigned char func_treshold(unsigned char inval,void *tresh)
{
	return inval > *((unsigned char *)tresh) ? 0xFF : 0x00;
}	/* func_treshold */


static unsigned char func_invert(unsigned char inval, void *unused)
{
	return 0xFF - inval;
}	/* func_invert */



unsigned char *ggiBseChangeAlphaByFunction(ggiBse_t bse, 
		unsigned char *alpha,
		unsigned char (*func)(unsigned char in,void *parms),
		void *parms)
{
	int xx,yy;

	if (bse == NULL) return NULL;
	if (alpha == NULL) return NULL;
	if (func == NULL) return NULL;


	for (yy = 0; yy < bse->h; yy++) {
		for (xx = 0; xx < bse->w; xx++) {
			*alpha = func(*alpha,parms);
		}	/* for */
	}	/* for */

	return alpha;
}	/* ggiBseChangeAlphaByFunction */


unsigned char *ggiBseChangeAlphaTreshold(ggiBse_t bse,
					 unsigned char *alpha,
					 unsigned char treshold)
{
	return ggiBseChangeAlphaByFunction(bse, alpha,
					   func_treshold, &treshold);
}	/* ggiBseChangeAlphaTreshold */


unsigned char *ggiBseChangeAlphaInvert(ggiBse_t bse, unsigned char *alpha)
{
	return ggiBseChangeAlphaByFunction(bse, alpha,
					   func_invert, NULL);
}	/* ggiBseChangeAlphaInvert */
