/* $Id: context.c,v 1.1.1.1 2001/04/22 19:02:56 cegger Exp $
******************************************************************************

   LibGIC - Contexts

   Copyright (C) 1999 Andreas Beck	[becka@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdlib.h>
#include <string.h>
#include <ggi/gic.h>

gic_context    *gicContextAllocate	(gic_handle_t hand,char *name)
{
	gic_context *ret;
	
	if ((ret=malloc(sizeof(gic_context)))) {
		strncpy(ret->name,name,65);
		ret->name[65-1]=0;
		ret->controls=NULL;
	}
	return ret;
}

void		gicContextFree		(gic_handle_t hand,gic_context *context)
{
	free(context);
}

int gicContextAttachControl 		(gic_handle_t hand,gic_context *context, gic_control *control)
{
	gic_controllist *listitem, *cur;

	if (NULL==(cur=malloc(sizeof(gic_controllist)))) {
		return -1;	/* NOMEM */
	}

	cur->next = NULL;
	cur->control = control;

	if (context->controls == NULL) {
		context->controls = cur;
		return 0;
	}

        for (listitem = context->controls;
             listitem->next;
             listitem = listitem->next) {
        }

	listitem->next = cur;
	return 0;
}

int		gicContextDetachControl	(gic_handle_t hand,gic_context *context,gic_control *control)
{
	gic_controllist *listitem,**last;

	for(listitem=*(last=&context->controls);
		listitem!=NULL;listitem=*(last=&listitem->next))
	{
		if (listitem->control==control) {
			*last=listitem->next;
			free(listitem);
			return 0;
		}
	}
	return -1;
}

gic_control  *gicContextLookupControl	(gic_handle_t hand,gic_context *context,const char *name)
{
	gic_controllist *listitem;

	for(listitem=context->controls;
		listitem!=NULL;listitem=listitem->next)
	{
		if (0==strcmp(listitem->control->name,name)) {
			return listitem->control;
		}
	}
	return NULL;
}

gic_control *gicContextGetControl	(gic_handle_t hand,gic_context *context, int number)
{
	gic_controllist *listitem;

	for (listitem = context->controls;
	     listitem != NULL;
	     listitem = listitem->next) {
		if (number-- == 0)
			return listitem->control;
	}
	return NULL;
}

int gicContextNumControls		(gic_handle_t hand,gic_context *context)
{
	int count=0;
	gic_controllist *listitem;

	for (listitem = context->controls;
	     listitem != NULL;
	     listitem = listitem->next) {
		count++;
	}
	return count;
}

int             gicContextHandleEvent	(gic_handle_t hand,gic_context *context, gii_event *event)
{
	gic_controllist *listitem;
	int rc=0;

	/* Check all features ... */
	for(listitem=context->controls;listitem!=NULL;
		listitem=listitem->next) {
			rc+=gicControlHandleEvent(hand,listitem->control,event);
	}
	return rc;
}

int	gicContextWrite			(gic_handle_t hand,gic_context *context,FILE *where)
{
	gic_controllist *listitem;

	fprintf(where,"gic:  Context \"%s\"\n",context->name);
	for(listitem=context->controls;listitem!=NULL;listitem=listitem->next)
	{
		gicControlWrite(hand,listitem->control,where);
	}
	fprintf(where,"gic:  Context END\n");

	return 0;
}

gic_context	*gicContextRead(gic_handle_t hand,FILE *where)
{
	gic_context	*context;
	gic_control	*control;
	char buffer[1024],*hlp;

	fgets(buffer, sizeof(buffer), where);
	if ( 0==strncmp("gic:  Context \"",buffer,15) && 
	     (hlp=strchr(buffer+15,'"')) ) {
		*hlp='\0';	/* Terminate string */
		context=gicContextAllocate(hand,buffer+15);
		if (NULL==context) return NULL;

		/* Attention: The last gicControlRead will fail, as it reads
		 * the gic:  Context END. This is intended behaviour. It detects
		 * the end of the controls.
		 */
		while( (control=gicControlRead(hand,where)) )
			gicContextAttachControl(hand,context,control);

		return context;
	}
	return NULL;
}

int	gicContextMapActions	(gic_handle_t hand,gic_context *context,gic_actionlist *actions)
{
	gic_controllist *listitem;

	for(listitem=context->controls;listitem!=NULL;listitem=listitem->next)
	{
		gicControlMapActions(hand,listitem->control,actions);
	}

	return 0;
}

int		gicContextFindConflict	  (gic_handle_t hand,gic_context *context, gic_recognizer *rec,gic_recognizer **start_and_return,gic_feature **optional)
{
	int rc;
	gic_controllist *listitem;

	for(listitem=context->controls;listitem!=NULL;listitem=listitem->next)
	{
		if ((rc=gicControlFindConflict(hand,listitem->control,rec,start_and_return,optional))) {
			return rc;
		}
	}
	return 0;
}

int gicContextGetName (gic_handle_t hand,gic_context *context, char *string, int maxlen)
{
	strncpy(string, context->name, maxlen);
	string[maxlen-1] = 0;

	return 0;
}
