/*
 ******************************************************************************

 LIBGPF: internal functions

 Copyright (C) 2001-2002 Christoph Egger   [Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include <ggi/internal/gpf.h>
#include <ggi/internal/gpf_internal.h>
#include <ggi/internal/gpf_debug.h>

#include <stdlib.h>
#include <string.h>


/*
******************************************************************************
 Default functions for op-structs
******************************************************************************
*/

static int _default_error(void)
{
	GPFDPRINT_CORE("_default_error() called\n");
	return GGI_ENOFUNC;
}	/* _default_error */


/*
******************************************************************************
	op-structs initialisation
******************************************************************************
*/

static void *_gpf_init_opproto(struct gpf_handle_opproto_t *proto)
{
	proto->pipe = NULL;
	proto->head.dlhandle = NULL;
	proto->head.dummy = NULL;

	proto->is_eos = 0;
	proto->ptype = GPF_PROTO_NONE;

	proto->open = (void *)_default_error;
	proto->close = (void *)_default_error;
	proto->read = (void *)_default_error;
	proto->write = (void *)_default_error;
	proto->lseek = (void *)_default_error;
	proto->flush = (void *)_default_error;
	proto->eos = (void *)_default_error;

	return &(proto->head);
}	/* _gpf_init_opproto */


static void *_gpf_init_opio(struct gpf_handle_opio_t *io)
{
	io->pipe = NULL;
	io->head.dlhandle = NULL;
	io->head.dummy = NULL;

	_gpf_init_opproto(io->opproto);

	io->open = (void *)_default_error;
	io->close = (void *)_default_error;
	io->timestamp = (void *)_default_error;
	io->read = (void *)_default_error;
	io->write = (void *)_default_error;

	io->doio = (void *)_default_error;

	io->pipe_getinfolist = (void *)_default_error;

	return &(io->head);
}	/* _gpf_init_opio */


static void *_gpf_init_pipe(struct gpf_pipeline_t *pipeline)
{
	memset(pipeline, 0, sizeof(struct gpf_pipeline_t));

	pipeline->head.dlhandle = NULL;
	pipeline->head.dummy = NULL;

	return &(pipeline->head);
}	/* _gpf_init_pipe */


static void _gpf_init_allops(gpf_handle_t hnd, int initall)
{
	int idx;

	if (initall) {
		for_each_handle_input(hnd, idx) {
			_gpf_init_opio(LIBGPF_INPUT(hnd, idx));
			LIBGPF_INPUT(hnd, idx)->head.version
				= GPF_VERSION_HANDLE_OPIO;
		}	/* for */

		for_each_handle_output(hnd, idx) {
			_gpf_init_opio(LIBGPF_OUTPUT(hnd, idx));
			LIBGPF_OUTPUT(hnd, idx)->head.version
				= GPF_VERSION_HANDLE_OPIO;
		}	/* for */

		for_each_handle_pipeline(hnd, idx) {
			_gpf_init_pipe(LIBGPF_PIPELINE(hnd, idx)->pipeline);
			LIBGPF_PIPELINE(hnd, idx)->pipeline->head.version
				= GPF_VERSION_HANDLE_PIPELINE;
		}	/* for */
	}	/* if */
}	/* _gpf_init_allops */




/*
******************************************************************************
	static functions
******************************************************************************
*/

/* Make sure str contains a valid variable name. Kill everything but
 * [a-zA-Z0-9].
 */
static void mangle_variable(char *str)
{
        for (;*str; str++) {
                /**/ if ( ( *str>='A' && *str<='Z' ) ||
                          ( *str>='0' && *str<='9' ) ) continue;
                else if (   *str>='a' && *str<='z' ) *str+='A'-'a';
                else *str='_';
        }       /* for */
}       /* mangle_variable */




/*
******************************************************************************
	Helpers for creation and destruction of handles
******************************************************************************
*/


static int _gpf_handle_opio_realloc(gpf_handle_t hnd, int type, int *idx)
{
	idx[0] = -1;

	if (type & GPF_DLTYPE_INPUT) {
		if (hnd->opio_input == NULL) {
			LIB_ASSERT(hnd->num_inputs == 0);
			hnd->opio_input =
				calloc(1, sizeof(struct gpf_handle_opio_t *));
			if (!hnd->opio_input) {
				return GGI_ENOMEM;
			}	/* if */

		} else {
			struct gpf_handle_opio_t **tmp;

			LIB_ASSERT(hnd->num_inputs > 0);

			tmp = realloc(hnd->opio_input,
				(hnd->num_inputs+1) *
				sizeof(struct gpf_handle_opio_t *));
			if (!tmp) {
				return GGI_ENOMEM;
			}	/* if */

			hnd->opio_input = tmp;
		}	/* if */


		idx[0] = hnd->num_inputs;
		hnd->num_inputs++;

		return GGI_OK;
	}	/* if */

	if (type & GPF_DLTYPE_OUTPUT) {
		if (hnd->opio_output == NULL) {
			LIB_ASSERT(hnd->num_outputs == 0);
			hnd->opio_output =
				calloc(1, sizeof(struct gpf_handle_opio_t *));
			if (!hnd->opio_output) {
				return GGI_ENOMEM;
			}	/* if */

		} else {
			struct gpf_handle_opio_t **tmp;

			LIB_ASSERT(hnd->num_outputs > 0);

			tmp = realloc(hnd->opio_output,
				(hnd->num_outputs+1) *
				sizeof(struct gpf_handle_opio_t *));
			if (!tmp) {
				return GGI_ENOMEM;
			}	/* if */

			hnd->opio_output = tmp;
		}	/* if */


		idx[0] = hnd->num_outputs;
		hnd->num_outputs++;

		return GGI_OK;
	}	/* if */

	return GGI_ENOMATCH;
}	/* _gpf_handle_opio_realloc */


/*
******************************************************************************
	Open/Close a target
******************************************************************************
*/

int _gpfOpenTarget(gpf_handle_t hnd, int *idx, const char *target,
		const char *argf, va_list ap, int type)
{
	struct gpf_handle_opio_t *io = NULL;
	char protocol[GPF_MAXPARAM_LEN];
	char proto_argf[GPF_MAXPARAM_LEN];
	va_list proto_ap;
	int rc;

	io = calloc(1, sizeof(struct gpf_handle_opio_t));
	if (!io) {
		rc = GGI_ENOMEM;
		goto err0;
	}	/* if */

	io->pipe = _gpfPipelineItemCreate();
	if (!io->pipe) {
		rc = GGI_ENOMEM;
		goto err1;
	}	/* if */


	GPFDPRINT_CORE("_gpfOpenTarget: Loading target %s\n", target);

	do {
		char *tmp;

		memset(protocol, 0, sizeof(char) * GPF_MAXPARAM_LEN);
		memset(proto_argf, 0, sizeof(char) * GPF_MAXPARAM_LEN);

		rc = _gpfParseProtocol(argf, ap,
				protocol, proto_argf, &proto_ap);
		if (rc != GGI_OK) {
			break;
		}	/* if */


		if (strlen(target) == 0) {
			fprintf(stderr, "LIBGPF: Missing target descriptor!\n");
			rc = GGI_ENOMATCH;
			break;
		}	/* if */

		tmp = malloc(strlen(target) + 4);
		if (!tmp) {
			rc = GGI_ENOMEM;
			break;
		}	/* if */

		if (strncmp(target, "io-", 3) != 0) {
			snprintf(tmp, strlen(target) + 4, "io-%s", target);
		} else {
			strncpy(tmp, target, strlen(target));
		}	/* if */

		rc = _gpfOpenDL(io, io->pipe, tmp, argf, ap, type);
		free(tmp);
		if (rc != GGI_OK) {
			goto err2;
		}	/* if */

	} while (0);

	/* If protocol == NULL || proto_args == NULL, then
	 * _gpfOpenProtocol fails always here
	 */
	rc = _gpfOpenProtocol(io, protocol, proto_argf, proto_ap);
	if (rc != GGI_OK) {
		rc = GGI_ENOMEM;
		goto err3;
	}	/* if */

	io->type = type;

	LIB_ASSERT(io != NULL);
	LIB_ASSERT(io->opproto != NULL);

	if (LIBGPF_IO_ISOUTPUT(io)) {
		if (hnd->opio_input != NULL) {
			/* Ah - input target is already open
			 */
			LIB_ASSERT(LIBGPF_INPUT(hnd, 0)->pipe != NULL);
			io->pipe->width = LIBGPF_INPUT(hnd, 0)->pipe->width;
			io->pipe->height = LIBGPF_INPUT(hnd, 0)->pipe->height;
			io->pipe->depth = LIBGPF_INPUT(hnd, 0)->pipe->depth;
			io->pipe->size = LIBGPF_INPUT(hnd, 0)->pipe->size;
		}	/* if */
	}	/* if */

	rc = io->open(io);
	if (rc != GGI_OK) {
		goto err3;
	}	/* if */


	if (LIBGPF_IO_ISINPUT(io)) {

		rc = _gpf_handle_opio_realloc(hnd, GPF_DLTYPE_INPUT, idx);
		if (rc != GGI_OK) {
			goto err3;
		}	/* if */

		LIBGPF_INPUT(hnd, idx[0]) = io;
		if (hnd->opio_output != NULL) {
			/* Ah - output target is already open
			 */
			LIB_ASSERT(LIBGPF_OUTPUT(hnd, 0)->pipe != NULL);
			LIBGPF_OUTPUT(hnd, 0)->pipe->width = io->pipe->width;
			LIBGPF_OUTPUT(hnd, 0)->pipe->height = io->pipe->height;
			LIBGPF_OUTPUT(hnd, 0)->pipe->depth = io->pipe->depth;
			LIBGPF_OUTPUT(hnd, 0)->pipe->size = io->pipe->size;
		}	/* if */
	}	/* if */
	if (LIBGPF_IO_ISOUTPUT(io)) {

		rc = _gpf_handle_opio_realloc(hnd, GPF_DLTYPE_OUTPUT, idx);
		if (rc != GGI_OK) {
			goto err3;
		}	/* if */

		LIBGPF_OUTPUT(hnd, idx[0]) = io;
	}	/* if */


	GPFDPRINT_CORE("target %p successful loaded\n", io);
	LIB_ASSERT(rc == GGI_OK);
	return GGI_OK;

err3:
	_gpfCloseDL(io->pipe);
err2:
	_gpfPipelineItemDestroy(&(io->pipe));
err1:
	free(io);
err0:
	idx[0] = -1;
	return rc;
}	/* _gpfOpenTarget */



int _gpfCloseTarget(gpf_handle_t hnd, int idx, int type)
{
	int rc = GGI_OK;
	struct gpf_handle_opio_t **opio = NULL;
	gpf_dlhandle_l *dlh = NULL;


	if (hnd == NULL) {
		/* happens, when trying to close an
		 * already closed target
		 */
		return GGI_OK;
	}	/* if */

	if (type & GPF_DLTYPE_INPUT) {
		if ((hnd->opio_input == NULL)
		  || (LIBGPF_INPUT(hnd, idx) == NULL)) 
		{
			/* this happens, when opening a target failed */
			return GGI_OK;
		}	/* if */

		opio = &(hnd->opio_input[idx]);
		dlh = LIBGPF_IO_DLHANDLE(opio[0]);
	}	/* if */
	if (type & GPF_DLTYPE_OUTPUT) {
		if ((hnd->opio_output == NULL)
		  || (LIBGPF_OUTPUT(hnd, idx) == NULL))
		{
			/* this happens, when opening a target failed */
			return GGI_OK;
		}	/* if */

		opio = &(hnd->opio_output[idx]);
		dlh = LIBGPF_IO_DLHANDLE(opio[0]);
	}	/* if */

	if (dlh == NULL) {
		/* happens, when opening a target failed */
		return GGI_OK;
	}	/* if */

	LIB_ASSERT(opio[0] != NULL);
	opio[0]->close(opio[0]);

	do {
		GPFDPRINT_CORE("Close protocol\n");
		rc = _gpfCloseProtocol(opio[0]);
		if (rc != GGI_OK) {
			break;
		}	/* if */

		GPFDPRINT_CORE("Close target %p\n", opio);
		rc = _gpfCloseDL(opio[0]->pipe);
		if (rc != GGI_OK) {
			break;
		}	/* if */

		_gpfPipelineItemDestroy(&(opio[0]->pipe));
		free(opio[0]);
		opio[0] = NULL;
	} while (0);

	return rc;
}	/* _gpfCloseTarget */


/*
******************************************************************************
	Open/Close the protocol
******************************************************************************
*/


int _gpfOpenProtocol(struct gpf_handle_opio_t *opio,
		const char *protocol, const char *argf, va_list ap)
{
	struct gpf_handle_opproto_t *proto = NULL;
	int rc;

	LIB_ASSERT(opio != NULL);

	if (protocol == NULL) {
		rc = GGI_ENOMATCH;
		goto err0;
	}	/* if */

	LIB_ASSERT(protocol != NULL);

	proto = calloc(1, sizeof(struct gpf_handle_opproto_t));
	if (!proto) {
		rc = GGI_ENOMEM;
		goto err0;
	}	/* if */

	proto->pipe = _gpfPipelineItemCreate();
	if (!proto->pipe) {
		rc = GGI_ENOMEM;
		goto err1;
	}	/* if */
	opio->opproto = proto;


	GPFDPRINT_CORE("Loading protocol: %s\n", protocol);

	do {
		char *tmp;

		if (strlen(protocol) == 0) {
			fprintf(stderr, "LIBGPF: Missing protocol descriptor!\n");
			rc = GGI_ENOMATCH;
			break;
		}	/* if */

		tmp = malloc(strlen(protocol) + 7);
		if (!tmp) {
			rc = GGI_ENOMEM;
			break;
		}	/* if */

		if (strncmp(protocol, "proto-", 6) != 0) {
			snprintf(tmp, strlen(protocol) + 7,
				"proto-%s", protocol);
		} else {
			strncpy(tmp, protocol, strlen(protocol));
		}	/* if */

		rc = _gpfOpenDL(opio, opio->opproto->pipe, tmp,
				argf, ap, GPF_DLTYPE_LOCAL);
		free(tmp);
		if (rc != GGI_OK) {
			goto err2;
		}	/* if */

	} while (0);


	GPFDPRINT_CORE("protocol %p successful loaded\n", opio->opproto);

	LIB_ASSERT(rc == GGI_OK);
	return GGI_OK;

err2:
	_gpfPipelineItemDestroy(&(proto->pipe));
err1:
	free(proto);
	opio->opproto = NULL;
err0:
	return rc;
}	/* _gpfOpenProtocol */


int _gpfCloseProtocol(struct gpf_handle_opio_t *opio)
{
	int rc = GGI_OK;

	LIB_ASSERT(opio != NULL);
	LIB_ASSERT(opio->opproto != NULL);


	GPFDPRINT_CORE("Close protocol %p\n", opio->opproto);

	do {
		rc = _gpfCloseDL(opio->opproto->pipe);
		if (rc != GGI_OK) {
			break;
		}	/* if */

		_gpfPipelineItemDestroy(&(opio->opproto->pipe));
		free(opio->opproto);
		opio->opproto = NULL;
	} while (0);

	return rc;
}	/* _gpfCloseProtocol */


/*
******************************************************************************
	Creation and destruction of handles
******************************************************************************
*/


gpf_handle_t _gpfNewHandle(void)
{
	gpf_handle_t handle;

	handle = calloc(1, sizeof(*handle));
	if (handle == NULL) return NULL;

	handle->mutex = ggLockCreate();
	if (handle->mutex == NULL) goto out_freehandle;

	handle->version = GPF_VERSION_HANDLE;

	handle->num_inputs = 0;
	handle->num_outputs = 0;
	handle->num_pipelines = 0;

	handle->opio_input = NULL;
	handle->opio_output = NULL;
	handle->pipeline = NULL;

	handle->started = 0;
	handle->mutex_valid = 0;

	gpfClearProperties(&handle->props);

	_gpf_init_allops(handle, 0);


	return handle;


	/* Error occured. */
  out_freehandle:
	free(handle);

	return NULL;
}	/* _gpfNewHandle */


int _gpfDestroyHandle(gpf_handle_t handle)
{
	int i, rc;

	for_each_handle_input(handle, i) {
		rc = _gpfCloseTarget(handle, i, GPF_DLTYPE_INPUT);
		LIB_ASSERT(rc == GGI_OK);
	}	/* for */

	for_each_handle_output(handle, i) {
		rc = _gpfCloseTarget(handle, i, GPF_DLTYPE_OUTPUT);
		LIB_ASSERT(rc == GGI_OK);
	}	/* for */

	free(handle->opio_input);
	free(handle->opio_output);

	ggLockDestroy(handle->mutex);
	free(handle);

	return GGI_OK;
}	/* _gpfDestroyHandle */

