/*
 ******************************************************************************

 LIBGPF: ppm io pipeline communication functions

 Copyright (C) 2001 Christoph Egger     [Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include "ppm_gpf.h"


static int _PPM_getinfonr(int chain)
{
	switch (chain) {
	case INFOLIST_INPUT:
		return 2;

	case INFOLIST_OUTPUT:
		return 2;

	}	/* switch */

	return 0;
}	/* _PPM_getinfonr */


/* fills in the information of pixelformats it can handle
 * from the input
 */
static void _PPM_fillinfo_input(struct gpf_handle_opio_t *opio,
			struct gpf_infolist_t *item,
			int nr)
{
	struct ppmgpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = PPM_PRIV(opio);


	memset(&item->pf, 0, sizeof(struct gpf_pixelformat_t));

	item->pf.cf = GPF_CF_RGBA;
	item->pf.flags = GPF_PIPE_DF_RGBA;


	switch (nr) {
	case 0:
		/* RGB format: r8g8b8
		 *
		 * we can always provide RGB_888, because we can
		 * scale the colors down, if necessary
		 */
		item->pf.depth = 24;
		item->pf.size = 24;

		item->pf.type.rgba.red_mask = 0x0000FF;
		item->pf.type.rgba.red_shift = 0;
		item->pf.type.rgba.green_mask = 0x00FF00;
		item->pf.type.rgba.red_shift = 8;
		item->pf.type.rgba.blue_mask = 0xFF0000;
		item->pf.type.rgba.red_shift = 16;

		item->memory_usage = PF_ByPP(item->pf.size)
				* priv->width * priv->height;

		switch (priv->bypp) {
		case 1*3:
			/* raw transfer to the next pipe is
			 * a linear algorithm
			 */
			item->runtime = item->memory_usage * 1.0;
			break;

		case 2*3:
			/* we convert the data down from 2 to 1 byte.
			 * This is an extra loop, which doubles the
			 * runtime
			 */
			item->runtime = item->memory_usage * 2.0;
			break;

		default:
			fprintf(stderr, "unsupported pixelformat (bypp = %i)\n",
				priv->bypp);
			break;

		}	/* switch */
		break;

	case 1:
		/* RGB format: r16g16b16
		 *
		 * we can always provide RGB_FFF, because we can
		 * scale the colors up, if necessary
		 */
		item->pf.depth = 24 * 2;	/* rgb * 2 */
		item->pf.size = 24 * 2;		/* rgb * 2 */

		item->pf.type.rgba.red_mask = 0xFFFF;
		item->pf.type.rgba.green_mask = 0xFFFF;
		item->pf.type.rgba.blue_mask = 0xFFFF;


		item->memory_usage = PF_ByPP(item->pf.size)
				* priv->width * priv->height;

		switch (priv->bypp) {
		case 1*3:
			/* we convert the data up from 1 to 2 bytes.
			 * This is an extra loop, which doubles the
			 * runtime
			 */
			item->runtime = item->memory_usage * 2.0;
			break;

		case 2*3:
			/* raw transfer to the next pipe is
			 * a linear algorithm
			 */
			item->runtime = item->memory_usage * 1.0;
			break;

		default:
			fprintf(stderr, "unsupported pixelformat (bypp = %i)\n",
				priv->bypp);
			break;
		}	/* switch */
		break;

	default:
		fprintf(stderr, "Unknown nr: %i\n", nr);
		break;
	}	/* switch */

	return;
}	/* _PPM_fillinfo_input */



/* fills in the information of pixelformats it can convert to
 */
static void _PPM_fillinfo_output(struct gpf_handle_opio_t *opio,
			struct gpf_infolist_t *item,
			int nr)
{
	struct ppmgpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = PPM_PRIV(opio);


	memset(&item->pf, 0, sizeof(struct gpf_pixelformat_t));

	item->pf.cf = GPF_CF_RGBA;
	item->pf.flags = GPF_PIPE_DF_RGBA;

	switch (nr) {
	case 0:
		/* RGB foramt: r8g8b8
		 *
		 * we can always provide RGB_888, because we can
		 * scale the colors down, if necessary
		 */
		item->pf.depth = 24;
		item->pf.size = 24;

		item->pf.type.rgba.red_mask = 0x0000FF;
		item->pf.type.rgba.red_shift = 0;
		item->pf.type.rgba.green_mask = 0x00FF00;
		item->pf.type.rgba.green_shift = 8;
		item->pf.type.rgba.blue_mask = 0xFF0000;
		item->pf.type.rgba.blue_shift = 16;


		item->memory_usage = PF_ByPP(item->pf.size)
				* priv->width * priv->height;

		/* Assumed: priv->bypp == 1 */

		/* raw transfer to the next pipe is
		 * a linear algorithm
		 */
		item->runtime = item->memory_usage * 1.0;
		break;

	case 1:
		/* RGB format: r16g16b16
		 *
		 * we can always provide RGB_FFF, because we can
		 * scale the colors up, if necessary
		 */
		item->pf.depth = 24 * 2;
		item->pf.size = 24 * 2;

		item->pf.type.rgba.red_mask = 0xFFFF;
		item->pf.type.rgba.green_mask = 0xFFFF;
		item->pf.type.rgba.blue_mask = 0xFFFF;


		item->memory_usage = PF_ByPP(item->pf.size)
				* priv->width * priv->height;

		/* Assumed: priv->bypp == 2 */

		/* raw transfer to the next pipe is
		 * a linear algorithm
		 */
		item->runtime = item->memory_usage * 1.0;
		break;

	default:
		fprintf(stderr, "Unknown nr: %i\n", nr);
		break;
	}	/* switch */

	return;
}	/* _PPM_fillinfo_output */




static void _PPM_fillinfo(struct gpf_handle_opio_t *opio,
			struct gpf_infolist_t *item,
			int nr, int chain)
{
	switch (chain) {
	case INFOLIST_INPUT:
		_PPM_fillinfo_input(opio, item, nr);
		break;

	case INFOLIST_OUTPUT:
		_PPM_fillinfo_output(opio, item, nr);
		break;

	}	/* switch */

	return;
}	/* _PPM_fillinfo */



static struct gpf_infolist_t *_PPM_getInfolist(
					struct gpf_handle_opio_t *opio,
					int chain)
{
	int nr, i;
	struct gpf_infolist_t *head = NULL;
	struct gpf_infolist_t *item = NULL;


	nr = _PPM_getinfonr(chain);
	item = _gpfInfoListCreate();
	if (!item) {
		goto err0;
	}	/* if */
	_PPM_fillinfo(opio, item, 0, chain);
	head = item;

	for (i = 1; i < nr; i++) {
		item = _gpfInfoListCreate();
		if (!item) {
			goto err1;
		}	/* if */

		_PPM_fillinfo(opio, item, i, chain);

		_gpfInfoListAppend(head, item);
	}	/* for */


	return head;


err1:
	_gpfInfoListDestroy(&item);
	_gpfInfoListDestroy(&head);
err0:
	return NULL;
}	/* _PPM_getInfolist */


int GPF_PPM_pipe_getinfolist(struct gpf_handle_opio_t *opio)
{
	int rc = GGI_OK;
	struct gpf_pipeline_item_t *pipe;

	pipe = opio->pipe;

	LIB_ASSERT(pipe != NULL);

	LIB_ASSERT(pipe->input_head == NULL);
	LIB_ASSERT(pipe->output_head == NULL);


	switch (opio->type) {
	case GPF_DLTYPE_INPUT:
		/* This target interacts as an input, so we need the
		 * information, what it offers...
		 */
		pipe->output_head = _PPM_getInfolist(opio, INFOLIST_OUTPUT);
		if (!pipe->output_head) {
			rc = GGI_ENOMEM;
			goto err0;
		}	/* if */
		break;


	case GPF_DLTYPE_OUTPUT:
		/* This target interacts as an output, so we need the
		 * information, what it can handle from the input.
		 */
		pipe->input_head = _PPM_getInfolist(opio, INFOLIST_INPUT);
		if (!pipe->input_head) {
			rc = GGI_ENOMEM;
			goto err0;
		}	/* if */
		break;

	}	/* switch */

err0:
	return rc;
}	/* GPF_PPM_pipe_getinfolist */
