/*
 ******************************************************************************

 LIBGPF: raw io target pre/post conversion functions

 Copyright (C) 2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include "raw_gpf.h"


ssize_t GPF_RAW_transfernextstreamblock_input(
				gpf_handle_t handle,
				struct gpf_handle_pipeline_t *hnd_pipeline,
				struct gpf_pipeline_item_t *pipe,
				struct timeval *transfered_timestamps)
{
	ssize_t rc;
	size_t transfer_size;
	size_t frame_size;
	struct gpf_pipeline_t *pipeline = NULL;
	struct gpf_handle_opio_t *opio = NULL;
	struct rawgpf_priv *priv = NULL;

	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(hnd_pipeline != NULL);
	LIB_ASSERT(pipe != NULL);

	GPFDPRINT_TRANSFER("io-raw (input): used in handle: %p, pipe: %p\n",
			handle, pipe);

	pipeline = hnd_pipeline->pipeline;
	opio = hnd_pipeline->opio_input;
	priv = RAW_PRIV(opio);

	LIB_ASSERT(PIPE_READBUF(pipe) != NULL);
	LIB_ASSERT(PIPE_READBUFSIZE(pipe) > 0);
	LIB_ASSERT(PIPE_WRITEBUF(pipe) != NULL);
	LIB_ASSERT(PIPE_WRITEBUFSIZE(pipe) > 0);

	FIXME("%s:%s:%i: Limit timestamp to SEEK_END, if it exceeds\n",
		DEBUG_INFO);

	/* It is known that 1 frame == 1 second
	 * and never transfer more than one frame at once
	 * because you would overwrite the previous one before
	 * it has a chance to complete its transfer.
	 */
	if (transfered_timestamps->tv_sec > 1.0) {
		transfered_timestamps->tv_sec = 1.0;
		transfered_timestamps->tv_usec = 0.0;
	}	/* if */

	LIB_ASSERT(pipe->buf_conv == NULL);
	LIB_ASSERT(pipe->buf_conv_size == 0);
	pipe->buf_conv = PIPE_READBUF(pipe);
	pipe->buf_conv_size = PIPE_READBUFSIZE(pipe);

	/* convert timeval to bytes to transfer */
	frame_size = PIPE_READBUFSIZE(pipe);
	transfer_size = _gpf_gettransfersize(frame_size, transfered_timestamps);
	LIMIT_TRANSFERSIZE;

	rc = opio->read(opio, (PIPE_READBUF(pipe) + PIPE_READCACHESIZE(pipe)),
			transfer_size - PIPE_READCACHESIZE(pipe));
	if ((rc == 0) && (PIPE_READCACHESIZE(pipe) == 0)) {
		hnd_pipeline->pipe_index = NULL;
		goto exit;
	}	/* if */
	if (rc < 0) {
		goto exit;
	}	/* if */

	pipe->readbytes = rc;
	PIPE_WRITEBUFPOS(pipe) = PIPE_WRITECACHESIZE(pipe);

	GPFDPRINT_TRANSFER("io-raw (input): %i bytes read, %i bytes are cached, try %i bytes to write\n",
			pipe->readbytes, PIPE_READCACHESIZE(pipe),
			pipe->readbytes + PIPE_READCACHESIZE(pipe));

	rc = pipeline->write(pipe,
			pipe->readbytes + PIPE_READCACHESIZE(pipe),
			PIPE_WRITEBUFSIZE(pipe));
	if (rc < 0) {
		GPFDPRINT("io-libggi (input): error %i occured during write\n", rc);
		goto exit;
	}	/* if */

	pipe->writebytes = rc;
	GPFDPRINT_TRANSFER("io-raw (input): %i bytes finally written, %i bytes to cache\n",
			pipe->writebytes, pipe->readbytes - pipe->writebytes);

	if (pipe->readbytes > pipe->writebytes) {
		PIPE_READCACHEDATA(pipe, pipe->readbytes - pipe->writebytes);
	}	/* if */
	PIPE_READBUFPOS(pipe) = 0;

	hnd_pipeline->pipe_index = hnd_pipeline->pipe_index->next;

exit:
	pipe->buf_conv = NULL;
	pipe->buf_conv_size = 0;

	return rc;
}	/* GPF_RAW_transfernextstreamblock_input */



ssize_t GPF_RAW_transfernextstreamblock_output(
				gpf_handle_t handle,
				struct gpf_handle_pipeline_t *hnd_pipeline,
				struct gpf_pipeline_item_t *pipe,
				struct timeval *transfered_timestamps)
{
	ssize_t rc;
	size_t tmp_size;
	size_t transfer_size;
	size_t frame_size;
	struct gpf_handle_opio_t *opio = NULL;
	struct gpf_pipeline_t *pipeline = NULL;
	struct rawgpf_priv *priv = NULL;

	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(hnd_pipeline != NULL);
	LIB_ASSERT(pipe != NULL);

	GPFDPRINT_TRANSFER("io-libggi (output): used in handle: %p, pipe: %p\n",
			handle, pipe);

	pipeline = hnd_pipeline->pipeline;
	opio = hnd_pipeline->opio_output;
	priv = RAW_PRIV(opio);

	LIB_ASSERT(PIPE_READBUF(pipe) != NULL);
	LIB_ASSERT(PIPE_READBUFSIZE(pipe) > 0);
	LIB_ASSERT(pipe->prev != NULL);


        /* convert timeval to bytes to transfer */
        frame_size = PIPE_READBUFSIZE(pipe);
        transfer_size = _gpf_gettransfersize(frame_size, transfered_timestamps);

	GET_TARGET_TRANSFERSIZE;
	LIMIT_TRANSFERSIZE;


	/* make sure, that transfer_size is actually what you want. Sometimes
	 * you must change it and use another variable, i.e. write_size
	 */
	REALLOC_BUFCONVSIZE(transfer_size);

	GPFDPRINT_TRANSFER("io-raw (output): try to transfer %i bytes, %i bytes are cached\n",
			transfer_size, PIPE_READCACHESIZE(pipe));
	GPFDPRINT_TRANSFER("io-raw (output): %i bytes in read buffer (minus cached bytes)\n",
			PIPE_WRITEBUFPOS(pipe->prev));

	FIXME("%s:%s:%i: When interframe delay not exceeded,"
		" then give control back to application here\n",
		DEBUG_INFO);

	tmp_size = transfer_size;
	PIPE_READBUFPOS(pipe) = 0;
	do {
		rc = pipeline->read(pipe, tmp_size, transfer_size);
		if (rc < 0) {
			fprintf(stderr, "io-raw (output): error %i during reading occured\n",
				rc);
			return rc;
		}	/* if */
		if (rc == 0) break;     /* ready? */

		pipe->readbytes += rc;
		GPFDPRINT_TRANSFER("io-raw (output): %i bytes were read, now try to write them\n", rc);

		/* performs conversion if needed */
		rc = opio->write(opio, pipe->buf_conv, rc);
		if (rc < 0) {
			fprintf(stderr, "io-raw (output): error %i during writing occured\n",
				rc);
			return rc;
		}	/* if */

		GPFDPRINT_TRANSFER("io-raw (output): %i bytes written\n", rc);

		pipe->writebytes += rc;
		tmp_size -= rc;
	} while (rc > 0);

	rc = pipe->writebytes;

	GPFDPRINT_TRANSFER("io-raw (output): %i bytes transfered, %i bytes must be cached\n",
			pipe->writebytes, PIPE_WRITEBUFPOS(pipe->prev) - pipe->writebytes);

	if (PIPE_WRITEBUFPOS(pipe->prev) < pipe->writebytes) {
		PIPE_READCACHEDATA(pipe, pipe->writebytes -
				PIPE_WRITEBUFPOS(pipe->prev) +
				PIPE_READCACHESIZE(pipe));
	} else {
		PIPE_READCACHEDATA(pipe,
				PIPE_WRITEBUFPOS(pipe->prev) -
				pipe->writebytes);
	}	/* if */

	if (PIPE_WRITEBUFPOS(pipe->prev) <= pipe->writebytes) {
		hnd_pipeline->pipe_index = hnd_pipeline->pipe_index->prev;
		pipe->readbytes = pipe->writebytes = 0;
	}	/* if */

	GPFDPRINT_TRANSFER("io-raw (output): %i bytes cached\n",
		PIPE_READCACHESIZE(pipe));

	return rc;
}	/* GPF_RAW_transfernextstreamblock_output */
