/*
 ******************************************************************************

 LIBGPF: raw io target functions

 Copyright (C) 2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include "raw_gpf.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>



int GPF_RAW_open(struct gpf_handle_opio_t *opio)
{
	int rc = GGI_OK;
	int flags = 0;
	struct rawgpf_priv *priv;


	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);

	priv = RAW_PRIV(opio);


	/* When your target doesn't require a distingstion
	 * whether it accts as input or output target,
	 * then you can remove this
	 */
	if (LIBGPF_IO_ISINPUT(opio)) {
		goto read;
	}	/* if */

	if (LIBGPF_IO_ISOUTPUT(opio)) {
		goto write;
	}	/* if */


read:
	rc = opio->opproto->open(opio, flags);
	if (rc != GGI_OK) {
		goto err0;
	}	/* if */

	/* A must: Tell the pipeline management some information
	 * to solve a chicken/egg problem.
	 */
	opio->pipe->width = GGI_AUTO;
	opio->pipe->height = GGI_AUTO;
	opio->pipe->depth = GGI_AUTO;	/* Significant bits per pixel */
	opio->pipe->size = GGI_AUTO;	/* Bits per pixel */

	return rc;


write:
	rc = opio->opproto->open(opio, flags);
	if (rc != GGI_OK) {
		goto err0;
	}	/* if */


	return rc;

err0:
	GPFDPRINT_TARGET("%s:%s:%i: leave with rc: %i\n",
			DEBUG_INFO, rc);
	return rc;
}	/* GPF_RAW_open */


int GPF_RAW_close(struct gpf_handle_opio_t *opio)
{
	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);

	return opio->opproto->close(opio);
}	/* GPF_RAW_close */



int GPF_RAW_timestamp(struct gpf_handle_opio_t *opio,
			struct timeval *tv, int whence)
{
	struct rawgpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = RAW_PRIV(opio);

	switch (whence & GPF_SEEK_OPMASK) {
	case GPF_SEEK_GET:
		switch (whence & GPF_SEEK_POSMASK) {

		/* case GPF_SEEK_REL and GPF_SEEK_ABS are
		 * already handled in libgpf/gpf/gpf.c
		 */
		case GPF_SEEK_LAST:
			if (LIBGPF_IO_ISINPUT(opio)) {
				memcpy(tv, &opio->last_ts,
					sizeof(struct timeval));
				return GGI_OK;
			}	/* if */
			if (LIBGPF_IO_ISOUTPUT(opio)) {
				return GGI_ENOMATCH;
			}	/* if */
			return GGI_ENOMATCH;

		case GPF_SEEK_END:

			/* Return the value GPF_SEEK_END - tv seconds,
			 * where GPF_SEEK_END is the last timestamp
			 * of a video (for example). tv is simply the
			 * given parameter.
			 * If this is not possible, then just fail here
			 * and return GGI_ENOMATCH.
			 *
			 * Hint: Use _gpf_ts_sub() to do the subtraction.
			 */

			FIXME("%s:%s:%i: GPF_SEEK_GET | GPF_SEEK_END "
				"not implemented\n",
				DEBUG_INFO);
			return GGI_ENOMATCH;

		}	/* switch */
		break;

	case GPF_SEEK_SET:

		switch (whence & GPF_SEEK_POSMASK) {
		case GPF_SEEK_REL:
			FIXME("%s:%s:%i: GPF_SEEK_SET | GPF_SEEK_REL "
				"not implemented\n",
				DEBUG_INFO);
			break;
		case GPF_SEEK_ABS:
			FIXME("%s:%s:%i: GPF_SEEK_SET | GPF_SEEK_ABS "
				"not implemented\n",
				DEBUG_INFO);
			break;

		case GPF_SEEK_LAST:
			FIXME("%s:%s:%i: GPF_SEEK_SET | GPF_SEEK_LAST "
				"not implemented\n",
				DEBUG_INFO);
			break;
		case GPF_SEEK_END:
			FIXME("%s:%s:%i: GPF_SEEK_SET | GPF_SEEK_END "
				"not implemented\n",
				DEBUG_INFO);
			break;
		}	/* switch */
		break;
	}	/* switch */

	return GGI_ENOMATCH;
}	/* GPF_RAW_timestamp */





ssize_t GPF_RAW_read(struct gpf_handle_opio_t *opio,
			void *buf, size_t size)
{
	ssize_t rc;
	struct rawgpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: opio: %p, buf: %p, size: %i\n",
			DEBUG_INFO, opio, buf, size);
	priv = RAW_PRIV(opio);


	/* Read the data here */

	/* No conversion is required here
	 */

	rc = opio->opproto->read(opio, buf, size);

	GPFDPRINT_TARGET("%s:%s:%i: read data: %i\n",
			DEBUG_INFO, rc);
	if (rc < 0) {
		/* error occured */
		goto exit;
	}	/* if */

exit:
	return rc;
}	/* GPF_RAW_read */


ssize_t GPF_RAW_write(struct gpf_handle_opio_t *opio,
			void *buf, size_t size)
{
	ssize_t rc;
	struct rawgpf_priv *priv;

	GPFDPRINT_TARGET("%s:%s:%i: called\n", DEBUG_INFO);
	priv = RAW_PRIV(opio);

	rc = opio->opproto->write(opio, buf, size);

	return rc;
}	/* GPF_RAW_write */




int GPF_RAW_setup_input(gpf_handle_t handle,
			struct gpf_handle_pipeline_t *hnd_pipeline,
			struct gpf_pipeline_item_t *pipe,
			int pipeline_channel)
{
	int rc;
	size_t data_size;
	struct rawgpf_priv *priv = NULL;

	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(hnd_pipeline != NULL);
	LIB_ASSERT(pipe != NULL);

	priv = RAW_PRIV(hnd_pipeline->opio_input);

	LIB_ASSERT(PIPE_READBUF(pipe) == NULL);
	LIB_ASSERT(PIPE_WRITEBUF(pipe) == NULL);

	GPFDPRINT("io-raw (input): setting it up to act as input target\n");

	LIB_ASSERT(pipe->width > 0);
	LIB_ASSERT(pipe->height > 0);

	LIB_ASSERT(PIPELINEITEM_MARK(pipe, INFOLIST_INPUT) != NULL);
	LIB_ASSERT(PIPELINEITEM_MARK(pipe, INFOLIST_OUTPUT) == NULL);

	memcpy(&priv->pf, &PIPELINEITEM_MARK(pipe, INFOLIST_INPUT)->pf,
		sizeof(struct gpf_pixelformat_t));
	priv->nr = PIPELINEITEM_MARK(pipe, INFOLIST_INPUT)->nr;

	/* Set up some internal function pointers, if necessary
	 */

	data_size = pipe->width * pipe->height * 1;
	LIB_ASSERT(data_size > 0);

	/* set total amount of transfering timestamps */
	LIBGPF_TS_CLEAR(&LIBGPF_TS_TOTAL(handle));
	LIBGPF_TS_TOTAL(handle).tv_sec = -1;	/* -1 means unknown */

	/* set transfer rate
	 * if no timestamps are available then using 1 second as 
	 * 1 frame is recommended
	 */
	LIBGPF_TS_CLEAR(&LIBGPF_TS_TRANSFER(handle));
	LIBGPF_TS_TRANSFER(handle).tv_sec = 1;

	/* the amount of memory, that we have to transfer */
	rc = _gpf_alloc_readbuf(pipe, data_size);
	if (rc != GGI_OK) {
		GPFDPRINT("%s:%s:%i: io-raw (input): memory allocation failed\n",
			DEBUG_INFO);
		goto err0;
	}	/* if */

	rc = _gpf_alloc_writebuf(pipe, data_size);
	if (rc != GGI_OK) {
		GPFDPRINT("%s:%s:%i: io-raw (input): memory allocation failed\n",
			DEBUG_INFO);
		goto err1;
	}	/* if */

	LIB_ASSERT(pipe->buf_conv == NULL);
	LIB_ASSERT(pipe->buf_conv_size == 0);


	return GGI_OK;

err1:
	_gpf_free_readbuf(pipe);
err0:
	return GGI_ENOMEM;
}	/* GPF_RAW_setup_input */



int GPF_RAW_setup_output(gpf_handle_t handle,
			struct gpf_handle_pipeline_t *hnd_pipeline,
			struct gpf_pipeline_item_t *pipe,
			int pipeline_channel)
{
	struct rawgpf_priv *priv = NULL;
	struct gpf_pipeline_t *pipeline = NULL;

	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(hnd_pipeline != NULL);
	LIB_ASSERT(pipe != NULL);

	pipeline = hnd_pipeline->pipeline;
	priv = RAW_PRIV(hnd_pipeline->opio_output);

	GPFDPRINT("io-raw (output): setting it up to act as output target\n");

	LIB_ASSERT(PIPELINEITEM_MARK(pipe, INFOLIST_INPUT) == NULL);
	LIB_ASSERT(PIPELINEITEM_MARK(pipe, INFOLIST_OUTPUT) != NULL);

	memcpy(&priv->pf, &PIPELINEITEM_MARK(pipe, INFOLIST_OUTPUT)->pf,
		sizeof(struct gpf_pixelformat_t));
	priv->nr = PIPELINEITEM_MARK(pipe, INFOLIST_OUTPUT)->nr;

	/* Set up some internal function pointers, if necessary
	 */

	GPFDPRINT("io-raw (output): performs no conversion\n");

	return GGI_OK;
}	/* GPF_RAW_setup_output */


int GPF_RAW_getformat(struct gpf_pipeline_item_t *pipe,
			struct gpf_pipeline_item_t *from_pipe,
			int chain)
{
	struct rawgpf_priv *priv;
	priv = (struct rawgpf_priv *)LIBGPF_PIPE_PRIVATE(pipe);

	switch (chain) {
	case INFOLIST_INPUT:
		GPFDPRINT_PIPEMGR("io-raw: get needed memory size for the output\n");

		LIB_ASSERT(PIPELINEITEM_STRMARK(from_pipe, INFOLIST_INPUT) == NULL);
		LIB_ASSERT(PIPELINEITEM_STRMARK(from_pipe, INFOLIST_OUTPUT) != NULL);


		LIB_ASSERT(PIPELINEITEM_STRMARK(from_pipe, INFOLIST_OUTPUT)->memory_usage > 0);
		priv->output_data_size =
			PIPELINEITEM_STRMARK(from_pipe, INFOLIST_OUTPUT)->memory_usage;
		return GGI_OK;

	case INFOLIST_OUTPUT:
		GPFDPRINT_PIPEMGR("io-raw: get needed memory size for the input\n");

		LIB_ASSERT(PIPELINEITEM_STRMARK(from_pipe, INFOLIST_INPUT)->memory_usage > 0);
		priv->input_data_size =
			PIPELINEITEM_STRMARK(from_pipe, INFOLIST_INPUT)->memory_usage;
		return GGI_OK;

	}	/* switch */

	return GGI_OK;
}	/* GPF_RAW_getformat */
