/* $Id: debug.c,v 1.1.1.1 2001/04/21 16:55:09 cegger Exp $
******************************************************************************

   LibGGI3D debug module
   Debugging module. Take input stream and print it out to stderr.
   Then pass it on.

   Copyright (C) 1998 Andreas Beck [becka@ggi-project.org]
                 1999 Jon Taylor [taylorj@ggi-project.org]
 
   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/
#include <stdio.h>
#include <stdarg.h>
#include <ggi3d/module.h>
#include <ggi3d/modules/globals.h>
#include <ggi3d/modules/debug.h>

char identstr_debug[]="Debug 0.1";

typedef struct {

	int junk;
	/* How many bytes are there per "len" unit ? */
	int bytes_per_sample;

} debug_private;

ggi3d_modp ggi3d_dlinit()
{
	ggi3d_modp modinfo;
	debug_private *priv;

	modinfo = malloc(sizeof(ggi3d_module));
	modinfo->destroy = debug_destroy;
	modinfo->handle_input = debug_handle;
	modinfo->command = debug_command;
	
	priv = malloc(sizeof(debug_private));
	modinfo->priv = priv;
	
	return modinfo;
}


void dump_vertex(vertex *the_vertex);
void dump_polygon(polygon *the_polygon);
void dump_camera(camera *the_camera);
void dump_light(light *the_light);
void dump_object(object *the_object);
void dump_list(GList *list, int list_type);
void dump_world(renderworld_input_t *the_world);

void dump_vertex(vertex *the_vertex)
{
	printf("[X Y Z]=[%f %f %f]\n", the_vertex->c.x, the_vertex->c.y, the_vertex->c.z);
}

void dump_polygon(polygon *the_polygon)
{
	
	GList *index;
	vertex *the_vertex;

	printf("Vertices:\n");
	printf("---------\n");
	
	index = g_list_first(the_polygon->vertex_list);
	
	do
	{
		the_vertex = (vertex *)index->data;
		dump_vertex(the_vertex);
		index = index->next;
	}
	while (index != g_list_last(index));
}

void dump_camera(camera *the_camera)
{
        printf("Camera:\n-------\nx=%f, y=%f, z=%f, theta=%f, phi=%f, alpha=%f\nnear clip=%f, far clip=%f\n\n",
	       the_camera->x, the_camera->y, the_camera->z, the_camera->t, the_camera->p, the_camera->a, 
	       the_camera->nc, the_camera->fc);
}
 
void dump_light(light *the_light)
{
	printf("Origin: x=%f, y=%f, z=%f\n", the_light->origin.x, the_light->origin.y, the_light->origin.z);
}

void dump_object(object *the_object)
{
	printf("Type: ");
	
	switch (the_object->type)
	{
		case OBJ_POLYGON:
		printf("polygon\n");
		dump_polygon((polygon *)the_object->the_object);
		break;
		case OBJ_ASSEMBLY:
		printf("assembly\n");
		break;
	}
	
}

void dump_list(GList *list, int list_type)
{
	int i;
	GList *index = list;

	printf("Light list:\n");
	printf("-----------\n");
	
	i = 0;
	
	if (index != NULL)
	{
		i++;
		while (index != g_list_last(index))
		{
			switch (list_type)
			{
				case LIST_LIGHT:
				printf("Light %d:\n", i);
				printf("---------\n");
				dump_light((light *)index->data);
				break;
				case LIST_OBJECT:
				printf("Object %d:\n", i);
				printf("---------\n");
				dump_object((object *)index->data);
				break;
				case LIST_POLYGON:
				printf("Polygon %d:\n", i);
				printf("---------\n");
				dump_polygon((polygon *)index->data);
				break;
			}
			i++;
			index = index->next;
		}
		
	}
	printf("\n");
}

void dump_world(renderworld_input_t *the_world)
{
	dump_list(the_world->light_list, LIST_LIGHT);
	dump_list(the_world->object_list, LIST_OBJECT);
}

void dump_zbuffer_input(zbuffer_input_t *zbuffer_input)
{
	dump_list(zbuffer_input->light_list, LIST_LIGHT);
	dump_list(zbuffer_input->polygon_list, LIST_POLYGON);
}



/* Free the private data
 */
void debug_destroy(ggi3d_modp self)
{
	free(self->priv);
	return;
}

int debug_command(ggi3d_modp self, int cmd, void *data_in, void *data_out)
{
	switch(cmd)
	{
		case CMD_GET_IDENT:
		data_out = (void *)identstr_debug;
		return 1;
		break;
		case CMD_PROCESS:
		debug_handle(self, data_in, data_out);
		return 1;
		break;
	}
	
	return 0;	/* Not yet used. */
}

int debug_handle(ggi3d_modp self, void *data_in, void *data_out)
{
	object *the_object = (object *)data_in;
	
	switch(the_object->type)
	{
		case DEBUG_TYPE_WORLD:
		dump_world((renderworld_input_t *)the_object->the_object);
		break;
		case DEBUG_TYPE_ZBUFFER:
		dump_zbuffer_input((zbuffer_input_t *)the_object->the_object);
		break;
	}
	
	return 0;
}
